/** CFile **************************************************************

  FileName    [ list.c ]

  PackageName [ cv ]

  Synopsis    [ Utilities to manipulate lists ]

  Description [ ]

  SeeAlso     [ cv.h ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include <Errors.h>
#include "cvInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static cvn copy_list(cvn l);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Computes the union of two lists ]
  Description [ If the parameters kind is not kList, an error message
  is sent to stdout and 0 is returned. If the parameters are OK, then
  the result is a new list the union of the two given lists (with
  two list elements being equal if they have the same declaration). 
  However, if the parameter list2 has duplicates, the result has 
  duplicates. ]
  SideEffects [ If an error occurs, output is sent to stdout.
                Memory is allocated to store the result. ]
  SeeAlso     [ cv_ListAdd ]
 */

cvn 
cv_list_union
(cvn list1,
 cvn list2,
 int (* eq_fn) (const cvn, const cvn))
{
  cvn result, ptr ;

  if (list1 == 0) {
    return list2;
  }
  if (list2 == 0) {
    return list1;
  }

  if ((qKind(list1) != kList) || (qKind(list2) != kList)) {
    Message("cv_list_union: a parameter is not a List",
            xxError, NoPosition);
    return 0;
  }

  result = copy_list(list2);
  ptr = list1;
  while (ptr) {
    result = cv_list_add(result, qValue(ptr), eq_fn);
    ptr = qNext(ptr);
  }
  return result;
}

/** Function **
  Synopsis    [ Adds a value to a list considered as a set ]
  Description [ If the list parameter is not of type kList,
  then an error message is printed to stdout, and 0 is returned.
  The list parameter is considered as a set of values. If the
  given value does not belong to the set (two values being equal
  if they have the same declaration), it is added to the the list
  and the result is returned. ]
  SideEffects [ Allocated memory for the new list cell. ]
  SeeAlso     [ cv_list_union ]
 */
cvn
cv_list_add
(cvn set,
 cvn value,
 int (* eq_fn)(const cvn, const cvn))
{
  cvn ptr = set;
  while (ptr && !(* eq_fn)(qValue(ptr), value)) {
    ptr = qNext(ptr) ;
  }
  if (NullNode(ptr)) {
    return mList(0, set, 0, value);
  } else {
    return set;
  }
}

/** Function **
  Synopsis    [ Computes the number of elements in a list ]
  Description [ If the list parameter is not of type kList
  or does not belong to the abstract class Listed, then an 
  error message is printed to stdout, and -1 is returned.
  Otherwise returns the number of elements in the list. ]
  SideEffects [ If an error occurs, output is sent to stdout. ]
  SeeAlso     [ ]
 */
int
cv_list_size
(const cvn l)
{
  int res;
  cvn ptr;
  if (!IsA(l, kList)) {
    Message("cv_list_size: parameter is not a kList",
            xxError, NoPosition);
    return -1;
  }
  for (res = 0 , ptr = l; ptr ; ptr = qNext(ptr), ++res );

  return res;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Copies of a list ]
  Description [ The parameter must be of kind kList. ]
  SideEffects [ Allocates memory to store the result. ]
  SeeAlso     [ ]
 */
static cvn
copy_list
(cvn l)
{
  cvn res, prev, curr;
  res = mList(0, 0, 0, qValue(l));
  prev = res;
  for (l = qNext(l); l; l = qNext(l)) {
    curr = mList(0, 0, prev, qValue(l));
    sNext(prev, curr);
    prev = curr;
  }
  return res;
}

