/** CFile **************************************************************

  FileName    [ display.c ]

  PackageName [ cv ]

  Synopsis    [ Display routines ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include "cvInt.h"

static char rcsid [] = "$Id$";

#include <string.h>

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/
static int syntax_vhdl_93 = 0;
static unsigned indent;

static const char * direction_string_tab[] =
   {"", "to", "downto"};

static const char * mode_string_tab[] =
   {"", "in", "out", "", "inout"} ;

static const char * op_string_tab[] =
   {"",
    "not", "abs", 
    "and", "or", "nand", "nor", "xor", "xnor",
    "=", "/=", ">", ">=", "<", "<=",
    "+", "-", "&", 
    "*", "/", "mod", "rem"} ;

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void _nl (FILE *);
static void _dec_indent(void);
static void _inc_indent(void);

typedef void (* display_fn) (FILE *, cvn);

static void display_aux (FILE *, cvn);
static void display_aux_list (FILE *, cvn);
static void dList (FILE *, cvn);
static void dEntityDeclaration (FILE *, cvn);
static void dArchitectureBody (FILE *, cvn);
static void dPackageDeclaration (FILE *, cvn);
static void dPackageBody (FILE *, cvn);
static void dIntegerType (FILE *, cvn);
static void dEnumerationType (FILE *, cvn);
static void dSubtype (FILE *, cvn);
static void dTypeDeclaration (FILE *, cvn);
static void dSubtypeDeclaration (FILE *, cvn);
static void dSignalDeclaration (FILE *, cvn);
static void dVariableDeclaration (FILE *, cvn);
static void dConstantDeclaration (FILE *, cvn);
static void dUnary (FILE *, cvn);
static void dBinary (FILE *, cvn);
static void dIntegerValue (FILE *, cvn);
static void dRange (FILE *, cvn);
static void dIfStatement (FILE *, cvn);
static void dBranch (FILE *, cvn);
static void dCaseStatement (FILE *, cvn);
static void dWhileStatement (FILE *, cvn);
static void dForStatement (FILE *, cvn);
static void dAlternative (FILE *, cvn);
static void dWaitStatement (FILE *, cvn);
static void dVariableAssignmentStatement (FILE *, cvn);
static void dNullStatement (FILE *, cvn);
static void dWaveform (FILE *, cvn);
static void dSignalAssignmentStatement (FILE *, cvn);
static void dBlockStatement (FILE *, cvn);
static void dProcessStatement (FILE *, cvn);
static void dConditionalWaveform (FILE *, cvn);
static void dConditionalSignalAssignmentStatement (FILE *, cvn);
static void dSelectedSignalAssignmentStatement (FILE *, cvn);
static void dUseClause (FILE *, cvn);
static void dIdentifier (FILE *, cvn);
static void dCharacterLiteral (FILE *, cvn);
static void dEnumeratedLiteral (FILE *, cvn);

/**AutomaticEnd********************************************************/
static display_fn display_fn_tab [] = {
& dList,
& dEntityDeclaration,
& dArchitectureBody,
& dPackageDeclaration,
& dPackageBody,
& dIntegerType,
& dEnumerationType,
& dSubtype,
& dTypeDeclaration,
& dSubtypeDeclaration,
& dSignalDeclaration,
& dVariableDeclaration,
& dConstantDeclaration,
& dUnary,
& dBinary,
& dIntegerValue,
& dRange,
& dIfStatement,
& dBranch,
& dCaseStatement,
& dWhileStatement,
& dForStatement,
& dAlternative,
& dWaitStatement,
& dVariableAssignmentStatement,
& dNullStatement,
& dWaveform,
& dSignalAssignmentStatement,
& dBlockStatement,
& dProcessStatement,
& dConditionalWaveform,
& dConditionalSignalAssignmentStatement,
& dSelectedSignalAssignmentStatement,
& dUseClause,
& dIdentifier,
& dCharacterLiteral,
& dEnumeratedLiteral};


/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/


/** Function **
  Synopsis    [ Displays a VHDL text ]
  Description [ Given a CV node, sends a VHDL-like display to the given
  stream. The output format should be satisfying in most cases. ]
  SideEffects [ Sends output to the given stream ]
  SeeAlso     [ ]
 */
void cvn_display
(FILE * outstream,
 cvn t)
{
  indent = 0;
  display_aux(outstream, t);
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/**Function**
  Synopsis           [required]
  Description        [optional]
  SideEffects        [required]
  SeeAlso            [optional]
 */
static void display_aux
(FILE * outstream,
 cvn t)
{
  (* display_fn_tab[qKind(t)])(outstream, t);
}

/**Function**
  Synopsis           [required]
  Description        [optional]
  SideEffects        [required]
  SeeAlso            [optional]
 */
static void display_aux_list
(FILE * outstream,
 cvn t)
{
  while (t) {
    display_aux(outstream, t);
    if (!NullNode(t = qNext(t))) _nl(outstream);
  }
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dList
(FILE * outstream,
 cvn t)
{
  while(t) {
    display_aux(outstream, qValue(t));
    if ((t = qNext(t)) != 0) _nl(outstream);
  }
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dEntityDeclaration
(FILE * outstream,
 cvn t)
{
  cvn port; 
  fputs("entity ", outstream);
  fputs(qName(t), outstream);
  fputs(" is", outstream);
  port = qPorts(t);
  if (port) {
    _inc_indent(); _nl(outstream);
    fputs("port(", outstream);
    _inc_indent();
    do {
      fputs(qName(port), outstream);
      fputs(" : ", outstream);
      fputs(mode_string_tab[qMode(port)], outstream);
      fputc(' ', outstream);
      fputs(qName(qDeclaration(qBaseType(qSubtype(port)))), outstream);
      fputs(" := ", outstream);
      display_aux(outstream, qDefaultValue(port));
      if ((port = qNext(port)) != 0) {
        fputc(';', outstream);
        _nl(outstream);
      }
    } while(port);
    fputs(");", outstream);
    _dec_indent();
    _dec_indent(); _nl(outstream);
  } else {
    _nl(outstream);
  }
  fputs("end entity ", outstream) ;
  fputs(qName(t), outstream);
  putc(';', outstream);
  _nl(outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dArchitectureBody
(FILE * outstream,
 cvn t)
{
  cvn declarative_part;
  fputs("architecture body ", outstream);
  fputs(qName(t), outstream);
  fputs(" of ", outstream);
  fputs(qName(qPrimary(t)), outstream);
  fputs(" is", outstream);
  declarative_part = qDeclarations(t);
  if (declarative_part) {
    _inc_indent(); _nl(outstream);
    display_aux_list(outstream, declarative_part);
    _dec_indent(); _nl(outstream);
  } else {
    _nl(outstream);
  }
  fputs("begin", outstream) ;
  _inc_indent(); _nl(outstream);
  display_aux_list(outstream, qStatements(t));
  _dec_indent(); _nl(outstream);
  fputs("end architecture ", outstream) ;
  if (syntax_vhdl_93)
    fputs(qName(t), outstream);
  putc(';', outstream);
  _nl(outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dPackageDeclaration
(FILE * outstream,
 cvn t)
{
  fputs("package ", outstream);
  fputs(qName(t), outstream);
  fputs(" is", outstream);
  _inc_indent(); _nl(outstream);
  display_aux_list(outstream, qDeclarations(t));
  _dec_indent(); _nl(outstream);
  fputs("end package ", outstream) ;
  fputs(qName(t), outstream);
  putc(';', outstream);
  _nl(outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dPackageBody
(FILE * outstream,
 cvn t)
{
  fputs("package body of", outstream);
  fputs(qName(qPrimary(t)), outstream);
  fputs(" is", outstream);
  _inc_indent(); _nl(outstream);
  display_aux_list(outstream, qDeclarations(t));
  _dec_indent(); _nl(outstream);
  fputs("end package body;", outstream) ;
  _nl(outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dIntegerType
(FILE * outstream,
 cvn t)
{
  fprintf(outstream, "%s ", qName(qDeclaration(t)));
  display_aux(outstream, qRange(t));
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dEnumerationType
(FILE * outstream,
 cvn t)
{
  cvn element;
  fputc('(', outstream);
  element = qElements(t); 
  while(element) {
    display_aux(outstream, qValue(element));
    if ((element = qNext(element)) != 0) {
      fputs(", ", outstream);
    }
  }
  fputc(')', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dSubtype
(FILE * outstream,
 cvn t)
{
  cvn constraint = qConstraint(t);
  display_aux(outstream, qBaseType(t));
  if (constraint && (qKind(constraint) == kRange)) {
    display_aux(outstream, constraint);
  } else {
    while (constraint) {
      display_aux(outstream, constraint);
      constraint = qNext(constraint);
    }
  }
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dTypeDeclaration
(FILE * outstream,
 cvn t)
{
  cvn type;
  fputs("type ", outstream);
  fputs(qName(t), outstream);
  fputs(" is ", outstream);
  type = qBaseType(qSubtype(t));
  display_aux(outstream, type);
  fputc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dSubtypeDeclaration
(FILE * outstream,
 cvn t)
{
  fputs("subtype ", outstream);
  fputs(qName(t), outstream);
  fputs(" is ", outstream);
  display_aux(outstream, qSubtype(t));
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dVariableDeclaration
(FILE * outstream,
 cvn t)
{
  fputs("variable ", outstream);
  fputs(qName(t), outstream);
  fputs(" : ", outstream);
  fputs(qName(qDeclaration(qBaseType(qSubtype(t)))), outstream);
  fputs(" := ", outstream);
  display_aux(outstream, qInitialValue(t));
  fputc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dSignalDeclaration
(FILE * outstream,
 cvn t)
{
  fputs("signal ", outstream);
  fputs(qName(t), outstream);
  fputs(" : ", outstream);
  fputs(qName(qDeclaration(qBaseType(qSubtype(t)))), outstream);
  fputs(" := ", outstream);
  display_aux(outstream, qDefaultValue(t));
  fputc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dConstantDeclaration
(FILE * outstream,
 cvn t)
{
  fputs("constant ", outstream);
  fputs(qName(t), outstream);
  fputs(" : ", outstream);
  fputs(qName(qDeclaration(qBaseType(qSubtype(t)))), outstream);
  fputs(" := ", outstream);
  display_aux(outstream, qValue(t));
  fputc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dUnary
(FILE * outstream,
 cvn t)
{
  fprintf(outstream, "(%s ", op_string_tab[(int) qOperator(t)]);
  display_aux(outstream, qOperand(t));
  fprintf(outstream, ")");
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dBinary
(FILE * outstream,
 cvn t)
{
  fprintf(outstream, "(");
  display_aux(outstream, qLeft(t));
  fprintf(outstream, " %s ", op_string_tab[(int) qOperator(t)]);
  display_aux(outstream, qRight(t));
  fprintf(outstream, ")");
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dIntegerValue
(FILE * outstream,
 cvn t)
{
  fprintf(outstream, "%i", qIntegerValue(t));
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dRange
(FILE * outstream,
 cvn t)
{
  display_aux(outstream, qLeft(t));
  fprintf(outstream, " %s ", direction_string_tab[(int) qDirection(t)]);
  display_aux(outstream, qRight(t));
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dBlockStatement
(FILE * outstream,
 cvn t)
{
  char * label = qName(t);
  cvn declarative_part;
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs("block", outstream);
  declarative_part = qDeclarations(t);
  if (declarative_part) {
    _inc_indent(); _nl(outstream);
    display_aux_list(outstream, declarative_part);
    _dec_indent(); _nl(outstream);
  } else {
    _nl(outstream);
  }
  fputs("begin", outstream) ;
  _inc_indent(); _nl(outstream);
  display_aux_list(outstream, qStatements(t));
  _dec_indent(); _nl(outstream);
  fputs("end block", outstream) ;
  if (syntax_vhdl_93)
    fputs(qName(t), outstream);
  putc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dProcessStatement
(FILE * outstream,
 cvn t)
{
  char * label = qName(t);
  cvn sensitivity_list;
  cvn declarative_part;
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs("process", outstream);
  sensitivity_list = qSensitivityList(t);
  if(sensitivity_list) {
    fputs(" (", outstream);
    display_aux(outstream, sensitivity_list);
    fputs(")", outstream);
  }
  declarative_part = qDeclarations(t);
  if (declarative_part) {
    _inc_indent(); _nl(outstream);
    display_aux(outstream, declarative_part);
    _dec_indent(); _nl(outstream);
  } else {
    _nl(outstream);
  }
  fputs("begin", outstream) ;
  _inc_indent(); _nl(outstream);
  display_aux_list(outstream, qStatements(t));
  _dec_indent(); _nl(outstream);
  fputs("end process ", outstream) ;
  if (syntax_vhdl_93)
    fputs(qName(t), outstream);
  putc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dSignalAssignmentStatement
(FILE * outstream,
 cvn t)
{
  char * label = qName(t);
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs(qName(qTarget(t)), outstream) ;
  fputs(" <= ", outstream);
  display_aux(outstream, qValue(qSource(t)));
  putc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dConditionalSignalAssignmentStatement
(FILE * outstream,
 cvn t)
{
  char * label = qName(t);
  cvn cond_wave;
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs(qName(qTarget(t)), outstream) ;
  fputs(" <= ", outstream);
  cond_wave = qSource(t);
  _inc_indent();
  do {
    cvn condition = qCondition(cond_wave);
    display_aux(outstream, qValue(cond_wave));
    if (condition) {
      fputs(" when ", outstream);
      display_aux(outstream, condition);
    }
    if ((cond_wave = qNext(cond_wave)) != 0) {
      fputs(" else", outstream);
      _nl(outstream);
    }
  } while (cond_wave);
  putc(';', outstream);
  _dec_indent();
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dSelectedSignalAssignmentStatement
(FILE * outstream,
 cvn t)
{
  cvn alternative;
  char * label = qName(t);
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs("with ", outstream) ;
  display_aux(outstream, qSelector(t));
  fputs(" select", outstream) ;
  _inc_indent(); _nl(outstream);
  fputs(qName(qTarget(t)), outstream) ;
  fputs(" <= ", outstream);
  _inc_indent();
  for (alternative = qAlternatives(t);
       alternative;
       alternative = qNext(alternative)) {
    cvn choices;
    display_aux(outstream, qValue(qStatements(alternative)));
    fputs(" when ", outstream);
    choices = qChoices(alternative);
    if (choices) {
      do {
	display_aux(outstream, qValue(choices));
	if ((choices =qNext(choices)) != 0)
	  fputs(" | ", outstream);
      } while (choices);
    } else {
      fputs("others", outstream);
    }
    if (qNext(alternative)) {
      fputc(',', outstream);
      _nl(outstream);
    }
  }
  _dec_indent();
  _dec_indent();
  putc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dVariableAssignmentStatement
(FILE * outstream,
 cvn t)
{
  char * label = qName(t);
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs(qName(qTarget(t)), outstream) ;
  fputs(" := ", outstream);
  display_aux(outstream, qSource(t));
  putc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dWaitStatement
(FILE * outstream,
 cvn t)
{
  char * label = qName(t);
  cvn condition, signals;
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs("wait ", outstream);
  condition = qCondition(t);
  signals = qSignals(t);
  if (condition) {
    fputs("until ", outstream);
    display_aux(outstream, condition);
  }
  if (signals) {
    fputs("on ", outstream);
    do {
      fputs(qName(qValue(signals)), outstream);
      if ((signals = qNext(signals)) != 0)
	fputs(", ", outstream);
    } while (signals);
  }
  fputc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dIfStatement
(FILE * outstream,
 cvn t)
{
  char * label = qName(t);
  cvn branch;
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs("if ", outstream) ;
  branch = qBranches(t) ; 
  display_aux(outstream, qCondition(branch));
  fputs(" then", outstream);
  display_aux(outstream, branch);
  for (branch = qNext(branch) ; branch ; branch = qNext(branch)) {
    if (qCondition(branch)) {
      fputs("elsif ", outstream) ;
      display_aux(outstream, qCondition(branch)) ;
      fputs(" then", outstream) ;
    } else {
      fputs("else ", outstream) ;
    }
    display_aux(outstream, branch);
  }
  fputs("end if", outstream) ;
  if (syntax_vhdl_93)
    fputs(qName(t), outstream);
  putc(';', outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dBranch
(FILE * outstream,
 cvn t)
{
  _inc_indent(); _nl(outstream);
  display_aux_list(outstream, qStatements(t));
  _dec_indent(); _nl(outstream);
  _inc_indent(); _nl(outstream);
  display_aux_list(outstream, qStatements(t));
  _dec_indent(); _nl(outstream);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dCaseStatement
(FILE * outstream,
 cvn t)
{
  char * label = qName(t);
  cvn alternative;
  if (label[0] != '\0') {
    fputs(label, outstream);
    fputc(':', outstream);
  }
  fputs("case ", outstream) ;
  display_aux(outstream, qSelector(t));
  fputs(" is", outstream) ;
  _inc_indent(); _nl(outstream);
  for (alternative = qAlternatives(t);
       alternative;
       alternative = qNext(alternative)) {
    cvn choices;
    fputs("when ", outstream);
    choices = qChoices(alternative);
    if (choices) {
      do {
	display_aux(outstream, qValue(choices));
	if ((choices = qNext(choices)) != 0)
	  fputs(" | ", outstream);
      } while (choices);
    } else {
      fputs(" others", outstream);
    }
      fputs(" =>", outstream);
    _inc_indent(); _nl(outstream);
    display_aux_list(outstream, qStatements(alternative));
    _dec_indent(); _nl(outstream);
  }
  fputs("end case", outstream) ;
  if (syntax_vhdl_93)
    fputs(qName(t), outstream);
  putc(';', outstream);
  _dec_indent();
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dWhileStatement
(FILE * outstream,
 cvn t)
{
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dForStatement
(FILE * outstream,
 cvn t)
{
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dAlternative
(FILE * outstream,
 cvn t)
{
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dNullStatement
(FILE * outstream,
 cvn t)
{
  fprintf(outstream, "null;");
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dWaveform
(FILE * outstream,
 cvn t)
{
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dConditionalWaveform
(FILE * outstream,
 cvn t)
{
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dUseClause
(FILE * outstream,
 cvn t)
{
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dIdentifier
(FILE * outstream,
 cvn t)
{
  fprintf(outstream, qName(t));
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dCharacterLiteral
(FILE * outstream,
 cvn t)
{
  fprintf(outstream, qName(t));
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void dEnumeratedLiteral
(FILE * outstream,
 cvn t)
{
  fprintf(outstream, qName(t));
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void _inc_indent 
(void)
{
  indent +=2;
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void _dec_indent 
(void)
{
  indent -=2;
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static void _nl 
(FILE * outstream)
{
  unsigned cpt = indent;
  fputc('\n', outstream);
  fflush(outstream);
  while (cpt--) fputc(' ', outstream);
}

