/** CFile **************************************************************

  FileName    [ ctlBasic.c ]

  PackageName [ ctl ]

  Synopsis    [ Construction, destruction, query routines ]

  Description [ This file contains routines to create, delete
  and query CTL formulas. ]

  SeeAlso     [ ctl.h ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Log$ ]

***********************************************************************/

#include "ctlInt.h"

#include <stdarg.h>

static char rcsid [] = "$Log$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/
static rec_mgr _memory_manager = 0;

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Constructor ]
  Description [ Creates a new CTL term.
  If type equals ctl_AtomicTerm, there is a third parameter of
  type pointer representing an atomic proposition.
  If type equals ctl_UnaryTerm, there are a fourth and fifth
  parameters of types ctl_Operator_t and ctl_term representing
  the unary operator and the argument of the created CTL formula.
  If type equals ctl_BinaryTerm, there are a fourth, fifth, and sixth
  parameters of types ctl_Operator_t, ctl_term, ctl_term 
  representing the binary operator and the arguments of the created 
  CTL formula. 
  The parameter info is a pointer to user-defined data that is 
  associated to the term and can later be retrieved or overwritten
  with the functions ctl_BasicQInfo and ctl_BasicSInfo. ]
  SideEffects [ Allocates memory via the package mem(3). ]
  SeeAlso     [ ctl_BasicFree ]
 */
ctl_term
ctl_BasicNew
(ctl_term link,
 pointer info,
 ctl_Type_t type,
 ...)
{
  va_list ap;
  ctl_term res;
  if (_memory_manager == 0) {
    _memory_manager = mem_new_rec_mgr(sizeof(ctlTerm_t_));
  }
  res = (ctl_term) mem_new_rec(_memory_manager);
  res->link = link;
  res->info = info;
  res->type = type;
  va_start(ap, type);
  switch (type) {
  case ctl_AtomicTerm:
    res->cell.atom = va_arg(ap, pointer);
    break;
  case ctl_BinaryTerm:
    res->cell.bin.op = va_arg(ap, ctl_Operator_t);
    res->cell.bin.arg1 = va_arg(ap, ctl_term);
    res->cell.bin.arg2 = va_arg(ap, ctl_term);
    break;
  case ctl_UnaryTerm:
    res->cell.un.op = va_arg(ap, ctl_Operator_t);
    res->cell.un.arg = va_arg(ap, ctl_term);
    break;
  case ctl_MacroTerm:
    res->cell.macro.op = va_arg(ap, ctl_Operator_t);
    break;
  }
  va_end(ap);
  return res;
}

/** Function **
  Synopsis    [ Destructor ]
  SideEffects [ ]
 */
void
ctl_BasicFree
(ctl_term t)
{
  if (_memory_manager == 0 || t == 0) 
    return;
  mem_free_rec(_memory_manager, t);
}

/** Function **
  Synopsis    [ Query the type of a formula ]
  Description [ If t is not null, returns the type of t in 
  {ctl_AtomicTerm, ctl_UnaryTerm, ctl_BinaryTerm}; otherwise,
  returns 0. ]
  SideEffects [ None ]
 */
ctl_Type_t
ctl_BasicQType
(ctl_term t)
{
  return t ? t->type : 0;
}

/** Function **
  Synopsis    [ Sets the type of a formula ]
  SideEffects [ Modifies the state of the formula representation . ]
 */
void
ctl_BasicSType
(ctl_term t,
 ctl_Type_t type)
{
  if (t) {
    t->type = type;
  }
}


/** Function **
  Synopsis    [ Query the successor of a formula ]
  Description [ If t is not null, returns the link of t; otherwise,
  returns 0. ]
  SideEffects [ None ]
 */
ctl_term
ctl_BasicQNext
(ctl_term t)
{
  return t ? t->link : 0;
}

/** Function **
  Synopsis    [ Sets the successor of a formula ]
  Description [ If t is not null, sets the successor of t to link. ]
  SideEffects [ Modifies the state of the formula representation. ]
 */
ctl_term
ctl_BasicSNext
(ctl_term t,
 ctl_term link)
{
  if (t) {
    t->link = link;
  }
  return t;
}

/** Function **
  Synopsis    [ Formula information ]
  Description [ If t is not null, returns the client information
  associated to t; otherwise returns 0. ]
  SideEffects [ none ]
  SeeAlso     [ ctlBasicSInfo ]
 */
pointer
ctl_BasicQInfo
(ctl_term t)
{
  return t ? t->info : 0;
}

/** Function **
  Synopsis    [ Sets the info of a formula ]
  Description [ If t is not null, associates generic data info 
  to t (to be later retrieved with ctl_BasicQInfo); otherwise
  does nothing. ]
  SideEffects [ Modifies the state of the formula representation . ]
 */
void
ctl_BasicSInfo
(ctl_term t,
 pointer info)
{
  if (t) {
    t->info = info;
  }
}

/** Function **
  Synopsis    [ Query the operator ]
  Description [ If t is not null, returns the operator of t in 
  {ctl_EX, ctl_AX, ctl_EF, ctl_AF, ctl_EG, ctl_AG, ctl_EU, 
   ctl_AU, ctl_EW, ctl_AW, ctl_Implies, ctl_Not, ctl_And, ctl_Or,
   ctl_Xor, ctl_Xnor, ctl_Nand, ctl_Nor}; otherwise returns 0. ]
  SideEffects [ None ]
 */
ctl_Operator_t
ctl_BasicQOperator
(ctl_term t)
{
  if (!t || t->type == ctl_AtomicTerm) {
    return 0;
  } else {
    return t->cell.bin.op;
  }
}

/** Function **
  Synopsis    [ Query the (first) argument ]
  SideEffects [ None ]
 */
ctl_term
ctl_BasicQArgument
(ctl_term t)
{
  if (!t || t->type == ctl_AtomicTerm) {
    return 0;
  } else {
    return t->cell.bin.arg1;
  }
}


/** Function **
  Synopsis    [ Query the second argument ]
  SideEffects [ None ]
 */
ctl_term
ctl_BasicQArgument2
(ctl_term t)
{
  return ((t && t->type == ctl_BinaryTerm) ? t->cell.bin.arg2 : 0);
}

/** Function **
  Synopsis    [ Query atomic proposition ]
  SideEffects [ None ]
 */
pointer
ctl_BasicQAtomicTerm
(ctl_term t)
{
  return (t && t->type == ctl_AtomicTerm) ? t->cell.atom : 0;
}

/** Function **
  Synopsis    [ Query macro term ]
  SideEffects [ None ]
 */
ctl_Operator_t
ctl_BasicQMacro
(ctl_term t)
{
  return (t && t->type == ctl_MacroTerm) ? t->cell.macro.op : 0;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Sets the operator of a formula ]
  SideEffects [ Modifies the state of the formula representation . ]
 */
void
ctlBasicSOperator
(ctl_term t,
 ctl_Operator_t op)
{
  if (t && t->type != ctl_AtomicTerm) {
    t->cell.bin.op = op;
  }
}

/** Function **
  Synopsis    [ Sets the (first) argument of formula ]
  SideEffects [ Modifies the state of the formula representation . ]
 */
void
ctlBasicSArgument
(ctl_term t,
 ctl_term arg)
{
  if (t && t->type != ctl_AtomicTerm) {
    t->cell.bin.arg1 = arg;
  }
}

/** Function **
  Synopsis    [ Sets the second argument of formula ]
  SideEffects [ Modifies the state of the formula representation . ]
 */
void
ctlBasicSArgument2
(ctl_term t,
 ctl_term arg)
{
  if (t && t->type == ctl_BinaryTerm) {
    t->cell.bin.arg2 = arg;
  }
}

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

