/** CFile **************************************************************

  FileName    [ cacheX.c ]

  PackageName [ cache ]

  Synopsis    [ Generic implementation of a one-way, extensible size cache. ]

  Description [ Definition of the basic routines providing a 
  generic implementation of a one-way cache. When the load of
  the cache exceeds a certain ratio, the size of the cache is
  automatically grown to accomodate new entries without loss of
  performances. ]

  SeeAlso     [ cache.h cacheInt.h ]

  Author      [ David Dharbe, K. McMillan and David Long ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include <string.h>

#include "cacheInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/* hashing general data structure */
struct cacheX_ {
  long size;          /* size of the hash table */
  long entries;          /* number of entries currently in the hash table */
  hash_function hash; /* hashing function */
  eq_function equal;   /* equality predicate */
  del_function delete; /* delete predicate */
  pointer environment;      /* environment passed to the delete function */
  rec_mgr memory;      /* environment passed to the delete function */
  float threshold;      /* the rehash threshold */
  long rehash_limit;    /* maximum number of entries allowed before rehash */
  float resize_coef;    /* the resize factor */
  int size_index;  
  rec_ptr *tab;    /* the hash table */
  int resize;      /* number of times the table was resized */
  long find;       /* find calls */
  long insert;     /* insert calls */
  long hit;        /* successful find operations */
  long collision;  /* colliding inserts operations */
};

/* data structure for the elements of the hash table */
struct rec_ {
  rec_ptr link; /* next element with same key */
} rec;

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void _Resize(cacheX_mgr cache);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Provides a new cache manager ]
  Description [ Creates and returns a manager to cache generic data. 
  The only prescription on this data is that its type T must be a
  pointer to a structure such that the first field is T.
  The parameters are the size of the cache table, the hashing function, 
  the equality predicate, the delete function, the environment passed
  to the delete function, the rehash threshold and the resize factor.
  The equality predicate has to be C-style: returns 0 when the 
  comparison fails, a non null integer otherwise.
  The value returned by the hashing function is corrected to respect
  the table boundaries (cast to unsigned modulo table size).
  The delete function must take two arguments. One is of type pointer
  and will be passed the element to delete, and the other is also a 
  pointer and will be passed env. 
  The argument memory should be the record manager for the
  cached structures (see mem(3)).
  The rehash threshold is the maximum "load" for the table (i.e. the
  number of entries per line). When the number of entries exceeds this
  load, the table is resized and the entries are rehashed.
  The rehash threshold must be strictly greater than 0 (otherwise it
  is actually set to 4).
  The resize factor specifies the multiplying factor of this resize
  operation (if it is given smaller than 1, then it is will 
  actually be set to 2).
  Memory manager are defined in mem(3); the given memory manager should
  have been defined to get blocks of the size of the items that the
  table will hold. ]
  SideEffects [ Allocates memory via David Long's mem(3) library. ]
  SeeAlso     [ mem(3) ]
 */
cacheX_mgr /* New cache manager */
cache_XNew
(long init_size         /* Size of the cache table */,
 hash_function hashcode /* Hash code routine */,
 eq_function equality   /* Equality predicate */,
 del_function delete    /* Delete function */,
 pointer env            /* Environment for the delete function */,
 rec_mgr memory         /* memory manager for the hashed structures */,
 float rehash_threshold /* Table rehash threshold - must be greater than 0 */,
 float resize_factor    /* Rehash resize factor - must be greater than 1 */)
{
  long size;
  int size_index;
  cacheX_mgr res = (cacheX_mgr) mem_get_block(sizeof(struct cacheX_));
  size_index = 0;
  while ((primes[size_index] < init_size) && (size_index < primes_size)) {
    ++size_index;
  }
  size = primes[size_index];
  res->size = size;
  res->size_index = size_index;
  res->entries = 0;
  res->environment = env;
  res->hash = hashcode;
  res->equal = equality;
  res->delete = delete;
  res->memory = memory;
  res->threshold = rehash_threshold > 0 ? rehash_threshold : 4;
  res->rehash_limit = (unsigned long) res->threshold * res->size;
  res->resize_coef = (resize_factor > 1) ? resize_factor : 2;
  /* res->memory = memory; */
  res->tab = (rec_ptr *) mem_get_block(size * sizeof(rec_ptr));
  mem_zero((pointer) res->tab, (SIZE_T) size * sizeof(rec_ptr));
  res->resize = 0;
  res->find = 0;
  res->insert = 0;
  res->hit = 0;
  res->collision = 0;
  return res;
}

/**Function**
  Synopsis           [ Deallocates a cache ]
  Description        [ Resets the cache contents and deallocates the
  memory used. The cache is no longer usable. ]
  SideEffects        [ Returns memory. ]
  SeeAlso            [ cache_1New cache_1Reset ]
**/
void
cache_XFree
(cacheX_mgr c)
{
  cache_XReset(c);
  mem_free_block((pointer) c->tab);
  mem_free_block((pointer) c);
}

/** Function **
  Synopsis    [ Clears a cache manager ]
  Description [ Resets the cache manager to its initial
 state by emptying the table of its contents. ]
  SideEffects [ Memory used to represent lists of stored items
                is freed. ]
  SeeAlso     [ ]
 */
void 
cache_XReset
(cacheX_mgr cache /* Manager to clear */)
{
  long i;
  del_function delete;
  pointer env;
  delete = cache->delete;
  env = cache->environment;
  for(i=0;i<cache->size;i++){
    rec_ptr p = cache->tab[i];
    while(p){
      rec_ptr q = p;
      p = p->link;
      (* delete)(q, env);
      mem_free_rec(cache->memory,(pointer) q);
    }
    cache->tab[i] = 0;
  }
  cache->find = 0;
  cache->insert = 0;
  cache->hit = 0;
  cache->collision = 0;
}

/** Function **
  Synopsis    [ Finds data in cache structure ]
  Description [ Looks for the data in the table equal to the 
  given parameter based on the equality predicate. Returns it 
  if found, otherwise returns 0. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
pointer
cache_XFind
(cacheX_mgr cache,
 rec_ptr ptr /* Data to look for in the cache table */)
{
  eq_function equal = cache->equal;
  rec_ptr *p = cache->tab + (((unsigned long)(*cache->hash)(ptr, cache->environment)) % cache->size);
  ++cache->find;
  while(*p){
    if ((*equal)(*p, ptr, cache->environment)) {
      ++cache->hit;
      return *p;
    }
    p = &((*p)->link);
  }
  return 0;
}

/** Function **
  Synopsis    [ Inserts data in cache structure ]
  Description [ Looks for the data in the structure table equal to 
  the given parameter according to the equality predicate defined
  for this structure and inserts it in the structure if not found.
  Be careful that it is only of the address of the location of the
  data that it inserted, not the data itself. ]
  SideEffects [ If insertion is actually performed, memory
  is allocated to store the new rec_ptr. The cache table might
  be resized (with routine mem_get_block) if the number of entries
  in the table exceeds the four times its size. ]
  SeeAlso     [ ]
 */
void
cache_XInsert
(cacheX_mgr cache,
 rec_ptr ptr /* Data to insert in the cache table */)
{
  eq_function equal = cache->equal;
  rec_ptr *p = cache->tab + (((unsigned long)(*cache->hash)(ptr, cache->environment)) % cache->size);
  int collision = 0;
  while (*p) {
    if ((*equal)(*p, ptr, cache->environment)) {
      return;
    }
    p = &((*p)->link);
    collision = 1;
  }
  /* really inserting */
  ++cache->entries;
  ++cache->insert;
  if (collision) {
    ++cache->collision;
  }
  *p = ptr;
  (*p)->link = 0;
  if (cache->entries == cache->rehash_limit) {
    _Resize(cache);
  }
}


/** Function **
  Synopsis    [ Removes data from the cache structure ]
  Description [ Looks for the data in the table equal to 
  the given parameter according to the equality predicate.
  If the data is found it is removed from the table and freed,
  Otherwise value 0 is returned. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
int
cache_XRemove
(cacheX_mgr cache,
 rec_ptr ptr /* Data to remove from the cache table */)
{
  eq_function equal = cache->equal;
  rec_ptr *p = cache->tab + (((unsigned long)(*cache->hash)(ptr, cache->environment)) % cache->size);
  while (*p){
    if ((*equal)(*p, ptr, cache->environment)) {
      rec_ptr q = *p;
      *p = (*p)->link;
      (* cache->delete)(q, cache->environment);
      mem_free_rec(cache->memory, q);
      --cache->entries;
      return 1;
    }
    p = &((*p)->link);
  }
  return 0;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Resize a table (approximately doubling its capacity) ]
  Description [ This routine resizes and rehashes the table of the 
  given hash structure. It does so by allocating a new larger table,
  copying the contents of the original table into the new table, and
  then desallocates the original table. ]
  SideEffects [ Memory is allocated (with mem_get_block) ]
  SeeAlso     [ ]
 */
static void
_Resize
(cacheX_mgr cache)
{
  rec_ptr * newtable;
  int * size_index;
  unsigned long old_size, new_size, idx, entry;
  rec_ptr p, next;
  if (cache->size_index == primes_size)
    return;
  ++cache->resize;
  old_size = cache->size;
  new_size = (unsigned long) ((float) old_size * cache->resize_coef);
  size_index = & cache->size_index;
  while ((primes[*size_index] < new_size) && (*size_index < primes_size)) {
    ++*size_index;
  }
  cache->size = primes[*size_index];
  newtable= (rec_ptr *) mem_get_block((SIZE_T) sizeof(rec_ptr) * cache->size);
  mem_zero((pointer) newtable, (SIZE_T) sizeof(rec_ptr) * cache->size);
  for (idx = 0; idx < old_size ; ++idx) {
    for (p = cache->tab[idx]; p; p = next) {
      next = p->link;
      entry = ((unsigned long)(*cache->hash)(p, cache->environment)) % cache->size;
      p->link = newtable[entry];
      newtable[entry] = p;
    }
  }
  mem_free_block((pointer) cache->tab);
  cache->tab = newtable;
  cache->rehash_limit = (unsigned long) cache->threshold * cache->size;
}

