/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    vbdd_arith.c.                                             */
/*                                                              */
/* Description:                                                 */
/*    Extra routines that manipulate BDDs.                      */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

/* ------------------------------------------------------------ */
/* File management information:                                 */
/* $Revision$ */
/* $Date$ */
/* $Locker$ */
/* $Log$ */
/* ------------------------------------------------------------ */
#include "vbdd_arith.h"
#include "vbdd_logic.h"

/* Arithmetic operators */

vbdd vbdd_plus_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   vbdd result ;
   result = vbdd_pluscarry(bddm, v1, v2, bdd_zero(bddm)) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

vbdd vbdd_plus
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   return(vbdd_pluscarry(bddm, v1, v2, bdd_zero(bddm))) ;
}

/* Sub(V1, V2) = Xor(100..0, Adc(V1, Not(V2), 1)
   The first bit of the result is equal to one when V2 > V1 (overflow).
   A direct implementation would _probably_ be faster. */
vbdd vbdd_minus_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register vbdd result ;
   result = vbdd_minus(bddm, v1, v2) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

vbdd vbdd_minus
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   int length ;
   vbdd cache ;
   vbdd result ;
   vbdd pluscarry ;

   length = vbdd_length(v1) ;
   cache = vbdd_zero(bddm, length) ;
   (void) vbdd_set(bddm, cache, 0, bdd_one(bddm)) ;
   pluscarry = vbdd_pluscarry(bddm, v1, vbdd_not(bddm, v2), bdd_one(bddm)) ;
   result = vbdd_xor(bddm, cache, pluscarry) ;
   vbdd_unprotect(bddm, pluscarry) ;
   vbdd_free(bddm, pluscarry) ;
   vbdd_free(bddm, cache) ;
   return(result) ;
}

vbdd vbdd_pluscarry_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2, bdd carry)
#else
   (bddm, v1, v2, carry)
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
   bdd carry ;
#endif
{
   vbdd result ;
   result = vbdd_pluscarry(bddm, v1, v2, carry) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   bdd_free(bddm, carry) ;
   return(result) ;
}

vbdd vbdd_pluscarry
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2, bdd carry)
#else
   (bddm, v1, v2, carry)
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
   bdd carry ;
#endif
{
   vbdd result ;
   int length ;
   bdd * elem1 ;
   bdd * elem2 ;

   length = vbdd_length(v1) ;
   result = vbdd_zero(bddm, length) ;

/*
           <--------- Length ------>
            ___ ___ ___     ___ ___
           | B | B | B |...| B |nil|
            --- --- ---     --- ---
           V1^
            ___ ___ ___     ___ ___
           | B | B | B |...| B |nil|
            --- --- ---     --- ---
           V2^
            ___ ___ ___     ___ ___ ___
           | 0 | 0 | 0 |...| 0 | 0 |nil|
            --- --- ---     --- --- ---
       result^
   X represents raw data,
   B   ''       a bdd,
   nil ''       a NULL pointer.
*/
   elem1 = vbdd_elements(v1) ;
   elem2 = vbdd_elements(v2) ;
   elem1 += length - 1 ;
   elem2 += length - 1 ;
/*
            ___ ___ ___     ___ ___
           | B | B | B |...| B |nil|
            --- --- ---     --- ---
                           V1^
            ___ ___ ___     ___ ___
           | B | B | B |...| B |nil|
            --- --- ---     --- ---
                           V2^
            ___ ___ ___     ___ ___ ___
           | 0 | 0 | 0 |...| 0 | 0 |nil|
            --- --- ---     --- --- ---
                              result^
*/
   do {
      bdd Newcarry, OrV2carry, AndV2carry, XorV2carry ;
      XorV2carry = bdd_xor(bddm, * elem2, carry) ;
      OrV2carry = bdd_or(bddm, * elem2, carry) ;
      AndV2carry = bdd_and(bddm, * elem2, carry) ;
      bdd_free(bddm, carry) ;
      Newcarry = bdd_ite(bddm, * elem1, OrV2carry, AndV2carry) ;
      bdd_free(bddm, OrV2carry) ;
      bdd_free(bddm, AndV2carry) ;
      vbdd_set(bddm, result, length--,
               bdd_xor(bddm, * elem1, XorV2carry)) ;
      bdd_free(bddm, XorV2carry) ;
      carry = Newcarry ;
      elem1-- ;
      elem2-- ;
   } while(length) ;
/*
            ___ ___ ___     ___ ___
           | B | B | B |...| B |nil|
            --- --- ---     --- ---
        V1^
            ___ ___ ___     ___ ___
           | B | B | B |...| B |nil|
            --- --- ---     --- ---
        V2^
            ___ ___ ___     ___ ___ ___
           | X | B | B |...| B | B |nil|
            --- --- ---     --- --- ---
       result^
*/
   vbdd_set(bddm, result, 0, carry) ;
/*
            ___ ___ ___     ___ ___
           | B | B | B |...| B |nil|
            --- --- ---     --- ---
           V1^
            ___ ___ ___     ___ ___
           | B | B | B |...| B |nil|
            --- --- ---     --- ---
           V2^
            ___ ___ ___     ___ ___ ___
           | B | B | B |...| B | B |nil|
            --- --- ---     --- --- ---
       result^
*/
   return(result) ;
}

/* ------------------------------------------------------------ */
