/**CFile***********************************************************************

  FileName    [ ant.c ]

  PackageName [ ant ]

  Synopsis    [ Creation, destruction, etc. ]

  Description [ This file provides a constructor and a destructor for
  the type ant, as well as miscellaneous auxiliary routines. ]

  SeeAlso     [ ant.h ]

  Author      [ David Deharbe ]

  Copyright   [ Carnegie Mellon University - 1996 ]

******************************************************************************/

#include "antInt.h" 

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int eq_fn(cache_entry p, cache_entry q, ant_manager env);
static void del_fn(cache_entry p, ant_manager env);
static long hash_fn(cache_entry p, ant_manager env);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function**
  Synopsis           [ Antecedent manager creation ]
  Description        [ Creates a data structure to manager antecedent
  computations with the given vector of transition functions. ]
  SideEffects        [ Allocates memory ]
**/
ant_manager
ant_New
(bdd_manager bddm, 
 vbdd variables, 
 vbdd functions)
{
  ant_manager result;
  long count;

  result = (ant_manager) mem_get_block((SIZE_T) sizeof(ant_manager_rec)) ;
  result->bddm = bddm;
  result->memory = mem_new_rec_mgr((SIZE_T) sizeof(cache_entry_rec));
  count = vbdd_length(variables);
  result->cache = cache_XNew(count * count, (hash_function) &hash_fn,
                        (eq_function) &eq_fn, (del_function) &del_fn,
                        (pointer) result, result->memory,
                        (float) 0.67, (float) 2);
  {
    long i;
    bdd * array, * v;
    array = (bdd *) mem_get_block((SIZE_T)(sizeof(bdd) * (1 + count)));
    for (i = 0, v = vbdd_elements(variables); i < count; ++i, ++v) {
      array[i] = * v;
    }
    array[count] = 0;
    result->var_assoc = bdd_new_assoc(bddm, array, 0);
  }


  count = 1 + bdd_vars(bddm);
  result->functions_map_size = count;
  result->functions_map = (bdd *) mem_get_block((SIZE_T)(sizeof(bdd) * count));
  mem_zero((pointer) result->functions_map, (SIZE_T)(sizeof(bdd) * count));
  {
    bdd * v, *f;
    for (v = vbdd_elements(variables), f = vbdd_elements(functions); 
         * v;
         ++v, ++f) {
      result->functions_map[bdd_if_id(bddm, * v)] = * f;
    }
  }
  
  return result;
}

/**Function**
  Synopsis           [ Antecedent manager destruction ]
  SideEffects        [ Deallocates memory and decrements reference
  counters of the internal cache entries. t is no longer usable. ]
**/
void 
ant_Free
(ant_manager t)
{
  mem_free_block((pointer) t->functions_map) ;
  cache_XReset(t->cache);
  mem_free_block((pointer) t) ;
}

/**Function**
  Synopsis           [ Antecedent manager reset ]
  Description        [ Flushes the internal cache ]
  SideEffects        [ Deallocates memory and decrements reference
  counters of the internal cache entries. ]
**/
void
ant_Flush
(ant_manager t)
{
  cache_XReset(t->cache);
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function**
  Synopsis           [ Transition functions of variable with id ]
  Description        [ id being a variable identifier, returns the
  transition function of this variable, or 0 if there is none. ]
  SideEffects        [ None]
**/
bdd
antGetFunction
(ant_manager t,
 long id)
{
  if (id >= t->functions_map_size) {
    return 0;
  } else {
    return t->functions_map[id];
  }
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function**
  Synopsis           [ Antecedent equality ]
  Description        [ Predicate testing equality of two antecedent
  cache entries. ]
  SideEffects        [ None ]
**/
static int eq_fn (cache_entry p, cache_entry q, ant_manager env) {
  return (p && (p->set == q->set));
}

/**Function**
  Synopsis           [ Antecedent destruction ]
  Description        [ Routine deleting an antecedent cache entry. ]
  SideEffects        [ Decrements BDD reference counters. 
  Deallocates memory. ]
**/
static void del_fn (cache_entry p, ant_manager env) {
  if (p) {
    bdd_free(env->bddm, p->set);
    bdd_free(env->bddm, p->ant);
    mem_free_rec(env->memory, p);
  }
}

/**Function**
  Synopsis           [ Antecedent hash ]
  Description        [ Returns a hash code for an
  antecedent cache entry. ]
  SideEffects        [ None ]
**/
static long hash_fn (cache_entry p, ant_manager env) {
  return (long) p->set >> 2;
}
