/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    process-statement.c                                       */
/*                                                              */
/* Description:                                                 */
/*    Routines to compute the semantics of a VHDL process       */
/*    statement, that is to say, elaborate the abstract machine */
/*    that models its statement part.                           */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>

#include "cvc.h"
#include "graph.h"

/* ------------------------------------------------------------ */
/* Declaration of types                                         */
typedef struct zone_ {
   cvn graph ;
   cvn cond_clause ;
   cvn svty_list ;
} zone;

typedef zone * sliced_graph ;

typedef struct node_info_ {
   int mark ;
   void * misc ;
} * node_info ;

#define NOMARK 0
#define UCWMARK 1
#define PRINTMARK 2
#define LOOPMARK 4

/* ------------------------------------------------------------ */
/* Declaration of constants local to the file                   */

char * Tab = "   " ;

/* ------------------------------------------------------------ */
/* Declaration of the profile of routines local to the file */

static void exec_graph (cvn stm, cvn next, int * nb_waits, cvn * asg_obj);
static void sensitivity_list_modify (cvn process);
static int detect_cycles (cvn stm);
static cvn create_curr_wait (cvn process, int nb_waits);
static void asg_curr_wait(cvn stm, cvn first_stm, cvn process, 
	                  sliced_graph * proc_zones, cvn curr_wait);
static void insert_cw_assignment
   (cvn prev, cvn wait, sliced_graph * proc_zones, cvn curr_wait);
static int asg_curr_wait_except (cvn stm, sliced_graph * proc_zones);
static tGraph gen_trans (cvn, cvn);
static tGraph graph_insert (cvn Expression, tGraph graph);
static tGraph graph_compact (tGraph If);
static vbdd gen_sym_trans (expression_context, bdd_manager, vbdd, tGraph);
static void reduce_tree (cvn);
static cvn graph_tree (cvn);

static node_info node_info_copy (node_info i);
static node_info mark_new_mark (int mark, void * misc);

#define mark_add(i, m) {(i)->mark |= (m);}
#define mark_del(i, m) {(i)->mark &= ~(m);}
#define mark_test(i, m) ((int) ((i)->mark & (m)))
#define mark_get(i) ((void *) ((i)->mark))
#define mark_set(i, m)  (((node_info) (i))->mark = (m))
#define mark_get_misc(i) (((node_info) (i))->misc)
#define mark_set_misc(i, m)  (((node_info) (i))->misc = ((void *) m))

/* ------------------------------------------------------------ */

/* --                ---------------------------                */
/* -- Routine:       elaborate_process_statement                */
/* --                ---------------------------                */
/* -- Description: Takes as argument an kProcessStatement node  */
/* -- and a model under construction, and adds state variables  */
/* -- and transition functions, etc. to the model.              */

void
elaborate_process_statement
(cvc_context c, cvn process)
{
  int nb_waits ; /* nb of wait statements in the process */
  int nb_asg_obj ; /* nb of objects assigned in the process */
  cvn asg_objs ; /* list of identifiers of the objects assigned */
  sliced_graph proc_zones ;   
  /* proc_zones is an array of zone. 
     If the process has n wait statements, the length of this array is n+1.
     The fields of each zone is initialized in the routines 
     asg_curr_wait (zones 1 to n) and ElaborateProcessStatement 
     (zone 0). The Graph field is modified in the routine
     gen_zones. */
  tGraph * transitions;
  cvn curr_wait;
  int zone_idx;
  vbdd resump_conds;
  bdd resume, intermediate;

  /* check sensitivity_list */
  if(qSensitivityList(process))
    sensitivity_list_modify(process);
  /* generate the execution graph */
  nb_waits = 0 ;
  asg_objs = 0;
  exec_graph(qStatements(process), 0, & nb_waits, & asg_objs) ;
  assert (nb_waits != 0);
  assert (detect_cycles(qStatements(process)) == 0);
  /* proc_zones is an array if so-called zones ; a zone is
     the subgraph of the execution graph located between two
     wait statements. */
  assert ((proc_zones = calloc((nb_waits) + 1, sizeof(zone))) != NULL);
  (proc_zones[0]).graph = qStatements(process);
  (proc_zones[0]).cond_clause = InitCondition ;
  (proc_zones[0]).svty_list = (cvn) NULL ;

  /* add the current wait variable, compute the set of assigned
     objects, sets elements 1 to nb_waits of the array proc_zones */
  
  if (nb_waits > 1) {
    curr_wait = create_curr_wait(process, nb_waits);
    asg_objs = cv_list_add(asg_objs, curr_wait, & cv_eq_decl) ;
    asg_curr_wait(qStatements(process), qStatements(process),
		  process, & proc_zones, curr_wait) ;
  } else { /* nb_waits == 1 */
    curr_wait= 0;
    asg_curr_wait_except(qStatements(process), & proc_zones) ;
  } ;
  nb_asg_obj = cv_list_size(asg_objs) ;

  /* for each zone: */

  /* derive the execution trees */
  for (zone_idx= 0 ; zone_idx <= nb_waits ; zone_idx++ ) {
    proc_zones[zone_idx].graph= graph_tree(proc_zones[zone_idx].graph) ;
  }

  
  /* reduce the reducible variable occurences */
  for (zone_idx = 0 ; zone_idx <= nb_waits ; zone_idx++ )
    reduce_tree(proc_zones[zone_idx].graph) ;

  /* generation of the transitions */
  {
    cvn obj_list = asg_objs ;
    int idx;
    assert ((transitions = calloc(nb_asg_obj*(nb_waits+1), sizeof(tGraph)))
	    != 0) ;
    for (idx = 0 ; idx < nb_asg_obj ; idx++) {
      for (zone_idx = 0 ; zone_idx <= nb_waits ; ++zone_idx) {
	transitions[((nb_waits + 1) * idx) + zone_idx] =
	  gen_trans(proc_zones[zone_idx].graph, qValue(obj_list));
      }
      obj_list = qNext(obj_list) ;
    }
  }

  /* generation of the symbolic representation */
  add_declarations(c->model, qDeclarations(process));
  for (zone_idx = 1 ; zone_idx <= nb_waits ; ++zone_idx) {
    cvn signal ;
    
    for (signal= proc_zones[zone_idx].svty_list;
	 signal; signal = qNext(signal))
      add_previous(c->model, qDeclaration(qValue(signal)));
  }

  resump_conds = vbdd_newn(nb_waits+1);
  vbdd_set(bddm, resump_conds, 0, condition_code(InitCondition, c->expr_mgr));
  if (curr_wait) {
    cvn wait_value, wait_cond ;
    int zone_idx ;
    cvn curr_wait_type ;
    curr_wait_type = qSubtype(qDeclaration(curr_wait)) ;
    for (zone_idx = 1 ; zone_idx <= nb_waits ; zone_idx++) {
      wait_value = mIntegerValue(0, curr_wait_type, (int) zone_idx-1) ;
      wait_cond= mBinary(0, boolean_type, kEq, curr_wait, wait_value) ;
      vbdd_set(bddm, resump_conds, zone_idx,
               condition_code(wait_cond, c->expr_mgr)) ;
    }
  } else {
    vbdd_set(bddm, resump_conds, 1, bdd_not(bddm, vbdd_nth(resump_conds, 0)));
  }
  resume = bdd_identity(bddm, vbdd_nth(resump_conds, 0));
  intermediate = bdd_identity(bddm, bdd_one(bddm));
  for (zone_idx = 1 ; zone_idx <= nb_waits ; zone_idx++) {
    cvn sig ;
    bdd svty_clause, trigger, cond_clause, zone_resume, tmp1, tmp2;
    do {
      svty_clause = bdd_identity(bddm, bdd_zero(bddm)) ;
      zone_resume = bdd_identity(bddm, bdd_zero(bddm)) ;
      for (sig= proc_zones[zone_idx].svty_list; sig; sig = qNext(sig)) {
        bdd event;
        event = model_RepresentationQSignalEvent(c->model, qDeclaration(qValue(sig))); 
        svty_clause= bdd_orup2(bddm, event, svty_clause) ;
        if (qMode(qDeclaration(qValue(sig))) != kIn) {
	  zone_resume= bdd_orup2(bddm, event, zone_resume) ;
        }
      }
    
      /* no explicit condition clause is an implicit TRUE */
      if (proc_zones[zone_idx].cond_clause == 0) 
        proc_zones[zone_idx].cond_clause = true_literal ;
      cond_clause= condition_code(proc_zones[zone_idx].cond_clause, 
                                  c->expr_mgr);
      cond_clause= bdd_and(bddm, vbdd_nth(resump_conds, zone_idx), cond_clause) ;
      trigger = bdd_andup2(bddm, cond_clause, svty_clause) ;
      zone_resume = bdd_andup(bddm, zone_resume, cond_clause);
      tmp1 = bdd_andup2(bddm, intermediate, zone_resume);
      tmp2 = bdd_or(bddm, resume, trigger);
      if (bdd_overflow(bddm)) {
        bdd_free(bddm, tmp1);
        bdd_free(bddm, tmp2);
        bdd_resize_manager(bddm);
      } else {
        bdd_free(bddm, intermediate);
        bdd_free(bddm, resume);
        intermediate = tmp1;
        resume = tmp2;
        break;
      }
    } while (1);
    vbdd_set(bddm, resump_conds, zone_idx, trigger) ;
  }
   
  {
    int obj_ct;
    cvn obj_list;
    bdd stable;

    stable = bdd_notup(bddm, intermediate);
    model_RepresentationAddStable(c->model, stable);

    for (obj_ct = 0, obj_list=asg_objs ; obj_ct < nb_asg_obj ; 
	 ++obj_ct, obj_list=qNext(obj_list)) {
      cvn obj = qValue(obj_list) ;
      vbdd coding = encode_object(obj)->coding ;
      int zone_idx ;
      vbdd trans ;
      trans= vbdd_zero(bddm, vbdd_length(coding)) ;
      for (zone_idx = 0 ; zone_idx <= nb_waits ; zone_idx++ ) {
        do {
          vbdd tmp0, tmp1, tmp2;
	  tmp0 = 
	    gen_sym_trans(c->expr_mgr,
                          bddm, coding,
			  transitions[((nb_waits + 1) * obj_ct) + zone_idx]) ;
          tmp1 = vbdd_distribute(bdd_and, bddm, 
                                 vbdd_nth(resump_conds, zone_idx), 
				 tmp0, 1) ;
          vbdd_free(bddm, tmp0);
	  tmp2 = vbdd_or(bddm, trans, tmp1);
          vbdd_free(bddm, tmp1);
          if (bdd_overflow(bddm)) {
            vbdd_free(bddm, tmp2);
            bdd_resize_manager(bddm);
          } else {
            vbdd_free(bddm, trans);
            trans = tmp2;
            break;
          }
        } while (1);
      }
      /* If the process is stable (i.e. not active), the objects
	 retain their previous value */
      {
	cvn decl = qDeclaration(obj);
        do {
          vbdd tmp1, tmp2;
	  tmp1 = vbdd_distribute(bdd_and, bddm, bdd_not(bddm, resume), 
				 coding, 1);
	  tmp2 = vbdd_or(bddm, trans, tmp1);
          vbdd_free(bddm, tmp1);
          if (bdd_overflow(bddm)) {
            vbdd_free(bddm, tmp2);
            bdd_resize_manager(bddm);
          } else {
            vbdd_free(bddm, trans);
            trans = tmp2;
            break;
          }
        } while (1);
	if (IsA(decl, kSignalDeclaration)) 
	  model_RepresentationAddDriver(c->model, qDeclaration(obj), 
                                        process, trans);
	else
	  model_RepresentationAddTransition(c->model, qDeclaration(obj), 
                                            trans);
      }
    }
    bdd_free(bddm, stable);
  }
  vbdd_free(bddm, resump_conds);
  /* free memory used by local variables */
  free(proc_zones) ;
  free(transitions) ;
  /* following does not appear to work */
  /*
     {
     int Obj, zone_idx ;
     for (Obj = 0 ; Obj < nb_asg_obj ; Obj += 1) {
     for (zone_idx = 0 ; zone_idx < nb_waits ; zone_idx += 1) {
     graph_free(transitions[((nb_waits + 1) * Obj) + zone_idx],
     0, 0) ;
     }
     }
     }
     */
}


/* --                 ----------------------                    */
/* -- Routine:        exec_graph                    */
/* --                 ----------------------                    */
/* -- Description: This routine creates a so-called execution   */
/* -- graph of the process statement part. This graph contains  */
/* -- one node per statement. The edges associated to the node  */
/* -- have as label a cvn union node representing a boolean   */
/* -- expression, such that:                                    */
/* --   . if the statement is a wait, signal assignment or      */
/* --     variable assignment statement, there is a unique edge */
/* --     equal to (cvn) NULL (interpreted as TRUE)           */
/* --   . if the statement is an if statement, the set of edges */
/* --     correspond to the different branches of the if        */
/* --     statement: the label and the target of the edge are   */
/* --     the condition and the first statement of the branch;  */
/* --     the condition of the else part (either implicit or    */
/* --     explicit is represented by the label to (cvn) NULL  */
/* -- For instance:                                             */
/* --    V := A and B ;                                         */
/* --    if (V <> S) then                                       */
/* --        S <= V ;                                           */
/* --    end if ;                                               */
/* --    wait on A, B ;                                         */
/* -- is represented by the graph:                              */
/* --     ---->  V := A and B <-----------+                     */
/* --                |                    |                     */
/* --                | (cvn) NULL       |                     */
/* --               \-/                   |                     */
/* --           +-- if--+                 |                     */
/* --           |       |                 |                     */
/* --    V <> S |       |                 | (cvn) NULL        */
/* --          \-/      | (cvn) NULL    |                     */
/* --        S <= V     |                 |                     */
/* --           |       |                 |                     */
/* --           |      \-/                |                     */
/* --           +-> wait on A, B----------+                     */

static void
exec_graph
(cvn stm,
 cvn next,
 int * nb_waits,
 cvn * asg_objs)
{
  cvn next_stm ;
  int arity ;

  /* Update nb_waits, asg_objs */
  /* Set the ToolInfo attribute of If and Wait statements */
  sToolInfo(stm, mark_new_mark(NOMARK, NULL)) ;
  if (IsA(stm, kIfStatement)) {
    arity = CountBranches(stm) ;
    mark_set_misc(qToolInfo(stm), arity) ;
  } else if (IsA(stm, kCaseStatement)) {
    arity = CountBranches(stm) ;
    mark_set_misc(qToolInfo(stm), arity) ;
  } else if (IsA(stm, kWaitStatement)) {
    * nb_waits += 1 ;
    mark_set_misc(qToolInfo(stm), * nb_waits) ;
  } else if (IsA(stm, kSignalAssignmentStatement) ||
	     IsA(stm, kVariableAssignmentStatement)) {
    * asg_objs = cv_list_add(* asg_objs, qTarget(stm), & cv_eq_decl) ;
  }

  /* This happens exactly once, for the first instruction */
  if (next == 0) next = stm ;

  /* Link and apply to the rest of the graph */
  next_stm = qNext(stm) ;
   
  if (next_stm) {
    exec_graph(next_stm, next, nb_waits, asg_objs) ;
  } else {
    sNext(stm, next) ;
    next_stm = next ;
  }
  if (IsA(stm, kIfStatement)) {
     cvn branch ;
     int cpt ;

     branch = qBranches(stm) ;
     for (cpt = 0 ; cpt < arity ; cpt ++) {
       exec_graph(qStatements(branch), next_stm, nb_waits, asg_objs) ;
       if (cpt == arity - 2) {
	 /* This is the but-last branch of an incomplete if statement */
	 if (qNext(branch) == 0) {
	   cvn new_branch ; /* Creates a new branch that directly
				 points to the statement following
				 the if statement. The condition is
				 NULL */
	  new_branch= mBranch(0, 0, 0, 0, next_stm);
	  sNext(branch, new_branch) ;
	  /* Don't go down the new branch: exit loop */
	  ++cpt;
	} else {
	  branch = qNext(branch) ;
        }
      } else {
        branch = qNext(branch) ;
      }
    }
  } else if (IsA(stm, kCaseStatement)) {
     cvn alternative ;
     for (alternative = qAlternatives(stm);
          !NullNode(alternative);
          alternative = qNext(alternative)) {
       exec_graph(qStatements(alternative), next_stm, nb_waits, asg_objs) ;
     }
  } else if (IsA(stm, kWhileStatement) || IsA(stm, kForStatement)) {
    exec_graph(qStatements(stm), stm, nb_waits, asg_objs);
  }
}

/* tests if the execution flow graph contains cycles without wait stm */
static int 
detect_cycles
(cvn stm)
{
  int result = 0;
  if (mark_test((node_info) qToolInfo(stm), LOOPMARK)) {
    return 1;
  }
  if (IsA(stm, kWaitStatement)) {
    return 0;
  }
  mark_set((node_info) qToolInfo(stm), LOOPMARK);
  if (IsA(stm, kIfStatement)) {
    cvn branch = qBranches(stm);
    while (branch && (result == 0)) {
      result = detect_cycles(qStatements(branch));
      branch = qNext(branch);
      if ((qNext(branch) == 0) && (qCondition(branch) == 0)) {
        return 0;
      }
    }
  } else if (IsA(stm, kCaseStatement)) {
    cvn alternative = qAlternatives(stm);
    while (alternative && (result == 0)) {
      result = detect_cycles(qStatements(alternative));
      alternative = qNext(alternative);
    }
    return 0;
  } else if (IsA(stm, kWhileStatement) || IsA(stm, kForStatement)) {
    result = detect_cycles(qStatements(stm));
  }
  if (result == 0) {
    result = detect_cycles(qNext(stm));
  }
  mark_del((node_info) qToolInfo(stm), LOOPMARK);
  return result;
}

/* ---------------------------------- sensitivity_list_modify -- *
 * The condition of entering this function is that the process   *
 * has the nSensitivityList field. This function will traverse   *
 * the graph of the process to find the end of the process and   *
 * add a new wait statement at the end of it.                    */

static void
sensitivity_list_modify
(cvn process)
{
    cvn wait;
    cvn statement = qStatements(process);

    while(qNext(statement)) {
      statement = qNext(statement);
    }
    wait = mWaitStatement(0, 0, 0, statement, (char *) strdup(""), 0,
                          qSensitivityList(process));
    sNext(statement, wait);
}

/* ----------------------------------------- create_curr_wait -- *
 * Adds a variable declaration to process declaration part.      *
 * This variable has an integer type range 0 to (* nb_waits)     *
 * Returns an identifier node for this variable.                 */

static cvn
create_curr_wait
(cvn process, int nb_waits)
{
    cvn cw_id, cw_decl, left, right, constraint, cw_subtype_decl,
            cw_subtype, val, next ;
    unsigned line;
    line = qLine(process) ;
    next = qDeclarations(process) ;
    left= mIntegerValue(0, integer_type, 0) ;
    right= mIntegerValue(0, integer_type, (int) nb_waits-1);
    constraint = mRange(0, kTo, left, right);
    cw_subtype_decl=
      mSubtypeDeclaration(0, 0, 0, 0,
                           (char *) strdup("CURR-WAIT-TYPE-DECL"), 0);
    cw_subtype= mSubtype(0, cw_subtype_decl, integer_type, constraint);
    sSubtype(cw_subtype_decl, cw_subtype);
    val= mIntegerValue(0, cw_subtype, 0);
    cw_decl= mVariableDeclaration(0, line, next, 0, 
				  (char *) strdup("CURRENT-WAIT"),
				  cw_subtype, kInternal, val);
    cw_id= mIdentifier(0, (char *) strdup("CURRENT-WAIT"), cw_decl) ;
    sDeclarations(process, cw_decl);
    return cw_id;
}

/* --                 -------------------                       */
/* -- Routine:        gen_curr_wait                       */
/* --                 -------------------                       */
/* -- Description: This routine modifies the data structure     */
/* -- built by the routine gen_exec_graph. If the root  */
/* -- vertex of the given graph is not marked with one, then if */
/* -- the attached data corresponds to a wait statement it is   */
/* -- modified such that:                                       */
/* --   . the attached data is a cvn node that corresponds to */
/* --     a variable assignment statement, the target is the    */
/* --     implicit variable CURRENT-WAIT (created in the routine*/
/* --     BeginElaborate), and the source is the label of the   */
/* --     corresponding wait statement (a natural between 1 and */
/* --     nb_waits);                                             */
/* --   . there is one outgoing edge that goes to a new vertex  */
/* --     that contains the corresponding wait statement.       */
/* --   . the i^{th} entry of the global variable proc_zones  */
/* --     (where i is the label of the wait statement) is       */
/* --     initialized to a structure made of the sensitivity and*/
/* --     condition clauses of the wait statement, and the      */
/* --     current vertex.                                       */
/* -- The routine is called recursively on the vertices that are*/
/* -- targets of some edge of the root of the given graph. The  */
/* -- root of the graph, and the prossibly created vertex is    */
/* -- marked with 1.                                            */
/* -- For instance, the graph:                                  */
/* --     ---->  V := A and B <-----------+                     */
/* --                |                    |                     */
/* --                | (cvn) NULL       |                     */
/* --               \-/                   |                     */
/* --           +-- if--+                 |                     */
/* --           |       |                 |                     */
/* --    V <> S |       |                 | (cvn) NULL        */
/* --          \-/      | (cvn) NULL    |                     */
/* --        S <= V     |                 |                     */
/* --           |       |                 |                     */
/* --           |      \-/                |                     */
/* --           +-> wait on A, B----------+                     */
/* -- is transformed into:                                      */
/* --     ---->  V := A and B <-----------+                     */
/* --                |                    |                     */
/* --                | (cvn) NULL       |                     */
/* --               \-/                   |                     */
/* --           +-- if--+                 |                     */
/* --           |       |                 |                     */
/* --    V <> S |       |                 | (cvn) NULL        */
/* --          \-/      | (cvn) NULL    |                     */
/* --        S <= V     |                 |                     */
/* --           |       |                 |                     */
/* --           |      \-/                |                     */
/* --           +-> CURRENT-WAIT := 1     |                     */
/* --                   |                 |                     */
/* --                   | (cvn) NULL    |                     */
/* --                  \-/                |                     */
/* --               wait on A, B----------+                     */
/* -- and:                                                      */
/* --   proc_zones[1] = {(A, B), (cvn) NULL, <wait on A, B>}*/

static void
asg_curr_wait
(cvn stm,
 cvn first_stm,
 cvn process, 
 sliced_graph * proc_zones, 
 cvn curr_wait)
{
  if (mark_test((node_info) qToolInfo(stm), UCWMARK)) return ;
  mark_add((node_info) qToolInfo(stm), UCWMARK) ;

  if (IsA(stm, kWaitStatement) && cv_eq(stm, first_stm))
    insert_cw_assignment(process, stm, proc_zones, curr_wait);

  if (IsA(stm, kIfStatement)) {
    int cpt ;
    cvn branch = qBranches(stm) ;
    int arity = (int) mark_get_misc(qToolInfo(stm)) ;
    for (cpt = 0 ; cpt < arity ; cpt += 1) {
      stm = qStatements(branch) ;
      if (!cv_eq(stm, first_stm)) {
	if (IsA(stm, kWaitStatement))
	  insert_cw_assignment(branch, stm, proc_zones, curr_wait) ;
	asg_curr_wait(stm, first_stm, process, 
		      proc_zones, curr_wait) ;
      } else if (IsA(stm, kWaitStatement)) {
	sStatements(branch, qStatements(process));
      }
      branch = qNext(branch) ;
    }
  } else if (IsA(stm, kCaseStatement)) {
    cvn alternative;
    for (alternative = qAlternatives(stm);
         !NullNode(alternative);
         alternative = qNext(alternative)) {
      stm = qStatements(alternative);
      if (IsA(stm, kWaitStatement)) {
	insert_cw_assignment(alternative, stm, proc_zones, curr_wait) ;
      } 
      asg_curr_wait(stm, first_stm, process, proc_zones, curr_wait);
    }
  } else if (IsA(stm, kWhileStatement) || IsA(stm, kForStatement)) {
    cvn bodystm;
    bodystm = qStatements(stm);
    if (IsA(bodystm, kWaitStatement)) {
      insert_cw_assignment(bodystm, first_stm, proc_zones, curr_wait) ;
    } 
    asg_curr_wait(bodystm, first_stm, process, proc_zones, curr_wait);
  } else { /* Signal and variable assignment, wait statements */
    cvn next_stm ;
    next_stm = qNext(stm) ;
    /* if the following statement is a wait statement... */
    if (IsA(next_stm, kWaitStatement)) { 
      /* ... that is not a wait that was the first statement of the process */
      if (!cv_eq(next_stm, first_stm)) {
	/* then insert the CW assignment and process the rest */
	insert_cw_assignment(stm, next_stm, proc_zones, curr_wait);
	asg_curr_wait(next_stm, first_stm, process, proc_zones, curr_wait);
	/* ... that is a wait that was the first statement of the process */
      } else {
	/* then Statement is a last statement of the process, set
	   the new CW assignment as the next. */
	sNext(stm, qStatements(process)) ;
      }
      /* else if all statements have not been processed */
    } else if (!cv_eq(next_stm, first_stm)) {
      /* then keep working */
      asg_curr_wait(next_stm, first_stm, process, proc_zones, curr_wait) ;
    }
  }
}

/* inserts the assignment statement between prev
   and Wait (a wait statement) */
static void
insert_cw_assignment
(cvn prev,
 cvn wait,
 sliced_graph * proc_zones, 
 cvn curr_wait)
{
  int zone_n = (int) mark_get_misc(qToolInfo(wait)) ;
  
  /* two nodes are created in the intermediate format:
     - one for the assignment statement,
     - one for the source of the assignment. */
  cvn src= 
    mIntegerValue(0, qSubtype(qDeclaration(curr_wait)), (int) zone_n - 1) ;
  cvn asg_stm = 
    mVariableAssignmentStatement((void *) mark_new_mark(UCWMARK, 0), 0,
				 wait, 0,
				 (char *)
				 strdup("implicit variable assignment"),
				 curr_wait, src) ;

  /* a node is created in the execution graph, the wait statement
     is place in this node and the assignment statement is placed
     in the current node; the current node points to the created
     node */
  if (IsA(prev, kProcessStatement)) {
    sStatements(prev, asg_stm);
  } else if (IsA(prev, kBranch) || 
             IsA(prev, kWhileStatement) || 
             IsA(prev, kForStatement)) {
    sStatements(prev, asg_stm);
  } else {
    sNext(prev, asg_stm) ;
  }
   
  (* proc_zones)[zone_n].cond_clause = qCondition(wait) ;
  (* proc_zones)[zone_n].svty_list = 
    cv_wait_sensitivity_list(wait) ;
  (* proc_zones)[zone_n].graph = qNext(wait) ;
}

/* ------------------------------------- asg_curr_wait_except -- *
 * find the first (and only) wait statement of the process and   *
 * sets the element 1 of proc_zones with the different clauses.  */

static int
asg_curr_wait_except
(cvn stm, sliced_graph * proc_zones)
{
  if (IsA(stm, kWaitStatement)) {
    (* proc_zones)[1].cond_clause = qCondition(stm) ;
    (* proc_zones)[1].svty_list = cv_wait_sensitivity_list(stm) ;
    (* proc_zones)[1].graph = qNext(stm) ;
    return 1;
  }
  if (mark_test((node_info) qToolInfo(stm), UCWMARK)) return 0;
  mark_add((node_info) qToolInfo(stm), UCWMARK) ;

  if (IsA(stm, kIfStatement)) {
    int done;
    cvn branch = qBranches(stm) ;
    done= 0;
    while (!done && branch) {
      if (asg_curr_wait_except(qStatements(branch), proc_zones)) 
	return 1;
      branch = qNext(branch);
    }
  } else if (IsA(stm, kCaseStatement)) {
    int done;
    cvn alternative = qAlternatives(stm) ;
    done= 0;
    while (!done && alternative) {
      if (asg_curr_wait_except(qStatements(alternative), proc_zones)) 
	return 1;
      alternative = qNext(alternative);
    }
  } else if (IsA(stm, kWhileStatement) || IsA(stm, kForStatement)) {
    if (asg_curr_wait_except(qStatements(stm), proc_zones)) {
      return 1;
    }
  }
  return asg_curr_wait_except(qNext(stm), proc_zones);
}

/* --                 ------------                              */
/* -- Routine:        graph_tree                              */
/* --                 ------------                              */
/* -- Description: This routine modifies the data structure     */
/* -- built by the routine gen_exec_graph. If the root  */
/* -- vertex of the given graph is not marked with one, then if */
/* -- the attached data corresponds to a wait statement it is   */
/* -- modified such that:                                       */
/* --   . the attached data is a cvn node that corresponds to */
/* --     a variable assignment statement, the target is the    */
/* --     implicit variable CURRENT-WAIT (created in the routine*/
/* --     BeginElaborate), and the source is the label of the   */
/* --     corresponding wait statement (a natural between 1 and */
/* --     nb_waits);                                             */
/* --   . there is one outgoing edge that goes to a new vertex  */
/* --     that contains the corresponding wait statement.       */
/* --   . the i^{th} entry of the global variable proc_zones  */
/* --     (where i is the label of the wait statement) is       */
/* --     initialized to a structure made of the sensitivity and*/
/* --     condition clauses of the wait statement, and the      */
/* --     current vertex.                                       */
/* -- The routine is called recursively on the vertices that are*/
/* -- targets of some edge of the root of the given graph. The  */
/* -- root of the graph, and the prossibly created vertex is    */
/* -- marked with 1.                                            */
/* -- For instance, the graph:                                  */
/* --     ---->  V := A and B <-----------+                     */
/* --                |                    |                     */
/* --                | (cvn) NULL       |                     */
/* --               \-/                   |                     */
/* --           +-- if--+                 |                     */
/* --           |       |                 |                     */
/* --    V /= S |       |                 | (cvn) NULL        */
/* --          \-/      | (cvn) NULL    |                     */
/* --        S <= V     |                 |                     */
/* --           |       |                 |                     */
/* --           |      \-/                |                     */
/* --           +-> CURRENT-WAIT := 1     |                     */
/* --                   |                 |                     */
/* --                   | (cvn) NULL    |                     */
/* --                  \-/                |                     */
/* --               wait on A, B----------+                     */
/* -- is transformed into two trees that are ismorphic and look */
/* -- like this:                                                */
/* --   proc_zones[1].Graph = proc_zones[0].Graph =         */
/* --            V := A and B                                   */
/* --                |                                          */
/* --                | (cvn) NULL                             */
/* --               \-/                                         */
/* --           +-- if--+                                       */
/* --           |       |                                       */
/* --    V /= S |       |                                       */
/* --          \-/      | not (V /= S)                          */
/* --        S <= V     |                                       */
/* --           |       |                                       */
/* --           |      \-/                                      */
/* --           |   CURRENT-WAIT := 1                           */
/* --(cvn)NULL|                                                 */
/* --           |                                               */
/* --          \-/                                              */
/* --   CURRENT-WAIT := 1                                       */

static cvn
graph_tree
(cvn graph)
{
  cvn res ;
  cvn stm = graph;

  if (IsA(stm, kWaitStatement)) {
    res = 0 ;
  } else if(IsA(stm, kIfStatement)) {
    int arity = (int) mark_get_misc(qToolInfo(stm)) ;
    int idx ;
    cvn new_stm ;
    cvn branch, prev_branch;
    cvn sum_condition = cvn_copy(false_literal);
    new_stm = cvn_copy(stm) ;
    sToolInfo(new_stm, (void *) node_info_copy((node_info) qToolInfo(stm)));

    branch = qBranches(stm) ;
    for (idx = 0 ; idx < arity ; idx++) {
      cvn new_branch = cvn_copy(branch) ;
      cvn not_sum_condition;
      not_sum_condition =
        mUnary(0, mSubtype(0, 0, boolean_type, 0), kNot, sum_condition);
      if (NullNode(qCondition(branch))) {
        sCondition(new_branch, not_sum_condition);
      } else {
        sCondition(new_branch,
                   mBinary(0, mSubtype(0, 0, boolean_type, 0), kAnd, 
                         not_sum_condition, qCondition(branch)));
        sum_condition = mBinary(0, mSubtype(0, 0, boolean_type, 0), kOr,
                            sum_condition, qCondition(branch));
      }
      sToolInfo(new_branch, 
		(void *) node_info_copy((node_info) qToolInfo(branch)));
      sStatements(new_branch, graph_tree(qStatements(branch)));
      if (idx) {
	sNext(prev_branch, new_branch) ;
	if (idx == arity - 1) sNext(new_branch, 0) ;
      } else {
	sBranches(new_stm, new_branch);
      }
      prev_branch = new_branch ;
      branch = qNext(branch) ;
    }
    res = new_stm ;
  } else if (IsA(stm, kCaseStatement)) {
    int arity = (int) mark_get_misc(qToolInfo(stm)) ;
    int idx ;
    cvn new_stm ;
    cvn selector, alternative, prev_alternative;
    cvn sum_conditions;

    new_stm = cvn_copy(stm) ;
    sToolInfo(new_stm, (void *) node_info_copy((node_info) qToolInfo(stm)));

    selector = qSelector(stm) ;
    sum_conditions = cvn_copy(false_literal);
    for (idx = 0, alternative = qAlternatives(stm);
         idx < arity ;
         idx++, alternative = qNext(alternative)) {
      cvn new_alternative = cvn_copy(alternative);
      cvn condition = cvn_copy(false_literal);
      cvn choice;
      if (qChoices(alternative)) {
        for (choice = qChoices(alternative); 
             !NullNode(choice);
             choice = qNext(choice)) {
          condition =
            mBinary(0, mSubtype(0, 0, boolean_type, 0), kOr, condition, 
                    mBinary(0, mSubtype(0, 0, boolean_type, 0), kEq,
                            selector, qValue(choice)));
          sum_conditions = mBinary(0, mSubtype(0, 0, boolean_type, 0), kOr,
                                   sum_conditions, condition);
        }
      } else {
        condition = 
          mUnary(0, mSubtype(0, 0, boolean_type, 0), kNot, sum_conditions);
      }
      sToolInfo(new_alternative, 
                (void *) node_info_copy((node_info) qToolInfo(alternative)));
      sChoices(new_alternative, condition);
      sStatements(new_alternative, graph_tree(qStatements(alternative)));
      if (idx) {
	sNext(prev_alternative, new_alternative) ;
	if (idx == arity - 1) sNext(new_alternative, 0) ;
      } else {
	sAlternatives(new_stm, new_alternative);
      }
      prev_alternative = new_alternative ;
    }
    res = new_stm;
    /* Signal and variable assignment, null statements */
  } else {
    cvn new_stm ;
    new_stm = cvn_copy(stm) ;
    sToolInfo(new_stm, 
	      (void *) node_info_copy((node_info) qToolInfo(stm)));
    if (IsA(stm, kSignalAssignmentStatement) || 
        IsA(stm, kVariableAssignmentStatement)) {
      sSource(new_stm, cvn_copy(qSource(stm)));
    } else if (IsA(stm, kWhileStatement)) {
      sCondition(new_stm, cvn_copy(qCondition(stm)));
      sStatements(new_stm, graph_tree(qStatements(stm)));
    }
    sNext(new_stm, graph_tree(qNext(stm))) ;
    res = new_stm ;
  }
  return res;
}

/* --                 -----------------                         */
/* -- Routine:        var_reduc                         */
/* --                 -----------------                         */
/* -- Description: This routine modifies the data structure     */
/* -- of the Graph field of each entry of the global variable   */
/* -- proc_zones.                                             */
/* -- This data structure is supposed to be a tree as produced  */
/* -- by the routine gen_zones. The modification is such that*/
/* -- when a vertex represents a variable assignment, the tree  */
/* -- under this vertex is modified to take into account this   */
/* -- assignment. The instances of the assigned variable in the */
/* -- labels of the edges, and the data of the vertices are     */
/* -- replaced by the source of the assignment.                 */
/* -- This routine and the associated subroutines use the       */
/* -- ToolInfo attribute of the VHDL intermediate format of the */
/* -- assigned variable to store the source of the assignment.  */
/* -- A new variable assignment overrides the previous ones.    */
/* -- reduce_tree and ReduceExpression are the two main routines */
/* -- that are called by var_reduc.                     */
/* -- For instance, the tree:                                   */
/* --            V := A and B                                   */
/* --                |                                          */
/* --                | (cvn) NULL                             */
/* --               \-/                                         */
/* --           +-- if--+                                       */
/* --           |       |                                       */
/* --    V <> S |       |                                       */
/* --          \-/      | (cvn) NULL                          */
/* --        S <= V     |                                       */
/* --           |       |                                       */
/* --           |      \-/                                      */
/* --           |   CURRENT-WAIT := 1                           */
/* --(cvn)NULL|                                               */
/* --           |                                               */
/* --          \-/                                              */
/* -- is transformed into:                                      */
/* --   CURRENT-WAIT := 1                                       */
/* --            V := A and B                                   */
/* --                |                                          */
/* --                | (cvn) NULL                             */
/* --               \-/                                         */
/* --           +-- if--+                                       */
/* -- (A and B) |       |                                       */
/* --    <> S   |       |                                       */
/* --          \-/      | (cvn) NULL                          */
/* --   S <= (A and B)  |                                       */
/* --           |       |                                       */
/* --           |      \-/                                      */
/* --           |   CURRENT-WAIT := 1                           */
/* --(cvn)NULL|                                               */
/* --           |                                               */
/* --          \-/                                              */
/* --   CURRENT-WAIT := 1                                       */


/* This routine and the associated subroutines use the
   ToolInfo attribute to store results: whenever a variable is
   assigned, the source expression of the assignment is stored
   in this generic attribute */
static void
reduce_tree
(cvn stm)
{
  if (!stm) return ;

  /* For an if statement, the conditions of the different branches
     are reduced, and the branches are reduced. */
  if (IsA(stm, kIfStatement)) {
    cvn branch ;
    int idx ;
    branch = qBranches(stm);
    for (idx = 0; idx < (int) mark_get_misc(qToolInfo(stm)) ; 
	 ++idx) {
      sCondition(branch, ReduceExpression(qCondition(branch))) ;
      reduce_tree(qStatements(branch)) ;
      branch = qNext(branch) ;
    }
  } else if (IsA(stm, kCaseStatement)) {
    cvn alternative ;
    int idx;
    alternative = qAlternatives(stm);
    for (idx = 0; idx < (int) mark_get_misc(qToolInfo(stm)) ; ++idx) {
      reduce_tree(qStatements(alternative)) ;
      alternative = qNext(alternative) ;
    }
  } else if (IsA(stm, kWhileStatement)) {
    sCondition(stm, ReduceExpression(qCondition(stm)));
    reduce_tree(qStatements(stm));
    reduce_tree(qNext(stm));
  } else if (IsA(stm, kNullStatement)) {
    reduce_tree(qNext(stm)) ;
  } else if (IsA(stm, kSignalAssignmentStatement)) {
    cvn src;
    src = ReduceExpression(qValue(qSource(stm)));
    sValue(qSource(stm), src);
    reduce_tree(qNext(stm)) ;
  } else if (IsA(stm, kVariableAssignmentStatement)) {
    void * info;
    cvn src;
    src = ReduceExpression(qSource(stm)) ;
    sSource(stm, src);
    info = qToolInfo(qDeclaration(qTarget(stm))) ;
    sToolInfo(qDeclaration(qTarget(stm)), src) ;
    reduce_tree(qNext(stm)) ;
    sToolInfo(qDeclaration(qTarget(stm)), info) ;
    /* A wait statement should not occur here */
  } else if (IsA(stm, kWaitStatement)) {
    fprintf(stderr, 
	    "Internal error: unexpected wait statement [reduce_tree].\n") ;
    exit(ERROR_EXIT_CODE) ;

    /* For an assignment statement, the source expression is reduced. */
  }
}

/* --                 -----------------------                   */
/* -- Routine:        gen_trans                                 */
/* --                 -----------------------                   */
/* -- Description: This routine sets the global variable        */
/* -- transitions. Each object assigned in the process          */
/* -- statement has n+1 one entries in transitions, where n     */
/* -- is the number of wait statements of the process statement */
/* -- part. The i^{th} entry of some object O is set to a       */
/* -- decision tree, where the non terminal vertices correspond */
/* -- to if statements, and the leaves to values assigned to O. */
/* -- This decision tree is built from an execution tree as     */
/* -- produced by the routine ReorderStatements.                */
/* -- The main used subroutine is gen_trans.                    */
/* -- If no value is assigned the data field of the leaf is set */
/* -- to null (i.e. (cvn) NULL).                                */
/* -- For instance:                                             */
/* --           +-- if--+                                       */
/* -- (A and B) |       |                                       */
/* --    <> S   |       | (cvn) NULL                            */
/* --          \-/     \-/                                      */
/* --  V := A and B  V := A and B                               */
/* --           |       |                                       */
/* --(cvn)NULL|       | (cvn) NULL                              */
/* --           |       |                                       */
/* --          \-/      |                                       */
/* --   S <= (A and B)  |                                       */
/* --           |       |                                       */
/* --           |      \-/                                      */
/* --           |   CURRENT-WAIT := 1                           */
/* --(cvn)NULL|                                                 */
/* --           |                                               */
/* --          \-/                                              */
/* --   CURRENT-WAIT := 1                                       */
/* -- gives the following base transitions:                     */
/* --    . for V:                                               */
/* --           +-- if--+                                       */
/* -- (A and B) |       |                                       */
/* --    <> S   |       | (cvn) NULL                            */
/* --          \-/     \-/                                      */
/* --  V := A and B  V := A and B                               */
/* --    . for S:                                               */
/* --           +-- if--+                                       */
/* -- (A and B) |       |                                       */
/* --    <> S   |       | (cvn) NULL                            */
/* --          \-/     \-/                                      */
/* -- S <= (A and B) (cvn) NULL                                 */
/* --    . for CURRENT-WAIT:                                    */
/* --           +-- if--+                                       */
/* -- (A and B) |       |                                       */
/* --    <> S   |       | (cvn) NULL                            */
/* --          \-/     \-/                                      */
/* --CURRENT-WAIT:=1  CURRENT-WAIT:=1                           */

static tGraph
gen_trans
(cvn graph,
 cvn obj)
{
  if (!graph) return 0;

  if (IsA(graph, kNullStatement)) {
    return gen_trans(qNext(graph), obj);
  } else if (IsA(graph, kSignalAssignmentStatement) ||
      IsA(graph, kVariableAssignmentStatement)) {
    if (qNext(graph) == 0) {
      if (cv_eq_decl(qTarget(graph), obj))
	if (IsA(graph, kSignalAssignmentStatement))
	  return graph_new_vertex(qValue(qSource(graph)), 0, NOMARK);
	else
	  return graph_new_vertex(qSource(graph), 0, NOMARK);
      else 
	return 0;
    } else {
      tGraph tgt ;
      tgt = gen_trans(qNext(graph), obj) ;
      if (cv_eq_decl(qTarget(graph), obj))
	if (IsA(graph, kSignalAssignmentStatement))
	  return graph_insert(qValue(qSource(graph)), tgt);
	else
	  return graph_insert(qSource(graph), tgt);
      else
	return tgt;
    }
  } else if (IsA(graph, kWhileStatement)) {
    tGraph res;
    res = graph_new_vertex(0, 2, NOMARK);
    graph_set_edge(res, 0, qCondition(graph),
                   gen_trans(qStatements(graph), obj));
    graph_set_edge(res, 1, mUnary(0, mSubtype(0, 0, boolean_type, 0), kNot, 
                                  qCondition(graph)),
                   gen_trans(qNext(graph), obj));
    return res;
  } else if (IsA(graph, kIfStatement)) {
    tGraph res ;
    cvn branch ;
    int idx ;
    res = 
      graph_new_vertex((cvn) NULL, 
		       (int) mark_get_misc(qToolInfo(graph)), 
		       NOMARK) ;
    branch = qBranches(graph) ;
    for (idx = 0 ; 
	 idx < (int) mark_get_misc((node_info) qToolInfo(graph)); 
	 ++idx) {
      graph_set_edge(res, idx, qCondition(branch),
		     gen_trans(qStatements(branch), obj)) ;
      branch = qNext(branch) ;
    }
    return graph_compact(res);
  } else if (IsA(graph, kCaseStatement)) {
    tGraph res ;
    cvn alternative ;
    int idx ;
    res = 
      graph_new_vertex((cvn) NULL, 
		       (int) mark_get_misc(qToolInfo(graph)), 
		       NOMARK) ;
    alternative = qAlternatives(graph) ;
    for (idx = 0 ; 
	 idx < (int) mark_get_misc((node_info) qToolInfo(graph)); 
	 ++idx) {
      graph_set_edge(res, idx, qChoices(alternative),
		     gen_trans(qStatements(alternative), obj)) ;
      alternative = qNext(alternative) ;
    }
    return graph_compact(res);
  } else {
    fprintf(stderr,
	    "Internal error: vertex of unknown type [gen_trans].\n") ;
    exit(ERROR_EXIT_CODE) ;
  }
}

static tGraph
graph_insert
(cvn expr,
 tGraph node)
{
  tGraph res ;
  if (!node)
    res = graph_new_vertex(expr, 0, NOMARK) ;
  else if (!graph_get_data(node)) {
    int idx ;
    for (idx = 0 ; idx < graph_get_arity(node) ; idx += 1)
      graph_set_target(node, idx,
		       graph_insert(expr, graph_get_target(node, idx))) ;
    res = node ;
  } else
    res = node ;
  return res;
}

static tGraph
graph_compact
(tGraph graph)
{
   return graph;
}

static vbdd
gen_sym_trans
(expression_context expr_mgr,
 bdd_manager bddm,
 vbdd obj,
 tGraph graph)
{
  if (graph == 0) { /* no assignment, object keeps its value */
    return vbdd_copy(bddm, obj);
  } else {
    cvn root;
    root = (cvn) graph_get_data(graph) ;
    if (root != 0) { /* terminal node */
      return vbdd_copy(bddm, expression_value(expr_mgr, root)->coding);
    } else { /* non-terminal node */
      vbdd result;
      int i;
      bdd cond;
      for (cond = bdd_one(bddm),
           result = vbdd_zero(bddm, (int) vbdd_length(obj)),
           i = 0;
           i < graph -> Arity ; 
           ++i) {
        bdd tmp_cond;
        vbdd rest, ith_val, tmp_result;
        cvn label;
        do {
          rest = gen_sym_trans(expr_mgr, bddm, obj, graph_get_target(graph, i)) ;
          label = (cvn) graph_get_label(graph, i);
          if (label == 0) { /* last branch of an incomplete if stm */
            ith_val = vbdd_distribute(bdd_and, bddm, cond, rest, 1);
            /* assign a meaningful value to tmp_cond, so that free works. */
            tmp_cond = bdd_identity(bddm, bdd_one(bddm)); 
          } else {
            bdd ith_cond;
            ith_cond = bdd_and(bddm, condition_code(label, expr_mgr), cond);
            ith_val = vbdd_distribute(bdd_and, bddm, ith_cond, rest, 1);
            if (i != (graph -> Arity) - 1) { /* not last branch */
              tmp_cond = bdd_and(bddm, cond, bdd_not(bddm, ith_cond)) ;
            } else {
              /* assign a meaningful value to tmp_cond, so that free works. */
              tmp_cond = bdd_identity(bddm, bdd_one(bddm)); 
            }
            bdd_free(bddm, ith_cond);
          }
          vbdd_free(bddm, rest);
          tmp_result = vbdd_or(bddm, result, ith_val) ;
          if (bdd_overflow(bddm)) {
            vbdd_free(bddm, tmp_result);
            vbdd_free(bddm, ith_val);
            bdd_free(bddm, tmp_cond);
            bdd_resize_manager(bddm);
          } else {
            vbdd_free(bddm, result);
            vbdd_free(bddm, ith_val);
            bdd_free(bddm, cond);
            result = tmp_result;
            cond = tmp_cond;
            break;
          }
        } while (1);
      }
      return result;
    }
  }
}

static node_info
node_info_copy
(node_info i)
{
   node_info res;
   if (i == 0) return 0;

   res = malloc(sizeof(struct node_info_));
   memcpy(res, i, sizeof(struct node_info_));

   return res;
}

static node_info
mark_new_mark
(int mark, void * misc)
{
   node_info res ;
   assert ((res = malloc(sizeof(struct node_info_))) != 0) ;
   res -> mark = mark ;
   res -> misc = misc ;
   return res;
}

