/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    expression.c                                              */
/*                                                              */
/* Description:                                                 */
/*    Routines that compute the expression of semantics, and    */
/*    manipulate their result.                                  */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <stdlib.h>
#include <malloc.h>
#include <assert.h>

#include <bdduser.h>
#include <vbdd.h>
#include <vbdd_relat.h>
#include <vbdd_logic.h>
#include <vbdd_arith.h>

#include <cache.h>

#include "cvc.h"

/* ------------------------------------------------------------ */
typedef struct id_hash_entry_rec_ {
  struct id_hash_entry_rec_ * link;
  bdd_value value;
  cvn decl; 
} id_hash_entry_rec, * id_hash_entry;
static inline unsigned long id_hash (pointer id, pointer env);
static inline int id_eq (pointer id1, pointer id2, pointer env);
static inline void id_del (pointer id, pointer env);
static inline bdd_value id_value (expression_context c, cvn id);

/* ------------------------------------------------------------ */
typedef struct lit_hash_entry_rec_ {
  struct lit_hash_entry_rec_ * link;
  bdd_value value;
  cvn type; 
  unsigned pos; 
} lit_hash_entry_rec, * lit_hash_entry;
static inline unsigned long lit_hash (pointer lit, pointer env);
static inline int lit_eq (pointer lit1, pointer lit2, pointer env);
static inline void lit_del (pointer lit, pointer env);
static inline bdd_value lit_value (expression_context c, cvn lit);

/* ------------------------------------------------------------ */
/* there is a single unary operator (kNot), thus the key is
 * simply the operand */
typedef struct un_hash_entry_rec_ {
  struct un_hash_entry_rec_ * link;
  pointer operand;
  bdd_value value;
} un_hash_entry_rec, * un_hash_entry;
static inline unsigned long un_hash (pointer un, pointer env);
static inline int un_eq (pointer un1, pointer un2, pointer env);
static inline void un_del (pointer un, pointer env);
static inline bdd_value un_value (expression_context c, cvn un);

/* ------------------------------------------------------------ */
typedef struct bin_hash_entry_rec_ {
  struct bin_hash_entry_rec_ * link;
  tOperator operator;
  bdd_value left;
  bdd_value right;
  bdd_value value;
} bin_hash_entry_rec, * bin_hash_entry;
static inline unsigned long bin_hash (pointer bin, pointer env);
static inline int bin_eq (pointer bin1, pointer bin2, pointer env);
static inline void bin_del (pointer bin, pointer env);
static inline bdd_value bin_value (expression_context c, cvn bin);

/* ------------------------------------------------------------ */
static bdd_value
  arith_value(vbdd (* fn) (bdd_manager, vbdd, vbdd), bdd_value, bdd_value);
static bdd_value 
  logic_bin_value(vbdd (* fn) (bdd_manager, vbdd, vbdd), bdd_value, bdd_value);
static bdd_value 
  logic_un_value(vbdd (* fn) (bdd_manager, vbdd), bdd_value);
static bdd_value 
  relation_value(bdd (* fn) (bdd_manager, vbdd, vbdd), bdd_value, bdd_value);

/* ------------------------------------------------------------ */
#define is_relat(op) (op >= kEq) && (op <= kLe)
#define is_logic_bin(op) (op >= kAnd) && (op <= kXnor)
#define is_arith(op) (op == kPlus) || (op == kMinus)

/**Function**
  Synopsis           [ Initializes context for expression evaluation  ]
  Description        [ Creates a context that is later used to 
  evaluate VHDL expressions as vectors of BDDs. This context is composed
  of various caches. There is one such cache for each family of VHDL
  expressions. Uses the library libcache. ]
  SideEffects        [ Allocates memory ]
  SeeAlso            [ expression_reset expression_finish ]
*/
expression_context
expression_init
(void)
{
  expression_context res;
  res=(expression_context)mem_get_block((SIZE_T)sizeof(expression_context_rec));
  res->id_rec_mgr = mem_new_rec_mgr((SIZE_T) sizeof(struct id_hash_entry_rec_));
  res->id_cache = cache_XNew(512, 
                             (hash_function) & id_hash,
                             (eq_function) & id_eq,
                             (del_function) & id_del,
                             0, res->id_rec_mgr, 0.67, 2.0);
  res->lit_rec_mgr=mem_new_rec_mgr((SIZE_T) sizeof(struct lit_hash_entry_rec_));
  res->lit_cache=cache_XNew(64, 
                            (hash_function) & lit_hash,
                            (eq_function) & lit_eq,
                            (del_function) & lit_del,
                            0, res->lit_rec_mgr, 0.67, 2.0);
  res->un_rec_mgr=mem_new_rec_mgr((SIZE_T) sizeof(struct un_hash_entry_rec_));
  res->un_cache=cache_XNew(64, 
                           (hash_function) & un_hash,
                           (eq_function) & un_eq,
                           (del_function) & un_del,
                           0, res->un_rec_mgr, 0.67, 2.0);
  res->bin_rec_mgr=mem_new_rec_mgr((SIZE_T) sizeof(struct bin_hash_entry_rec_));
  res->bin_cache=cache_XNew(742, 
                            (hash_function) & bin_hash,
                            (eq_function) & bin_eq,
                            (del_function) & bin_del,
                            0, res->bin_rec_mgr, 0.67, 2.0);
  expression_value(res, InitCondition);
  return res;
}

void
expression_reset
(expression_context c)
{
/*  cache_XReset(c->id_cache); */
  cache_XReset(c->lit_cache);
  cache_XReset(c->un_cache);
  cache_XReset(c->bin_cache);
}

void
expression_finish
(expression_context c)
{
  cache_XFree(c->id_cache);
  cache_XFree(c->lit_cache);
  cache_XFree(c->un_cache);
  cache_XFree(c->bin_cache);
  mem_free_rec_mgr(c->id_rec_mgr);
  mem_free_rec_mgr(c->lit_rec_mgr);
  mem_free_rec_mgr(c->un_rec_mgr);
  mem_free_rec_mgr(c->bin_rec_mgr);
  mem_free_block((pointer) c);
}

/* ------------------------------------------------------------ */
static inline unsigned long
id_hash
(pointer id,
 pointer env) 
{
  cvn decl = ((id_hash_entry) id)->decl;
  return (((unsigned long) decl) >> 2);
}

static inline int
id_eq
(pointer id1,
 pointer id2,
 pointer env) 
{
  id_hash_entry e1 = (id_hash_entry) id1,
                e2 = (id_hash_entry) id2;
  return (e1 && e2 && cv_eq(e1->decl, e2->decl));
}

static inline void
id_del
(pointer id,
 pointer env)
{
  bdd_value_free(((id_hash_entry)id)->value);
}

static inline bdd_value
id_value
(expression_context c, cvn id)
{
  id_hash_entry_rec entry;
  id_hash_entry r;
  entry.decl = qDeclaration(id);
  if ((r = cache_XFind(c->id_cache, (pointer) & entry)) == 0) {
    r = (id_hash_entry) mem_new_rec(c->id_rec_mgr);
    r->link = 0;
    r->decl = qDeclaration(id);
    r->value = encode_object(id);
    cache_XInsert(c->id_cache, (pointer) r);
  }
  return r->value;
}

/* ------------------------------------------------------------ */
static inline unsigned long
lit_hash
(pointer lit,
 pointer env)
{
  lit_hash_entry entry = (lit_hash_entry) lit;
  return ((unsigned long) entry->type + (entry->pos >> 2));
}

static inline int
lit_eq
(pointer lit1,
 pointer lit2,
 pointer env)
{
  lit_hash_entry e1 = (lit_hash_entry) lit1,
                 e2 = (lit_hash_entry) lit2;
  return (e1 && e2 && cv_eq(e1->type, e2->type) && (e1->pos == e2->pos));
}

static inline void
lit_del
(pointer lit,
 pointer env)
{
  bdd_value_free(((lit_hash_entry)lit)->value);
}

static inline bdd_value
lit_value
(expression_context c,
 cvn lit)
{
  lit_hash_entry_rec entry;
  lit_hash_entry r;
  {
    cvn subtype, basetype, range;
    subtype= qSubtype(lit);
    entry.type = subtype;
    basetype = qBaseType(subtype);
    range = qConstraint(subtype);
    if ((range == 0) && IsA(basetype, kIntegerType)) {
      range = qRange(basetype);
    }
    if (IsA(basetype, kIntegerType)) 
      entry.pos = qIntegerValue(lit) - qIntegerValue(qLeft(range));
    else /* type = kEnumerationType */
      entry.pos= qOrdinalValue(lit);
  }
  if ((r = cache_XFind(c->lit_cache, (pointer) & entry)) == 0) {
    r = mem_new_rec(c->lit_rec_mgr);
    r->link = 0;
    r->type = entry.type;
    r->pos = entry.pos;
    r->value = encode_constant(lit);
    cache_XInsert(c->lit_cache, (pointer) r);
  }
  return r->value;
}

/* ------------------------------------------------------------ */
static inline unsigned long
un_hash
(pointer un,
 pointer env)
{
  un_hash_entry e = (un_hash_entry) un;
  return ((unsigned long) e->operand >> 2);
}

static inline int
un_eq
(pointer un1,
 pointer un2,
 pointer env)
{
  un_hash_entry e1 = (un_hash_entry) un1,
                e2 = (un_hash_entry) un2;
  return (e1 && e2 && (e1->operand == e2->operand));
}

static inline void
un_del
(pointer un,
 pointer env)
{
  bdd_value_free(((un_hash_entry)un)->value);
}

static inline bdd_value
un_value
(expression_context c,
 cvn un)
{
  un_hash_entry_rec entry;
  un_hash_entry r;
  entry.operand = expression_value(c, qOperand(un));
  if ((r = cache_XFind(c->un_cache, (pointer) & entry)) == 0) {
    r = (un_hash_entry) mem_new_rec(c->un_rec_mgr);
    r->link = 0;
    r->operand = entry.operand;
/*
   ! works as long as the only unary operator is kNot 
*/
    r->value = logic_un_value(vbdd_not, entry.operand);
    cache_XInsert(c->un_cache, (pointer) r);
  }
  return r->value;
}

/* ------------------------------------------------------------ */
static inline unsigned long
bin_hash
(pointer bin,
 pointer env)
{
  bin_hash_entry e = (bin_hash_entry) bin;
  unsigned long l = (unsigned long) e->left,
                r = (unsigned long) e->right;
  const int SLOT = 64;
  switch (e->operator) {
  case kEq:
    return 0*SLOT + ((l >> 6) + (r >> 6)) % SLOT;
  case kNeq:
    return 1*SLOT + ((l >> 7) + (r >> 7)) % SLOT;
  case kLt:
    return 2*SLOT + ((l >> 5) + (r >> 7)) % SLOT;
  case kGt:
    return 2*SLOT + ((l >> 7) + (r >> 5)) % SLOT;
  case kLe: 
    return 3*SLOT + ((l >> 5) + (r >> 7)) % SLOT;
  case kGe:
    return 3*SLOT + ((l >> 7) + (r >> 5)) % SLOT;
  case kAnd:
    return 4*SLOT + ((l >> 5) + (r >> 5)) % SLOT;
  case kNand:
    return 5*SLOT + ((l >> 5) + (r >> 5)) % SLOT;
  case kOr:
    return 6*SLOT + ((l >> 5) + (r >> 5)) % SLOT;
  case kNor:
    return 7*SLOT + ((l >> 5) + (r >> 5)) % SLOT;
  case kXor:
    return 8*SLOT + ((l >> 5) + (r >> 5)) % SLOT;
  case kXnor:
    return 9*SLOT + ((l >> 5) + (r >> 5)) % SLOT;
  case kPlus:
    return 10*SLOT + ((l >> 5) + (r >> 5)) % SLOT;
  default: /* i.e. kMinus */
    return 11*SLOT + ((l >> 5) + (r >> 5)) % SLOT;
  }
}

static inline int
bin_eq
(pointer bin1,
 pointer bin2,
 pointer env)
{
  bin_hash_entry e1 = (bin_hash_entry) bin1,
                 e2 = (bin_hash_entry) bin2;
  return (e1 && e2 && 
          (e1->operator == e2->operator) && 
          (e1->left == e2->left) && 
          (e1->right == e2->right));
}

static inline void
bin_del
(pointer bin,
 pointer env)
{
  bdd_value_free(((bin_hash_entry)bin)->value);
}

static inline bdd_value
bin_value
(expression_context c,
 cvn bin)
{
  bin_hash_entry_rec entry;
  bin_hash_entry r;
  tOperator operator;
  entry.operator = operator = qOperator(bin);
  entry.left= expression_value(c, qLeft(bin));
  entry.right= expression_value(c, qRight(bin));
  if ((r = cache_XFind(c->bin_cache, (pointer) & entry)) == 0) {
    r = (bin_hash_entry) mem_new_rec(c->bin_rec_mgr);
    r->link = 0;
    r->operator = entry.operator;
    r->left = entry.left;
    r->right = entry.right;
    /* If the operation is a relation */
    if (is_relat(operator)) {
      bdd (* fn) (bdd_manager, vbdd, vbdd);
      switch(operator) {
      case kEq: fn = vbdd_eq;
	break;
      case kNeq: fn = vbdd_neq;
	break;
      case kGt: fn = vbdd_gt;
	break;
      case kGe: fn = vbdd_ge;
	break;
      case kLe: fn = vbdd_le;
	break;
      default: fn = vbdd_lt;
      } 
      r->value = relation_value(fn, entry.left, entry.right);
       
    } else if (is_logic_bin(operator)) {
      vbdd (* fn) (bdd_manager, vbdd, vbdd);
      switch (operator) {
      case kAnd: fn = vbdd_and;
	break;
      case kNand: fn = vbdd_nand;
	break;
      case kOr: fn = vbdd_or;
	break;
      case kNor: fn = vbdd_nor;
	break;
      case kXor: fn = vbdd_xor;
	break;
      default: fn = vbdd_xnor;
      } 
      r->value = logic_bin_value(fn, entry.left, entry.right);
    } else if (is_arith(operator)) {
      vbdd (* fn) (bdd_manager, vbdd, vbdd);
      switch (operator) {
      case kPlus: fn = vbdd_plus;
	break;
      case kMinus: fn = vbdd_minus;
	break;
      default:
	printf("Can't handle operator: %u.\n", (unsigned) operator);
	fn = vbdd_plus;
      }
      r->value = arith_value(fn, entry.left, entry.right);
    }
    cache_XInsert(c->bin_cache, (pointer) r);
  }
  return r->value;
}

/* ------------------------------------------------------------ */
/* condition_code
      args: cvn exp
      result: bdd
*/
bdd
condition_code 
(cvn exp,
 expression_context c)
{
  bdd_value value;
  bdd res;
  value= expression_value(c, exp);
  res= vbdd_nth(value->coding, 0);
  return res;
}

/* ------------------------------------------------------------ */
/* expression_value
      args: cvn exp
      result: bdd_value
*/

bdd_value
expression_value
(expression_context c,
 cvn exp)
{
  bdd_value res;

  if ((IsA(exp, kCharacterLiteral)) || 
      (IsA(exp, kEnumeratedLiteral)) ||
      (IsA(exp, kIntegerValue))) {
    res = lit_value(c, exp);
    return res;
  }

  if (IsA(exp, kIdentifier)) {
    cvn decl = qDeclaration(exp);
    if (IsA(decl, kSignalDeclaration) || (IsA(decl, kVariableDeclaration)))
      res = id_value(c, exp);
    else if (IsA(decl, kEnumeratedLiteral))
      res = lit_value(c, decl);
    else
      return 0;
    return res;
  }

  if (IsA(exp, kUnary)) {
    res = un_value(c, exp);
    return res;
  }

  if (IsA(exp, kBinary)) {
    res = bin_value(c, exp);
    return res;
  }
  return 0;
}


/* ------------------------------------------------------------ */
/* logic_un_value
      args: vbdd (* fn) (bdd_manager, vbdd)
            bdd_value arg
      result: bdd_value
   fn is a predefined unary VHDL operator. exp is the
   semantics of the argument of fn.
   The result of the expression is a set of atomic expressions.
   There might be several atomic expressions for each zone that has
   a corresponding continuation in the environment, but none in
   when there is no continuation.
*/

static bdd_value
logic_un_value
(vbdd (* fn) (bdd_manager, vbdd),
 bdd_value arg)
{
  bdd_value res;
  res= malloc(sizeof(struct bdd_value_));
  res->type = arg->type;
  res->valid = arg->valid;
  res->coding = fn(bddm, arg->coding);
  return res;
}

/* ------------------------------------------------------------ */
/* relation_value
      args: bdd (* fn) (bdd_manager, vbdd, vbdd)
            bdd_value arg1
            bdd_value arg2
      result: bdd_value
   fn is a predefined relational VHDL operator. Expression is the
   semantics of the argument of fn.
   The result of the expression is a set of atomic expressions.
   There might be several atomic expressions for each zone that has
   a corresponding continuation in the environment, but none in
   when there is no continuation.
*/

static bdd_value
relation_value 
(bdd (* fn) (bdd_manager, vbdd, vbdd),
 bdd_value arg1, bdd_value arg2)
{

   /* res codes the symbolic value of the
      expression of the result triple */
   bdd_value res= 0;
   bdd valid;

   if ((valid = bdd_and(bddm, arg1->valid, arg2->valid))
       != bdd_zero(bddm)) {
      /* adj_values is the normalization of the
         expressions of the argument triples */
      bdd_value * adj_values;
      /* coding is a null-terminated one-element vector of BDD that 
	 codes the expression of the triple */
      vbdd symbolic_value;
      symbolic_value = vbdd_newn(1);
      adj_values = adjust_values(arg1, arg2);
      res = malloc(sizeof(struct bdd_value_));
      if (vhdl_type_equal(adj_values[0]->type, adj_values[1]->type)) {
	res->valid =  bdd_and(bddm, adj_values[0]->valid, adj_values[1]->valid);
      } else {
	res->valid = bdd_zero(bddm);
     }  
     vbdd_set(bddm, symbolic_value, 0, 
              bdd_and(bddm, res->valid,
		      fn(bddm, adj_values[0]->coding, adj_values[1]->coding)));
     res->coding = symbolic_value;
     {
       vhdl_type * type = malloc(sizeof(t_enumeration_type));
       type->enum_t.kind = kEnumType;
       type->enum_t.node = boolean_type;
       res->type = * type;
     }
   }
   return res;
}

/* ------------------------------------------------------------ */
/* logic_bin_value
      args: vbdd (* fn) (bdd_manager, vbdd, vbdd)
            bdd_value arg1
            bdd_value arg2
      result: bdd_value
*/
static bdd_value
logic_bin_value
(vbdd (* fn) (bdd_manager, vbdd, vbdd),
 bdd_value arg1,
 bdd_value arg2)
{
   bdd_value res = malloc(sizeof(struct bdd_value_));

   memcpy(res, arg1, sizeof(struct bdd_value_));
   res->valid = bdd_and(bddm, arg1->valid, arg2->valid);
   res->coding = fn(bddm, arg1->coding, arg2->coding);
   return res;

}

/* ------------------------------------------------------------ */
/* arith_value
      args: vbdd (* fn) (bdd_manager, vbdd, vbdd)
            bdd_value arg1
            bdd_value arg2
      result: bdd_value
*/

static bdd_value
arith_value
(vbdd (* fn) (bdd_manager, vbdd, vbdd),
 bdd_value arg1,
 bdd_value arg2)
{
  bdd valid;
  bdd_value res = malloc(sizeof(struct bdd_value_));
  /* type is the are the informations of the encoding of
     the (vhdl) type of the result */
  vhdl_type * type;

  if ((valid = bdd_and(bddm, arg1->valid , arg2->valid))
      != bdd_zero(bddm)) {
    bdd_value * adj_values;
    vbdd symbolic_value;
    unsigned length;

    adj_values = adjust_values(arg1, arg2);
    length = vbdd_length(adj_values[0]->coding);

    if (vhdl_type_equal(adj_values[0]->type, adj_values[1]->type)) {
      res->valid =
	bdd_and(bddm, adj_values[0]->valid, adj_values[1]->valid);
      type = malloc(sizeof(t_integer_type));
      type->int_t.kind = kSRType;
      type->int_t.offset = (adj_values[0]->type).int_t.offset;
      type->int_t.range = 2 << length;
    } else {
      res->valid = bdd_zero(bddm);
    }
    symbolic_value= fn(bddm, adj_values[0]->coding, adj_values[1]->coding);
    res->coding = symbolic_value;
    res->type = * type;
  }
  return res;
}
