/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    cvc.c                                                     */
/*                                                              */
/* Description:                                                 */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <stdlib.h>
#include <ctype.h>
#include <Errors.h>

#include "Scanner.h"
#include "Parser.h"
#include "cvc.h"

#if !defined(ARGS)
#   if defined(__STDC__)
#      define ARGS(args) args
#   else
#      define ARGS(args) ()
#   endif
#endif

/* ------------------------------------------------------------ */
static void _usage (cvc_context) ;
static void _version (cvc_context) ;

static FILE * vhdl_stream;

/* ------------------------------------------------------------ */
static void _initialize_data (cvc_context context);
static void _parse_command_line (int, char **, cvc_context context);
static int _open_specification (char * name, char ** path);
static char * _parse_specification (cvc_context);
static int _parse_specification_file (char * path);
static void _elaborate_model (cvc_context context);
static void _compute_reachable_states (cvc_context context);
static void _check_model (cvc_context);
static void _check_property (cvc_context, spec_Node_t);
static int _check_property_aux (cvc_context, ctl_term p);
static void _finalize_data (cvc_context);

/* ------------------------------------------------------------ */
static void _DiagnoseCounterExample (cvc_context, ctl_term);
static void _CounterExample(cvc_context, bdd s, ctl_term f, char * label);
static void _Witness(cvc_context, bdd s, ctl_term f, char * label);
static void _NotAvailable(cvc_context, char * label);
static void _VHDL(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _Macro(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _Not(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _Imply(cvc_context, bdd s, ctl_term f, char * label);
static void _And(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _Or(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _Nor(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _Nand(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _Xor(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _Xnor(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _EX(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _EF(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _EG(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _EU(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _EW(cvc_context, bdd s, ctl_term f, char * label, int positive);
static void _BannerLimitProof(cvc_context, char * label, int l);

/* ------------------------------------------------------------ */
static void cvc_bdd_overflow_fn_1 (bdd_manager bddm, pointer env) {
  cvc_context context = (cvc_context) env;
  fprintf(context->outstream, " O ");
  fflush(context->outstream);
}

static void cvc_bdd_overflow_fn_2 (bdd_manager bddm, pointer env) {
  Model m = (Model) env;
  model_RepresentationFlush(m);
}

/* ------------------------------------------------------------ */
const int SUCCESS_EXIT_CODE = 0;
const int ERROR_EXIT_CODE = 1;

/* ------------------------------------------------------------ */
static char * port_dir[5] =  { "default", "in", "out",
                               "internal", "inout" };
static unsigned indent;

static void indent_inc(void);
static void indent_dec(void);
static void space_print(FILE * outstream);

static void indent_inc
(void)
{
  indent += 2;
}

static void indent_dec
(void)
{
  indent -= 2;
}

static void space_print
(FILE * outstream)
{
  unsigned i = indent;

  while(i--!=0) fprintf(outstream, " ");
}

static bdd * _build_specification_support (cvc_context c);
static void * union_fn (void *, void *, void *);
static void * support_fn (void *, void *);
static ctl_SupportUnionFn_t union_fn_ptr = & union_fn;
static ctl_TerminalSupportFn_t support_fn_ptr = & support_fn;

static void _compute_invariant (cvc_context context);

/* ---------------------------------------------------- main -- */
int
main
(int argc,
 char ** argv)
{

  /* declaration part */

  cvc_context_rec context; /* program context */

  bdd_manager bddm; /* aliases frequently used context fields */
  FILE * outstream;

  char * specification_path;
  timer total_time, partial_time;

  /* statement part */

  total_time = timer_New();
  partial_time = timer_New();
  timer_Start(total_time);

  _initialize_data(& context);
  bddm = context.bdd.mgr;
  outstream = context.outstream;

  _parse_command_line(argc, argv, & context);

  specification_path = _parse_specification(& context);

  _version(& context);
  fprintf(outstream,
          "specification: %s\n"
          "       design: %s.%s(%s)\n",
          specification_path, spec_design_library, 
          qName(qPrimary(spec_design)), qName(spec_design));

  timer_Start(partial_time);
  fprintf(outstream, "elaborating model\n");
  _elaborate_model(& context);

  _compute_invariant(& context);

  if (context.opt_still_inputs) {
    model_ModifyStillInputs(context.model);
  }

  timer_Display(outstream, "done", partial_time);

  timer_Reset(partial_time);
  timer_Start(partial_time);

  fprintf(outstream, "simplifying [dynamic reodering]\n");
  expression_reset(context.expr_mgr);
  bdd_gc(bddm);
  if (context.reorder_coeff >= 3) {
    bdd_dynamic_reordering(bddm, bdd_reorder_sift);
    bdd_reorder(bddm);
    bdd_dynamic_reordering(bddm, bdd_reorder_none);
  }

  timer_Display(outstream, "done", partial_time);

  timer_Reset(partial_time);
  timer_Start(partial_time);

  context.bdd.node_limit = bdd_total_size(bddm) * context.size_mult_coeff;
  bdd_node_limit(bddm, context.bdd.node_limit); 

  _compute_reachable_states(& context);

  /* turn off dynamic reordering */
  bdd_dynamic_reordering(bddm, bdd_reorder_none);

  /* compute fairness constraints and fair states */
  if (spec_fairness) {
    timer_Reset(partial_time);
    timer_Start(partial_time);
    model_RepresentationSetFair(context.model, context.outstream, spec_fairness);
    timer_Display(context.outstream, "fair states computed", partial_time);
  }

  _check_model(& context);

  _finalize_data(& context);

  timer_Display(stdout, "total time", total_time);
  timer_Dispose(total_time);

  exit(SUCCESS_EXIT_CODE); 
}

/* ---------------------------------------- _initialize_data -- */
static void 
_initialize_data
(cvc_context context)
{
  /* statement part */

  context->reorder_coeff = 3 ;
  context->size_mult_coeff = 300;
  context->opt_projection = 1 ;
  context->opt_scc = 1 ;
  context->opt_still_inputs = 0;
  context->opt_vhdl_testbench = 0;
  context->opt_witness_display = 0;
  context->opt_witness_counterexample = 0;
  context->specification_name = 0;
  context->outstream = stdout;
  context->model = 0;
  context->expr_mgr = 0;
  bddm = context->bdd.mgr = bdd_init();
  context->bdd.node_limit = 1 << 16;
  context->expr_mgr = 0;
  context->model = 0;

  cv_initialize();
  cv_load_unit("std", "standard", "");

  context->proof_label_mgr = label_NewManager("proof");

  bdd_dynamic_reordering(context->bdd.mgr, bdd_reorder_none);
  bdd_node_limit(context->bdd.mgr, context->bdd.node_limit); 
  bdd_overflow_closure(context->bdd.mgr, cvc_bdd_overflow_fn_1, & context);

  encode_init ();
  context->expr_mgr = expression_init();

  model_Init();

}

/* ------------------------------------------ _finalize_data -- */
static void 
_finalize_data
(cvc_context context)
{
  bdd_gc(context->bdd.mgr);
  model_Final();
  expression_finish(context->expr_mgr);
  encode_close();
  label_FreeManager(context->proof_label_mgr);
}

/* ------------------------------------- _parse_command_line -- */
static void
_parse_command_line
(int argc,
 char ** argv,
 cvc_context context)
{
  /* Process the arguments */
  argv++ ;
  argc-- ;

  while (argc) {
    if (**argv == '-') {
      if ((strcmp(*argv, "-h") == 0) ||
          (strcmp(*argv, "--help") == 0)) {
        _usage(context);
        exit(SUCCESS_EXIT_CODE);
      } else if ((strcmp(*argv, "-v") == 0) || 
               (strcmp(*argv, "--version") == 0)) {
        _version(context);
        exit(SUCCESS_EXIT_CODE);
      } else if (strcmp(*argv, "-fproject") == 0) {
        context->opt_projection = 1;
      } else if (strcmp(*argv, "-fno-project") == 0) {
        context->opt_projection = 0;
      } else if (strcmp(*argv, "-fscc") == 0) {
        context->opt_scc = 1;
      } else if (strcmp(*argv, "-fno-scc") == 0) {
        context->opt_scc = 0;
      } else if (strcmp(*argv, "-freorder-level") == 0) {
        argv++ ;
        argc-- ;
        context->reorder_coeff = atoi(* argv);
        if (context->reorder_coeff < 0) {
          context->reorder_coeff = 0 ;
        } else if (context->reorder_coeff > 4) {
          context->reorder_coeff = 4 ;
        }
      } else if (strcmp(*argv, "-fsize-mult-coeff") == 0) {
        argv++ ;
        argc-- ;
        context->size_mult_coeff = atoi(* argv);
        if (context->size_mult_coeff < 1) {
	  context->size_mult_coeff = 2 ;
        }
      } else if (strcmp(*argv, "-hstill-inputs") == 0) {
        context->opt_still_inputs = 1;
      } else if ((strcmp(*argv, "-W") == 0) || 
               (strcmp(*argv, "--Witness") == 0)) {
        context->opt_witness_counterexample = 1;
        context->opt_witness_display = 1;
      } else if ((strcmp(*argv, "-Wvhdl") == 0) || 
               (strcmp(*argv, "--Witness-vhdl") == 0)) {
        context->opt_witness_counterexample = 1;
        context->opt_vhdl_testbench = 1;
      } else {
        char * message;
        message = (char *) 
	  mem_get_block((SIZE_T) sizeof(char) *
			(strlen("unknown option ") + strlen(*argv) + 1));
        sprintf(message, "unknown option %s", * argv);
        Message(message, xxFatal, NoPosition) ;
        mem_free_block((pointer) message);
        _usage(context);
        exit(ERROR_EXIT_CODE) ;
      }
    } else if (context->specification_name != 0) {
      char * message;
      message = (char *) 
	mem_get_block((SIZE_T) sizeof(char) *
		      (strlen("a single specification file at a time ( or )") +
		       strlen(context->specification_name) + strlen(*argv) + 1));
      sprintf(message, "a single specification file at a time (%s or %s)",
              * argv, context->specification_name);
      Message(message, xxFatal, NoPosition) ;
      mem_free_block((pointer) message);
      _usage (context);
      exit(ERROR_EXIT_CODE) ;
    } else {
      context->specification_name = * argv;
    }
    argv++;
    argc--;
  }
  if (context->specification_name == 0) {
    _usage (context);
    exit(ERROR_EXIT_CODE) ;
  }
}

/* -------------------------------------------------- _usage -- */
static void
_usage
(cvc_context context)
{

  /* statement part */

  fprintf(context->outstream, 
  "cvc(1)\n"
  "usage: cvc [options] filename\n"
  "where options are:\n"
  "<general options>\n"
  "-v .............................  prints CVC version number\n" 
  "-h .............................  prints this\n"
  "<optimization options>\n"
  "-fproject ......................  eliminates parts of the model that\n"
  "                                  are not relevant to the specification\n"
  "-fno-project ...................  turns of projection\n"
  "-fscc ..........................  sets experimental heuristic \n"
  "-fno-scc .......................  turns of experimental heuristic \n"
  "-freorder-level n ..............  sets BDD variable reordering level to n:\n"
  "                                  0 <= n <= 4 (default: 3) \n"
  "-fsize-mult-coeff n ............  multiplies n to model size to set maximum\n"
  "                                  number of BDD nodes\n"
  "                                  2 <= n (default: 300) \n"
  "<witness options>\n"
  "-W .............................  prints witnesses when a specification\n"
  "                                  property is not verified\n"
  "-Wvhdl .........................  prints a testbench to simulate the VHDL\n"
  "                                  description for each generated witness\n"
  "<miscellaneous options>\n"
  "-hstill-inputs .................  models only simulation where signals of\n"
  "                                  mode IN do not change during delta cycles\n"
  );
}

/* -------------------------------------------------- _version -- */
static
void
_version
(cvc_context context)
{
  fprintf(context->outstream,
          "cvc: Cmu Vhdl model Checker (release 1.b1)\n");
}

/* --------------------------------------- _open_specification -- */
/* int _open_specification (char * name, char ** path)

   -1 if a system call failed,
   0 if no file matching "name" was found,
   > 0 if a file matching "name" was found, the file descripted
   by "ptr" is set to the descriptor for this file, mode "read". 
   Searches in the comma-separated list of directories specified by
   the environment variable CV_SPECDIR.
   Also tries to append the suffix ".spec". */

static 
int
_open_specification
(char * name,
 char **path)
{
  /* declaration part */

  FILE * stream;

  /* statement part */

  if ((stream = fopen(name, "r")) != 0) {
    fclose(stream);
    * path = (char *) strdup(name);
    return 1;
  }

  * path = (char *) mem_get_block((SIZE_T) sizeof(char) *
                                  (strlen(name) + strlen(".spec") + 1));
  sprintf(* path, "%s.spec", name);

  if ((stream = fopen(* path, "r")) != 0) {
    fclose(stream);
    return 1;
  }
  return 0;
}

/**Function**
  Synopsis           [ Parses specification file ]
  Description        [ Invokes parser on specification file and
  sets the following global variables: 
  spec_name is a string that identifies the specification;
  spec_design is the root node of the specified VHDL description;
  spec_abbreviations is the set of abbreviations;
  spec_invariant is the set of invariant assumptions;
  spec_fairness is the set of fairness assumptions;
  spec_properties is the set of guarantees to be model checked. 
  Errors abort the program execution. The value returned is the
  actual path of the specification file. ]
  SideEffects        [ Memory is dynamically allocated to store
  these values ]
  SeeAlso            [ _parse_specification_file _open_specification ]
*/
static 
char *
_parse_specification
(cvc_context context)
{
  char * path;
  switch (_open_specification(context->specification_name, & path)) {
  case -1:
    Message("A system call failed", xxFatal, NoPosition);
       exit(ERROR_EXIT_CODE) ;
  case 0:
     Message("Cannot find specification file", xxFatal, NoPosition);
	       exit(ERROR_EXIT_CODE) ;
  default:
    break;
  }

  if (_parse_specification_file(path) == 0) {
    Message("specification cannot be checked", xxFatal, NoPosition);
    exit(ERROR_EXIT_CODE) ;
  }
  {
    cvn n;
    for (n = spec_invariant; n; n = qNext(n)) {
      sToolInfo(qValue(n),
                bdd_identity(context->bdd.mgr,
                             condition_code(qValue(n), context->expr_mgr)));
    }
    for (n = spec_fairness; n; n = qNext(n)) {
      sToolInfo(qValue(n),
                bdd_identity(context->bdd.mgr,
                             condition_code(qValue(n), context->expr_mgr)));
    }
  }

  return path;
}   

/* ----------------------------------- _parse_specification_file -- */
static
int
_parse_specification_file
(char * path)
{

  /* declaration part */

  int nb_parsing_errors;

  /* statement part */

  BeginScanner();
  BeginFile(path);
  nb_parsing_errors=Parser() ;
  if (nb_parsing_errors) {
    char * message;
    if (nb_parsing_errors == 1) {
      message = (char *)
        mem_get_block(1+strlen("1 error found while parsing specification")
                      * sizeof(char));
      sprintf(message, "%s",
              "1 error found while parsing specification");
    } else {
      message = (char *)
        mem_get_block((10 + strlen(" errors found while parsing ."))
                      * sizeof(char));
      sprintf(message, "%u errors found while parsing specification", 
	      nb_parsing_errors);
    }
    Message(message, xxNote, NoPosition);
    mem_free_block(message);
    return 0;
  } else {
    return 1;
  }
}

/* ---------------------------------------- _elaborate_model -- */
static
void
_elaborate_model
(cvc_context context)
{
  bdd * specification_support;
  context->model = model_RepresentationNew(context->bdd.mgr, spec_design);
  elaborate_design_entity(context, spec_design) ;
  specification_support = _build_specification_support(context);
  model_RepresentationBuildTransition(context->model, 
                                      specification_support, 50);
  bdd_free_support(bddm, specification_support);
}

/* ---------------------------- _build_specification_support -- */
static 
bdd *
_build_specification_support
(cvc_context context)
{
  bdd * result;
  spec_Node_t ptr;
  int success = 1;

  result = bdd_new_support(context->bdd.mgr);
  * result = 0;
  for (ptr = spec_properties;  success && ptr; ptr = spec_BasicQNext(ptr)) {
    success = ctl_MiscSupport((ctl_term) spec_BasicQValue(ptr), 
                              support_fn_ptr, union_fn_ptr, context,
                              (void **) & result);
  }
  if ((success == 0) || (context->opt_projection == 0)) {
    bdd_free_support(context->bdd.mgr, result);
    result = 0;
  }
  return result;
}

/* --------------------------------------------- _union_fn -- */
static
void *
union_fn
(void * l1,
 void * l2,
 void * env)
{
  bdd_manager bddm = ((cvc_context) env)->bdd.mgr;
  bdd * res = bdd_new_support(bddm);
  bdd_support_union(bddm, (bdd *) l1, (bdd *) l2, (bdd *) res);
  bdd_free_support(bddm, (bdd *) l1);
  bdd_free_support(bddm, (bdd *) l2);
  return (void *) res;
}

/* --------------------------------------------- _support_fn -- */
static
void *
support_fn
(void * exp,
 void * env)
{
  cvc_context context = (cvc_context) env;
  bdd_manager bddm = context->bdd.mgr;
  bdd code, * res;
  code = condition_code((cvn) exp, context->expr_mgr);
  sToolInfo((cvn) exp, (void *) bdd_identity(bddm, code));
  res = bdd_new_support(bddm);
  bdd_support(bddm, code, res);
  return (void *) res;
}

/* -------------------------------------- _compute_invariant -- */
static
void
_compute_invariant
(cvc_context context)
{
  bdd_manager bddm = context->bdd.mgr;
  bdd invariant;
  cvn port;

  /* structural interface invariant */
  for (invariant = bdd_identity(bddm, bdd_one(bddm)),
       port = qPorts(qPrimary(spec_design));
       !NullNode(port);
       port = qNext(port)) {
    if (qMode(port) == kIn) {
      invariant = bdd_andup2(bddm, encode_decl(port)->valid, invariant);
    }
  }

  /* assumption invariant */
  if (spec_invariant) {
    cvn l;
    for (l = spec_invariant; !NullNode(l); l = qNext(l)) {
      invariant = bdd_andup(bddm, invariant, (bdd) qToolInfo(qValue(l)));
    }

    if (invariant == bdd_zero(bddm)) {
      Message("No transition in the model (check specification invariant)",
              xxWarning, NoPosition);
    }

    model_RepresentationAddInvariant(context->model, invariant);
  }
}

/* ------------------------------- _compute_reachable_states -- */
static
void
_compute_reachable_states
(cvc_context context)
{
  timer clock;

  clock = timer_New();
  timer_Start(clock);

  fprintf(context->outstream, "computing reachable states\n");

  bdd_overflow_closure(context->bdd.mgr, cvc_bdd_overflow_fn_2, 
                       (void *) context->model);
  bdd_resize_data(context->bdd.mgr, bdd_reorder_sift, 0.67, 2);

  if (context->opt_scc) {
    model_ModifyReduceSCC(context->model);
  }

  model_ReachabilityCompute(context->model, context->outstream);

  timer_Display(context->outstream, "done", clock);
  timer_Reset(clock);
  
  fprintf(context->outstream, "simplifying transition representation\n");

  model_ModifyReduceTransition(context->model, context->outstream, 
                               model_ReachabilityQAll(context->model));
  timer_Display(context->outstream, "done", clock);

  timer_Dispose(clock);
}

/* -------------------------------------------- _check_model -- */
static
void
_check_model
(cvc_context context)
{
  spec_Node_t p = spec_properties;
     
  fprintf(context->outstream, "Model checking started\n");
  while (p) {
    spec_BasicDisplay(stdout, p);
    fprintf(stdout, "\n");
    _check_property(context, p);
    p = spec_BasicQNext(p);
  }
}

/* ------------------------------------------ _check_property -- */
/* auxiliary routine */

void _check_property
(cvc_context context,
 spec_Node_t p)
{
  /* declaration part */
  char * message;
  int valid;
  ctl_term term;
  timer time_counter;

  /* statement part */
  time_counter = timer_New();
  timer_Start(time_counter);
  message = (char *) 
    mem_get_block(sizeof(char) * (strlen("checking property ...\n") + 
                                  strlen(spec_BasicQName(p)) + 1));
  sprintf(message, "checking property %s...\n", spec_BasicQName(p));
  fprintf(context->outstream, message);
  mem_free_block(message);
  term = (ctl_term) spec_BasicQValue(p);
  valid = _check_property_aux(context, term);
  if (valid) {
    char * message;
    message = (char *)
      mem_get_block((SIZE_T) sizeof(char) *
                    (strlen(spec_BasicQName(p)) +
                      strlen("++ property %s is valid ++") + 1));
    sprintf(message, "++ property %s is valid ++",spec_BasicQName(p));
    timer_Display(context->outstream, message, time_counter);
    mem_free_block(message);
  } else {
    char * message;
    message = (char *)
      mem_get_block((SIZE_T) sizeof(char) *
                    (strlen(spec_BasicQName(p)) +
                      strlen("-- property %s is invalid --") + 1));
    sprintf(message, "-- property %s is invalid --",spec_BasicQName(p));
    timer_Display(context->outstream, message, time_counter);
    mem_free_block(message);
    if (context->opt_vhdl_testbench) {
      char * identifier, * filename;
      cvn entity = qPrimary(spec_design);
      cvn signal;

      indent = 0;
      identifier = calloc(strlen(qName(entity)) +
                          strlen(qName(spec_design)) +
                          strlen(spec_BasicQName(p)) + 3,
                          sizeof(char));
      filename = calloc(strlen(".vhdl") +
                        strlen(qName(entity)) +
                        strlen(qName(spec_design)) +
                        strlen(spec_BasicQName(p)) + 3,
                        sizeof(char));
      sprintf(identifier, "%s_%s_%s", 
              qName(entity), 
              qName(spec_design), 
              spec_BasicQName(p));
      sprintf(filename, "%s-%s-%s.vhdl", 
              qName(entity), 
              qName(spec_design), 
              spec_BasicQName(p));
      vhdl_stream = fopen(filename, "w");

      fprintf(vhdl_stream, "\nentity TESTBENCH is\n");
      fprintf(vhdl_stream, "end;\n\n");

      fprintf(vhdl_stream, "architecture %s of TESTBENCH is\n", identifier);
      indent_inc();

      /* Signal Declarations */

      signal = qPorts(entity);
      while(signal) {
        if(qKind(signal) == kSignalDeclaration) {
          space_print(vhdl_stream);
          fprintf(vhdl_stream, "signal %s : %s := %s;\n", qName(signal),
                  qName(qDeclaration(qSubtype(signal))),
                  qName(qDefaultValue(signal)));
        } else {
          Message("Compiler Error in Ports", xxError, NoPosition);
        }
        signal = qNext(signal);
      }
      fprintf(vhdl_stream, "\n");

      space_print(vhdl_stream);
      fprintf(vhdl_stream, "component %s\n", qName(entity));
      indent_inc(); space_print(vhdl_stream);
      fprintf(vhdl_stream, "port(");

      /* Ports Declaration */

      signal = qPorts(entity);
      while(signal) {
        if(qKind(signal) == kSignalDeclaration) {
          fprintf(vhdl_stream, "%s: %s %s", qName(signal),
                  port_dir[qMode(signal)], qName(qDeclaration(qSubtype(signal))));
        } else {
          Message("Compiler Error in Ports", xxError, NoPosition);
        }
        signal = qNext(signal);
        if(signal) fprintf(vhdl_stream, "; ");
      }

      fprintf(vhdl_stream, ");\n");
      indent_dec(); space_print(vhdl_stream);
      fprintf(vhdl_stream, "end component;\n");

      space_print(vhdl_stream);
      fprintf(vhdl_stream, "for all: %s use entity work.%s(%s);\n",
              qName(entity),
              qName(entity),
              qName(spec_design));
      indent_dec(); space_print(vhdl_stream);
      fprintf(vhdl_stream, "begin\n");
      indent_inc(); space_print(vhdl_stream);
      fprintf(vhdl_stream, "TEST: %s port map(", qName(entity));
  
      /* Port Map with Signals */

      signal = qPorts(entity);
      while(signal) {
        if(qKind(signal) == kSignalDeclaration) {
          fprintf(vhdl_stream, "%s", qName(signal));
        } else {
          Message("Compiler Error in Ports", xxError, NoPosition);
        }
        signal = qNext(signal);
        if(signal) fprintf(vhdl_stream, ", ");
      }

      fprintf(vhdl_stream, ");\n\n");
    
    /* Signal Stimulus */

      space_print(vhdl_stream);
      fprintf(vhdl_stream, "stimulus: process\n");
      space_print(vhdl_stream);
      fprintf(vhdl_stream, "begin\n");
    }
    if(context->opt_witness_counterexample) {
      _DiagnoseCounterExample(context, term);
    }
    if (context->opt_vhdl_testbench) {
      indent_dec(); space_print(vhdl_stream);
      fprintf(vhdl_stream, "    wait;\n");
      fprintf(vhdl_stream, "  end process;\n");
      fprintf(vhdl_stream, "end;\n");
      fclose(vhdl_stream);
    }
  }
  timer_Dispose(time_counter);
}

static
bdd
terminal_fn
(void * p,
 void * e)
{
  return (bdd) qToolInfo((cvn) p);
}

int _check_property_aux
(cvc_context context,
 ctl_term t)
{
  if (ctl_BasicQType(t) == ctl_UnaryTerm) {
    ctl_Operator_t op;
    op = ctl_BasicQOperator(t);
    if ((op == ctl_AG) || (op == ctl_EF)) {
      ctl_term arg;
      arg = ctl_BasicQArgument(t);
      model_Check(context->model, context->outstream, arg, 
                  (terminal_check_fn) & terminal_fn, context->expr_mgr);
      if (op == ctl_AG) {
        return (bdd_imply(bddm, model_ReachabilityQAll(context->model),
                          model_CheckQSolution(ctl_BasicQInfo(arg)))
                == bdd_one(bddm));
      } else {
        return (bdd_and(bddm, model_ReachabilityQAll(context->model),
                        model_CheckQSolution(ctl_BasicQInfo(arg)))
                != bdd_zero(bddm));
      }
    }
  }
  model_Check(context->model, context->outstream,
              t, (terminal_check_fn) & terminal_fn, context->expr_mgr);
  return (bdd_zero(bddm) ==
          bdd_implies(bddm, model_RepresentationQInitialState(context->model),
                      model_CheckQSolution(ctl_BasicQInfo(t))));
}

/*--------------------------------------------------------------------*/
static const int BEGIN_PROOF_LABEL = 0;
static const int END_PROOF_LABEL = 1;

/** Function **
  Synopsis    [ Displays a counterexample of a CTL formula ]
  Description [ The CTL formula should not be valid in the state s
  of the model m. _DiagnoseCounterExample prints a counterexample
  path to the stream context->outstream. It uses auxialiary routines to
  recurse over the structure of the formula. ]
  SideEffects [ Modifies the local counter used to guarantee unique
                labeling of proofs and states. ]
  SeeAlso     [ ]
*/
void
_DiagnoseCounterExample
(cvc_context context /* stream where the counterexample is displayed (must be writable) */, 
 ctl_term t /* CTL term */)
{
  bdd config;
  char * l;
  l = label_NewLabel(context->proof_label_mgr);

  if ((ctl_BasicQType(t) == ctl_UnaryTerm) &&
      (ctl_BasicQOperator(t) == ctl_EF)) {
    return;
  } else if ((ctl_BasicQType(t) == ctl_UnaryTerm) &&
      (ctl_BasicQOperator(t) == ctl_AG)) {
  /* An "AG f" property is not valid.
   * First find an initial configuration "i" where "AG f" is not valid
   * (that determines initial values of the inputs).
   * Then compute a path from "i" to a configuration "c" where "f" is not
   * valid. 
   * Display path from "i" to "c" and show that "f" is not valid in "c".
   */
    bdd not_f_cond;
    ModelPath path;
    ctl_term arg;
    arg = ctl_BasicQArgument(t);
    not_f_cond = 
      bdd_andup2(bddm,
                 model_ReachabilityQAll(context->model),
                 bdd_not(bddm, model_CheckQSolution(ctl_BasicQInfo(arg))));
    config = model_ExplicitComputeCondition(context->model, not_f_cond, MODEL_ALLMASK);
    path = model_WitnessConfig(context->model, config);
    if (context->opt_witness_display) {
      ctl_DisplayText(context->outstream, t, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, 
              "\n"
              "is not valid since there is a path that leads to a state where\n");
      ctl_DisplayText(context->outstream, ctl_BasicQArgument(t), 
                      (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream,
              "\n"
              "is not valid (%s).\n", l);
      model_ExplicitDisplayPath(context->model, context->outstream, path);
    }
    if (context->opt_vhdl_testbench) {
      model_ExplicitDisplayVHDL(context->model, vhdl_stream, path);
    }
    _CounterExample(context, model_ExplicitQPathDestination(path), 
                    ctl_BasicQArgument(t), l);
    bdd_free(bddm, config);
    model_ExplicitFreePath(path);
  } else {
    config = bdd_andup2(bddm, model_RepresentationQInitialState(context->model), 
                        bdd_not(bddm, model_CheckQSolution(ctl_BasicQInfo(t))));
    _CounterExample(context, config, t, l);
  }
  label_FreeLabel(context->proof_label_mgr, l);
  bdd_free(bddm, config);
}

/** Function **
  Synopsis    [ Auxiliary routine to model_DiagnoseCounterExample ]
  Description [ According to the type of CTL formula, calls the
                appropriate specialized routine.]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
static void
_CounterExample
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label)
{
  switch (ctl_BasicQType(f)) {
  case ctl_AtomicTerm:
    _VHDL(context, s, f, label, 0);
    break;
  case ctl_UnaryTerm:
    switch (ctl_BasicQOperator(f)) {
    case ctl_Not:
      _Not(context, s, f, label, 0);
      break;
    case ctl_EX:
      _NotAvailable(context, label);
      break;
    case ctl_AX: 
      _EX(context, s, f, label, 0);
      break;
    case ctl_EG:
      _NotAvailable(context, label);
      break;
    case ctl_AG:
      _EF(context, s, f, label, 0);
      break;
    case ctl_EF:
      _NotAvailable(context, label);
      break;
    case ctl_AF:
      _EG(context, s, f, label, 0);
      break;
    default:
      break;
    }
  case ctl_BinaryTerm:
    switch (ctl_BasicQOperator(f)) {
    case ctl_Implies:
      _Imply(context, s, f, label);
      break;
    case ctl_And:
      _Or(context, s, f, label, 0);
      break;
    case ctl_Or:
      _And(context, s, f, label, 0);
      break;
    case ctl_Xor:
      _Xnor(context, s, f, label, 0);
      break;
    case ctl_Xnor:
      _Xor(context, s, f, label, 0);
      break;
    case ctl_Nand:
      _Nor(context, s, f, label, 0);
      break;
    case ctl_Nor:
      _Nand(context, s, f, label, 0);
	break;
    case ctl_EU:
      _NotAvailable(context, label);
      break;
    case ctl_AU:
      _EW(context, s, f, label, 0);
      break;
    case ctl_EW:
      _NotAvailable(context, label);
      break;
    case ctl_AW:
      _EU(context, s, f, label, 0);
      break;
    default:
      break;
    }
    break;
  case ctl_MacroTerm:
    _Macro(context, s, f, label, 0);
    break;
  }
}

/** Function **
  Synopsis    [ Auxiliary routine to model_DiagnoseCounterExample ]
  Description [ According to the type of CTL formula, calls the
                appropriate specialized routine.]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
static void
_Witness
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label)
{
  switch (ctl_BasicQType(f)) {
  case ctl_AtomicTerm:
    _VHDL(context, s, f, label, 1);
    break;
  case ctl_UnaryTerm:
    switch (ctl_BasicQOperator(f)) {
    case ctl_Not:
      _Not(context, s, f, label, 1);
      break;
    case ctl_EX:
      _EX(context, s, f, label, 1);
      break;
    case ctl_AX: 
      _NotAvailable(context, label);
      break;
    case ctl_EG:
      _EG(context, s, f, label, 1);
      break;
    case ctl_AG:
      _NotAvailable(context, label);
      break;
    case ctl_EF:
      _EF(context, s, f, label, 1);
      break;
    case ctl_AF:
      _NotAvailable(context, label);
      break;
    default:
      break;
    }
  case ctl_BinaryTerm:
    switch (ctl_BasicQOperator(f)) {
    case ctl_Implies:
      _Imply(context, s, f, label);
      break;
    case ctl_And:
      _And(context, s, f, label, 1);
      break;
    case ctl_Or:
      _Or(context, s, f, label, 1);
      break;
    case ctl_Xor:
      _Xor(context, s, f, label, 1);
      break;
    case ctl_Xnor:
      _Xnor(context, s, f, label, 1);
      break;
    case ctl_Nand:
      _Nand(context, s, f, label, 1);
      break;
    case ctl_Nor:
      _Nor(context, s, f, label, 1);
	break;
    case ctl_EU:
      _EU(context, s, f, label, 1);
      break;
    case ctl_AU:
      _NotAvailable(context, label);
      break;
    case ctl_EW:
      _EW(context, s, f, label, 1);
      break;
    case ctl_AW:
      _NotAvailable(context, label);
      break;
    default:
      break;
    }
    break;
  case ctl_MacroTerm:
    _Macro(context, s, f, label, 1);
    break;
  }
}

/** Function **
  Synopsis    [ Display message for proofs not available. ]
  Description [ Displaying a witness (counterexample) of a 
                universal (existential) formula would generate
                huge amounts of data and probably take too much
                time. Instead a laconic N/A message is printed. ]
  SideEffects [ Data is sent to context->outstream ]
  See Also    []
*/
static void 
_NotAvailable (
  cvc_context context,
  char * label)
{
  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  fprintf(context->outstream, "\tN/A\n");
  _BannerLimitProof(context, label, END_PROOF_LABEL);
}

/** Function **
  Synopsis    [ Prints a witness or a counterexample for a VHDL expression ]
  Description [ Prints to context->outstream why the CTL formula f that
                consists of a VHDL expression is or is not valid in the
                state s using the string opt to distinguish both
                cases. 
                This proof is identified with label. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
static void 
_VHDL
(cvc_context context, 
 bdd s,
 ctl_term f, 
 char * label, 
 int positive)
{
  char * opt;
  ModelConfig config;
  opt = positive ? "" : "not ";

  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if (context->opt_witness_display) {
    config = model_ExplicitNewConfig(context->model, s, MODEL_ALLMASK);
    fprintf(context->outstream, 
            "In the propositional logic, it is %sthe case that\n", opt);
    model_ExplicitDisplayConfig(context->outstream, config);
    fprintf(context->outstream,
            " => \n");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\n");
    model_ExplicitFreeConfig(config);
  }
  _BannerLimitProof(context, label, END_PROOF_LABEL);
}

/** Function **
  Synopsis    [ Prints a witness or a counterexample for a macro ]
  Description [ Prints to context->outstream why the CTL macro f that
  is or is not valid in the state s using the string opt to
  distinguish both cases. This proof is identified with label. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
static void 
_Macro
(cvc_context context, 
 bdd s,
 ctl_term f, 
 char * label, 
 int positive)
{
  char * opt;
  ModelConfig config;
  opt = positive ? "" : "not ";

  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if (context->opt_witness_display) {
    config = model_ExplicitNewConfig(context->model, s, MODEL_ALLMASK);
    fprintf(context->outstream, 
            "It is %sthe case that\n", opt);
    model_ExplicitDisplayConfig(context->outstream, config);
    fprintf(context->outstream,
            " is \n");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\n");
    model_ExplicitFreeConfig(config);
  }
  _BannerLimitProof(context, label, END_PROOF_LABEL);
}

/** Function **
  Synopsis    [ Prints a witness or a counterexample for a negative CTL term ]
  Description [ The CTL formula must have the operator not and contain a
                embedded temporal operator. If positive is 0, this formula 
                should not hold in the state s of the model m, otherwise
                it should hold.
                A proof identified with label is printed to context->outstream. ]
  SideEffects [ The label counter is incremented.
                Output is sent to context->outstream. ]
  SeeAlso     [ ]
 */
static void
_Not
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label,
 int positive)
{
  if (positive) {
    _CounterExample(context, s, ctl_BasicQArgument(f), label);
  } else {
    _Witness(context, s, ctl_BasicQArgument(f), label);
  }
}

/** Function **
  Synopsis    [ Prints a witness or a counterexample for a "=>" CTL term ]
  Description [ The CTL formula must have the operator "=>" and contain an
                embedded temporal operator. If positive is 0, this formula 
                should not hold in the state s of the model m, otherwise
                it should hold.
                A proof identified with label is printed to context->outstream. ]
  SideEffects [ The label counter is incremented.
                Output is sent to context->outstream. ]
  SeeAlso     [ ]
 */
static void
_Imply
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label)
{
  char * l1, * l2;
  l1 = label_NewLabel(context->proof_label_mgr);
  l2 = label_NewLabel(context->proof_label_mgr);

  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if(context->opt_witness_display) {
    fprintf(context->outstream, "The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis not valid since\n\t");
    ctl_DisplayText(context->outstream, ctl_BasicQArgument(f), (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis valid (%s) and \n\t", l1);
    ctl_DisplayText(context->outstream, ctl_BasicQArgument2(f), (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis not valid in (%s)\n", l2);
  }
  _BannerLimitProof(context, label, END_PROOF_LABEL);
  _Witness(context, s, ctl_BasicQArgument(f), l1);
  _CounterExample(context, s, ctl_BasicQArgument2(f), l2);
  label_FreeLabel(context->proof_label_mgr, l1);
  label_FreeLabel(context->proof_label_mgr, l2);
}

/** Function **
  Synopsis    [ Witness (counterexample) of a OR (AND) property ]
  Description [ The CTL formula must have either the operator OR or
  AND.  s is a state of the model m: f = g OR h (f = g AND h). The 
  fields node_info of f, g and h must contain the result of the model
  checking computation.
  If positive is null (non null) the operator is OR (AND) and the proof
  is a witness (counterexample). If the operator is OR (AND), the term
  should (not) be valid in s.  
  A proof identified with label is printed to context->outstream.
  A formula f = g OR h (g AND h) is valid (not valid) 
  thus there is a witness (counterexample) to g (not g) or there is 
  a witness (counterexample) to h (not h). ]
  SideEffects [ The label counter is incremented.
  Output is sent to context->outstream. ]
  SeeAlso     [ optional ]
 */
static void
_Or
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label,
 int positive)
{
  char * opt;
  char * newlabel;
  ctl_term gterm, hterm;
  bdd g, h, tmp;

  opt = positive ? "" : "not ";
  newlabel = label_NewLabel(context->proof_label_mgr);
  gterm = ctl_BasicQArgument(f);
  hterm = ctl_BasicQArgument2(f);
  g = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(gterm));
  h = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(hterm));
  if (!positive) {
    g = bdd_not(bddm, g);
  }
  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if(context->opt_witness_display) {
    fprintf(context->outstream, "The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid since\n\t", opt);
  }
  if ((tmp = bdd_and(bddm, s, g)) == bdd_zero(bddm)) {
    /*         OR:                            AND:
     *  s |= g or h, s |/= g           s|/= g and h, s|=g
     *  --------------------           -------------------
     *        s |= h                         s |/= h
     */
    if(context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, hterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt, newlabel);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _Witness(context, s, hterm, newlabel);
    } else {
      _CounterExample(context, s, hterm, newlabel);
    }
  } else {
    /*         OR:                            AND:
     *  s |= g or h, s |= g           s|/= g and h, s|/=g
     */
    bdd_free(bddm, tmp);
    if(context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, gterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt, newlabel);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _Witness(context, s, gterm, newlabel);
    } else {
      _CounterExample(context, s, gterm, newlabel);
    }
  }
  label_FreeLabel(context->proof_label_mgr, newlabel);
  if (!positive) {
    bdd_free(bddm, g);
  }
}

/** Function **
  Synopsis    [ Witness (counterexample) of a XNOR (XOR) property ]
  Description [ The CTL formula must have either the operator XNOR or
  XOR.  s is a state of the model m: f = g XNOR h (f = g XOR h). The 
  fields node_info of f, g and h must contain the result of the model
  checking computation.
  If positive is null (non null) the operator is XNOR (XOR) and the proof
  is a witness (counterexample). If the operator is XNOR (XOR), the term
  should (not) be valid in s.  
  A proof identified with label is printed to context->outstream.
  A formula f = g XNOR h (g XOR h) is valid (not valid) 
  thus either there is a witness (counterexample) to g (not g) and there 
  is a witness (counterexample) to h (not h), or there is a counterexample 
  (witness) to g (not g) and there is a counterexample (witness) to h 
  (not h). ]
  SideEffects [ The label counter is incremented.
  Output is sent to context->outstream. ]
  SeeAlso     [ optional ]
 */
static void
_Xnor
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label,
 int positive)
{
  char * opt1, * opt2;
  char * newlabel1, * newlabel2;
  ctl_term gterm, hterm;
  bdd g, h, tmp;

  opt1 = positive ? "" : "not ";
  opt2 = positive ? "not " : "";

  newlabel1 = label_NewLabel(context->proof_label_mgr);
  newlabel2 = label_NewLabel(context->proof_label_mgr);
  gterm = ctl_BasicQArgument(f);
  hterm = ctl_BasicQArgument2(f);
  g = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(gterm));
  h = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(hterm));
  if (!positive) {
    g = bdd_not(bddm, g);
    h = bdd_not(bddm, h);
  }

  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if(context->opt_witness_display) {
    fprintf(context->outstream, "The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid since\n\t", opt1);
  }
  if ((tmp = bdd_and(bddm, s, g)) == bdd_zero(bddm)) {
    /*         XNOR:                            XOR:
     *  s |= g xor h, s |/= g           s|/= g xor h, s|=g
     *  ---------------------           ------------------
     *        s |/= h                         s |= h
     */
    if(context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, gterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt2, newlabel1);
      fprintf(context->outstream, "and the formula\n\t");
      ctl_DisplayText(context->outstream, hterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt2, newlabel2);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _CounterExample(context, s, gterm, newlabel1);
      _CounterExample(context, s, hterm, newlabel2);
    } else {
      _Witness(context, s, gterm, newlabel1);
      _Witness(context, s, hterm, newlabel2);
    }
  } else {
    /*         XNOR:                            XOR:
     *  s |= g xnor h, s |/= g           s|/= g xor h, s|=g
     *  ----------------------           --------------------
     *        s |/= h                         s |= h
     */
    bdd_free(bddm, tmp);
    if(context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, gterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt1, newlabel1);
      fprintf(context->outstream, "and the formula\n\t");
      ctl_DisplayText(context->outstream, hterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt1, newlabel2);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _Witness(context, s, gterm, newlabel1);
      _Witness(context, s, hterm, newlabel2);
    } else {
      _CounterExample(context, s, gterm, newlabel1);
      _CounterExample(context, s, hterm, newlabel2);
    }
  }
  label_FreeLabel(context->proof_label_mgr, newlabel1);
  label_FreeLabel(context->proof_label_mgr, newlabel2);
  if (!positive) {
    bdd_free(bddm, g);
  }
}

/** Function **
  Synopsis    [ Witness (counterexample) of a XOR (XNOR) property ]
  Description [ The CTL formula must have either the operator XOR or
  XNOR.  s is a state of the model m: f = g XOR h (f = g XNOR h). The 
  fields node_info of f, g and h must contain the result of the model
  checking computation.
  If positive is null (non null) the operator is XOR (XNOR) and the proof
  is a witness (counterexample). If the operator is XOR (XNOR), the term
  should (not) be valid in s.  
  A proof identified with label is printed to context->outstream.
  A formula f = g XOR h (g XNOR h) is valid (not valid) 
  thus either there is a witness (counterexample) to g (not g) and there 
  is a counterexample (witness) to h (not h), or there is a counterexample 
  (witness) to g (not g) and there is a witness (counterexample)
  to h (not h). ]
  SideEffects [ The label counter is incremented.
  Output is sent to context->outstream. ]
  SeeAlso     [ optional ]
 */
static void
_Xor
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label,
 int positive)
{
  char * opt1, * opt2;
  char * newlabel1, * newlabel2;
  ctl_term gterm, hterm;
  bdd g, h, tmp;

  opt1 = positive ? "" : "not ";
  opt2 = positive ? "not " : "";

  newlabel1 = label_NewLabel(context->proof_label_mgr);
  newlabel2 = label_NewLabel(context->proof_label_mgr);
  gterm = ctl_BasicQArgument(f);
  hterm = ctl_BasicQArgument2(f);
  g = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(gterm));
  h = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(hterm));
  if (!positive) {
    g = bdd_not(bddm, g);
    h = bdd_not(bddm, h);
  }

  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if(context->opt_witness_display) {
    fprintf(context->outstream, "The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid since\n\t", opt1);
  }
  if ((tmp = bdd_and(bddm, s, g)) == bdd_zero(bddm)) {
    /*         XOR:                            XNOR:
     *  s |= g xor h, s |/= g           s|/= g xnor h, s|=g
     *  --------------------           -------------------
     *        s |= h                         s |/= h
     */
    if(context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, gterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt2, newlabel1);
      fprintf(context->outstream, "and the formula\n\t");
      ctl_DisplayText(context->outstream, hterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt1, newlabel2);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _CounterExample(context, s, gterm, newlabel1);
      _Witness(context, s, hterm, newlabel2);
    } else {
      _Witness(context, s, gterm, newlabel1);
      _CounterExample(context, s, hterm, newlabel2);
    }
  } else {
    /*         XOR:                            XNOR:
     *  s |= g xor h, s |= g           s|/= g xnor h, s|/=g
     *  --------------------           --------------------
     *        s |/= h                         s |= h
     */
    bdd_free(bddm, tmp);
    if(context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, gterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt1, newlabel1);
      fprintf(context->outstream, "and the formula\n\t");
      ctl_DisplayText(context->outstream, hterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt2, newlabel2);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _Witness(context, s, gterm, newlabel1);
      _CounterExample(context, s, hterm, newlabel2);
    } else {
      _CounterExample(context, s, gterm, newlabel1);
      _Witness(context, s, hterm, newlabel2);
    }
  }
  label_FreeLabel(context->proof_label_mgr, newlabel1);
  label_FreeLabel(context->proof_label_mgr, newlabel2);
  if (!positive) {
    bdd_free(bddm, g);
  }
}

/** Function **
  Synopsis    [ Witness (counterexample) of a AND (OR) property ]
  Description [ The CTL formula must have either the operator AND or
  OR.  s is a state of the model m: f = g AND h. The fields node_info
  of f, g or h must contain the result of the model checking computation.
  If positive is null (non null) the operator is AND (OR) and the proof
  is a witness (counterexample). If the operator is AND (OR), the term
  should (not) be valid in s.  
  A proof identified with label is printed to context->outstream.
  A formula f = g AND h (g OR h) is valid (not valid) 
  thus there is a witness (counterexample) to g (not g) and there is 
  a witness (counterexample) to h (not h). ]
  SideEffects [ The label counter is incremented.
  Output is sent to context->outstream. ]
  SeeAlso     [ optional ]
 */
static void
_And
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label,
 int positive)
{
  char * opt;
  char * newlabel1, * newlabel2;
  ctl_term gterm, hterm;

  opt = positive ? "" : "not ";
  newlabel1 = label_NewLabel(context->proof_label_mgr);
  newlabel2 = label_NewLabel(context->proof_label_mgr);

  gterm = ctl_BasicQArgument(f);
  hterm = ctl_BasicQArgument2(f);

  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if(context->opt_witness_display) {
    fprintf(context->outstream, "The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid since\n\t", opt);
    fprintf(context->outstream, "the formula\n\t");
    ctl_DisplayText(context->outstream, gterm, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid (%s) and the formula\n\t", opt, newlabel1);
    ctl_DisplayText(context->outstream, hterm, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid (%s)\n", opt, newlabel2);
  }
  _BannerLimitProof(context, label, END_PROOF_LABEL);
  if (positive) {
    _Witness(context, s, gterm, newlabel1);
    _Witness(context, s, hterm, newlabel2);
  } else {
    _CounterExample(context, s, gterm, newlabel1);
    _CounterExample(context, s, hterm, newlabel2);
  }
  label_FreeLabel(context->proof_label_mgr, newlabel1);
  label_FreeLabel(context->proof_label_mgr, newlabel2);
}

/** Function **
  Synopsis    [ Witness (counterexample) of a NAND (NOR) property ]
  Description [ The CTL formula must have either the operator NAND or
  NOR.  s is a state of the model m: f = g NAND h. The fields node_info
  of f, g or h must contain the result of the model checking computation.
  If positive is null (non null) the operator is NAND (NOR) and the proof
  is a witness (counterexample). If the operator is NAND (NOR), the term
  should (not) be valid in s.  
  A proof identified with label is printed to context->outstream.
  f = g NAND h (g NOR h) is valid (not valid) thus there is a 
  counterexample (witness) to g (not g) or there is a counterexample 
  (witness) to h (not h). ]
  SideEffects [ The label counter is incremented.
  Output is sent to context->outstream. ]
  SeeAlso     [ optional ]
 */
static void
_Nand
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label,
 int positive)
{
  char * opt2;
  char * newlabel;
  ctl_term gterm, hterm;
  bdd g, h, tmp;

  opt2 = positive ? "not " : "";
  newlabel = label_NewLabel(context->proof_label_mgr);
  gterm = ctl_BasicQArgument(f);
  hterm = ctl_BasicQArgument2(f);
  g = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(gterm));
  h = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(hterm));
  if (!positive) {
    g = bdd_not(bddm, g);
    h = bdd_not(bddm, h);
  }

  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if(context->opt_witness_display) {
    fprintf(context->outstream, "The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid since\n\t", opt2);
  }
  if ((tmp = bdd_and(bddm, s, g)) == bdd_zero(bddm)) {
    /*         NAND:                           NOR:
     *  s |= g nand h, s |/= g           s|/= g nor h, s|=g
     */
    if(context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, hterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt2, newlabel);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _CounterExample(context, s, gterm, newlabel);
    } else {
      _Witness(context, s, gterm, newlabel);
    }
  } else {
    /*         OR:                            AND:
     *  s |= g or h, s |= g           s|/= g and h, s|/=g
     *  -------------------           -------------------
     *        s |= h                         s |/= h
     */
    bdd_free(bddm, tmp);
    fprintf(context->outstream, "the formula\n\t");
    ctl_DisplayText(context->outstream, gterm, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid (%s)\t", opt2, newlabel);
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _CounterExample(context, s, hterm, newlabel);
    } else {
      _Witness(context, s, hterm, newlabel);
    }
  }
  label_FreeLabel(context->proof_label_mgr, newlabel);
  if (!positive) {
    bdd_free(bddm, g);
  }
}

/** Function **
  Synopsis    [ Witness (counterexample) of a NOR (NAND) property ]
  Description [ The CTL formula must have either the operator NOR or
  NAND.  s is a state of the model m: f = g NOR h (f = g NAND h). The 
  fields node_info of f, g and h must contain the result of the model
  checking computation.
  If positive is null (non null) the operator is NOR (NAND) and the proof
  is a witness (counterexample). If the operator is NOR (NAND), the term
  should (not) be valid in s.  
  A proof identified with label is printed to context->outstream.
  A formula f = g NOR h (g NAND h) is valid (not valid) 
  thus there is a counterexample (witness) to g (not g) or there is 
  a counterexample (witness) to h (not h). ]
  SideEffects [ The label counter is incremented.
  Output is sent to context->outstream. ]
  SeeAlso     [ optional ]
 */
static void
_Nor
(cvc_context context, 
 bdd s, 
 ctl_term f, 
 char * label,
 int positive)
{
  char * opt, * opt2;
  char * newlabel;
  ctl_term gterm, hterm;
  bdd g, h, tmp;

  opt = positive ? "" : "not ";
  opt2 = positive ? "not " : "";
  newlabel = label_NewLabel(context->proof_label_mgr);
  gterm = ctl_BasicQArgument(f);
  hterm = ctl_BasicQArgument2(f);
  g = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(gterm));
  h = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(hterm));
  if (!positive) {
    g = bdd_not(bddm, g);
    h = bdd_not(bddm, h);
  }
  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  if (context->opt_witness_display) {
    fprintf(context->outstream, "The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis %svalid since\n\t", opt);
  }
  if ((tmp = bdd_and(bddm, s, g)) == bdd_zero(bddm)) {
    /*         NOR:                            NAND:
     *  s |= g nor h, s |/= g           s|/= g nand h, s|=g
     *                                     
     */
    if (context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, gterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt2, newlabel);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _CounterExample(context, s, gterm, newlabel);
    } else {
      _Witness(context, s, gterm, newlabel);
    }
  } else {
    /*         NOR:                            NAND:
     *  s |= g nor h, s |/= g            s|/= g nand h, s|=g
     *  ---------------------            --------------------
     *        s |/= h                         s |= h
     */
    bdd_free(bddm, tmp);
    if (context->opt_witness_display) {
      fprintf(context->outstream, "the formula\n\t");
      ctl_DisplayText(context->outstream, hterm, (ctl_TerminalDisplayFn_t) &cvn_display);
      fprintf(context->outstream, "\nis %svalid (%s)\t", opt2, newlabel);
    }
    _BannerLimitProof(context, label, END_PROOF_LABEL);
    if (positive) {
      _CounterExample(context, s, hterm, newlabel);
    } else {
      _Witness(context, s, hterm, newlabel);
    }
  }
  label_FreeLabel(context->proof_label_mgr, newlabel);
  if (!positive) {
    bdd_free(bddm, g);
    bdd_free(bddm, h);
  }
}

/** Function **
  Synopsis    [ Prints a counterexample (witness) of a AX (EX) term ]
  Description [ The CTL formula must have either the operator EX or AX.
                s is a state of the model m: f = EX g.
                The fields node_info of f and g must contain the result
                of the model checking computation.
                If positive is null (non null) the operator is EX (AX)
                and the proof is a witness (counterexample).
                If the operator is AX, the term should not be valid in s.
                If the operator is EX, the term should be valid in s.
                A proof identified with label is printed to context->outstream.
                The proof consists of an input vector that makes m 
                transition from s to a state s', and to show that g is 
                not valid in s', if the operator is AX, or to show that 
                g is valid in s', it the operator is EX. ]
  SideEffects [ The label counter is incremented.
                Output is sent to context->outstream. ]
  SeeAlso     [ ]
 */
static void
_EX
(cvc_context context, 
 bdd s, 
 ctl_term f,
 char * label, 
 int positive)
{

  /* declarations */
  ModelPath witness;    /* the path exhibiting counterexample */
  char * newlabel;  /* label of subproof */
  bdd g;
  char * opt;
  opt = positive ? "" : "not ";

  /* statements */
  g = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(ctl_BasicQArgument(f)));
  if (!positive) {
    g = bdd_not(bddm, g);
  }
  newlabel = label_NewLabel(context->proof_label_mgr);

  /* compute and print proof argument */
  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  witness = model_WitnessEX(context->model, s, g);
  if (positive) {
    bdd_free(bddm, g);
  }
  if (context->opt_witness_display) {
    fprintf(context->outstream,"The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, 
            "\nis %svalid since there is a successor where the formula\n\t", opt);
    ctl_DisplayText(context->outstream, ctl_BasicQArgument(f), (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "is %svalid (%s)", opt, newlabel);
    model_ExplicitDisplayPath(context->model, context->outstream, witness);
  }
  if (context->opt_vhdl_testbench) {
    model_ExplicitDisplayVHDL(context->model, vhdl_stream, witness);
  }
  /* sign off */
  _BannerLimitProof(context, label, END_PROOF_LABEL);

  /* perform the subproof */
  if (positive) {
    _Witness(context, model_ExplicitQPathDestination(witness),
             ctl_BasicQArgument(f), newlabel);
  } else {
    _CounterExample(context, model_ExplicitQPathDestination(witness),
                    ctl_BasicQArgument(f), newlabel);
  }

  /* free allocated memory */
  label_FreeLabel(context->proof_label_mgr, newlabel);
  model_ExplicitFreePath(witness);
}

/** Function **
  Synopsis    [ Prints a counterexample (witness) of a AG (EF) term ]
  Description [ The CTL formula must have either the operator EF or AG.
                s is a state of the model m: f = EF g.
                The fields node_info of f and g must contain the result
                of the model checking computation.
                If positive is null (non null) the operator is EF (AG)
                and the proof is a witness (counterexample).
                If the operator is AG, the term should not be valid in s.
                If the operator is EF, the term should be valid in s.
                A proof identified with label is printed to context->outstream.
                The proof consists in a path that makes m go from to a 
                state s' (possibly in several transitions) where g is 
                valid and to show that g is valid in s', if the operator 
                is EF, or to show that g is not valid in s', if the 
                operator is AG. ]
  SideEffects [ The label counter is incremented.
                Output is sent to context->outstream. ]
  SeeAlso     [ ]
 */
static void
_EF
(cvc_context context, 
 bdd s, 
 ctl_term f,
 char * label, 
 int positive)
{

  /* declarations */
  ModelPath witness;    /* the path exhibiting counterexample */
  char * newlabel;  /* label of subproof */
  bdd g;
  vbdd fixpoint;
  char * opt;
  opt = positive ? "" : "not ";

  /* statements */
  g = model_CheckQSolution(
    (ModelCheckComputation) ctl_BasicQInfo(ctl_BasicQArgument(f))); 
  if (!positive) {
    g = bdd_not(bddm, g);
  }
  fixpoint = (vbdd) model_CheckQInfo((ModelCheckComputation) ctl_BasicQInfo(f));
  newlabel = label_NewLabel(context->proof_label_mgr);

  /* compute and print proof argument */
  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  witness = model_WitnessEU(context->model, s, bdd_one(bddm), g, fixpoint);
  if (!positive) {
    bdd_free(bddm, g);
  }
  if (context->opt_witness_display) {
    fprintf(context->outstream,"The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\n is %s valid since there is a path where the formula\n\t", opt);
    ctl_DisplayText(context->outstream, ctl_BasicQArgument(f), (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis eventually %svalid (proof %s):\n", opt, newlabel);
    model_ExplicitDisplayPath(context->model, context->outstream, witness);
  }
  if (context->opt_vhdl_testbench) {
    model_ExplicitDisplayVHDL(context->model, vhdl_stream, witness);
  }
  /* sign off */
  _BannerLimitProof(context, label, END_PROOF_LABEL);

  /* perform the subproof */
  if (positive) {
    _Witness(context, model_ExplicitQPathDestination(witness), 
             ctl_BasicQArgument(f), newlabel);
  } else {
    _CounterExample(context, 
                    model_ExplicitQPathDestination(witness), 
                    ctl_BasicQArgument(f), newlabel);
  }

  /* free allocated memory */
  label_FreeLabel(context->proof_label_mgr, newlabel);
  model_ExplicitFreePath(witness);
}

/** Function **
  Synopsis    [ Prints a counterexample (witness) of a AF (EG) term ]
  Description [ The CTL formula must have either the operator EG or AF.
                s is a state of the model m: f = EG g or f = AF g.
                The fields node_info of f and g must contain the result
                of the model checking computation.
                If positive is 0 (non 0) the operator is EG (AF) and the 
                proof is a witness (counterexample), and thus f should
                be valid (non valid) in s.
                A proof identified with label is printed to context->outstream.
                The proof consists in an infinite m-path that starts at
                s and where g is always valid, if the operator is EG,
                always invalid if the operator is AF. ]
  SideEffects [ The label counter is incremented.
                Output is sent to context->outstream. ]
  SeeAlso     [ ]
 */
static void
_EG
(cvc_context context, 
 bdd s, 
 ctl_term f,
 char * label, 
 int positive)
{

  /* declarations */
  ModelPath witness;    /* the path exhibiting counterexample */
  bdd bddg, bddf;
  vbdd * approximations;
  char * opt;
  opt = positive ? "" : "not ";

  /* statements */
  bddg = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(ctl_BasicQArgument(f)));
  bddf = model_CheckQSolution((ModelCheckComputation) ctl_BasicQInfo(f));
  approximations = (vbdd *)
    model_CheckQInfo((ModelCheckComputation) ctl_BasicQInfo(f));
  if (!positive) {
    bddg = bdd_not(bddm, bddg);
    bddf = bdd_andup2(bddm, model_ReachabilityQAll(context->model), bdd_not(bddm, bddf));
  }

  /* compute and print proof argument */
  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  witness = model_WitnessEG(context->model, context->outstream, s, bddg, bddf, approximations);
  if (!positive) {
    bdd_free(bddm, bddg);
  }
  if (context->opt_witness_display) {
    fprintf(context->outstream,"The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\n is %svalid since there is an infinite path where the formula\n\t", opt);
    ctl_DisplayText(context->outstream, ctl_BasicQArgument(f), (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nis continuously %svalid.\n", opt);
    model_ExplicitDisplayPath(context->model, context->outstream, witness);
  }
  if (context->opt_vhdl_testbench) {
    model_ExplicitDisplayVHDL(context->model, vhdl_stream, witness);
  }
  /* sign off */
  _BannerLimitProof(context, label, END_PROOF_LABEL);

  /* free allocated memory */
  model_ExplicitFreePath(witness);
}

/** Function **
  Synopsis    [ Prints a counterexample (witness) of a AW (EU) term ]
  Description [ The CTL formula must have either the operator EU or AW.
                s is a state of the model m: f = E\[gUh\] or f = A\[gWh\].
                The fields node_info of f and g must contain the result
                of the model checking computation.
                If positive is 0 (non 0) the operator is EU (AW) and the 
                proof is a witness (counterexample), and thus f should
                be valid (non valid) in s.
                A proof identified with label is printed to context->outstream.
                The proof consists in an finite m-path that starts at
                s and where g is always valid until a state where h
                is valid is reached, if the operator is EU, or until a 
                state where h is invalid is reached, if the operator is 
                AW. ]
  SideEffects [ The label counter is incremented.
                Output is sent to context->outstream. ]
  SeeAlso     [ ]
 */
static void
_EU
(cvc_context context, 
 bdd s, 
 ctl_term f,
 char * label, 
 int positive)
{

  /* declarations */
  ModelPath witness; /* the path exhibiting counterexample */
  char * newlabel;  /* label of subproof */
  bdd hsolution;
  vbdd fixpoint;
  char * opt;
  opt = positive ? "" : "not ";

  /* statements */
  hsolution =  model_CheckQSolution(
    (ModelCheckComputation) ctl_BasicQInfo(ctl_BasicQArgument2(f)));
  if (!positive) {
    hsolution = bdd_not(bddm, hsolution);
  }
  fixpoint = (vbdd)
    model_CheckQInfo((ModelCheckComputation) ctl_BasicQInfo(f));
  newlabel = label_NewLabel(context->proof_label_mgr);

  /* compute and print proof argument */
  _BannerLimitProof(context, label, BEGIN_PROOF_LABEL);
  witness = model_WitnessEU(context->model, s, bdd_one(bddm), hsolution, fixpoint);
  if (context->opt_witness_display) {
    fprintf(context->outstream,"The formula\n\t");
    ctl_DisplayText(context->outstream, f, (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, 
            "\nis %svalid since there is a path where the formula\n\t", 
            opt);
    ctl_DisplayText(context->outstream, ctl_BasicQArgument(f), (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, 
          "is valid until the formula\n");
    ctl_DisplayText(context->outstream, ctl_BasicQArgument2(f), (ctl_TerminalDisplayFn_t) &cvn_display);
    fprintf(context->outstream, "\nbecomes %svalid (proof %s):\n", opt, newlabel);
    model_ExplicitDisplayPath(context->model, context->outstream, witness);
  }
  if (context->opt_vhdl_testbench) {
    model_ExplicitDisplayVHDL(context->model, vhdl_stream, witness);
  }
  /* sign off */
  _BannerLimitProof(context, label, END_PROOF_LABEL);

  /* perform the subproof */
  if (positive) {
    _Witness(context, 
             model_ExplicitQPathDestination(witness), 
             ctl_BasicQArgument2(f), newlabel);
  } else {
    _CounterExample(context,
                    model_ExplicitQPathDestination(witness),
                    ctl_BasicQArgument2(f), newlabel);
  }

  /* free allocated memory */
  label_FreeLabel(context->proof_label_mgr, newlabel);
  model_ExplicitFreePath(witness);
}

/** Function **
  Synopsis    [ Prints a counterexample (witness) of a AU (EW) term ]
  Description [ The CTL formula must have either the operator EW or AU.
                s is a state of the model m: f = E\[gWh\] or f = A\[gUh\].
                The fields node_info of f and g must contain the result
                of the model checking computation.
                If positive is 0 (non 0) the operator is EW (AU) and the 
                proof is a witness (counterexample), and thus f should
                be valid (non valid) in s.
                A proof identified with label is printed to context->outstream.
                The proof consists in an finite m-path that starts at
                s and where g is always valid until a state where h
                is valid is reached, if the operator is EW, or until a 
                state where h is invalid is reached, if the operator is 
                AU. ]
  SideEffects [ The label counter is incremented.
                Output is sent to context->outstream. ]
  SeeAlso     [ ]
 */
static void
_EW
(cvc_context context, 
 bdd s, 
 ctl_term f,
 char * label, 
 int positive)
{
  _NotAvailable(context, label);
}

/** Function **
  Synopsis    [ Prints a banner to delimit a proof ]
  Description [ If l is equal to BEGIN_PROOF_LABEL sends a starting 
                banner for the proof named "label" to the stream 
                context->outstream, otherwise sends an ending banner for this 
                proof. ]
  SideEffects [ A string is sent to context->outstream using fputs. ]
  SeeAlso     [ ]
 */

static void _BannerLimitProof
(cvc_context context, 
 char * label, 
 int l)
{
  if (context->opt_witness_display) {
    fputs(" **", context->outstream);
    fputs((l == BEGIN_PROOF_LABEL) ? " begin " : " end ", context->outstream);
    fputs(label, context->outstream);
    fputs(" ** \n\n", context->outstream);
  }
}


/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


