/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    type.c                                                    */
/*                                                              */
/* Description:                                                 */
/*    routines and declarations that handle type related actions*/
/*    in the VHDL analyzer.                                     */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that compiles VHDL text descriptions into the   */
/*    internal format.                                          */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <string.h>

#include <Errors.h>

#include "actions.h"

/* ---------------------------------------- type_declaration --- */

/* cvn type_declaration
     (char * name, tPosition position, cvn type)

   . creates a kTypeDeclaration node;
   . sets the aName and aLine attributes, the nType child to the value 
   of the corresponding parameters;
   . set the Declaration child of definition to the created node;
   . if definition is an enumerated, sets the declaration child of
   each of the elements to the created node. */

cvn type_declaration
(char * name,
 int * nb_errors,
 tPosition position,
 cvn type)
{
  cvn declaration;
  cvn elem, subtype;

  subtype = mSubtype(0, 0, type, 0);
  declaration = mTypeDeclaration(0, position.Line, 0, 0, name, subtype);
  sDeclaration(subtype, declaration);
  sDeclaration(type, declaration);
  add_declarative_part(declaration);
  switch (qKind(type)) {
  case kEnumerationType:
    {
     /* unsigned ordinal_value = 0; */
      for (elem = qElements(type); elem; elem=qNext(elem)) {
        sDeclaration(qValue(elem), declaration);
      }
    }
    break;
  case kIntegerType:
    sSubtype(qLeft(qRange(type)), subtype);
    sSubtype(qRight(qRange(type)), subtype);
    break;
  default:
    Message("unknown type definition in type declaration", xxError, position);
    if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
    break;
  }
  return declaration;
}

/* ------------------------------------------- enumeration_type --- */

/* cvn enumeration_type (cvn list)

   . creates a kEnumerationType node;
   . sets the nElements child to enumeration_list;
   . for each element of nElements, sets the aOrdinalValue 
   attribute. */

cvn
enumeration_type
(cvn list)
{
  cvn type, elem;
  unsigned ordinal;
  type = mEnumerationType(0, 0, list);
  for (elem = list, ordinal = 0; elem; elem = qNext(elem), ++ordinal) {
    sOrdinalValue(qValue(elem), ordinal);
  }
  return type;
}

/* ------------------------------------------- integer_type_d ------- */

/* cvn integer_type_d (cvn range)

   . creates a kIntegerType node;
   . sets the nElements child to range; */

cvn
integer_type_d
(cvn range)
{
  cvn type;

  type = mIntegerType(0, 0, range);
  return type;
}

/* ------------------------------------------- type_compatible --- */

/* int type_compatible (cvn t1, cvn t2)

   . checks that the two types t1 and t2 are compatible.

   This routine will be replaced by a more complex treatment and,
   hopefully, more accurate implementation of VHDL typing rules
   in a future release. */


int
type_compatible
(cvn t1,
 cvn t2)
{
  if (t1 == t2) {
    return 1;
  }
  if (IsA(t1, kSubtype)) {
    return type_compatible(qBaseType(t1), t2);
  }
  if (IsA(t2, kSubtype)) {
    return type_compatible(t1, qBaseType(t2));
  }
  if (IsA(t1, kIdentifier) || 
      IsA(t1, kEnumeratedLiteral) || 
      IsA(t1, kCharacterLiteral)) {
    return type_compatible(qBaseType(qSubtype(qDeclaration(t1))), t2);
  }
  if (IsA(t2, kIdentifier) || 
      IsA(t2, kEnumeratedLiteral) || 
      IsA(t2, kCharacterLiteral)) {
    return type_compatible(t1, qBaseType(qSubtype(qDeclaration(t2))));
  }
  return 0;
}

/* ---------------------------------------- subtype_indication --- */

/* cvn subtype_indication (cvn identifier, cvn range)

      . implements (simplistically ?) the action performed on a
      subtype indication is parsed. */

cvn
subtype_indication
(cvn identifier,
 cvn range)
{
   cvn declaration ;

   declaration = get_declaration(name_match, qName(identifier)) ;
   if (range == 0)
     return qSubtype(declaration);
   else
     return mSubtype(0, 0, qSubtype(declaration), range);
}

/* ---------------------------------------- subtype_declaration ---- */

/* cvn subtype_declaration (char * name,
   tPosition pos, cvn subtype)
   make a subtype node */

cvn subtype_declaration
(char * name,
 int * nb_errors,
 tPosition pos,
 cvn subtype)
{
   cvn declaration;

   declaration = mSubtypeDeclaration(0, pos.Line, 0, 0,
                 name, subtype) ;
   sDeclaration(subtype, declaration);
   add_declarative_part(declaration);
   return declaration;
}

