/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    statement.c                                               */
/*                                                              */
/* Description:                                                 */
/*    Routines to create intermediate format nodes related to   */
/*    VHDL statements.                                          */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <memuser.h>

#include "Errors.h"

#include "actions.h"

/* ------------------------------------------------------------ */
static int _check_alternatives (cvn, cvn, int *, tPosition);
static int _check_sig_asg_target (cvn, int *, tPosition);
static char * string0 (void);

/* ------------------------------------- _check_alternatives -- */

/* int _check_alternatives
      (cvn selector, cvn alternatives, tPosition position)

      . performs checks for the alternatives in case statement
      and selected signal assignment statement. */

int
_check_alternatives
(cvn selector,
 cvn alternatives,
 int * nb_errors,
 tPosition position)
{
  /* checks performed:
     - type of selector is a discrete type,
     - for all alternatives, for all choices, type of choice
     is equal to type of selector, 
     - choices must be locally static expressions,
     - choices are exclusive,
     - choices are exhaustive. */
  cvn subtype = qSubtype(selector),
      constraint = qConstraint(subtype),
      basetype = qBaseType(subtype),
      alternative;
  char * flags; /* one entry per element in the type */
  unsigned size, idx;
  int lower, upper;
  unsigned error;
  
  if ((qKind(basetype) != kEnumerationType) &&
      (qKind(basetype) != kIntegerType)) {
    Message("base type of selector is not a scalar type", xxError, position);
    if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
    return 0;
  }

  if (IsA(basetype, kIntegerType)) {
    if (NullNode(constraint)) {
      lower = qIntegerValue(qLeft(basetype));
      upper = qIntegerValue(qRight(basetype));
    } else {
      lower = qIntegerValue(qLower(constraint));
      upper = qIntegerValue(qUpper(constraint));
    }
  } else {
    if (NullNode(constraint)) {
      cvn enumerated = qElements(basetype);
      lower = qOrdinalValue(qValue(enumerated));
      while (qNext(enumerated)) enumerated = qNext(enumerated);
      upper = qOrdinalValue(qValue(enumerated));
    } else {
      lower = qOrdinalValue(qLower(constraint));
      upper = qOrdinalValue(qUpper(constraint));
    }
  }
  size = 1 + upper - lower;
  {
    SIZE_T buf_size = (SIZE_T) sizeof(char) * size;
    flags = (char *) mem_get_block(buf_size);
    mem_zero(flags, buf_size);
  }
  for (alternative = alternatives; 
       alternative && qChoices(alternative) ; 
       alternative = qNext(alternative)) {
    cvn choice_cell;
    for (choice_cell= qChoices(alternative);
         choice_cell; 
         choice_cell = qNext(choice_cell)) {
      cvn choice = qValue(choice_cell);
      if (qBaseType(qSubtype(choice)) != basetype) {
        Message("type of choice does not match selector type", xxError, position);
        if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
        error = 1;
      } else if (!is_locally_static(choice)) {
        Message("choice expression is not locally static", xxError, position);
        if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
        error = 1;
      } else {
        int index;
        cvn value;
        value = eval_locally_static_exp(choice);
        if (IsA(basetype, kIntegerType)) {
          index = qIntegerValue(value);
        } else {
          index = qOrdinalValue(value);
        }
        index -= lower;
        if (flags[index]) {
          Message("value present in different alternatives", xxError, position);
          if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
          error = 1;
        }
      }
    }
    for (choice_cell= qChoices(alternative);
         choice_cell; 
         choice_cell = qNext(choice_cell)) {
      cvn choice = qValue(choice_cell);
      cvn value = eval_locally_static_exp(choice);
      int index;
      if (IsA(basetype, kIntegerType)) {
        index = qIntegerValue(value);
      } else {
        index = qOrdinalValue(value);
      }
      index -= lower;
      if ((qBaseType(qSubtype(choice)) == basetype) && (is_locally_static(choice)))
        flags[index] = 1;
      }
    }
    if (!NullNode(alternative) && NullNode(qChoices(alternative))) {
      return 1;
    } else { /* look for missing values */
      for (idx = 0 ; idx < size; ++idx)
        if (!flags[idx]) {
          Message("choices do not exhaust values of selector type", 
                  xxError, position);
          if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
          error = 1;
      }
      return error;
  }
  return 0;
}

/* ----------------------------------- _check_sig_asg_target -- */

/*  int _check_sig_asg_target (cvn target, int * nb_errors, tPosition position)

    . performs checks common to all flavors of signal assignment
    statements; if ok returns 1 else returns 0. */

static int
_check_sig_asg_target
(cvn target,
 int * nb_errors,
 tPosition position)
{
  /* Check that the target is a signal */
  if (!(IsA(qDeclaration(target), kSignalDeclaration))) {
    Message("target of signal assignment is not a signal", xxError, position) ;
    if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
    return 0;
  }
  /* Check that the signal is not of mode IN */
  if (qMode(qDeclaration(target)) == kIn) {
    Message("cannot assign a port of mode IN", xxError, position) ;
    if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
    return 0;
  }
  return 1;
}

/* --------------------------------------------- sig_asg_statement --- */

/* cvn sig_asg_statement (cvn target, cvn source, int * nb_errors, tPosition position)

   . creates a signal assignment statement node and returns it as
   its result;
   . performs semantics checks. */

cvn
sig_asg_statement
(cvn target,
 cvn source,
 int * nb_errors,
 tPosition position)
{
  /* Check the target */
  _check_sig_asg_target(target, nb_errors, position);

  /* Check that source and target have compatible types */
  if (!type_compatible(qSubtype(target), qSubtype(qValue(source)))) {
    Message("target and source of assignment do not have the same type",
	    xxError, position) ;
    if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
  }

  return mSignalAssignmentStatement(0, position.Line, 0, 0, string0(), target, source);

}

/* ----------------------------------------- conc_sig_asg_statement --- */

/* cvn conc_sig_asg_statement 
     (cvn target, cvn source, int * nb_errors, tPosition position)

   . depending on the class of the source node parameter, creates either a 
   signal assignment statement or a conditional signal assignment statement 
   node and returns is as result;
   . performs semantics checks. */

cvn
conc_sig_asg_statement
(cvn target,
 cvn source,
 int * nb_errors,
 tPosition position)
{
  /* Check that the target is a signal */
  _check_sig_asg_target(target, nb_errors, position);
  /* Check if it is a conditional or a simple assignment statement */
  if (qKind(source) == kWaveform) {
    /* Check that source and target have compatible types */
    if (!type_compatible(qSubtype(target), qSubtype(qValue(source)))) {
      Message("target and source of assignment do not have the same type",
	      xxError, position) ;
    if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
    }
    return mSignalAssignmentStatement(0, position.Line, 0, 0, string0(), 
                                      target, source);
  } else if (qKind(source) == kConditionalWaveform) {
    cvn waveform ;    /* conditional waveform */
    cvn type ;           /* target type */
    type = qSubtype(target) ;
    for (waveform = source ; waveform ;
	 waveform = qNext(waveform)) {
      cvn transaction; /* waveform transaction */
      cvn type2;       /* transaction type */
      transaction = qValue(waveform) ;
      type2 = qSubtype(transaction) ;
      if (!type_compatible(type, type2)) {
        Message("target and source of assignment do not have the same type",
		xxError, position) ;
        if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
      }
    }
    return mConditionalSignalAssignmentStatement(0, position.Line, 0, 0, string0(), 
						 target, source);
  } else {
    Message("cannot recognize source of signal assigmnent statement",
	    xxError, position) ;
    if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
    return 0;
  }
}


/* ----------------------------------------- selected_waveform --- */

/* cvn selected_waveform (cvn source, cvn choices)

   . creates a node of class Alternative to represent a selected
   waveform and returns it. */

cvn
selected_waveform
(cvn source,
 cvn choices)
{
  return mAlternative(0, 0, 0, choices, source);
}

/* ------------------------------------ sel_sig_asg_statement --- */

/* cvn sel_sig_asg_statement
     (cvn selector, cvn target, 
      cvn alternatives, int * nb_errors, tPosition position)

   . creates a selected signal assignment statement node and 
   returns it as its result;
   . performs semantics checks. */

cvn
sel_sig_asg_statement
(cvn selector,
 cvn target,
 cvn alternatives,
 int * nb_errors,
 tPosition position)
{
  _check_sig_asg_target(target, nb_errors, position);
  _check_alternatives(selector, alternatives, nb_errors, position);
  /* Check that source and target have compatible types */
  {
    cvn alternative;
    for (alternative = alternatives; 
	 alternative && qChoices(alternative);
         alternative = qNext(alternative)) {
      if (!type_compatible(qSubtype(target), 
           qSubtype(qValue(qStatements(alternative))))) {
        Message(
"target and source of selected signal assignment do not have the same type",
	        xxError, position) ;
        if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
      }
    }
    if (alternative) {
      if (!type_compatible(qSubtype(target), 
           qSubtype(qValue(qStatements(alternative))))) {
        Message(
"target and source of selected signal assignment do not have the same type",
	        xxError, position) ;
        if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
      }
    }
  }
  return mSelectedSignalAssignmentStatement(0, position.Line, 0, 0, 
					    string0(), target, 
					    selector, alternatives);
}

/* ----------------------------------------- var_asg_statement --- */
/* cvn var_asg_statement
   (cvn target, cvn source, int * nb_errors, tPosition position)

   . creates a variable assignment statement node and returns it as 
   its result;
   . performs semantics checks. */

cvn
var_asg_statement
(cvn target,
 cvn source,
 int * nb_errors,
 tPosition position)
{
   /* Check that the target is a variable */
   if (!(IsA(qDeclaration(target), kVariableDeclaration))) {
      Message("target of variable assignment is not a variable",
              xxError, position) ;
      if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
   }
   /* Check that source and targets have compatible types */
   if (! type_compatible(qSubtype(target), qSubtype(source))) {
      Message("target and source of assignment do not have the same type",
              xxError, position) ;
      if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
   }
   return mVariableAssignmentStatement(0, position.Line, 0, 0, string0(), target, 
				       source) ;
}

/* ----------------------------------------- wait_statement --- */

/* cvn wait_statement
   (cvn target, cvn source, int * nb_errors, tPosition position)

   . if has_sensitivity_list is not set, then
   . creates a wait statement node and returns it as its 
   result;
   . otherwise, report the error signal. */

cvn
wait_statement
(cvn sensitivity_clause,
 cvn condition_clause,
 int * nb_errors,
 tPosition position)
{
  if(!has_sensitivity_list) {
    return mWaitStatement(0, position.Line, 0, 0, string0(), 
		          condition_clause, sensitivity_clause) ;
  } else {
    Message("Wait statement appear in the process with sensitivity list",
            xxError, position);
    if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
    return 0;
  }
}

/* ------------------------------------------------- branch --- */

/* cvn branch
   (cvn condition, cvn statements)

   . creates a branch node and returns it as its result. */

cvn
branch
(cvn condition,
 cvn statements)
{
  return mBranch(0, 0, 0, condition, statements);
}

/* ------------------------------------------- if_statement --- */

/* cvn wait_statement
   (cvn branch_if, cvn branches_elsif, cvn branch_else, 
    int * nb_errors, tPosition position)

   . creates a if statement node and returns it as its result. */

cvn
if_statement
(cvn branch_if,
 cvn branches_elsif,
 cvn branch_else,
 int * nb_errors,
 tPosition position)
{
  cvn branch ;
  if (branches_elsif) {
    sNext(branch_if, branches_elsif) ;
    sPrevious(branches_elsif, branch_if);
  }
  for (branch = branch_if ; qNext(branch); branch = qNext(branch));
  if (branch_else) {
    sNext(branch, branch_else) ;
    sPrevious(branch_else, branch);
  }
  return mIfStatement(0, position.Line, 0, 0, string0(), branch_if);
}

/* ------------------------------------------ case_alternative --- */

/* cvn case_alternative (cvn choices, cvn statements)

   . creates a node of class Alternative that represents one
   alternative of a case statement and returns it as its result. */

cvn
case_alternative
(cvn choices,
 cvn statements)
{
  return mAlternative(0, 0, 0, choices, statements);
}

/* -------------------------------------------- case_statement --- */

/* cvn case_statement 
     (cvn selector, cvn alternatives, int * nb_errors, tPosition position)

     . creates a node of class CaseStatement and returns it as its 
     result;
     . performs semantics checks. */

cvn
case_statement
(cvn selector,
 cvn alternatives,
 int * nb_errors,
 tPosition position)
{
  _check_alternatives(selector, alternatives, nb_errors, position);
  return mCaseStatement(0, position.Line, 0, 0, string0(), selector, alternatives);
}

/* -------------------------------------------- while_statement --- */

/* cvn while_statement
     (cvn condition, int * nb_errors, tPosition position)

     . creates a node of class WhileStatement and returns it as its
     result; */

cvn
while_statement
(cvn condition,
 int * nb_errors,
 tPosition position)
{
  return mWhileStatement(0, position.Line, 0, 0, string0(), 0, condition);
}

/* -------------------------------------------- for_statement --- */

/* cvn for_statement
     (cvn par, cvn range, int * nb_errors, tPosition position)

     . creates a node of class ForStatement and returns it as its
     result; */

cvn
for_statement
(cvn par,
 cvn range,
 int * nb_errors,
 tPosition position)
{
  return mForStatement(0, position.Line, 0, 0, string0(), 0, par, range);
}

/* -------------------------------------------- loop_statement --- */

/* cvn loop_statement
     (cvn ite, cvn seq, int * nb_errors, tPosition position)

     . creates a node of class ForStatement and returns it as its
     result; */

cvn
loop_statement
(cvn ite,
 cvn seq,
 int * nb_errors,
 tPosition position)
{
  sStatements(ite, seq);
  return ite;
}

/* -------------------------------------------- null_statement --- */

/* cvn null_statement
     (int * nb_errors, tPosition position)

     . creates a node of class NullStatement and returns it as its
     result; */

cvn
null_statement
(int * nb_errors,
 tPosition position)
{
  return mNullStatement(0, position.Line, 0, 0, string0());
}

/* ---------------------------------------- process_statement --- */

/* cvn process_statement
     (cvn declarations, cvn statements, int * nb_errors, tPosition position)

     . reset global variable has_sensitivity_list to 0.
     . creates a node of class ProcessStatement and returns it as 
     its result. */

cvn
process_statement
(cvn sensitivity_list,
 cvn declarations,
 cvn statements,
 int * nb_errors,
 tPosition position)
{
  reset_has_sensitivity_list();
  return mProcessStatement(0, sensitivity_list, declarations, statements,
			   position.Line, 0, 0, string0()) ;
}

/* ------------------------------------------ block_statement --- */

/* cvn block_statement
     (cvn declarations, cvn statements, int * nb_errors, tPosition position)

     . creates a node of class ProcessStatement and returns it as 
     its result. */

cvn
block_statement
(cvn declarations,
 cvn statements,
 int * nb_errors,
 tPosition position)
{
   return mBlockStatement(0, declarations, statements, position.Line, 
			  0, 0, string0());
}

/* ----------------------------------------------- transaction --- */
/* to be developed in a future implementation that includes time */

cvn
transaction
(cvn value,
 cvn delay)
{
   /* check that the type of the delay expression is time */
   /* creates the transaction */
   return value;
}

/* ------------------------------------------------------------ */

/* cvn conditional_waveform
      (cvn waveform, cvn condition, cvn previous, int *, tPosition)

      . creates a node of class ConditionalWaveform;
      . performs semantic checks;
      . appends this node at the end of the previously
      generated conditional waveforms (previous)
      . returns the concatenation as its result. */

cvn
conditional_waveform
(cvn waveform,
 cvn condition,
 cvn previous,
 int * nb_errors,
 tPosition position)
{
   cvn conditional_waveform ;
   /* this routine can be called in a conc. sig. asg. stm. such as:
      S <= E ; 
      on E. A conditional waveform shall not be built in this case */
   if (NullNode(previous) && NullNode(condition)) {
      return(waveform) ;
   }
   conditional_waveform = 
      mConditionalWaveform(0, 0, 0, qValue(waveform), condition) ;
   if (!NullNode(condition) &&
       !type_compatible(qSubtype(condition), boolean_type)) {
      Message("conditional assignment does not have a boolean condition",
              xxError, position) ;
      if (++*nb_errors > max_nb_errors) exit (MAX_NB_ERRORS_EXIT_CODE);
   }
   return append(conditional_waveform, previous) ;
}

/* ---------------------------------------------- waveform --- */

/* cvn waveform (cvn transaction)

      . creates a node of class Waveform
      . returns it as its result. */
cvn
waveform
(cvn transaction)
{
   return mWaveform(0, transaction) ;
}

/* ----------------------------------------- name_conc_statement --- */

/* cvn name_conc_statement (cvn stm, cvn name, int * nb_errors, tPosition position)
   
      In the VHDL grammar, a name that starts a statement can be 
      either the target of an assignment, or a label.

      . checks if the parser is on an assignment statement;
      . if yes, calls the routine for creation of signal assignment
      statements;
      . else set the aName and aLine attributes of stm to name,
      and position. */

cvn
name_conc_statement
(cvn stm,
 cvn name,
 int * nb_errors,
 tPosition position)
{
  if (is_assignment_statement()) {
    return conc_sig_asg_statement(name, stm, nb_errors, position);
  } else {
    return label_set(stm, qName(name), position.Line);
  }
}

/* -------------------------------------------------- label_set --- */

/* cvn label_set (cvn stm, char * label, int * nb_errors, tPosition position)

      . sets the attributes aLine and aName of a statement node
      and returns the modified node;
      . an error message is issued if stm is not a statement. */

cvn
label_set
(cvn stm,
 char * label,
 int line)
{
  sName(stm, label);
  sLine(stm, line);
  return stm;
}

/* -------------------------------------------------- string0 --- */

/* char * string0 (void)

      . creates a string of length 1, and fills it with the
      null character */

char *
string0
(void)
{
  char * Result = (char *) mem_get_block((SIZE_T) sizeof(char)) ;
  * Result = '\0' ;
  return Result;
}

