/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    design-unit.c                                             */
/*                                                              */
/* Description:                                                 */
/*    Routines called by the compiler on rules associated to    */
/*    design units.                                             */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that compiles VHDL text descriptions into the   */
/*    internal format.                                          */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <malloc.h>
#include <string.h>
#include <memuser.h>
#include "actions.h"

/* ------------------------------------------------------------ */
void
unit_begin
(int * nb_errors)
{
  * nb_errors = 0;
  begin_declarative_part();
}

void
unit_end
(cvn unit, int * nb_errors)
{
  if (* nb_errors) {
    char * message;
    if (* nb_errors == 1) {
      message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      strlen("1 compilation error") + 1);
      sprintf(message, "1 compilation error");
    } else {
      message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      strlen(" compilation errors") + 4);
      sprintf(message, "%u compilation errors", * nb_errors);
    }
    Message(message, xxInformation, NoPosition);
    mem_free_block(message);
  } else {
    cv_save_store(unit) ;
  }
}

/* ------------------------------------------------------------ */
cvn
entity_declaration
(char * name,
 cvn ports,
 int * nb_errors,
 tPosition pos) 
{
  return mEntityDeclaration(0, pos.Line, 0, 0, name, ports, 0);
}

cvn
architecture_body
(char * name,
 cvn decls,
 cvn stms, 
 cvn ent_name,
 int * nb_errors,
 tPosition pos)
{
  cvn primary;
  primary = get_declaration(name_match, qName(ent_name)) ;
  if (primary == 0) {
    Message("internal error - please report cmuvhdl@cs.cmu.edu", 
            xxError, pos);
    if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
  }
  return mArchitectureBody(0, pos.Line, 0, 0, name, decls, stms, primary);
}

/* --------------------------------------- package_declaration --- */

cvn
package_declaration
(char * name,
 cvn content,
 int * nb_errors,
 tPosition pos)
{
  return mPackageDeclaration(0, pos.Line, 0, 0, name, content);
}

/* --------------------------------------- package_body ---------- */

/* cvn package_body(cvn name, cvn content,
   tPosition pos) */

cvn
package_body
(cvn name,
 cvn content,
 int * nb_errors,
 tPosition pos)
{
  cvn primary;
  char * pack_name = qName(name);

  primary = get_declaration(name_match, pack_name);
  if (primary == 0) {
    Message("internal error - please report cmuvhdl@cs.cmu.edu", 
            xxError, pos);
    if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
  }
  return mPackageBody(0, pos.Line, 0, 0, pack_name, content, primary);
}

/* --------------------------------------- add_context_clause --- */
cvn
add_context_clause
(cvn unit,
 cvn clause)
{
  cvn default_use_clause;

  if (clause != 0) {
    sNext(clause, qDeclarations(unit));
    sDeclarations(unit, clause);
  }

  if (std_clause != 0) {
    default_use_clause = mUseClause(0, 0, 0, mList(0, 0, 0, std_clause));
    sNext(default_use_clause, qDeclarations(unit));
    sDeclarations(unit, default_use_clause);
  }

  return unit;
}

/* --------------------------------------- set_primary --- */

/* void set_primary (cvn name, sec_name)

      . looks up for a declaration named name.
      . if such declaration is not found, loads the file name.cv 
      which must contain the CV representation of an entity 
      declaration;
      . adds the ports of the entity to the environment.
      . failure to do so results in an error message. */

void set_primary
(cvn primary,
 char * sec_name,
 int * nb_errors,
 tPosition position)
{
/* name must be the top level name of a declarative region: entity
  declaration in our restricted subset */
  cvn toplevel ;
  char * prim_name = qName(primary) ;

  toplevel = cv_load_unit("work", prim_name, "");

  if (toplevel == 0) {
    Message("cannot load library unit of primary entity",
            xxFatal, position);
    exit(FATAL_EXIT_CODE);
  }
  add_declarative_part(toplevel);
  switch (qKind(toplevel)) {
    case kEntityDeclaration :
      add_declarative_part(qPorts(toplevel)) ;
      /* no local declarations in an entity declaration now */
      break ;
    case kPackageDeclaration :
      add_declarative_part(qDeclarations(toplevel));
      break;
    default :
      Message("declarative region not supported", xxError, position) ;
      if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
      break ;
  }
}

/* --------------------------- default_context_clause ----- */
void
default_context_clause
(tKind kind,
 char * primary,
 int * nb_errors,
 tPosition position)
{
  library_clause = 0;
  if ((kind == kPackageDeclaration) &&
      (strcasecmp(primary, "standard") == 0)) {
    std_clause = 0;
  } else {
    add_library_logical_name("std", nb_errors, position);
    add_library_logical_name("work", nb_errors, position);
    std_clause = open_external_reference("std", "standard", nb_errors, position);
  }
}

