/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    declaration.c                                             */
/*                                                              */
/* Description:                                                 */
/*    Routines to handle the analysis of declarations.          */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    analyze : a VHDL analyzer.                                */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <Errors.h>
#include <Positions.h>

#include "actions.h"

/* ------------------------------------------------------------ */
/* data structure for a declarative part:
     . a set of declarations,
     . an environment, which is a declarative part as well. */
struct declarative_part_ {
   cvn declarative_part ;
   struct declarative_part_ * environment ;
};

/* declaration_env                            (global variable)
   lists the different declarative regions the parser is
   currently nested in */
static tEnvironment declaration_env ;

/* -------------------------------------- declaration_init --- */

void
declaration_init
(cvn init_decls)
{
  declaration_env = (tEnvironment) malloc(sizeof(struct declarative_part_)) ;
  declaration_env -> environment = 0 ;
  declaration_env -> declarative_part = init_decls ;
}

/* ------------------------------------------------------------ */
/* _current_declarative_part                   (global variable)
   keeps track of the declarations already parsed when a
   declarative region (as process_declarative_part) is
   parsed */
static cvn _current_declarative_part;

/* -------------------------------- begin_declarative_part --- */
void
begin_declarative_part 
(void)
{
  _current_declarative_part = 0;
}

/* ---------------------------------- end_declarative_part --- */
void
end_declarative_part 
(void)
{
  _current_declarative_part = 0;
}

/* ---------------------------------- add_declarative_item --- */
/* cvn add_declarative_item (cvn item)

      . checks if item is not in conflict with the other
      declarative items of the current declarative part 
      and appends it at the end of the current declarative
      part;
      . returns the list of items already processed in the
      current declarative part. */

cvn
add_declarative_item
(cvn item,
 int * nb_errors,
 tPosition position)
{
  cvn declaration, last_declaration;
  char * name = qName(item);
  int clash;
  declaration = _current_declarative_part;
  last_declaration = 0;
  while (declaration) {
    if ((clash = (strcasecmp(qName(declaration), name) == 0))) {
      unsigned length =
	strlen("duplicate declaration of ") +
	  strlen(name) + 1 ;
      char * mess = calloc(length, sizeof(char)) ;
      sprintf(mess,"duplicate declaration of %s", name) ;
      Message(mess, xxError, NoPosition) ;
      free(mess) ;
      if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
    }
    last_declaration = declaration;
    declaration = qNext(declaration);
  }
  if (last_declaration)
    sNext(last_declaration, item);
  else
    _current_declarative_part = item;
  return _current_declarative_part;
}

/* ----------------------------------------- add_declarative_part --- */
/* void add_declarative_part (cvn declarations)

      . stacks declarations on the environment. */

void
add_declarative_part
(cvn declarations)
{
  tEnvironment env = malloc(sizeof(struct declarative_part_)) ;
  env-> environment = declaration_env ;
  env-> declarative_part = declarations ;
  declaration_env = env;
}

/* -------------------------------------- remove_declarative_part --- */

/* void remove_declarative_part (void)

      . pops up the last declarative part off the environment. */

void
remove_declarative_part
(void)
{
  tEnvironment tmp = declaration_env->environment;
  free(declaration_env);
  declaration_env = tmp;
}


/* -------------------------------------------- get_declaration --- */

/* cvn get_declaration
      (int (* fn) (cvn, void *), void * criterion)

      . looks up in the environment the declaration such that
      fn(declaration, criterion) yields a non null result
      (this includes the literals of an enumeration type declaration).
      . if a declaration is found, it is returned;
      . otherwise the value 0 is returned. */

static cvn get_declaration_
(cvn declaration, int (* fn) (cvn, void *), void * criterion);

cvn get_declaration
(int (* fn) (cvn, void *),
 void * criterion)
{
  tEnvironment env = declaration_env ;
  cvn result = 0;
  result = get_declaration_(_current_declarative_part, fn, criterion);
  while ((result == 0) && (env != 0)) {
    result = get_declaration_(env -> declarative_part, fn, criterion);
    env = env -> environment ;
  }
  return result;
}

static cvn
get_declaration_ 
(cvn declaration,
 int (* fn) (cvn, void *),
 void * criterion)
{
  int match ;
  match = 0 ;
  while (declaration && !(match = fn(declaration, criterion))) {
    if ((IsA(declaration, kTypeDeclaration)) &&
	(IsA(qBaseType(qSubtype(declaration)), kEnumerationType))) {
      cvn cell = qElements(qBaseType(qSubtype(declaration))) ;
      while (cell && !(match = fn(qValue(cell), criterion)))
	cell = qNext(cell) ;
      if (match) 
	return(qValue(cell)) ;
    }
    declaration = qNext(declaration) ;
  }
  if (match) {
    return(declaration) ;
  }
  return 0;
}

/* -------------------------------------------- add_declaration --- */

/* void add_declaration (cvn declaration) 

      . adds declaration to the top most declarative part of the
      environment (rem: at the first place). */

void
add_declaration
(cvn declaration) 
{
  cvn preceding = declaration_env->declarative_part;

  sNext(declaration, preceding) ;
  if (preceding) sPrevious(preceding, declaration);
  declaration_env -> declarative_part = declaration ;
}

/* -------------------------------------------- name_match --- */

/* int name_match (cvn declaration, void * name)
   
      . returns a non null result if the name of the 
      declaration is equal to name;
      . is not case-sensitive;
      . is used in conjunction with get_declaration */

int
name_match
(cvn declaration,
 void * name)
{
   return(strcasecmp(qName(declaration), (char *) name) == 0) ;
}

/* -------------------------------------------- symbol_literal --- */

/* int symbol_literal (cvn declaration, void * name)
   
      . returns a non null result if declaration is an enumeration
      type declaration and has an element whose name is name;
      . is not case-sensitive;
      . is used in conjunction with get_declaration */

int
symbol_literal
(cvn declaration,
 void * name)
{
   int match ;

   if ((IsA(declaration, kTypeDeclaration)) &&
       (IsA(qBaseType(qSubtype(declaration)), kEnumerationType))) {
      cvn cell = qElements(qBaseType(qSubtype(declaration))) ;
      while (cell && !(match = (strcasecmp(qName(qValue(cell)), name) == 0))) {
         cell = qNext(cell) ;
      }
      return match;
   }
   return 0;
}

/* ------------------------------------ object_declarations --- */

/* cvn object_declarations
      (tClass class, cvn subtype, cvn init, 
      tMode mode, identifier_list list)
      
      . is called when an object declaration rule has been parsed;
      . if the init (or default) parameter is null, sets it to
      the leftmost calue of type;
      . creates the object declaration nodes and links them
      together.
      . semantic checks (e.g. name conflicts are not performed). */

cvn
object_declarations
(tClass class,
 cvn subtype,
 cvn init,
 tMode mode, 
 identifier_list list,
 int * nb_errors,
 tPosition position)
{
  identifier_list l;
  cvn decl, preceding_decl;
  cvn range = qConstraint(subtype);
  cvn basetype = qBaseType(subtype);

  if (NullNode(init)) {
    if (NullNode(range)) {
      if (IsA(basetype, kIntegerType)) {
        init = qLeft(basetype) ;
      } else {
        init = qValue(qElements(basetype));
      }
    } else {
      init = qLeft(range);
    }
  }
  decl=0;
  for (l = list; l; l=l->tail) {
    char * name = l->string;
    tPosition pos = l->position;
    preceding_decl = decl;
    switch (class) {
    case kDefaultClass : case kSignal :
      decl =
	mSignalDeclaration(0, pos.Line, preceding_decl, 0, name, subtype, 
			   mode, init) ;
      if (preceding_decl) sPrevious(preceding_decl, decl);
      break ;
    case kVariable :
      decl =
	mVariableDeclaration(0, pos.Line, preceding_decl, 0, name, subtype, 
			     mode, init) ;
      if (preceding_decl) sPrevious(preceding_decl, decl);
      break ;
    case kConstant :
      decl =
        mConstantDeclaration(0, pos.Line, preceding_decl, 0, name,subtype,
                             init);
      if (preceding_decl) sPrevious(preceding_decl, decl);
      break ;
    default:
      Message("This type of declaration is not handled", xxError, pos) ;
      if(++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
      break ;
    }
  }
  free_identifier_list(list);
  return decl;
}

/* ------------------------------------ free_declaration --- */

/* void free_declaration()
   free the declaration_env */

void
free_declaration
(void)
{
  tEnvironment env;

  while(declaration_env->environment!=0) {
    if(declaration_env->declarative_part!=0)
      free(declaration_env->declarative_part);
    env = declaration_env;
    declaration_env = env->environment;
    free(env);
  }
  if(declaration_env->declarative_part!=0)
    free(declaration_env->declarative_part);
  declaration_env->declarative_part = 0;
}

