/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    cva.c                                                     */
/*                                                              */
/* Description:                                                 */
/*    process the on-line arguments and call the parser.        */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that compiles VHDL text descriptions into CV    */
/*    intermediate format.                                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "Memory.h"
#include "Scanner.h"
#include "Parser.h"
#include "cva.h"

/* ------------------------------------------------------------ */
const int SUCCESS_EXIT_CODE = 0;
const int MAX_NB_ERRORS_EXIT_CODE = -1;
const int FATAL_EXIT_CODE = -2;
int max_nb_errors = 20;

static char * _seek_vhdl_file (char * name) ;
static char * _add_suffix (char * name, char * suffix) ;

static void _usage (void) ;
static void _version (void) ;
/* ------------------------------------------------------------ */

int main
#if defined(__STDC__)
   (int argc, char ** argv)
#else
   (argc, argv)
   int argc ;
   char ** argv ;
#endif
{
   char * FileName ;
   char * VhdlFile ;

   /* Process the arguments */
   argv++ ;
   argc-- ;
   VhdlFile = (char *) NULL ;
   while (argc) {
     if (**argv == '-') {
       if ((strcmp(*argv, "-v") == 0) || (strcmp(*argv, "--version") == 0)) {
         _version();
         exit(SUCCESS_EXIT_CODE);
       } else if ((strcmp(*argv, "-h") == 0) || 
                  (strcmp(*argv, "--help") == 0)) {
         _usage();
         exit(SUCCESS_EXIT_CODE);
       } else if ((strcmp(*argv, "-m") == 0) ||
                  (strcmp(*argv, "--max-nb-errors") == 0)) {
         argv++ ;
         argc-- ;
         if (argc) {
           max_nb_errors = atoi(* argv);
         } else {
           _usage();
           exit(FATAL_EXIT_CODE) ;
         }
       }
     } else {
       VhdlFile = * argv ;
     }
     argv++ ;
     argc-- ;
   }

   if ((max_nb_errors <= 0) || (VhdlFile == 0)) {
     _usage() ;
     exit(FATAL_EXIT_CODE) ;
   }

   if ((FileName = _seek_vhdl_file(VhdlFile)) == NULL) {
     Message("cannot open vhdl file", xxFatal, NoPosition);
     exit(FATAL_EXIT_CODE) ;
   }

   /* Call the parser */
   BeginFile(FileName) ;
   cv_initialize();
   (void) Parser	();
   CloseParser	();
   /* sign off */
   return(SUCCESS_EXIT_CODE) ;
}

/* ------------------------------------------- _seek_vhdl_file --- */

/* char * _seek_vhdl_file (char * name)

   looks if a file named "name", "name.v", "name.vhd" or "name.vhdl"
   exists (in this order). If yes return the full name of the file,
   else a null pointer.
   the original string name is left untouched. */

char * _seek_vhdl_file
#if defined(__STDC__)
   (char * name)
#else
   (name)
   char * name ;
#endif
{
   FILE * fptr ;

   name = strdup(name) ;
   if ((fptr = fopen(name, "r")) != 0) {
      fclose(fptr);
      return name ;
   }
   name = _add_suffix(name, ".v") ;
   if ((fptr = fopen(name, "r")) != 0) {
      fclose(fptr);
      return name ;
   }
   name = _add_suffix(name, "hd") ;
   if ((fptr = fopen(name, "r")) != 0) {
      fclose(fptr);
      return name ;
   }
   name = _add_suffix(name, "l") ;
   if ((fptr = fopen(name, "r")) != 0) {
      fclose(fptr);
      return name ;
   }
   return 0 ;
}

/* ------------------------------------------- _add_suffix -------- */

/* char * _add_suffix (char * name, char * suffix)

   creates a string representing the concatenation of name and suffix,
   frees name, and return the result string */

char * _add_suffix
#if defined(__STDC__)
   (char * name, char * suffix)
#else
   (name, suffix)
   char * name ;
   char * suffix ;
#endif
{
   char * result ;
   result = calloc(strlen(name)+strlen(suffix)+1, sizeof(char)) ;
   sprintf(result,"%s%s", name, suffix) ;
   free(name) ;
   return(result) ;
}


/* ------------------------------------------- _usage ------------ */

/* void _usage (void)

   prints the usage for the program cva */

void _usage
#if defined(__STDC__)
   (void)
#else
   ()
#endif
{
  fprintf(stdout,
"Usage: cva [options] filename\n"
"Options:\n"
"  -h,     --help        prints this and quits\n"
"  -v,     --version     print version number and quits\n"
"  -m N,   --max-nb-errors N (with N positive)\n"
"                        stops compilation and quits after N errors\n"
         );
}

/* ------------------------------------------- _version ---------- */

/* void _version (void)

   prints the version for the program cva */

void _version
#if defined(__STDC__)
   (void)
#else
   ()
#endif
{
  fprintf(stdout, "cva release number: 1.b.1\n");
}
