/**************************** CPPHeaderFile ***************************

* FileName [SimplifyTP.h]

* PackageName [theorem-prover.simplify]

* Synopsis [Header file for SimplifyTP class.]

* Description [This class provides concrete implementations of the
* GenericTP interface that allow the tool to interface with the
* Simplify theorem prover.]

* SeeAlso [../../main/GenericTP.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __SIMPLIFYTP_H__
#define __SIMPLIFYTP_H__

namespace magic {

//other classes needed
class BasicExpr;

class SimplifyTP : public GenericTP
{
 private:
#ifdef WIN32
  //the simplify process info
  PROCESS_INFORMATION proc; 
  //the input and output file handles for communicating with Simplify
  HANDLE out;
  HANDLE in;
#else
  //the simplify process id
  pid_t proc;
  //the output stream piped to stdin of the process
  FILE *out;
  //the input stream piped to the stdout of the process
  FILE *in;
#endif //WIN32

  //map structure fields to ids
  map<string,string> fieldMap;

  //the set of enumeration constants for which constraints have been
  //added already
  set<string> enumsAdded;
  
  //the caches for storing results. two caches for the valid and
  //invalid formulas.
  set<string> trueCache;
  set<string> falseCache;

  //the total size of all the queries made so far
  size_t totalQuerySize;

  //list of strings needed to check if a formula is of type Boolean
  static list<string> boolCheck;
  
  //the predicate for value
  static const string VALUE;
  //the predicate for field access
  static const string FIELD_ACCESS;
  //the predicate for pinter dereference
  static const string POINTER_DEREF;
  //the predicate for address
  static const string ADDRESS;
  //biwtise operators
  static const string BITWISE_NOT;
  static const string BITWISE_AND;
  static const string BITWISE_XOR;
  static const string BITWISE_OR;
  static const string LEFT_SHIFT;
  static const string RIGHT_SHIFT;
  //constant used for procedure calls
  static const string PROCEDURE;
  //replacement strings for special characters
  static const string UNDERSCORE;
  static const string COLON_COLON;
  static const string HASH;
  static const string VALID_PTR_PRED;
  //the maximum size of queries before restart
  static const size_t MAX_TOTAL_QUERY_SIZE;

  //exceptions
  enum SimplifyException { NOT_HANDLED };

  string ReadFromSimplify();
  void WriteToSimplify(const string &line);
  void SetupEnv();
  string Value(const BasicExpr *node) throw (SimplifyException);
  string Location(const BasicExpr *node) throw (SimplifyException);
  string ConvertToBoolean(const string &old);
  void SimplifyId(string &arg) const;
  string CreateConjunction(const list<const BasicExpr*> &exprs) throw (SimplifyException); 
  string CreateDisjunction(const list<const BasicExpr*> &exprs) throw (SimplifyException); 
  int CheckCache(const string &formula);
  void AddConstraints(const list<const BasicExpr*> &ante,const list<const BasicExpr*> &cons);
  void UpdateCache(const string &formula,bool type);

 public:
  SimplifyTP();
  virtual ~SimplifyTP() {}

  //initialise the theorem prover
  void Initialise();
  
  //shutdown the theorem prover
  void Shutdown();
  
  //prove that an antecedent implies a consequent. both the
  //antecedent and the consequent are provided as a list of
  //formulas. the result is true if the cojunction of all of the
  //antecedents implies the disjunction of all of the
  //consequents. return true if it can be proved and false
  //otherwise.
  bool ProveImplies(const list<const BasicExpr*> &ante,const list<const BasicExpr*> &cons);
  
  //add a constraint to the theorem prover environment. this
  //constraint is provided as a C expression.
  void AddConstraint(const BasicExpr *expr);
};

} //namespace magic

#endif //__SIMPLIFYTP_H__

/*********************************************************************/
//end of SimplifyTP.h
/*********************************************************************/
