/******************************** CPPFile *****************************

* FileName [SimplifyTP.cpp]

* PackageName [theorem-prover.simplify]

* Synopsis [This file contains the methods for the SimplifyTP class.]

* SeeAlso [SimplifyTP.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <sys/types.h>
#ifndef WIN32
#include <sys/wait.h>
#endif //WIN32
#include <unistd.h>
#include <signal.h>
#include <string>
#include <set>
#include <list>
#include <map>
#include <vector>
#include <typeinfo>
using namespace std;

#include "Util.h"
#include "Statistics.h"
#include "Node.h"
#include "GenericTP.h"
#include "SimplifyTP.h"
#include "Database.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//static data
/*********************************************************************/
list<string> SimplifyTP::boolCheck;
const string SimplifyTP::VALUE = "val";
const string SimplifyTP::FIELD_ACCESS = "extract";
const string SimplifyTP::POINTER_DEREF = "deref";
const string SimplifyTP::ADDRESS = "address";
const string SimplifyTP::BITWISE_NOT = "bwnot";
const string SimplifyTP::BITWISE_AND = "bwand";
const string SimplifyTP::BITWISE_XOR = "bwxor";
const string SimplifyTP::BITWISE_OR = "bwor";
const string SimplifyTP::LEFT_SHIFT = "lsh";
const string SimplifyTP::RIGHT_SHIFT = "rsh";
const string SimplifyTP::PROCEDURE = "proc_";
const string SimplifyTP::UNDERSCORE = "simplify_uscore_";
const string SimplifyTP::COLON_COLON = "_simplify_dcolon_";
const string SimplifyTP::HASH = "_simplify_hash_";
const string SimplifyTP::VALID_PTR_PRED = "simplify_valid_ptr";
const size_t SimplifyTP::MAX_TOTAL_QUERY_SIZE = 100000000;

/*********************************************************************/
//constructor and destructor
/*********************************************************************/
SimplifyTP::SimplifyTP()
{
  if(boolCheck.empty()) {
    boolCheck.push_back("(+");
    boolCheck.push_back("(-");
    boolCheck.push_back("(*");
    boolCheck.push_back("(" + VALUE);
    boolCheck.push_back("(" + ADDRESS);
    boolCheck.push_back("(" + BITWISE_AND);
    boolCheck.push_back("(" + BITWISE_XOR);
    boolCheck.push_back("(" + BITWISE_OR);
    boolCheck.push_back("(" + LEFT_SHIFT);
    boolCheck.push_back("(" + RIGHT_SHIFT);
    boolCheck.push_back("(" + PROCEDURE);
    boolCheck.push_back(PROCEDURE);
  }
}

/*********************************************************************/
//initialise the theorem prover
/*********************************************************************/
void SimplifyTP::Initialise()
{
#ifdef WIN32
  HANDLE hChildStdinRd,hChildStdinWr,hChildStdoutRd,hChildStdoutWr; 
  HANDLE hSaveStdin,hSaveStdout;
  SECURITY_ATTRIBUTES saAttr; 
  BOOL fSuccess; 
  //Set the bInheritHandle flag so pipe handles are inherited. 
  saAttr.nLength = sizeof(SECURITY_ATTRIBUTES); 
  saAttr.bInheritHandle = TRUE; 
  saAttr.lpSecurityDescriptor = NULL; 
  //The steps for redirecting child process's STDOUT: 
  //    1. Save current STDOUT, to be restored later. 
  //    2. Create anonymous pipe to be STDOUT for child process. 
  //    3. Set STDOUT of the parent process to be write handle to 
  //       the pipe, so it is inherited by the child process. 
  //    4. Create a noninheritable duplicate of the read handle and
  //       close the inheritable read handle. 
  //Save the handle to the current STDOUT. 
  hSaveStdout = GetStdHandle(STD_OUTPUT_HANDLE); 
  //Create a pipe for the child process's STDOUT. 
  if (! CreatePipe(&hChildStdoutRd, &hChildStdoutWr, &saAttr, 0)) {
    Util::Error("ERROR: Stdout pipe creation failed\n"); 
  }
  //Set a write handle to the pipe to be STDOUT.    
  if (! SetStdHandle(STD_OUTPUT_HANDLE, hChildStdoutWr)) {
    Util::Error("ERROR: Redirecting STDOUT failed");
  }
  //Create noninheritable read handle and close the inheritable read
  //handle.  
  fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStdoutRd,
			     GetCurrentProcess(), &in , 0,
			     FALSE,
			     DUPLICATE_SAME_ACCESS);
  if( !fSuccess ) Util::Error("ERROR: DuplicateHandle failed");
  CloseHandle(hChildStdoutRd);
  //The steps for redirecting child process's STDIN: 
  //    1.  Save current STDIN, to be restored later. 
  //    2.  Create anonymous pipe to be STDIN for child process. 
  //    3.  Set STDIN of the parent to be the read handle to the 
  //        pipe, so it is inherited by the child process. 
  //    4.  Create a noninheritable duplicate of the write handle, 
  //        and close the inheritable write handle. 
  //Save the handle to the current STDIN.
  hSaveStdin = GetStdHandle(STD_INPUT_HANDLE); 
  //Create a pipe for the child process's STDIN. 
  if (! CreatePipe(&hChildStdinRd, &hChildStdinWr, &saAttr, 0)) {
    Util::Error("ERROR: Stdin pipe creation failed\n"); 
  }
  //Set a read handle to the pipe to be STDIN. 
  if (! SetStdHandle(STD_INPUT_HANDLE, hChildStdinRd)) {
    Util::Error("ERROR: Redirecting Stdin failed"); 
  }
  //Duplicate the write handle to the pipe so it is not inherited. 
  fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStdinWr, 
			     GetCurrentProcess(), &out, 0, 
			     FALSE,                  //not inherited 
			     DUPLICATE_SAME_ACCESS); 
  if (! fSuccess) Util::Error("ERROR: DuplicateHandle failed"); 
  CloseHandle(hChildStdinWr); 
  //Now create the child process.
  STARTUPINFO siStartInfo;
  BOOL bFuncRetn = FALSE; 
  //Set up members of the PROCESS_INFORMATION structure. 
  ZeroMemory( &proc, sizeof(PROCESS_INFORMATION) );
  //Set up members of the STARTUPINFO structure. 
  ZeroMemory( &siStartInfo, sizeof(STARTUPINFO) );
  siStartInfo.cb = sizeof(STARTUPINFO);
  //create the name of simplify executable file
  char *envptr = getenv("MAGICROOT");
  if(envptr == NULL) Util::Error("ERROR: MAGICROOT not set ... Aborting ...\n");
  string command = string("\"") + string(envptr) + "\\theorem-prover\\simplify\\Simplify.exe\" -nosc";
  char commLine[1024];
  strcpy(commLine,command.c_str());
  //Create the child process. 
  bFuncRetn = CreateProcess(NULL, 
			    commLine,     //command line 
			    NULL,         //process security attributes 
			    NULL,         //primary thread security attributes 
			    TRUE,         //handles are inherited 
			    0,            //creation flags 
			    NULL,         //use parent's environment 
			    NULL,         //use parent's current directory 
			    &siStartInfo, //STARTUPINFO pointer 
			    &proc);       //receives PROCESS_INFORMATION 
  if (bFuncRetn == 0) Util::Error("ERROR: CreateProcess failed\n");
  else Util::Message(2,"Simplify process created ...\n");
  //restore the saved STDIN and STDOUT. 
  if (! SetStdHandle(STD_INPUT_HANDLE, hSaveStdin)) Util::Error("ERROR: Re-redirecting Stdin failed\n"); 
  if (! SetStdHandle(STD_OUTPUT_HANDLE, hSaveStdout)) Util::Error("ERROR: Re-redirecting Stdout failed\n");
  //reset data structures
  enumsAdded.clear();
  totalQuerySize = 0;
  //read up the initial Simplify prompt
  ReadFromSimplify();
  //setup the environment - load the axioms etc.
  SetupEnv();
#else
  //create the name of cvs executable file
  char *envptr = getenv("MAGICROOT");
  if(envptr == NULL) Util::Error("ERROR: MAGICROOT not set ... Aborting ...\n");
  string command = string(envptr) + "/theorem-prover/simplify/Simplify";

  //create the file descriptors for communication
  int toSimplify[2],fromSimplify[2];
  if((pipe(toSimplify) == -1) || (pipe(fromSimplify) == -1)) {
    Util::Error("ERROR: could not create file descriptors for communication ...\n");
  }

  //start the simplify process
  proc = fork();
  //failure
  if(proc == -1) Util::Error("ERROR: could not fork for creating Simplify process ...\n");
  //child process
  if(proc == 0) {
    //close file descriptors
    if((close(toSimplify[1]) == -1) || (close(fromSimplify[0]) == -1) ||
       (close(0) == -1) || (close(1) == -1)) {
      assert(false);
    }
    //duplicate file descriptors
    if((dup(toSimplify[0]) == -1) || (dup(fromSimplify[1]) == -1)) {
      assert(false);
    }
    //exec the simplify process
    if(execl(command.c_str(),command.c_str(),"-nosc",NULL) == -1) {
      assert(false);
    }
  }
  //parent process
  else {
    Util::Message(2,"Simplify process started ...\n");
    //close file descriptors
    if((close(toSimplify[0]) == -1) || (close(fromSimplify[1]) == -1)) {
      Util::Error("ERROR: could not close pipe file descriptors in parent ...\n");
    }
    //convert file descriptors into FILE*
    if(((out = fdopen(toSimplify[1],"w")) == NULL) || ((in = fdopen(fromSimplify[0],"r")) == NULL)) {
      Util::Error("ERROR: could not convert file descriptors to FILE* in parent ...\n");
    }
    //reset data structures
    enumsAdded.clear();
    totalQuerySize = 0;
    //read up the initial Simplify prompt
    ReadFromSimplify();
    //setup the environment - load the axioms etc.
    SetupEnv();
  }
#endif //WIN32
}

/*********************************************************************/
//shutdown the theorem prover
/*********************************************************************/
void SimplifyTP::Shutdown()
{
#ifdef WIN32
  //close process and thread handles
  if(TerminateProcess(proc.hProcess,1)) {
    CloseHandle(proc.hProcess);
    CloseHandle(proc.hThread);
    CloseHandle(in);
    CloseHandle(out);
    Util::Message(2,"Simplify process destroyed ...\n");
  } else Util::Error("ERROR: could not destroy Simplify process ...\n");
#else
  //destroy the simplify process
  if(kill(proc,SIGKILL) == -1) {
    Util::Error("ERROR: could not destroy Simplify process ...\n");
  } else {
    if(waitpid(proc,NULL,0) != proc) {
      Util::Error("ERROR: could not free resources of Simplify process ...\n");
    } else {
      Util::Message(2,"Simplify process destroyed ...\n");
    }
  }
#endif //WIN32
}

/*********************************************************************/
//simplify an identifier
/*********************************************************************/
void SimplifyTP::SimplifyId(string &arg) const
{
  if(arg == Database::VALID_PTR_PRED) arg = VALID_PTR_PRED;
  else {
    if(arg.at(0) == '_') arg = UNDERSCORE + arg;
    size_t pos;
    while((pos = arg.find(':')) != string::npos) {
      arg = arg.substr(0,pos) + COLON_COLON + arg.substr(pos+2,arg.size()-pos-2);
    }
    while((pos = arg.find('#')) != string::npos) {
      arg = arg.substr(0,pos) + HASH + arg.substr(pos+1,arg.size()-pos-1);
    }
  }
}

/*********************************************************************/
//given a list of expressions create a string of their conjunction
/*********************************************************************/
string SimplifyTP::CreateConjunction(const list<const BasicExpr*> &exprs) throw (SimplifyException)
{
  //first convert all the expressions to boolean formula strings
  list<string> x;
  for(list<const BasicExpr*>::const_iterator i = exprs.begin();i != exprs.end();++i) {
    try {
      x.push_back(ConvertToBoolean(Value(*i)));
    } catch(SimplifyException ex) {
      assert(ex == NOT_HANDLED);
      throw ex;
    }
  }
  
  //handle special cases
  if(x.size() == 0) return "TRUE";
  else if(x.size() == 1) return x.front();
  
  //create the conjunction
  string ret = "(AND";
  for(list<string>::iterator i = x.begin();i != x.end();++i) {
    ret = ret + " " + (*i);
  }
  return ret + ")";
}

/*********************************************************************/
//given a list of expressions create a string of their disjunction
/*********************************************************************/
string SimplifyTP::CreateDisjunction(const list<const BasicExpr*> &exprs) throw (SimplifyException)
{
  //first convert all the expressions to boolean formula strings
  list<string> x;
  for(list<const BasicExpr*>::const_iterator i = exprs.begin();i != exprs.end();++i) {
    try {
      x.push_back(ConvertToBoolean(Value(*i)));
    } catch(SimplifyException ex) {
      assert(ex == NOT_HANDLED);
      throw ex;
    }
  }
  
  //handle special cases
  if(x.size() == 0) return "FALSE";
  else if(x.size() == 1) return x.front();
  
  //create the conjunction
  string ret = "(OR";
  for(list<string>::iterator i = x.begin();i != x.end();++i) {
    ret = ret + " " + (*i);
  }
  return ret + ")";
}

/*********************************************************************/
//prove that an antecedent implies a consequent. both the antecedent
//and the consequent are provided as a list of formulas. the result is
//true if the cojunction of all of the antecedents implies the
//disjunction of all of the consequents. return true if it can be
//proved and false otherwise.
/*********************************************************************/
bool SimplifyTP::ProveImplies(const list<const BasicExpr*> &ante,const list<const BasicExpr*> &cons)
{
  try {
    //create the antecedent formula
    string anteStr = CreateConjunction(ante);
    
    //create the consequent formula
    string consStr = CreateDisjunction(cons);
    
    if(!Database::IGNORE_POINTERS) {
      //add constraints to antecedent to force locations to be distinct
      //and have valid addresses
      set<string> varSet;
      set<Expr> lvSet;
      for(list<const BasicExpr*>::const_iterator i = ante.begin();i != ante.end();++i) {
	set<string> ids = Util::ComputeIdLvalues(*i);    
	varSet.insert(ids.begin(),ids.end());
	set<Expr> lvs = Util::ComputeExprLvalues(*i);
	lvSet.insert(lvs.begin(),lvs.end());
      }
      for(list<const BasicExpr*>::const_iterator i = cons.begin();i != cons.end();++i) {
	set<string> ids = Util::ComputeIdLvalues(*i);    
	varSet.insert(ids.begin(),ids.end());
	set<Expr> lvs = Util::ComputeExprLvalues(*i);
	lvSet.insert(lvs.begin(),lvs.end());
      }
      if(varSet.size() > 1) {
	string x = "(DISTINCT";
	for(set<string>::const_iterator i = varSet.begin();i != varSet.end();++i) {
	  string y = *i; SimplifyId(y);
	  x += " " + y;
	}
	x += ")";
	anteStr = "(AND " + x + " " + anteStr + ")";
      }
      Expr vpp = ExprManager::GetIdExpr(Database::VALID_PTR_PRED);
      for(set<Expr>::const_iterator i = lvSet.begin();i != lvSet.end();++i) {
	Expr a = ExprManager::GetUnaryExpr(*i,'&');
	list<Expr> el; el.push_back(a);
	Expr b = ExprManager::GetParExpr(vpp,el);
	string c = ConvertToBoolean(Value(b.GetExpr()));
	anteStr = "(AND " + c + " " + anteStr + ")";
      }
    }

    //create the implication formula
    string formula = "(IMPLIES " + anteStr + " " + consStr + ")";
    
    //first check the cache
    int x = CheckCache(formula);
    if(x != -1) return (x == 1);
    
    //add appropriate constraints
    AddConstraints(ante,cons);
    
    //send the formula to simplify
    WriteToSimplify(formula);
    totalQuerySize += formula.size();
    
    //read the reply. return true if the reply is Valid and false
    //if the reply is Invalid. flush the input.
    string reply = ReadFromSimplify();  
    if(totalQuerySize > MAX_TOTAL_QUERY_SIZE) {
      Shutdown();
      Initialise();
    }
    if(reply.find("Valid") != string::npos) {
      UpdateCache(formula,true);
      return true;
    } else if(reply.find("Invalid") != string::npos) {
      UpdateCache(formula,false);
      return false;
    } else {
      string output = "antecedent : " + Util::ExprListToString(ante) + 
	"\nconsequent : " + Util::ExprListToString(cons) + "\nformula : " + 
	formula + "\nunknown reply from Simplify : " + reply;
      Util::Error(output);
    }
    return false;
  } catch(SimplifyException ex) {
    assert(ex == NOT_HANDLED);
    return false;
  }
}
  
/*********************************************************************/
//add a constraint to the theorem prover environment. this constraint
//is provided as a C expression.
/*********************************************************************/
void SimplifyTP::AddConstraint(const BasicExpr *expr)
{
  try {
    string x = "(BG_PUSH " + Value(expr) + ")";
    WriteToSimplify(x);
    ReadFromSimplify();
  } catch(SimplifyException ex) {
    Util::Error("ERROR: unhandled exception in SimplifyTP ...\n");
  }
}

/*********************************************************************/
//read from simplfy till u get a ">"
/*********************************************************************/
string SimplifyTP::ReadFromSimplify()
{
#ifdef WIN32
  string res;
  char y[2];
  while(true) {
    DWORD bytesRead;
    if((!ReadFile(in,y,1,&bytesRead,NULL)) || (bytesRead != 1)) {
      Util::Error("ERROR: error reading from Simplify ...\n");
    }
    y[1] = '\0';
    res += y;
    if(y[0] == '>') break;
  }
  return res;
#else
  string res;
  char y[2]; y[1] = '\0';
  while(true) {
    int a = fgetc(in);
    if(a == EOF) {
      Util::Error("ERROR: error reading from Simplify ...\n");
    }
    y[0] = static_cast<char>(a);
    res += y;
    if(y[0] == '>') break;
  }
  return res;
#endif //WIN32
}

/*********************************************************************/
//write a line to simplify
/*********************************************************************/
void SimplifyTP::WriteToSimplify(const string &line)
{
#ifdef WIN32
  string newLine = line + "\n";
  DWORD bytesWritten;
  if((!WriteFile(out,newLine.c_str(),newLine.size(),&bytesWritten,NULL)) ||
     (bytesWritten != newLine.size())) {
    Util::Error("ERROR: error writing to Simplify ...\n");
  }
#else
  fprintf(out,"%s\n",line.c_str());
  fflush(out);
#endif //WIN32
}

/*********************************************************************/
//setup the theorem proving environment. load the axioms etc.
/*********************************************************************/
void SimplifyTP::SetupEnv()
{
  //load the dereference-address axiom
  string s1 = "(BG_PUSH (FORALL (x) (EQ x (" + POINTER_DEREF + " (" + ADDRESS + " x)))))";
  WriteToSimplify(s1);
  ReadFromSimplify();
  //load the address-dereference axiom
  string s2 = "(BG_PUSH (FORALL (x) (EQ x (" + ADDRESS + " (" + POINTER_DEREF + " x)))))";
  WriteToSimplify(s2);
  ReadFromSimplify();
  //if we are not ignoring pointers
  if(!Database::IGNORE_POINTERS) {
    //load the theorem that says if a pointer is valid then its non-NULL
    string s3 = "(BG_PUSH (FORALL (x) (IMPLIES (NEQ (" + PROCEDURE + "1 (" + VALUE + " " + VALID_PTR_PRED + ") x) 0) (NEQ x 0))))";
    WriteToSimplify(s3);
    ReadFromSimplify();
    //load the theorem that says if a pointer is valid then it can only
    //point to a another NULL or valid pointer
    string a = "(" + POINTER_DEREF + " (" + VALUE + " x))";
    string b = "(" + VALUE + " (" + FIELD_ACCESS + " " + a + " y))";
    string s4 = "(BG_PUSH (FORALL (x y) (IMPLIES (AND (NEQ (" + PROCEDURE + "1 (" + VALUE + " " + VALID_PTR_PRED + ") (" + VALUE + " x)) 0) (NEQ " + b + " 0)) ((NEQ " + PROCEDURE + "1 " + VALID_PTR_PRED  + " " + b + ") 0))))";
    WriteToSimplify(s4);
    ReadFromSimplify();
    s4 = "(BG_PUSH (FORALL (x y) (IMPLIES (EQ (" + PROCEDURE + "1 (" + VALUE + " " + VALID_PTR_PRED + ") (" + VALUE + " x)) 0) (EQ " + b + " 0))))";
    WriteToSimplify(s4);
    ReadFromSimplify();
    //address operator is injective
    string s5 = "(BG_PUSH (FORALL (x y) (IMPLIES (NEQ x y) (NEQ (" + ADDRESS + " x) (" + ADDRESS + " y)))))"; 
    WriteToSimplify(s5);
    ReadFromSimplify();
    //the selection operator is injective
    string s6 = "(BG_PUSH (FORALL (a b c d) (IMPLIES (OR (NEQ a b) (NEQ c d)) (NEQ (" + FIELD_ACCESS + "  a c) (" + FIELD_ACCESS + " b d)))))"; 
    WriteToSimplify(s6);
    ReadFromSimplify();
    //all array elements contains valid pointers
    string s7 = "(BG_PUSH (FORALL (x y) (NEQ (" + PROCEDURE + "1 (" + VALUE + " " + VALID_PTR_PRED + ") (" + VALUE + " (" + POINTER_DEREF + " (+ x y)))) 0)))";
    WriteToSimplify(s7);
    ReadFromSimplify();
    //if X is a valid pointer then so is X + Y
    string s8 = "(BG_PUSH (FORALL (x y) (IMPLIES (NEQ (" + PROCEDURE + "1 (" + VALUE + " " + VALID_PTR_PRED + ") x) 0) (NEQ (" + PROCEDURE + "1 (" + VALUE + " " + VALID_PTR_PRED + ") (+ x y)) 0))))";
    WriteToSimplify(s8);
    ReadFromSimplify();
  }
}

/*********************************************************************/
//convert a C expression ast to a simplify formula that represents its
//value. throw an exception if the expression uses a construct that is
//not handled yet.
/*********************************************************************/
string SimplifyTP::Value(const BasicExpr *node) throw (SimplifyException)
{
  //id_expression
  if(typeid(*node) == typeid(IdExpr)) {
    string x = Util::TrimString(node->ToString());
    SimplifyId(x);
    return "(" + VALUE + " " + x + ")";
  }
  //int_const_expression
  else if(typeid(*node) == typeid(IntConstExpr)) {
    return Util::TrimString(node->ToString());
  }
  //const_expression
  else if(typeid(*node) == typeid(ConstExpr)) {
    throw NOT_HANDLED;
    //Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a value\n");
  }
  //str_expression
  else if(typeid(*node) == typeid(StrExpr)) {
    throw NOT_HANDLED;
    //Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a value\n");
  }
  //str_str_expression
  else if(typeid(*node) == typeid(StrStrExpr)) {
    throw NOT_HANDLED;
    //Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a value\n");
  }
  //stmt_expression
  else if(typeid(*node) == typeid(StmtExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a value\n");
  }
  //brack_expression
  else if(typeid(*node) == typeid(BrackExpr)) {
    const BrackExpr *x = static_cast<const BrackExpr*>(node);
    string a = Value(x->array);
    string b = Value(x->index);    
    return "(" + VALUE + " (" + POINTER_DEREF + " (+ " + a + " " + b + ")))";
  }
  //par_expression
  else if(typeid(*node) == typeid(ParExpr)) {
    const ParExpr *x = static_cast<const ParExpr*>(node);
    list<BasicExpr*> &a = x->args->data;
    char buf[50];
    snprintf(buf,50,"%d ",a.size());
    string y = "(" + PROCEDURE + buf + Value(x->proc);
    for(list<BasicExpr*>::const_iterator i = a.begin();i != a.end();++i) {
      y += " " + Value(*i);
    }
    y += ")";
    return y;
  }
  //dot_expression
  else if(typeid(*node) == typeid(DotExpr)) {
    const DotExpr *x = static_cast<const DotExpr*>(node);
    string y = Location(x->expr);
    string z = x->id; SimplifyId(z);
    if(fieldMap.count(z) == 0) {
      char buf[64];
      snprintf(buf,64,"%d",fieldMap.size());
      fieldMap[z] = buf;
    }
    z = fieldMap[z];
    return "(" + VALUE + " (" + FIELD_ACCESS + " " + y + " " + z + "))";
  }
  //arrow_expression
  else if(typeid(*node) == typeid(ArrowExpr)) {
    const ArrowExpr *x = static_cast<const ArrowExpr*>(node);
    string y = Value(x->expr);
    string z = x->id; SimplifyId(z);
    if(fieldMap.count(z) == 0) {
      char buf[64];
      snprintf(buf,64,"%d",fieldMap.size());
      fieldMap[z] = buf;
    }
    z = fieldMap[z];
    SimplifyId(z);
    string a = "(" + POINTER_DEREF + " " + y + ")";
    return "(" + VALUE + " (" + FIELD_ACCESS + " " + a + " " + z + "))";
  }
  //inc_expression
  else if(typeid(*node) == typeid(IncExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a value\n");
  }
  //unary_expression
  else if(typeid(*node) == typeid(UnaryExpr)) {
    const UnaryExpr *x = static_cast<const UnaryExpr*>(node);
    if(x->op == '&') {
      return "(" + ADDRESS + " " + Location(x->expr) + ")";
    } else if(x->op == '*') {
      return "(" + VALUE + " (" + POINTER_DEREF + Value(x->expr) + "))";
    } else if(x->op == '+') {
      return "(+ 0 " + Value(x->expr) + ")";
    } else if(x->op == '-') {
      return "(- 0 " + Value(x->expr) + ")";
    } else if(x->op == '~') {
      return "(" + BITWISE_NOT + " " + Value(x->expr) + ")";
    } else {
      return "(NOT " + ConvertToBoolean(Value(x->expr)) + ")";
    }
  }
  //empty_expression
  else if(typeid(*node) == typeid(EmptyExpr)) {    
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a value\n");
  }
  //binary_expression
  else if(typeid(*node) == typeid(BinaryExpr)) {
    const BinaryExpr *x = static_cast<const BinaryExpr*>(node);
    string y = Value(x->lhs);
    string z = Value(x->rhs);
 
    if(x->op == '*') return "(* " + y + " " + z + ")";
    else if(x->op == '/') Util::Error("ERROR: simplify cannot handle " + node->ToString() + "\n");
    else if(x->op == '%') Util::Error("ERROR: simplify cannot handle " + node->ToString() + "\n");
    else if(x->op == '+') return "(+ " + y + " " + z + ")";
    else if(x->op == '-') return "(- " + y + " " + z + ")";
    else if(x->op == MAGIC_LEFT_OP) return "(" + LEFT_SHIFT + " " + y + " " + z + ")";
    else if(x->op == MAGIC_RIGHT_OP) return "(" + RIGHT_SHIFT + " " + y + " " + z + ")";
    else if(x->op == '<') return "(< " + y + " " + z + ")";
    else if(x->op == '>') return "(> " + y + " " + z + ")";
    else if(x->op == MAGIC_LE_OP) return "(<= " + y + " " + z + ")";
    else if(x->op == MAGIC_GE_OP) return "(>= " + y + " " + z + ")";
    else if(x->op == MAGIC_EQ_OP) return "(EQ " + y + " " + z + ")";
    else if(x->op == MAGIC_NE_OP) return "(NEQ " + y + " " + z + ")";
    else if(x->op == '&') return "(" + BITWISE_AND + " " + y + " " + z + ")";
    else if(x->op == '^') return "(" + BITWISE_XOR + " " + y + " " + z + ")";
    else if(x->op == '|') return "(" + BITWISE_OR + " " + y + " " + z + ")";
    else if(x->op == MAGIC_AND_OP) {
      //convert both sub-expressions to boolean form by
      //comparing with zero
      y = ConvertToBoolean(y);
      z = ConvertToBoolean(z);
    
      return "(AND " + y + " " + z + ")";
    } else if(x->op == MAGIC_OR_OP) {
      //convert both sub-expressions to boolean form by
      //comparing with zero
      y = ConvertToBoolean(y);
      z = ConvertToBoolean(z);
    
      return "(OR " + y + " " + z + ")";
    } else Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a value\n");
  }
  //quest_expression
  else if(typeid(*node) == typeid(QuestExpr)) {
    const QuestExpr *x = static_cast<const QuestExpr*>(node);
    //convert an expression of the form A ? B : C to (A && B)
    //|| (!A && C)
    BasicExpr *a = new BinaryExpr(x->cond,x->tcase,MAGIC_AND_OP);
    BasicExpr *b = x->cond->Negate();
    BasicExpr *c = new BinaryExpr(b,x->ecase,MAGIC_AND_OP);
    delete b;
    BasicExpr *d = new BinaryExpr(a,c,MAGIC_OR_OP);
    delete a; delete c;
    string res = Value(d);
    delete d;
    return res;
  }
  //assign_expression
  else if(typeid(*node) == typeid(AssignExpr)) {
    const AssignExpr *x = static_cast<const AssignExpr*>(node);
    return Value(x->lhs);
  }
  //illegal expression
  else {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a value\n");
  }
}

/*********************************************************************/
//convert a C expression ast to a simplify formula that represents its
//value. throw an exception if the expression uses a construct that is
//not handled yet.
/*********************************************************************/
string SimplifyTP::Location(const BasicExpr *node) throw (SimplifyException)
{
  //id_expression
  if(typeid(*node) == typeid(IdExpr)) {
    string x = Util::TrimString(node->ToString());
    SimplifyId(x);
    return x;
  }
  //int_const_expression
  else if(typeid(*node) == typeid(IntConstExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //const_expression
  else if(typeid(*node) == typeid(ConstExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //str_expression
  else if(typeid(*node) == typeid(StrExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //str_str_expression
  else if(typeid(*node) == typeid(StrStrExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //stmt_expression
  else if(typeid(*node) == typeid(StmtExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //brack_expression
  else if(typeid(*node) == typeid(BrackExpr)) {
    const BrackExpr *x = static_cast<const BrackExpr*>(node);
    string a = Value(x->array);
    string b = Value(x->index);    
    return "(" + POINTER_DEREF + " (+ " + a + " " + b + "))";
  }
  //par_expression
  else if(typeid(*node) == typeid(ParExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //dot_expression
  else if(typeid(*node) == typeid(DotExpr)) {
    const DotExpr *x = static_cast<const DotExpr*>(node);
    string y = Location(x->expr);
    string z = x->id; SimplifyId(z);
    if(fieldMap.count(z) == 0) {
      char buf[64];
      snprintf(buf,64,"%d",fieldMap.size());
      fieldMap[z] = buf;
    }
    z = fieldMap[z];
    return "(" + FIELD_ACCESS + " " + y + " " + z + ")";
  }
  //arrow_expression
  else if(typeid(*node) == typeid(ArrowExpr)) {
    const ArrowExpr *x = static_cast<const ArrowExpr*>(node);
    string y = Value(x->expr);
    string z = x->id; SimplifyId(z);
    if(fieldMap.count(z) == 0) {
      char buf[64];
      snprintf(buf,64,"%d",fieldMap.size());
      fieldMap[z] = buf;
    }
    z = fieldMap[z];
    SimplifyId(z);
    string a = "(" + POINTER_DEREF + " " + y + ")";
    return "(" + FIELD_ACCESS + " " + a + " " + z + ")";
  }
  //inc_expression
  else if(typeid(*node) == typeid(IncExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //unary_expression
  else if(typeid(*node) == typeid(UnaryExpr)) {
    const UnaryExpr *x = static_cast<const UnaryExpr*>(node);
    if(x->op == '&') {
      Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
    } else if(x->op == '*') {
      return "(" + POINTER_DEREF + Value(x->expr) + ")";
    } else {
      Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
    }
  }
  //empty_expression
  else if(typeid(*node) == typeid(EmptyExpr)) {    
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //binary_expression
  else if(typeid(*node) == typeid(BinaryExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //quest_expression
  else if(typeid(*node) == typeid(QuestExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //assign_expression
  else if(typeid(*node) == typeid(AssignExpr)) {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
  //illegal expression
  else {
    Util::Error("ERROR: cannot convert expression " + node->ToString() + "to a location\n");
  }
}

/*********************************************************************/
//convert a formula to boolean form by comparing with 0
/*********************************************************************/
string SimplifyTP::ConvertToBoolean(const string &old)
{
  if(old.at(0) != '(') return "(NEQ " + old + " 0)";
  for(list<string>::const_iterator i = boolCheck.begin();i != boolCheck.end();++i) {
    if(strstr(old.c_str(),i->c_str()) == old.c_str()) return  "(NEQ " + old + " 0)";
  }
  return old;
}

/*********************************************************************/
//check if a formula belongs to a cache. returns 1 if the formula is
//in the true cache, 0 if it is in the false cache and -1
//otherwise. also updates the theorem prover statistics accordingly.
/*********************************************************************/
int SimplifyTP::CheckCache(const string &formula)
{
  if(trueCache.count(formula) != 0) {
    ++Statistics::tpInCache;
    return 1;
  } else if(falseCache.count(formula) != 0) {
    ++Statistics::tpInCache;
    return 0;
  } else {
    ++Statistics::tpCalls;
    ++Statistics::tpCacheSize;
    return -1;
  }
}

/*********************************************************************/
//add appropriate enumeration constraints given two lists of
//formulas
/*********************************************************************/
void SimplifyTP::AddConstraints(const list<const BasicExpr*> &ante,const list<const BasicExpr*> &cons)
{  
  //combine the two lists
  list<const BasicExpr*> all = ante;
  all.insert(all.end(),cons.begin(),cons.end());
  
  //compute the set of ids in the expressions
  set<string> enumList;
  for(list<const BasicExpr*>::const_iterator i = all.begin();i != all.end();++i) {
    set<string> ids = Util::ComputeIdLvalues(*i);    
    for(set<string>::const_iterator j = ids.begin();j != ids.end();++j) {
      if(enumsAdded.count(*j) == 0) enumList.insert(*j);
    }
  }  
  //get the enum map
  const map<string,const BasicExpr*> &enums = Database::enums;
  //now do the stuff
  while(!enumList.empty()) {
    string a = *(enumList.begin());
    enumList.erase(enumList.begin());
    if(enums.count(a) != 0) {
      if(enumsAdded.count(a) == 0) {
	const BasicExpr *lhs = new IdExpr(a);
	const BasicExpr *rhs = enums.find(a)->second;
	BasicExpr *eq = new BinaryExpr(lhs,rhs,MAGIC_EQ_OP);
	AddConstraint(eq);
	enumsAdded.insert(a);	
	set<string> b = Util::ComputeIdLvalues(rhs);
	for(set<string>::const_iterator i = b.begin();i != b.end();++i) {
	  if(enumsAdded.count(*i) == 0) {
	    enumList.insert(*i);
	  }
	}	
	delete lhs; delete eq;
      }
    }
  }
}

/*********************************************************************/
//add a new cache entry. the first argument is the entry. the second
//argument is the type of the cache to be updated.
/*********************************************************************/
void SimplifyTP::UpdateCache(const string &formula,bool type)
{
  if(Database::THEOREM_PROVER_CACHE) {
    if(type) {
      if(static_cast<int>(trueCache.size()) > Database::THEOREM_PROVER_CACHE_SIZE) {
	trueCache.clear();
      }
      trueCache.insert(formula);
    } else {
      if(static_cast<int>(falseCache.size()) > Database::THEOREM_PROVER_CACHE_SIZE) {
	falseCache.clear();
      }
      falseCache.insert(formula);
    }
  }
}

/*********************************************************************/
//end of SimplifyTP.cpp
/*********************************************************************/
