/**************************** CPPHeaderFile ***************************

* FileName [Stmt.h]

* PackageName [parser]

* Synopsis [Header file for Stmt classes.]

* Description [These classes encapsulate statement nodes in the AST.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __STMT_H__
#define __STMT_H__

namespace magic {

/*********************************************************************/
//statement
/*********************************************************************/
class Stmt : public Node {};

/*********************************************************************/
//list of statements
/*********************************************************************/
class StmtList : public Node
{
 private:
  list<Stmt*> data;
 public:
  StmtList() {}
  StmtList(const list<Stmt*> &d) { 
    for(list<Stmt*>::const_iterator i = d.begin();i != d.end();++i) {
      data.push_back(static_cast<Stmt*>((*i)->Clone()));
    }
  }
  StmtList(const StmtList &rhs) { *this = rhs; }
  ~StmtList() { Cleanup(); }
  const StmtList &operator = (const StmtList &rhs) {
    Cleanup();
    for(list<Stmt*>::const_iterator i = rhs.data.begin();i != rhs.data.end();++i) {
      data.push_back(static_cast<Stmt*>((*i)->Clone()));
    }
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new StmtList(data)); }
  void Cleanup() {
    for(list<Stmt*>::iterator i = data.begin();i != data.end();++i) {
      delete *i;
    }
    data.clear();
  }

  string ToString() const {
    string res;
    for(list<Stmt*>::const_iterator i = data.begin();i != data.end();++i) {
      res += (*i)->ToString();
    } 
    return res + " "; 
  }
  void Apply(DFSAdapter &a) const;

  //add a new statement at the end of the list
  void AddStmt(Stmt *e) { data.push_back(static_cast<Stmt*>(e->Clone())); }

  //add a list of statements at the end
  void AddStmtList(const StmtList &arg) {
    for(list<Stmt*>::const_iterator i = arg.data.begin();i != arg.data.end();++i) {
      data.push_back(static_cast<Stmt*>((*i)->Clone()));
    }
  }

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//labeled statement
/*********************************************************************/
class LabelStmt : public Stmt
{
 private:
  string id;
  Stmt *stmt;
 public:
  LabelStmt() { stmt = NULL; }
  LabelStmt(const string &s,const Stmt *d) { 
    id = s;
    stmt = (d == NULL) ? NULL : static_cast<Stmt*>(d->Clone()); 
  }
  LabelStmt(const LabelStmt &rhs) { stmt = NULL; *this = rhs; }
  ~LabelStmt() { Cleanup(); }
  const LabelStmt &operator = (const LabelStmt &rhs) {
    Cleanup();
    id = rhs.id;
    stmt = (rhs.stmt == NULL) ? NULL : static_cast<Stmt*>(rhs.stmt->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new LabelStmt(id,stmt)); }
  void Cleanup() {
    if(stmt != NULL) delete stmt;
    stmt = NULL;
  }

  string ToString() const { return id + ": " + stmt->ToString(); }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//case statement
/*********************************************************************/
class CaseStmt : public Stmt
{
 private:
  BasicExpr *expr;
  Stmt *stmt;
 public:
  CaseStmt() { expr = NULL; stmt = NULL; }
  CaseStmt(const BasicExpr *e,const Stmt *d) { 
    expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
    stmt = (d == NULL) ? NULL : static_cast<Stmt*>(d->Clone()); 
  }
  CaseStmt(const CaseStmt &rhs) { expr = NULL; stmt = NULL; *this = rhs; }
  ~CaseStmt() { Cleanup(); }
  const CaseStmt &operator = (const CaseStmt &rhs) {
    Cleanup();
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    stmt = (rhs.stmt == NULL) ? NULL : static_cast<Stmt*>(rhs.stmt->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new CaseStmt(expr,stmt)); }
  void Cleanup() {
    if(expr != NULL) delete expr;
    expr = NULL;
    if(stmt != NULL) delete stmt;
    stmt = NULL;
  }

  string ToString() const { return "case " + expr->ToString()+ ": " + stmt->ToString(); }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//default statement
/*********************************************************************/
class DefaultStmt : public Stmt
{
 private:
  Stmt *stmt;
 public:
  DefaultStmt() { stmt = NULL; }
  DefaultStmt(const Stmt *d) { stmt = (d == NULL) ? NULL : static_cast<Stmt*>(d->Clone()); }
  DefaultStmt(const DefaultStmt &rhs) { stmt = NULL; *this = rhs; }
  ~DefaultStmt() { Cleanup(); }
  const DefaultStmt &operator = (const DefaultStmt &rhs) {
    Cleanup();
    stmt = (rhs.stmt == NULL) ? NULL : static_cast<Stmt*>(rhs.stmt->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new DefaultStmt(stmt)); }
  void Cleanup() {
    if(stmt != NULL) delete stmt;
    stmt = NULL;
  }

  string ToString() const { return "default : " + stmt->ToString(); }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//compound statement
/*********************************************************************/
class CompStmt : public Stmt
{
 private:
  StmtList *data;
 public:
  CompStmt() { data = NULL; }
  CompStmt(const StmtList *d) { data = (d == NULL) ? NULL : static_cast<StmtList*>(d->Clone()); }
  CompStmt(const CompStmt &rhs) { data = NULL; *this = rhs; }
  ~CompStmt() { Cleanup(); }
  const CompStmt &operator = (const CompStmt &rhs) {
    Cleanup();
    data = (rhs.data == NULL) ? NULL : static_cast<StmtList*>(rhs.data->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new CompStmt(data)); }
  void Cleanup() {
    if(data != NULL) delete data;
    data = NULL;
  }

  string ToString() const { return "{ " + data->ToString() + "} "; }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend class ProcInfo;
};

/*********************************************************************/
//expression statement
/*********************************************************************/
class ExprStmt : public Stmt
{
 private:
  BasicExpr *expr;
 public:
  ExprStmt() { expr = NULL; }
  ExprStmt(const BasicExpr *d) { expr = (d == NULL) ? NULL : static_cast<BasicExpr*>(d->Clone()); }
  ExprStmt(const ExprStmt &rhs) { expr = NULL; *this = rhs; }
  ~ExprStmt() { Cleanup(); }
  const ExprStmt &operator = (const ExprStmt &rhs) {
    Cleanup();
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new ExprStmt(expr)); }
  void Cleanup() {
    if(expr != NULL) delete expr;
    expr = NULL;
  }

  string ToString() const { return expr->ToString() + "; "; }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend Stmt *ForToWhile(Stmt *one,Stmt *two,BasicExpr *three,Stmt *body);
};

/*********************************************************************/
//empty statement - just a placeholder
/*********************************************************************/
class EmptyStmt : public Stmt
{
 public:
  EmptyStmt() {}
  EmptyStmt(const EmptyStmt &rhs) { *this = rhs; }
  ~EmptyStmt() {}
  const EmptyStmt &operator = (const EmptyStmt &rhs) {
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new EmptyStmt()); }
  void Cleanup() { assert(false); }

  string ToString() const { return "{} "; }
  void Apply(DFSAdapter &a) const;
};

/*********************************************************************/
//if-then-else statement
/*********************************************************************/
class IfStmt : public Stmt
{
 private:
  BasicExpr *expr;
  Stmt *tstmt,*estmt;
 public:
  IfStmt() { expr = NULL; tstmt = estmt = NULL; }
  IfStmt(const BasicExpr *c,const Stmt *t,const Stmt *e) { 
    expr = (c == NULL) ? NULL : static_cast<BasicExpr*>(c->Clone());
    tstmt = (t == NULL) ? NULL : static_cast<Stmt*>(t->Clone()); 
    estmt = (e == NULL) ? NULL : static_cast<Stmt*>(e->Clone()); 
  }
  IfStmt(const IfStmt &rhs) { expr = NULL; tstmt = estmt = NULL; *this = rhs; }
  ~IfStmt() { Cleanup(); }
  const IfStmt &operator = (const IfStmt &rhs) {
    Cleanup();
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    tstmt = (rhs.tstmt == NULL) ? NULL : static_cast<Stmt*>(rhs.tstmt->Clone());
    estmt = (rhs.estmt == NULL) ? NULL : static_cast<Stmt*>(rhs.estmt->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new IfStmt(expr,tstmt,estmt)); }
  void Cleanup() {
    if(expr != NULL) delete expr;
    expr = NULL;
    if(tstmt != NULL) delete tstmt;
    tstmt = NULL;
    if(estmt != NULL) delete estmt;
    estmt = NULL;
  }

  string ToString() const { 
    return "if ( " + expr->ToString()+ ") " + tstmt->ToString() + "else " + estmt->ToString(); 
  }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//switch statement
/*********************************************************************/
class SwitchStmt : public Stmt
{
 private:
  BasicExpr *expr;
  Stmt *stmt;
 public:
  SwitchStmt() { expr = NULL; stmt = NULL; }
  SwitchStmt(const BasicExpr *e,const Stmt *d) { 
    expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
    stmt = (d == NULL) ? NULL : static_cast<Stmt*>(d->Clone()); 
  }
  SwitchStmt(const SwitchStmt &rhs) { expr = NULL; stmt = NULL; *this = rhs; }
  ~SwitchStmt() { Cleanup(); }
  const SwitchStmt &operator = (const SwitchStmt &rhs) {
    Cleanup();
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    stmt = (rhs.stmt == NULL) ? NULL : static_cast<Stmt*>(rhs.stmt->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new SwitchStmt(expr,stmt)); }
  void Cleanup() {
    if(expr != NULL) delete expr;
    expr = NULL;
    if(stmt != NULL) delete stmt;
    stmt = NULL;
  }

  string ToString() const { return "switch ( " + expr->ToString() + ") " + stmt->ToString(); }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//while statement
/*********************************************************************/
class WhileStmt : public Stmt
{
 private:
  BasicExpr *expr;
  Stmt *stmt;
 public:
  WhileStmt() { expr = NULL; stmt = NULL; }
  WhileStmt(const BasicExpr *e,const Stmt *d) { 
    expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
    stmt = (d == NULL) ? NULL : static_cast<Stmt*>(d->Clone()); 
  }
  WhileStmt(const WhileStmt &rhs) { expr = NULL; stmt = NULL; *this = rhs; }
  ~WhileStmt() { Cleanup(); }
  const WhileStmt &operator = (const WhileStmt &rhs) {
    Cleanup();
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    stmt = (rhs.stmt == NULL) ? NULL : static_cast<Stmt*>(rhs.stmt->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new WhileStmt(expr,stmt)); }
  void Cleanup() {
    if(expr != NULL) delete expr;
    expr = NULL;
    if(stmt != NULL) delete stmt;
    stmt = NULL;
  }

  string ToString() const { return "while ( " + expr->ToString() + ") " + stmt->ToString(); }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//do statement
/*********************************************************************/
class DoStmt : public Stmt
{
 private:
  BasicExpr *expr;
  Stmt *stmt;
 public:
  DoStmt() { expr = NULL; stmt = NULL; }
  DoStmt(const BasicExpr *e,const Stmt *d) { 
    expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
    stmt = (d == NULL) ? NULL : static_cast<Stmt*>(d->Clone()); 
  }
  DoStmt(const DoStmt &rhs) { expr = NULL; stmt = NULL; *this = rhs; }
  ~DoStmt() { Cleanup(); }
  const DoStmt &operator = (const DoStmt &rhs) {
    Cleanup();
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    stmt = (rhs.stmt == NULL) ? NULL : static_cast<Stmt*>(rhs.stmt->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new DoStmt(expr,stmt)); }
  void Cleanup() {
    if(expr != NULL) delete expr;
    expr = NULL;
    if(stmt != NULL) delete stmt;
    stmt = NULL;
  }

  string ToString() const { return "do " + stmt->ToString() + "while ( " + expr->ToString() + ") ; "; }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//for statement
/*********************************************************************/
class ForStmt : public Stmt
{
 private:
  Stmt *init;
  Stmt *cond;
  BasicExpr *update;
  Stmt *stmt;
 public:
  ForStmt() { init = cond = stmt = NULL; update = NULL; }
  ForStmt(const Stmt *i,const Stmt *c,const BasicExpr *u,const Stmt *s) {
    init = (i == NULL) ? NULL : static_cast<Stmt*>(i->Clone());
    cond = (c == NULL) ? NULL : static_cast<Stmt*>(c->Clone()); 
    update = (u == NULL) ? NULL : static_cast<BasicExpr*>(u->Clone());
    stmt = (s == NULL) ? NULL : static_cast<Stmt*>(s->Clone()); 
  }
  ForStmt(const ForStmt &rhs) { init = cond = stmt = NULL; update = NULL; *this = rhs; }
  ~ForStmt() { Cleanup(); }
  const ForStmt &operator = (const ForStmt &rhs) {
    Cleanup();
    init = (rhs.init == NULL) ? NULL : static_cast<Stmt*>(rhs.init->Clone());
    cond = (rhs.cond == NULL) ? NULL : static_cast<Stmt*>(rhs.cond->Clone()); 
    update = (rhs.update == NULL) ? NULL : static_cast<BasicExpr*>(rhs.update->Clone());
    stmt = (rhs.stmt == NULL) ? NULL : static_cast<Stmt*>(rhs.stmt->Clone()); 
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new ForStmt(init,cond,update,stmt)); }
  void Cleanup() {
    if(init != NULL) delete init;
    init = NULL;
    if(cond != NULL) delete cond;
    cond = NULL;
    if(update != NULL) delete update;
    update = NULL;
    if(stmt != NULL) delete stmt;
    stmt = NULL;
  }

  string ToString() const { 
    return "for ( " + init->ToString() + cond->ToString() + update->ToString() + ") " + stmt->ToString(); 
  }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
};

/*********************************************************************/
//goto statement
/*********************************************************************/
class GotoStmt : public Stmt
{
 private:
  string id;
 public:
  GotoStmt() {}
  GotoStmt(const string &s) { id = s; }
  GotoStmt(const GotoStmt &rhs) { *this = rhs; }
  ~GotoStmt() {}
  const GotoStmt &operator = (const GotoStmt &rhs) {
    id = rhs.id;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new GotoStmt(id)); }
  void Cleanup() { assert(false); }

  string ToString() const { return "goto " + id; }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class ContLocExtractor;
};

/*********************************************************************/
//continue statement
/*********************************************************************/
class ContinueStmt : public Stmt
{
 public:
  ContinueStmt() {}
  ContinueStmt(const ContinueStmt &rhs) { *this = rhs; }
  ~ContinueStmt() {}
  const ContinueStmt &operator = (const ContinueStmt &rhs) {
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new ContinueStmt()); }
  void Cleanup() { assert(false); }

  string ToString() const { return "continue ; "; }
  void Apply(DFSAdapter &a) const;
};

/*********************************************************************/
//break statement
/*********************************************************************/
class BreakStmt : public Stmt
{
 public:
  BreakStmt() {}
  BreakStmt(const BreakStmt &rhs) { *this = rhs; }
  ~BreakStmt() {}
  const BreakStmt &operator = (const BreakStmt &rhs) {
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new BreakStmt()); }
  void Cleanup() { assert(false); }

  string ToString() const { return "break ; "; }
  void Apply(DFSAdapter &a) const;
};

/*********************************************************************/
//return statement
/*********************************************************************/
class ReturnStmt : public Stmt
{
 private:
  BasicExpr *expr;
 public:
  ReturnStmt() { expr = NULL; }
  ReturnStmt(const BasicExpr *e) { expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone()); }
  ReturnStmt(const ReturnStmt &rhs) { expr = NULL; *this = rhs; }
  ~ReturnStmt() { Cleanup(); }
  const ReturnStmt &operator = (const ReturnStmt &rhs) {
    Cleanup();
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new ReturnStmt(expr)); }
  void Cleanup() {
    if(expr != NULL) delete expr;
    expr = NULL;
  }

  string ToString() const { return "return " + expr->ToString()+ "; "; }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

} //namespace magic

#endif //__STMT_H__

/*********************************************************************/
//end of Stmt.h
/*********************************************************************/
