%{
#include <cstdio>
#include <cassert>
#include <typeinfo>
#include <string>
#include <set>
#include <list>
#include <map>
#include <vector>
#include <typeinfo>
using namespace std;

#include "Util.h"
#include "CSymbolTable.h"
#include "Node.h"
#include "Action.h"
#include "ActionManager.h"
#include "Alias.h"
#include "ProgAbs.h"
#include "ProgInfo.h"
#include "ProcAbs.h"
#include "ProcAddInfo.h"
#include "LtsTrans.h"
#include "FspInfo.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Seaw.h"
#include "SeawManager.h"
#include "Database.h"
using namespace magic;

extern char Stdctext[];
extern int lineNum,column;
void Stdcerror(char *s);
int Stdclex();
__attribute__ ((noreturn)) void GlobalShutdown(int sig);

//the last value assigned to an enumerator constant
BasicExpr *lastEnumVal = new IntConstExpr(-1);

namespace magic {

//this type stores a pair of a variable and a statement. it represents
//a variable declaration and initialization. the statement is the
//assignment of the variable to its initial value.
class DeclStmt { public: string first; Stmt *second; };

//this type stores a pair of a list of strings and a list of
//statements. it is used to represent declarations and compound
//statements. the list of strings represents the variables declared
//and the statements represent the assignments of variables to initial
//values and other statements in case of a compound statement.
class DeclStmtList { public: list<string> first; StmtList *second; };

//this class is needed to represent a FSP process body. it could
//either be a string or a list of transitions without initial states.
class FspBody { public: string id; list<LtsTrans> tr; };

//this class is needed to represent a sequence of FSP transitions. it
//consists of a list of transitions at the beginning of the sequence
//and a list of transitions at the end of the sequence.
class FspTransSeq { public: list<LtsTrans> begin,end; };

//utility routines
BasicExpr *SimplifyExpr(BasicExpr *arg);
list<LtsTrans> * MakeTransList(FspTransSeq *seq,FspBody *body); 
Stmt *ForToWhile(Stmt *one,Stmt *two,BasicExpr *three,Stmt *body);

} //namespace magic
%}

%union {
  char cval;
  int ival;
  long long lval;
  string *strptr;
  list<string> *strlistptr;
  magic::Node *nptr;
  magic::BasicExpr *eptr;
  magic::ExprList *elptr;
  magic::Stmt *sptr;
  magic::StmtList *slptr;
  magic::CompStmt *csptr;
  magic::Decl *dptr;
  magic::DeclStmt *dsptr;
  magic::DeclStmtList *dslptr;
  magic::Proc *pptr;
  magic::StdcFile *stdcfptr;
  magic::ProgInfo *piptr;
  magic::ProcAddInfo *paiptr;
  magic::Alias *aliasptr;
  list<magic::Alias> *aliaslptr;
  magic::ProcAbs *pabsptr;
  list<magic::ProcAbs> *pabslptr;
  magic::ProgAbs *pgabsptr;
  list<magic::ProgAbs> *pgabslptr;
  magic::Action *aptr;
  set<magic::Action> *asptr;
  list<magic::LtsTrans> *tlptr;
  magic::FspBody *fbptr;
  magic::FspTransSeq *ftsptr;
  magic::BasicLtl *ltlptr;
  magic::OmegaExpr *oeptr;
  magic::BasicSeaw *bsptr;
  list<magic::BasicSeaw*> *bslptr;
}

%expect 1

%token MAGIC_IDENTIFIER MAGIC_INT_CONSTANT MAGIC_CONSTANT MAGIC_STRING_LITERAL 
%token MAGIC_SIZEOF MAGIC_PTR_OP MAGIC_INC_OP MAGIC_DEC_OP MAGIC_LEFT_OP 
%token MAGIC_RIGHT_OP MAGIC_LE_OP MAGIC_GE_OP MAGIC_EQ_OP MAGIC_NE_OP MAGIC_IMPLIES
%token MAGIC_AND_OP MAGIC_OR_OP MAGIC_MUL_ASSIGN MAGIC_DIV_ASSIGN MAGIC_MOD_ASSIGN 
%token MAGIC_ADD_ASSIGN MAGIC_SUB_ASSIGN MAGIC_LEFT_ASSIGN MAGIC_RIGHT_ASSIGN 
%token MAGIC_AND_ASSIGN MAGIC_XOR_ASSIGN MAGIC_OR_ASSIGN MAGIC_TYPE_NAME

%token MAGIC_TYPEDEF MAGIC_TYPEOF MAGIC_EXTERN MAGIC_STATIC MAGIC_AUTO MAGIC_REGISTER
%token MAGIC_CHAR MAGIC_SHORT MAGIC_INT MAGIC_LONG MAGIC_SIGNED MAGIC_UNSIGNED 
%token MAGIC_FLOAT MAGIC_DOUBLE MAGIC_CONST MAGIC_VOLATILE MAGIC_VOID
%token MAGIC_STRUCT MAGIC_UNION MAGIC_ENUM MAGIC_ELLIPSIS

%token MAGIC_CASE MAGIC_DEFAULT MAGIC_IF MAGIC_ELSE MAGIC_SWITCH MAGIC_WHILE 
%token MAGIC_DO MAGIC_FOR MAGIC_GOTO MAGIC_CONTINUE MAGIC_BREAK MAGIC_RETURN

%token MAGIC_PREDICATE MAGIC_AUXILIARY MAGIC_INLINE MAGIC_ALIAS MAGIC_ABSTRACT 
%token MAGIC_DOT_DOT MAGIC_RANGE MAGIC_SET MAGIC_STOP MAGIC_WHEN MAGIC_PROPERTY 
%token MAGIC_LOWER_ID MAGIC_UPPER_ID MAGIC_CPROG MAGIC_CPROC MAGIC_CONTEXT
%token MAGIC_NEXT_TIME MAGIC_GLOBALLY MAGIC_FUTURE MAGIC_UNTIL MAGIC_RELEASE
%token MAGIC_FAIR_LOOP MAGIC_BANG_BANG MAGIC_DUMMY_VAR MAGIC_OMEGA_OP_NAME
%token MAGIC_SEAW_NAME MAGIC_ALL_PATHS

%type <lval> int_constant
%type <strptr> identifier constant string_literal parameter_declaration

%type <eptr> primary_expression postfix_expression
%type <elptr> argument_expression_list
%type <eptr> unary_expression
%type <cval> unary_operator
%type <eptr> cast_expression multiplicative_expression additive_expression
%type <eptr> shift_expression relational_expression equality_expression
%type <eptr> and_expression exclusive_or_expression inclusive_or_expression
%type <eptr> logical_and_expression logical_or_expression conditional_expression
%type <eptr> assignment_expression expression constant_expression
%type <ival> assignment_operator

%type <sptr> statement labeled_statement expression_statement
%type <sptr> selection_statement iteration_statement jump_statement
%type <slptr> statement_list

%type <dsptr> init_declarator
%type <dslptr> declaration_list declaration init_declarator_list compound_statement

%type <dptr> declarator direct_declarator
%type <strlistptr> parameter_type_list identifier_list parameter_list
%type <eptr> initializer

%type <nptr> declaration_specifiers storage_class_specifier type_specifier type_qualifier
%type <nptr> struct_or_union_specifier enum_specifier type_name struct_or_union
%type <nptr> struct_declaration_list struct_declaration specifier_qualifier_list
%type <nptr> struct_declarator_list struct_declarator enumerator_list enumerator
%type <nptr> pointer type_qualifier_list abstract_declarator direct_abstract_declarator
%type <nptr> list_initializer initializer_list initializer_element translation_unit

%type <pptr> external_declaration function_definition
%type <stdcfptr> stdc_translation_unit

%type <strptr> lower_id upper_id process_id
%type <eptr> c_proc_name
%type <elptr> predicate_list c_proc_name_list predicate_decl fair_loop_decl
%type <elptr> auxiliary_decl inline_decl points_to_list
%type <piptr> c_prog_info
%type <paiptr> c_proc_decls c_proc_info
%type <aliaslptr> alias_decl alias_item_list
%type <pabslptr> abstract_decl abstract_item_list
%type <pgabslptr> c_prog_decls
%type <pgabsptr> c_prog_decl
%type <aliasptr> alias_item
%type <pabsptr> abstract_item
%type <lval> context_decl
%type <aptr> action_label action_labels
%type <asptr> action_label_list
%type <nptr> ext_def_list spec_translation_unit fsp_definition ltl_formula_def
%type <nptr> process_definition local_process_defs local_process_def
%type <nptr> property_definition action_guard
%type <tlptr> process_choice action_prefix
%type <fbptr> process_body local_process
%type <ftsptr> prefix_actions
%type <ltlptr> primary_ltl_formula unary_ltl_formula and_ltl_formula or_ltl_formula
%type <ltlptr> until_ltl_formula release_ltl_formula implies_ltl_formula ltl_formula

%type <oeptr> omega_expr implies_omega or_omega and_omega unary_omega atomic_omega
%type <oeptr> regular_expr implies_regular or_regular and_regular dot_regular
%type <oeptr> unary_regular atomic_regular until_omega release_omega
%type <strptr> dummy_var omega_op_name seaw_formula_name
%type <strlistptr> dummy_var_list
%type <nptr> omega_op_def seaw_formula_def
%type <bsptr> atomic_op_seaw and_op_seaw or_op_seaw implies_op_seaw op_seaw
%type <bsptr> atomic_nop_seaw unary_nop_seaw and_nop_seaw or_nop_seaw
%type <bsptr> implies_nop_seaw nop_seaw seaw_formula
%type <bslptr> seaw_formula_list

%start translation_unit

%%
/*********************************************************************/
//basics
/*********************************************************************/

identifier 
         : MAGIC_IDENTIFIER { $$ = new string(CSymbolTable::buffer); } 
         | lower_id { $$ = $1; }
         | upper_id { $$ = $1; } 
         | dummy_var  { $$ = $1; }
         ;

int_constant : MAGIC_INT_CONSTANT { $$ = CSymbolTable::longbuf; } ;

constant : MAGIC_CONSTANT { $$ = new string(CSymbolTable::buffer); } ;

string_literal : MAGIC_STRING_LITERAL { $$ = new string(CSymbolTable::buffer); } ;

omega_op_name : MAGIC_OMEGA_OP_NAME { $$ = new string(CSymbolTable::buffer); } ;

seaw_formula_name : MAGIC_SEAW_NAME { $$ = new string(CSymbolTable::buffer); } ;

lower_id : MAGIC_LOWER_ID { $$ = new string(CSymbolTable::buffer); } ;

upper_id : MAGIC_UPPER_ID { $$ = new string(CSymbolTable::buffer); } ;

dummy_var : MAGIC_DUMMY_VAR { $$ = new string(CSymbolTable::buffer); } ;

/*********************************************************************/
//expressions
/*********************************************************************/

primary_expression
        : identifier { assert($1 != NULL); $$ = new IdExpr(*$1); delete $1; }
        | int_constant { $$ = new IntConstExpr($1); }
        | constant { assert($1 != NULL); $$ = new ConstExpr(*$1); delete $1; }
        | string_literal { assert($1 != NULL); $$ = new StrExpr(*$1); delete $1; }
        /* this is needed for calls like printk("<1>" "hello\n") */
        | string_literal string_literal {
	  assert(($1 != NULL) && ($2 != NULL)); 
	  $$ = new StrStrExpr(*$1,*$2);
	  delete $1;
	  delete $2;
	}
        /* this is a GNU C extension */
        | '(' compound_statement ')' { 
	  assert(($2 != NULL) && ($2->second != NULL));
	  CompStmt *x = new CompStmt($2->second);
	  $$ = new StmtExpr(x); 
	  delete x; delete $2->second; delete $2; 
	}
        | '(' expression ')' { $$ = $2; }
	;

postfix_expression
        : primary_expression { $$ = $1; }
        | postfix_expression '[' expression ']' {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BrackExpr($1,$3); 
	  delete $1; delete $3;
	}
        | postfix_expression '(' ')' {
	  assert($1 != NULL);
	  ExprList *x = new ExprList();
	  $$ = new ParExpr($1,x);
	  delete x; delete $1;
	}
	| postfix_expression '(' argument_expression_list ')' {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new ParExpr($1,$3);
	  delete $1; delete $3;
	}
        | postfix_expression '.' identifier {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new DotExpr($1,*$3);
	  delete $1; delete $3;
	}
        | postfix_expression MAGIC_PTR_OP identifier {
	  assert(($1 != NULL) && ($3 != NULL));
	  /* convert (&A)->B to A.B */
	  if(typeid(*$1) == typeid(UnaryExpr)) {
	    UnaryExpr *x = static_cast<UnaryExpr*>($1);
	    if(x->GetOp() == '&') {
	      $$ = new DotExpr(x->GetExpr(),*$3);
	    } else {
	      $$ = new ArrowExpr($1,*$3);
	    }
	  } else {
	    $$ = new ArrowExpr($1,*$3);
	  }
	  delete $1; delete $3;
	}
	| postfix_expression MAGIC_INC_OP {
	  assert($1 != NULL);
	  $$ = new IncExpr($1,Database::POST_INC);
	  delete $1;
	}
	| postfix_expression MAGIC_DEC_OP {
	  assert($1 != NULL);
	  $$ = new IncExpr($1,Database::POST_DEC);
	  delete $1;
	}
	;

argument_expression_list
	: assignment_expression { 
	  assert($1 != NULL);
	  $$ = new ExprList();
	  $$->AddExpr($1); 
	  delete $1; 
        }
	| argument_expression_list ',' assignment_expression { 
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->AddExpr($3); 
	  delete $3; 
	}
	;

unary_expression
        : postfix_expression { $$ = $1; }
	| MAGIC_INC_OP unary_expression {
	  assert($2 != NULL);
	  $$ = new IncExpr($2,Database::PRE_INC);
	  delete $2;
	}
	| MAGIC_DEC_OP unary_expression {
	  assert($2 != NULL);
	  $$ = new IncExpr($2,Database::PRE_DEC);
	  delete $2;
	}
        | unary_operator cast_expression {
	  assert($2 != NULL);
	  /* reduce *&A and &*A to A */
	  if(typeid(*$2) == typeid(UnaryExpr)) {
	    UnaryExpr *x = static_cast<UnaryExpr*>($2);
	    if((($1 == '*') && (x->GetOp() == '&')) || (($1 == '&') && (x->GetOp() == '*'))) {
	      $$ = static_cast<BasicExpr*>(x->GetExpr()->Clone());
	    } else {
	      $$ = new UnaryExpr($2,$1);
	      $$ = SimplifyExpr($$);
	    }
	  } else {
	    $$ = new UnaryExpr($2,$1);
	    $$ = SimplifyExpr($$);
	  }
	  delete $2;
	}
	| MAGIC_SIZEOF unary_expression { 
	  $$ = $2;
	}
	| MAGIC_SIZEOF '(' type_name ')' {
	  $$ = new IntConstExpr(4);
	  /* $$ = new EmptyExpr(); */
	}
	;

unary_operator
	: '&' { $$ = '&'; }
	| '*' { $$ = '*'; }
	| '+' { $$ = '+'; }
	| '-' { $$ = '-'; }
	| '~' { $$ = '~'; }
	| '!' { $$ = '!'; }
	;

cast_expression
	: unary_expression { $$ = $1; }
	| '(' type_name ')' cast_expression { $$ = $4; }
	| '(' type_name ')' list_initializer { $$ = new EmptyExpr(); }
	;

multiplicative_expression
	: cast_expression { $$ = $1; }
	| multiplicative_expression '*' cast_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'*');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	| multiplicative_expression '/' cast_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'/');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	| multiplicative_expression '%' cast_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'%');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

additive_expression
	: multiplicative_expression { $$ = $1; }
	| additive_expression '+' multiplicative_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'+');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
        | additive_expression '-' multiplicative_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'-');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

shift_expression
        : additive_expression { $$ = $1; }
        | shift_expression MAGIC_LEFT_OP additive_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,MAGIC_LEFT_OP);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
        | shift_expression MAGIC_RIGHT_OP additive_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,MAGIC_RIGHT_OP);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

relational_expression
        : shift_expression { $$ = $1; }
	| relational_expression '<' shift_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'<');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	| relational_expression '>' shift_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'>');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	| relational_expression MAGIC_LE_OP shift_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,MAGIC_LE_OP);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	| relational_expression MAGIC_GE_OP shift_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,MAGIC_GE_OP);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

equality_expression
        : relational_expression {}
	| equality_expression MAGIC_EQ_OP relational_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,MAGIC_EQ_OP);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	| equality_expression MAGIC_NE_OP relational_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,MAGIC_NE_OP);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

and_expression
	: equality_expression { $$ = $1; }
	| and_expression '&' equality_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'&');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

exclusive_or_expression
	: and_expression { $$ = $1; }
	| exclusive_or_expression '^' and_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'^');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

inclusive_or_expression
	: exclusive_or_expression { $$ = $1; }
	| inclusive_or_expression '|' exclusive_or_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,'|');
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

logical_and_expression
	: inclusive_or_expression { $$ = $1; }
	| logical_and_expression MAGIC_AND_OP inclusive_or_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,MAGIC_AND_OP);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

logical_or_expression
	: logical_and_expression { $$ = $1; }
	| logical_or_expression MAGIC_OR_OP logical_and_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,MAGIC_OR_OP);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3;
	}
	;

conditional_expression
	: logical_or_expression { $$ = $1; }
	| logical_or_expression '?' ':' conditional_expression {
	  assert(($1 != NULL) && ($4 != NULL));
	  $$ = new QuestExpr($1,$1,$4);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $4;
	}
	| logical_or_expression '?' expression ':' conditional_expression {
	  assert(($1 != NULL) && ($3 != NULL) && ($5 != NULL));
	  $$ = new QuestExpr($1,$3,$5);
	  $$ = SimplifyExpr($$);
	  delete $1; delete $3; delete $5;
	}
	;

assignment_expression
        : conditional_expression { $$ = $1; }
	| unary_expression assignment_operator assignment_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  BasicExpr *rhs = NULL;
	  if($2 == '=') rhs = static_cast<BasicExpr*>($3->Clone());
	  else if($2 == MAGIC_MUL_ASSIGN) rhs = new BinaryExpr($1,$3,'*');
	  else if($2 == MAGIC_DIV_ASSIGN) rhs = new BinaryExpr($1,$3,'/');
	  else if($2 == MAGIC_MOD_ASSIGN) rhs = new BinaryExpr($1,$3,'%');
	  else if($2 == MAGIC_ADD_ASSIGN) rhs = new BinaryExpr($1,$3,'+');
	  else if($2 == MAGIC_SUB_ASSIGN) rhs = new BinaryExpr($1,$3,'-');
	  else if($2 == MAGIC_LEFT_ASSIGN) rhs = new BinaryExpr($1,$3,MAGIC_LEFT_OP);
	  else if($2 == MAGIC_RIGHT_ASSIGN) rhs = new BinaryExpr($1,$3,MAGIC_RIGHT_OP);
	  else if($2 == MAGIC_AND_ASSIGN) rhs = new BinaryExpr($1,$3,'&');
	  else if($2 == MAGIC_XOR_ASSIGN) rhs = new BinaryExpr($1,$3,'^');
	  else if($2 == MAGIC_OR_ASSIGN) rhs = new BinaryExpr($1,$3,'|');
	  else assert(false);
	  $$ = new AssignExpr($1,rhs);
	  delete rhs; delete $1; delete $3;
	}
	;

assignment_operator
	: '=' { $$ = '='; }
	| MAGIC_MUL_ASSIGN { $$ = MAGIC_MUL_ASSIGN; }
	| MAGIC_DIV_ASSIGN { $$ = MAGIC_DIV_ASSIGN; }
	| MAGIC_MOD_ASSIGN { $$ = MAGIC_MOD_ASSIGN; }
	| MAGIC_ADD_ASSIGN { $$ = MAGIC_ADD_ASSIGN; }
	| MAGIC_SUB_ASSIGN { $$ = MAGIC_SUB_ASSIGN; }
	| MAGIC_LEFT_ASSIGN { $$ = MAGIC_LEFT_ASSIGN; }
	| MAGIC_RIGHT_ASSIGN { $$ = MAGIC_RIGHT_ASSIGN; }
	| MAGIC_AND_ASSIGN { $$ = MAGIC_AND_ASSIGN; }
	| MAGIC_XOR_ASSIGN { $$ = MAGIC_XOR_ASSIGN; }
	| MAGIC_OR_ASSIGN { $$ = MAGIC_OR_ASSIGN; }
	;

expression
        : assignment_expression { $$ = $1; }
        | expression ',' assignment_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BinaryExpr($1,$3,',');
	  delete $1; delete $3;
	}
	;

constant_expression
	: conditional_expression { $$ = $1; }
	;

/*********************************************************************/
//C declarations
/*********************************************************************/

declaration
        : declaration_specifiers ';' { 
          CSymbolTable::inTypedef = false;
	  $$ = new DeclStmtList();
	  $$->second = new StmtList();
        }
        | declaration_specifiers init_declarator_list ';' { 
	  CSymbolTable::inTypedef = false; 
	  $$ = $2;
	}
	;

declaration_specifiers
        : storage_class_specifier { assert($1 == NULL); $$ = NULL; }
	| storage_class_specifier declaration_specifiers { assert(($1 == NULL) && ($2 == NULL)); $$ = NULL; }
	| type_specifier { assert($1 == NULL); $$ = NULL; }
	| type_specifier declaration_specifiers { assert(($1 == NULL) && ($2 == NULL)); $$ = NULL; }
	| type_qualifier { assert($1 == NULL); $$ = NULL; }
	| type_qualifier declaration_specifiers { assert(($1 == NULL) && ($2 == NULL)); $$ = NULL; }
	;

init_declarator_list
        : init_declarator {
          $$ = new DeclStmtList();
	  $$->second = new StmtList();
	  if($1 != NULL) {
	    $$->first.push_back($1->first);
	    if($1->second != NULL) {
	      $$->second->AddStmt($1->second);
	      delete $1->second;
	    }
	    delete $1;
	  }
        }
	| init_declarator_list ',' init_declarator {
	  assert($1 != NULL);
	  $$ = $1;
	  if($3 != NULL) {
	    $$->first.push_back($3->first);
	    if($3->second != NULL) {
	      $$->second->AddStmt($3->second);
	      delete $3->second;
	    }
	    delete $3;
	  }
	}
	;

init_declarator
        : declarator {
	  assert($1 != NULL);
	  if(CSymbolTable::inTypedef) {
	    CSymbolTable::AddTypeName($1->GetId());
	    $$ = NULL;
	  } else {
	    $$ = new DeclStmt();
	    $$->first = $1->GetId();
	    $$->second = NULL;
	  }
	  delete $1;
        }
	| declarator '=' initializer {
	  assert($1 != NULL);
	  if(CSymbolTable::inTypedef) {
	    CSymbolTable::AddTypeName($1->GetId());
	    $$ = NULL;
	  } else {
	    $$ = new DeclStmt();
	    $$->first = $1->GetId();
	    if($3 != NULL) {
	      BasicExpr *a = new IdExpr($1->GetId());
	      BasicExpr *b = new AssignExpr(a,$3);
	      $$->second = new ExprStmt(b);
	      delete a; delete b; delete $3;
	    } else $$->second = NULL;
	  }
	  delete $1;
	}
	;

storage_class_specifier
        : MAGIC_TYPEDEF { $$ = NULL; }
	| MAGIC_EXTERN { $$ = NULL; }
	| MAGIC_STATIC { $$ = NULL; }
	| MAGIC_AUTO { $$ = NULL; }
	| MAGIC_REGISTER { $$ = NULL; }
	;

type_specifier
	: MAGIC_VOID { $$ = NULL; }
	| MAGIC_CHAR { $$ = NULL; }
	| MAGIC_SHORT { $$ = NULL; }
	| MAGIC_INT { $$ = NULL; }
	| MAGIC_LONG { $$ = NULL; }
	| MAGIC_FLOAT { $$ = NULL; }
	| MAGIC_DOUBLE { $$ = NULL; }
	| MAGIC_SIGNED { $$ = NULL; }
	| MAGIC_UNSIGNED { $$ = NULL; }
	| struct_or_union_specifier { assert($1 == NULL); $$ = NULL; }
	| enum_specifier { assert($1 == NULL); $$ = NULL; }
	| MAGIC_TYPE_NAME { $$ = NULL; }
        /* these are GNU C extensions */
        | MAGIC_TYPEOF '(' type_name ')' { assert($3 == NULL); $$ = NULL; }
        | MAGIC_TYPEOF primary_expression { assert($2 != NULL); delete $2; $$ = NULL; }
	;

struct_or_union_specifier
	: struct_or_union identifier '{' struct_declaration_list '}' {
          assert(($1 == NULL) && ($2 != NULL) && ($4 == NULL));
          delete $2;
          $$ = NULL;
        }
	| struct_or_union MAGIC_TYPE_NAME '{' struct_declaration_list '}' {
          assert(($1 == NULL) && ($4 == NULL));
          $$ = NULL;
        }
	| struct_or_union identifier '{' '}' {
          assert(($1 == NULL) && ($2 != NULL));
          delete $2;
          $$ = NULL;
        }
	| struct_or_union MAGIC_TYPE_NAME '{' '}' {
          assert($1 == NULL);
	  $$ = NULL;
        }
	| struct_or_union '{' struct_declaration_list '}' {
          assert(($1 == NULL) && ($3 == NULL));
	  $$ = NULL;
	}
	| struct_or_union '{' '}' { assert($1 == NULL); $$ = NULL; }
	| struct_or_union identifier {
          assert(($1 == NULL) && ($2 != NULL));
	  delete $2;
	  $$ = NULL;
	}
	| struct_or_union MAGIC_TYPE_NAME {
          assert($1 == NULL);
	  $$ = NULL;
	}
	;

struct_or_union
	: MAGIC_STRUCT { $$ = NULL; }
	| MAGIC_UNION { $$ = NULL; }
	;

struct_declaration_list
	: struct_declaration {
          assert($1 == NULL);
	  $$ = NULL;
	}
	| struct_declaration_list struct_declaration {
          assert(($1 == NULL) && ($2 == NULL));
	  $$ = NULL;
	}
	;

struct_declaration
	: specifier_qualifier_list struct_declarator_list ';' {
          assert(($1 == NULL) && ($2 == NULL));
	  $$ = NULL;
	}
	;

specifier_qualifier_list
	: type_specifier specifier_qualifier_list {
          assert(($1 == NULL) && ($2 == NULL));
	  $$ = NULL;
	}
	| type_specifier { assert($1 == NULL); $$ = NULL; }
	| type_qualifier specifier_qualifier_list {
          assert(($1 == NULL) && ($2 == NULL));
	  $$ = NULL;
	}
	| type_qualifier { assert($1 == NULL); $$ = NULL; }
	;

struct_declarator_list
	: struct_declarator { assert($1 == NULL); $$ = NULL; }
	| struct_declarator_list ',' struct_declarator {
          assert(($1 == NULL) && ($3 == NULL));
	  $$ = NULL;
	}
	;

struct_declarator
        : declarator { assert($1 != NULL); delete $1; $$ = NULL; }
        | ':' constant_expression { assert($2 != NULL); delete $2; $$ = NULL; }
        | declarator ':' constant_expression {
          assert(($1 != NULL) && ($3 != NULL));
	  delete $1; delete $3; $$ = NULL;
	}
	;

enum_specifier
	: MAGIC_ENUM '{' enumerator_list '}' {
          assert($3 == NULL);
	  //reset last enum value
	  assert(lastEnumVal != NULL); delete lastEnumVal;
	  lastEnumVal = new IntConstExpr(-1);
	  $$ = NULL;
	}
        /* enum list can end with a comma */
	| MAGIC_ENUM '{' enumerator_list ',' '}' {
          assert($3 == NULL);
	  //reset last enum value
	  assert(lastEnumVal != NULL); delete lastEnumVal;
	  lastEnumVal = new IntConstExpr(-1);
	  $$ = NULL;
	}
	| MAGIC_ENUM identifier '{' enumerator_list '}' {
          assert(($2 != NULL) && ($4 == NULL));
	  //reset last enum value
	  assert(lastEnumVal != NULL); delete lastEnumVal;
	  lastEnumVal = new IntConstExpr(-1);
	  delete $2;
	  $$ = NULL;
	}
        /* enum list can end with a comma */
	| MAGIC_ENUM identifier '{' enumerator_list ',' '}' {
          assert(($2 != NULL) && ($4 == NULL));
	  //reset last enum value
	  assert(lastEnumVal != NULL); delete lastEnumVal;
	  lastEnumVal = new IntConstExpr(-1);
	  delete $2;
	  $$ = NULL;
	}
	| MAGIC_ENUM identifier {
          assert($2 != NULL);
	  //reset last enum value
	  assert(lastEnumVal != NULL); delete lastEnumVal;
	  lastEnumVal = new IntConstExpr(-1);
	  delete $2;
	  $$ = NULL;
	}
	;

enumerator_list
	: enumerator {
          assert($1 == NULL);
	  $$ = NULL;
	}
	| enumerator_list ',' enumerator {
          assert(($1 == NULL) && ($3 == NULL));
	  $$ = NULL;
	}
	;

enumerator
        : identifier {
          assert(($1 != NULL) && (lastEnumVal != NULL));
	  if(Database::enums.count(*$1) == 0) {
	    BasicExpr *a = new IntConstExpr(1);
	    BasicExpr *b = new BinaryExpr(lastEnumVal,a,'+');
	    b = SimplifyExpr(b);
	    delete a; delete lastEnumVal;
	    lastEnumVal = static_cast<BasicExpr*>(b->Clone());
	    Database::enums[*$1] = static_cast<BasicExpr*>(b->Clone());
	    delete b; 
          }
	  delete $1; $$ = NULL;
	}
	| identifier '=' constant_expression {
          assert(($1 != NULL) && ($3 != NULL) && (lastEnumVal != NULL));
	  if(Database::enums.count(*$1) == 0) {
	    delete lastEnumVal; lastEnumVal = static_cast<BasicExpr*>($3->Clone());
	    Database::enums[*$1] = static_cast<BasicExpr*>($3->Clone());
	  }
	  delete $1; delete $3; $$ = NULL;
	}
	;

type_qualifier
	: MAGIC_CONST { $$ = NULL; }
	| MAGIC_VOLATILE { $$ = NULL; }
	;

declarator
        : pointer direct_declarator { assert($1 == NULL); $$ = $2; }
        | direct_declarator { $$ = $1; }
	;

direct_declarator
        : identifier {
	  assert($1 != NULL);
	  $$ = new Decl();
	  $$->SetId(*$1);
	  delete $1;
        }
        | '(' declarator ')' { $$ = $2; }
        | direct_declarator '[' constant_expression ']' { $$ = $1; }
        | direct_declarator '[' ']' { $$ = $1; }
        | direct_declarator '(' parameter_type_list ')' { 
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1; 
	  $$->SetIdList(*$3);
	  delete $3;
	}
        | direct_declarator '(' identifier_list ')' { 
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->SetIdList(*$3);
	  delete $3;
	}
        | direct_declarator '(' ')' { assert($1 != NULL); $$ = $1; $$->ClearIdList(); }
	;

pointer
	: '*' { $$ = NULL; }
	| '*' type_qualifier_list { assert($2 == NULL); $$ = NULL; }
	| '*' pointer { assert($2 == NULL); $$ = NULL; }
	| '*' type_qualifier_list pointer { assert(($2 == NULL) && ($3 == NULL)); $$ = NULL; }
	;

type_qualifier_list
	: type_qualifier { assert($1 == NULL); $$ = NULL; }
	| type_qualifier_list type_qualifier { assert(($1 == NULL) && ($2 == NULL)); $$ = NULL; }
	;


parameter_type_list
        : parameter_list { $$ = $1; }
	| parameter_list ',' MAGIC_ELLIPSIS { $$ = $1; }
	;

parameter_list
	: parameter_declaration {
	  $$ = new list<string>;
	  if($1 != NULL) {
	    $$->push_back(*$1);
	    delete $1;
	  }
        }
	| parameter_list ',' parameter_declaration {
	  assert($1 != NULL);
	  $$ = $1;
	  if($3 != NULL) {
	    $$->push_back(*$3);
	    delete $3;
	  }
	} 
	;

parameter_declaration
        : declaration_specifiers declarator {
          assert(($1 == NULL) && ($2 != NULL));
	  $$ = new string($2->GetId());
	  delete $2;
        }
        | declaration_specifiers abstract_declarator {
          assert(($1 == NULL) && ($2 == NULL));
	  $$ = NULL;
	}
	| declaration_specifiers {
	  assert($1 == NULL);
	  $$ = NULL;
	}
	;

identifier_list
	: identifier {
	  assert($1 != NULL);
	  $$ = new list<string>;
	  $$->push_back(*$1);
	  delete $1;
        }
	| identifier_list ',' identifier {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->push_back(*$3);
	  delete $3;
	}
	;

type_name
	: specifier_qualifier_list { assert($1 == NULL); $$ = NULL; }
	| specifier_qualifier_list abstract_declarator { assert(($1 == NULL) && ($2 == NULL)); $$ = NULL; }
	;

abstract_declarator
        : pointer { assert($1 == NULL); $$ = NULL; }
	| direct_abstract_declarator { assert($1 == NULL); $$ = NULL; }
	| pointer direct_abstract_declarator { assert(($1 == NULL) && ($2 == NULL)); $$ = NULL; }
	;

direct_abstract_declarator
	: '(' abstract_declarator ')' { assert($2 == NULL); $$ = NULL; }
	| '[' ']' { $$ = NULL; }
	| '[' constant_expression ']' { assert($2 != NULL); delete $2; $$ = NULL; }
	| direct_abstract_declarator '[' ']' { assert($1 == NULL); $$ = NULL; }
	| direct_abstract_declarator '[' constant_expression ']' { 
	  assert(($1 == NULL) && ($3 != NULL)); 
	  delete $3; $$ = NULL; 
	}
	| '(' ')' { $$ = NULL; }
	| '(' parameter_type_list ')' { assert($2 != NULL); delete $2; $$ = NULL; }
	| direct_abstract_declarator '(' ')' { assert($1 == NULL); $$ = NULL; }
	| direct_abstract_declarator '(' parameter_type_list ')' { 
	  assert(($1 == NULL) && ($3 != NULL)); 
	  delete $3; $$ = NULL; 
	}
	;

initializer
	: assignment_expression { $$ = $1; }
	| list_initializer { assert($1 == NULL); $$ = NULL; }
	;

list_initializer
        : '{' '}' { $$ = NULL; }
        | '{' initializer_list '}' { assert($2 == NULL); $$ = NULL; }
        | '{' initializer_list ',' '}' { assert($2 == NULL); $$ = NULL; }
        ;

initializer_list
	: initializer_element { assert($1 == NULL); $$ = NULL; }
	| initializer_list ',' initializer_element { assert(($1 == NULL) && ($3 == NULL)); $$ = NULL; }
	;

initializer_element
        /*init*/
	: initializer { if($1 != NULL) delete $1; $$ = NULL; }
        /* the following cases are GNU C extensions */
        /*id_colon*/
        | identifier ':' initializer { 
	  assert($1 != NULL); delete $1; 
	  if($3 != NULL) delete $3;
	  $$ = NULL; 
	}
        /*dot_equal*/
        | '.' identifier '=' initializer { 
	  assert($2 != NULL); delete $2; 
	  if($4 != NULL) delete $4;
	  $$ = NULL; 
	}
        /*brack*/
	| '[' constant_expression ']' '=' initializer { 
	  assert($2 != NULL); delete $2; 
	  if($5 != NULL) delete $5;
	  $$ = NULL; 
	}
        /*ellipsis*/
	| '[' constant_expression MAGIC_ELLIPSIS constant_expression ']' '=' initializer { 
	  assert(($2 != NULL) && ($4 != NULL)); 
	  delete $2; delete $4; 
	  if($7 != NULL) delete $7;
	  $$ = NULL; 
	}
        ;

/*********************************************************************/
//C statements
/*********************************************************************/

statement
	: labeled_statement { $$ = $1; }
        | compound_statement {
	  assert(($1 != NULL) && ($1->second != NULL));
	  $$ = new CompStmt($1->second);
	  delete $1->second;
	  delete $1;
	}
	| expression_statement { $$ = $1; }
	| selection_statement { $$ = $1; }
	| iteration_statement { $$ = $1; }
	| jump_statement { $$ = $1; }
	;

labeled_statement
	: identifier ':' statement {
          assert(($1 != NULL) && ($3 != NULL));
	  $$ = new LabelStmt(*$1,$3);
	  delete $1; delete $3;
        }
	| MAGIC_CASE constant_expression ':' statement {
	  assert(($2 != NULL) && ($4 != NULL));
	  $$ = new CaseStmt($2,$4);
	  delete $2; delete $4;
	}
	| MAGIC_DEFAULT ':' statement {
	  assert($3 != NULL);
	  $$ = new DefaultStmt($3);
	  delete $3;
	}
	;

compound_statement
        : '{' '}' { 
	  $$ = new DeclStmtList();
	  $$->second = new StmtList();
        }
	| '{' statement_list '}' {
	  assert($2 != NULL);
	  $$ = new DeclStmtList();
	  $$->second = $2;
	}
	| '{' declaration_list '}' { 
	  $$ = $2;
	}
	| '{' declaration_list statement_list '}' {
	  assert(($2 != NULL) && ($3 != NULL));
	  $$ = $2;
	  if($3 != NULL) {
	    $$->second->AddStmtList(*$3);
	    delete $3;
	  }
	}
	;

declaration_list
	: declaration { $$ = $1; }
	| declaration_list declaration {
	  assert(($1 != NULL) && ($2 != NULL));
	  $$ = $1;
	  $$->first.insert($$->first.end(),$2->first.begin(),$2->first.end());
	  $$->second->AddStmtList(*($2->second));
	  delete $2->second;
	  delete $2;
	}
	;

statement_list
	: statement {
	  assert($1 != NULL);
	  $$ = new StmtList();
	  $$->AddStmt($1); 
	  delete $1;
        }
	| statement_list statement { 
	  assert(($1 != NULL) && ($2 != NULL));
	  $$ = $1;
	  $$->AddStmt($2); 
	  delete $2; 
	}
	;

expression_statement
	: ';' {
	  BasicExpr *x = new EmptyExpr();
	  $$ = new ExprStmt(x);
	  delete x;
        }
        | expression ';' { 
          assert($1 != NULL);
	  $$ = new ExprStmt($1);
	  delete $1;
	}
	;

selection_statement
	: MAGIC_IF '(' expression ')' statement {
          assert(($3 != NULL) && ($5 != NULL));
	  Stmt *x = new EmptyStmt();
	  $$ = new IfStmt($3,$5,x);
	  delete x; delete $3; delete $5;
        }
	| MAGIC_IF '(' expression ')' statement MAGIC_ELSE statement {
	  assert(($3 != NULL) && ($5 != NULL) && ($7 != NULL));
	  $$ = new IfStmt($3,$5,$7);
	  delete $3; delete $5; delete $7;
	}
	| MAGIC_SWITCH '(' expression ')' statement {
	  assert(($3 != NULL) && ($5 != NULL));
	  $$ = new SwitchStmt($3,$5);
	  delete $3; delete $5;
	}
	;

iteration_statement
	: MAGIC_WHILE '(' expression ')' statement {
          assert(($3 != NULL) && ($5 != NULL));
	  $$ = new WhileStmt($3,$5);
	  delete $3; delete $5;
        }
	| MAGIC_DO statement MAGIC_WHILE '(' expression ')' ';' {
	  assert(($2 != NULL) && ($5 != NULL));
	  $$ = new DoStmt($5,$2);
	  delete $2; delete $5;
	}
	| MAGIC_FOR '(' expression_statement expression_statement ')' statement {
	  assert(($3 != NULL) && ($4 != NULL) && ($6 != NULL));
	  BasicExpr *x = new EmptyExpr();
	  $$ = ForToWhile($3,$4,x,$6);
	  delete x; delete $3; delete $4; delete $6;
	}
	| MAGIC_FOR '(' expression_statement expression_statement expression ')' statement {
	  assert(($3 != NULL) && ($4 != NULL) && ($5 != NULL) && ($7 != NULL));
	  $$ = ForToWhile($3,$4,$5,$7);
	  delete $3; delete $4; delete $5; delete $7;
	}
	;

jump_statement
	: MAGIC_GOTO identifier ';' {
          assert($2 != NULL);
	  $$ = new GotoStmt(*$2);
	  delete $2;
        }
	| MAGIC_CONTINUE ';' { $$ = new ContinueStmt(); }
	| MAGIC_BREAK ';' { $$ = new BreakStmt(); }
	| MAGIC_RETURN ';' {
	  BasicExpr *x = new EmptyExpr();
	  $$ = new ReturnStmt(x);
	  delete x;
	}
	| MAGIC_RETURN expression ';' {
	  assert($2 != NULL);
	  $$ = new ReturnStmt($2);
	  delete $2;
	}
	;

/*********************************************************************/
//the C grammar part
/*********************************************************************/

stdc_translation_unit
        : external_declaration { 
	  $$ = new StdcFile();
	  if($1 != NULL) {
	    $$->AddProc($1);
	    delete $1;
	  } 
	}
	| stdc_translation_unit external_declaration { 
	  assert($1 != NULL);
	  $$ = $1;
	  if($2 != NULL) {
	    $$->AddProc($2);
	    delete $2;
	  }
	}
	;

external_declaration
        : function_definition { assert($1 != NULL); $$ = $1; } 
        | declaration { assert($1 != NULL); delete $1; $$ = NULL; }
	;

function_definition
	: declaration_specifiers declarator declaration_list compound_statement { assert(false); }
	| declaration_specifiers declarator compound_statement {
	  assert(($1 == NULL) && ($2 != NULL) && ($3 != NULL) && ($3->second != NULL));
	  CompStmt *x = new CompStmt($3->second);
	  $$ = new Proc($2->GetId(),$2->GetIdList(),$3->first,x);
	  delete x; delete $2; delete $3->second; delete $3;
	}
        | declarator declaration_list compound_statement { assert(false); }
        | declarator compound_statement {
	  assert(($1 != NULL) && ($2 != NULL) && ($2->second != NULL));
	  CompStmt *x = new CompStmt($2->second);
	  $$ = new Proc($1->GetId(),$1->GetIdList(),$2->first,x);
	  delete x; delete $1; delete $2->second; delete $2;
	}
	;

/*********************************************************************/
//the spec grammar part
/*********************************************************************/

/*********************************************************************/
//begin C procedure additional information  productions
/*********************************************************************/

c_proc_name 
        : postfix_expression { $$ = $1; }
        ;

c_proc_name_list
        /*name*/
	: c_proc_name {
          assert($1 != NULL);
	  $$ = new ExprList();
	  $$->AddExpr($1);
	  delete $1;
        }
        /*list*/
	| c_proc_name_list ',' c_proc_name {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->AddExpr($3);
	  delete $3;
	}
        ;

c_prog_info
        : MAGIC_CPROG lower_id '=' c_proc_name_list '{' c_prog_decls '}' {
          assert(($2 != NULL) && ($4 != NULL) && ($6 != NULL));
	  $$ = new ProgInfo(*$2,$4->ToStringList(),*$6);
	  delete $2; delete $4; delete $6;
        }
        | MAGIC_CPROG lower_id '=' c_proc_name_list '{' '}' {
          assert(($2 != NULL) && ($4 != NULL));
	  $$ = new ProgInfo(*$2,$4->ToStringList());
	  delete $2; delete $4;
        }
        ;

c_prog_decls
        : c_prog_decl {
	  assert($1 != NULL);
	  $$ = new list<ProgAbs>();
	  $$->push_back(*$1);
	  delete $1;
        } 
        | c_prog_decls c_prog_decl {
          assert($2 != NULL);
	  $$ = $1;
	  $$->push_back(*$2);
	  delete $2;
	}
        ;

c_prog_decl
        : MAGIC_ABSTRACT identifier ',' '{' argument_expression_list '}' ',' process_id ';' { 
          assert(($2 != NULL) && ($5 != NULL) && ($8 != NULL));
          $$ = new ProgAbs(*$2,*$5,*$8);
	  delete $2; delete $5; delete $8;
        }
        | MAGIC_ABSTRACT identifier ',' '{' argument_expression_list '}' ',' seaw_formula_name ';' { 
          assert(($2 != NULL) && ($5 != NULL) && ($8 != NULL));
          $$ = new ProgAbs(*$2,*$5,*$8);
	  delete $2; delete $5; delete $8;
        }
        ;

c_proc_info 
        : MAGIC_CPROC c_proc_name '{' c_proc_decls '}' {
          assert(($2 != NULL) && ($4 != NULL));
	  $$ = $4;
	  $$->SetName(Util::TrimString($2->ToString()));
	  delete $2;
        }
        | MAGIC_CPROC c_proc_name '{' '}' {
	  assert($2 != NULL);
	  $$ = new ProcAddInfo();
	  $$->SetName(Util::TrimString($2->ToString()));
	  delete $2;
	}
        ;

c_proc_decls
        /*pred*/
	: predicate_decl {
	  assert($1 != NULL);
	  $$ = new ProcAddInfo();
	  $$->AddPreds(*$1);
	  delete $1;
        }
        /*pred_list*/
	| c_proc_decls predicate_decl {
          assert(($1 != NULL) && ($2 != NULL));
	  $$ = $1;
	  $$->AddPreds(*$2);
	  delete $2;
	}
        /*fair_loop*/
	| fair_loop_decl {
	  assert($1 != NULL);
	  $$ = new ProcAddInfo();
	  $$->AddFairLoops(*$1);
	  delete $1;
        }
        /*fair_loop_list*/
	| c_proc_decls fair_loop_decl {
          assert(($1 != NULL) && ($2 != NULL));
	  $$ = $1;
	  $$->AddFairLoops(*$2);
	  delete $2;
	}
        /*aux*/
	| auxiliary_decl {
	  assert($1 != NULL);
	  $$ = new ProcAddInfo();
	  //auxiliary predicates only used for CEGAR
	  if(Database::CEGAR) $$->AddPreds(*$1);
	  delete $1;
        }
        /*aux_list*/
	| c_proc_decls auxiliary_decl {
          assert(($1 != NULL) && ($2 != NULL));
	  $$ = $1;
	  //auxiliary predicates only used for CEGAR
	  if(Database::CEGAR) $$->AddPreds(*$2);
	  delete $2;
	}
        /*inline*/
	| inline_decl {
	  assert($1 != NULL);
	  $$ = new ProcAddInfo();
	  $$->AddInlines($1->ToStringList());
	  delete $1;
        }
        /*inline_list*/
	| c_proc_decls inline_decl {
          assert(($1 != NULL) && ($2 != NULL));
	  $$ = $1;
	  $$->AddInlines($2->ToStringList());
	  delete $2;
	} 
        /*alias*/
        | alias_decl {
	  assert($1 != NULL);
	  $$ = new ProcAddInfo();
	  $$->AddAlias(*$1);
	  delete $1;
	}
        /*alias_list*/
        | c_proc_decls alias_decl {
          assert(($1 != NULL) && ($2 != NULL));
	  $$ = $1;
	  $$->AddAlias(*$2);
	  delete $2;
	}
        /*abstract*/
        | abstract_decl {
	  assert($1 != NULL);
	  $$ = new ProcAddInfo();
	  $$->AddAbstraction(*$1);
	  delete $1;
	}
        /*abstract_list*/
        | c_proc_decls abstract_decl {
          assert(($1 != NULL) && ($2 != NULL));
	  $$ = $1;
	  $$->AddAbstraction(*$2);
	  delete $2;
	}
        /*context*/
        | context_decl {
	  $$ = new ProcAddInfo();
	  $$->SetContext(static_cast<unsigned>($1));
	}
        /*context_list*/
        | c_proc_decls context_decl {
          assert($1 != NULL);
	  $$ = $1;
	  $$->SetContext(static_cast<unsigned>($2));
	}
        ;

predicate_decl 
        : MAGIC_PREDICATE predicate_list ';' { $$ = $2; }
        ;

fair_loop_decl 
        : MAGIC_FAIR_LOOP predicate_list ';' { $$ = $2; }
        ;

auxiliary_decl 
        : MAGIC_AUXILIARY predicate_list ';' { $$ = $2; }
        ;

predicate_list
        /*pred*/
        : conditional_expression {
          assert($1 != NULL);
	  $$ = new ExprList();
	  $$->AddExpr($1);
	  delete $1;
        }
        /*list*/
	| predicate_list ',' conditional_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->AddExpr($3);
	  delete $3;
	}
        ;

inline_decl 
        : MAGIC_INLINE c_proc_name_list ';' { $$ = $2; }
        ;

alias_decl
        : MAGIC_ALIAS alias_item_list ';' {
          $$ = $2;
        }
        ;

alias_item_list
        : alias_item {
          assert($1 != NULL);
	  $$ = new list<Alias>();
	  $$->push_back(*$1);
	  delete $1;
        }
        | alias_item_list ',' alias_item {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->push_back(*$3);
	  delete $3;
	}
        ;

alias_item
        : '{' conditional_expression ',' points_to_list '}' {
          assert(($2 != NULL) && ($4 != NULL));
	  $$ = new Alias($2,$4);
	  delete $2; delete $4;
        }
        ;

points_to_list
        /*name*/
	: conditional_expression {
          assert($1 != NULL);
	  $$ = new ExprList();
	  $$->AddExpr($1);
	  delete $1;
        }
        /*list*/
	| points_to_list ',' conditional_expression {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->AddExpr($3);
	  delete $3;
	}
        ;

abstract_decl
        : MAGIC_ABSTRACT abstract_item_list ';' {
          $$ = $2;
        }
        ;

abstract_item_list
        : abstract_item {
          assert($1 != NULL);
	  $$ = new list<ProcAbs>();
	  $$->push_back(*$1);
	  delete $1;
        }
        | abstract_item_list ',' abstract_item {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->push_back(*$3);
	  delete $3;
	}
        ;

abstract_item
        : '{' identifier ',' conditional_expression ',' process_id '}' {
          assert(($2 != NULL) && ($4 != NULL) && ($6 != NULL));
	  $$ = new ProcAbs(*$2,$4,*$6);
	  delete $2; delete $4; delete $6;
        }
        ; 

context_decl : MAGIC_CONTEXT int_constant ';' { 
          $$ = $2; 
        }
        ;
   
/*********************************************************************/
//begin FSP productions
/*********************************************************************/

/*
range_id 
        : upper_id {}
        ;
*/

/*
set_id 
        : upper_id {}
        ;
*/

/*
constant_id 
        : upper_id {}
        ;
*/

process_id 
        : upper_id { $$ = $1; }
        ;

action_label
        /*id*/
	: lower_id {
          assert($1 != NULL);
	  $$ = new Action(*$1);
	  delete $1;
        }
        /*normal return*/
	| MAGIC_RETURN '{' conditional_expression '}' {
	  assert($3 != NULL);
	  $$ = new Action($3);
	  delete $3;
	}
        /*void return*/
	| MAGIC_RETURN '{' '}' {
	  BasicExpr *x = new EmptyExpr();
	  $$ = new Action(x);
	  delete x;
	}
        /*assign*/
        | '{' conditional_expression  '=' '[' conditional_expression ']' '}' {
	  assert(($2 != NULL) && ($5 != NULL));
	  $$ = new Action($2,$5);
	  delete $2; delete $5;
	}
        /*broadcast*/
        | lower_id MAGIC_BANG_BANG '[' predicate_list ']' {
	  assert(($1 != NULL) && ($4 != NULL));
	  $$ = new Action(BasicAction::ACTION_BCAST,*$1,*$4);
	  delete $1; delete $4;
	}
        /*void send*/
        | lower_id '!' '[' ']' {
	  assert($1 != NULL);
	  $$ = new Action(BasicAction::ACTION_SEND,*$1,list<Expr>());
	  delete $1;
	}
        /*send*/
        | lower_id '!' '[' predicate_list ']' {
	  assert(($1 != NULL) && ($4 != NULL));
	  $$ = new Action(BasicAction::ACTION_SEND,*$1,*$4);
	  delete $1; delete $4;
	}
        /*void receive*/
        | lower_id '?' '[' ']' {
	  assert($1 != NULL);
	  $$ = new Action(BasicAction::ACTION_RECV,*$1,list<Expr>());
	  delete $1;
	}
        /*receive*/
        | lower_id '?' '[' predicate_list ']' {
	  assert(($1 != NULL) && ($4 != NULL));
	  $$ = new Action(BasicAction::ACTION_RECV,*$1,*$4);
	  delete $1; delete $4;
	}
        /*action_id*/
	/*action_label dot lower_id*/
        /*expr*/
	/*l_bracket expression r_bracket*/
        /*action_expr*/
	/*action_label l_bracket expression r_bracket*/
        ;

action_label_list
        : action_label {
	  assert($1 != NULL);
	  $$ = new set<Action>();
	  $$->insert(*$1);
	  delete $1;
        }
        | action_label_list ',' action_label {
	  assert($3 != NULL);
	  $$ = $1;
	  $$->insert(*$3);
	  delete $3;
        }
        ;

action_labels
        /*action*/
	: action_label { $$ = $1; }
        /*set*/
	/*set_decl*/
        /*range*/
	/*l_bracket action_range r_bracket*/
        /*action_dot*/
	/*action_labels dot action_label*/
        /*set_dot*/
	/*action_labels dot set_decl*/
        /*range_dot*/
	/*action_labels dot l_bracket action_range r_bracket*/
        ;

/*
action_range
	: range_decl
	| set_decl
	| lower_id ':' range_decl {}
	| lower_id ':' set_decl {}
        ;
*/

/*
range_decl
	: range_id
	| expression MAGIC_DOT_DOT expression {}
        ;
*/

/*
set_decl
	: set_id
	| '{' set_elements '}' 
        ;
*/

/*
set_elements
	: action_labels
	| set_elements ',' action_labels 
        ;
*/

/*
constant_definition 
        : MAGIC_CONST constant_id '=' additive_expression 
        ;
*/

/*
range_definition 
        : MAGIC_RANGE range_id '=' additive_expression MAGIC_DOT_DOT additive_expression 
        ;
*/

/*
set_definition 
        : MAGIC_SET set_id '=' '{' set_elements '}' 
        ;
*/

process_definition 
        : process_id '=' process_body '.' {
          assert(($1 != NULL) && ($3 != NULL));
	  assert((($3->id == "") && (!$3->tr.empty())) || (($3->id != "") && ($3->tr.empty())));
	  if($3->id == "") {
	    for(list<LtsTrans>::iterator i = $3->tr.begin();i != $3->tr.end();++i) {
	      i->SetInitState(*$1);
	      Database::fspInfo.AddTrans(*i);
	    }
	  } else {
	    Database::fspInfo.AddTrans(LtsTrans(*$1,$3->id,ActionManager::GetSpecEpsilonAction()));
	  }
	  delete $1; delete $3;
          $$ = NULL;
        }
        | process_id '=' process_body '+' '{' action_label_list '}' '.' {
          assert(($1 != NULL) && ($3 != NULL) && ($6 != NULL));
	  assert((($3->id == "") && (!$3->tr.empty())) || (($3->id != "") && ($3->tr.empty())));
	  if($3->id == "") {
	    for(list<LtsTrans>::iterator i = $3->tr.begin();i != $3->tr.end();++i) {
	      i->SetInitState(*$1);
	      Database::fspInfo.AddTrans(*i);
	    }
	  } else {
	    Database::fspInfo.AddTrans(LtsTrans(*$1,$3->id,ActionManager::GetSpecEpsilonAction()));
	  }
	  Database::fspInfo.AddExtAct(*$1,*$6);
	  delete $1; delete $3; delete $6;
          $$ = NULL;
        }
        ;

process_body
        /*local*/
	: local_process { $$ = $1; }
        /*list*/
	| local_process ',' local_process_defs {
	  assert($3 == NULL);
	  $$ = $1;
	} 
        ;

local_process_defs
        /*local*/
	: local_process_def {
	  assert($1 == NULL);
	  $$ = NULL;
        }
        /*list*/
	| local_process_defs ',' local_process_def {
	  assert(($1 == NULL) && ($3 == NULL));
	  $$ = NULL;
	} 
        ;

local_process_def 
        : process_id '=' local_process {
          assert(($1 != NULL) && ($3 != NULL));
	  assert((($3->id == "") && (!$3->tr.empty())) || (($3->id != "") && ($3->tr.empty())));
	  if($3->id == "") {
	    for(list<LtsTrans>::iterator i = $3->tr.begin();i != $3->tr.end();++i) {
	      i->SetInitState(*$1);
	      Database::fspInfo.AddTrans(*i);
	    }
	  } else {
	    Database::fspInfo.AddTrans(LtsTrans(*$1,$3->id,ActionManager::GetSpecEpsilonAction()));
	  }
	  delete $1; delete $3;
          $$ = NULL;
        }
        ;

local_process
        /*stop*/
        : MAGIC_STOP {
          $$ = new FspBody;
	  $$->id = Database::STOP_STATE;
        }
        /*error*/
	/*error*/
        /*id*/
	| process_id {
          assert($1 != NULL);
          $$ = new FspBody;
	  $$->id = *$1;
	  delete $1;
        }
        /*choice*/
	| '(' process_choice ')' {
	  assert($2 != NULL);
	  $$ = new FspBody;
	  $$->tr = *$2;
	  delete $2;
	}
        ;

process_choice
        /*action*/
	: action_prefix { $$ = $1; }
        /*choice*/
	| process_choice '|' action_prefix {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->insert($$->end(),$3->begin(),$3->end());	  
	  delete $3;
	}
        ;

action_prefix
        : action_guard prefix_actions MAGIC_PTR_OP local_process {
          assert(($1 == NULL) && ($2 != NULL) && ($4 != NULL));
	  assert((!$2->begin.empty() && !$2->end.empty()) || (!$2->begin.empty() && $2->end.empty()));
	  assert((($4->id == "") && (!$4->tr.empty())) || (($4->id != "") && ($4->tr.empty())));
	  $$ = MakeTransList($2,$4);
	  delete $2; delete $4;
        }
        | prefix_actions MAGIC_PTR_OP local_process {
	  assert(($1 != NULL) && ($3 != NULL));
	  assert((!$1->begin.empty() && !$1->end.empty()) || (!$1->begin.empty() && $1->end.empty()));
	  assert((($3->id == "") && (!$3->tr.empty())) || (($3->id != "") && ($3->tr.empty())));
	  $$ = MakeTransList($1,$3);
	  delete $1; delete $3;
	}
        ;

prefix_actions
        /*action*/
	: action_labels {
	  assert($1 != NULL);
	  $$ = new FspTransSeq();
	  $$->begin.push_back(LtsTrans(*$1));
	  delete $1;
        }
        /*prefix*/
	| prefix_actions MAGIC_PTR_OP action_labels {
	  assert(($1 != NULL) && ($3 != NULL));
	  assert((!$1->begin.empty() && !$1->end.empty()) || (!$1->begin.empty() && $1->end.empty()));
	  string tvar = Util::NewTempVar();
	  list<LtsTrans> x;
	  if($1->end.empty()) {
	    for(list<LtsTrans>::iterator i = $1->begin.begin();i != $1->begin.end();++i) {
	      i->SetFinalState(tvar);
	      x.push_back(*i);
	    }
	  } else {
	    x = $1->begin;
	    for(list<LtsTrans>::iterator i = $1->end.begin();i != $1->end.end();++i) {
	      i->SetFinalState(tvar);
	      Database::fspInfo.AddTrans(*i);
	    }
	  }
	  list<LtsTrans> y;
	  LtsTrans z = LtsTrans(*$3);
	  z.SetInitState(tvar);
	  y.push_back(z);
	  $$ = new FspTransSeq();
	  $$->begin = x; $$->end = y;
	  delete $1; delete $3;
	} 
        ;

action_guard 
        : MAGIC_WHEN expression {
          assert($2 != NULL);
	  delete $2;
	  $$ = NULL;
        }
        ;

/*composite_definition : ';' ;*/

property_definition
        /*safety*/
	: MAGIC_PROPERTY process_definition {
          assert($2 == NULL);
	  $$ = NULL;
        }
        ;

/*progress_definition : ';' ;*/

/*menu_definition : ';' ;*/

fsp_definition
        /*const*/
	/*constant_definition*/
        /*range*/
	/*range_definition*/
        /*set*/
	/*set_definition*/
        /*process*/
        : process_definition {
          assert($1 == NULL);
	  $$ = NULL;
        }
        /*composite*/
	/*composite_definition*/
        /*property*/
	| property_definition {
          assert($1 == NULL);
	  $$ = NULL;
        }
        /*progress*/
	/*progress_definition*/
        /*menu*/
	/*menu_definition*/
        ;

/*********************************************************************/
//begin LTL productions
/*********************************************************************/

primary_ltl_formula
        : '[' conditional_expression ']' {
	  assert($2 != NULL);
	  $$ = new BasicLtl($2);
	  delete $2;
        }
        | action_label {
          assert($1 != NULL);
	  $$ = new BasicLtl(*$1);
	  delete $1;
	}
        | '(' ltl_formula ')' { $$ = $2; }
        ;

unary_ltl_formula
        : primary_ltl_formula { $$ = $1; }
        | '!' unary_ltl_formula {
          assert($2 != NULL);
	  $$ = new BasicLtl(BasicLtl::LTL_NOT,$2);
	  delete $2;
	}
        | MAGIC_NEXT_TIME unary_ltl_formula {
          assert($2 != NULL);
	  $$ = new BasicLtl(BasicLtl::LTL_X,$2);
	  delete $2;
	}
        | MAGIC_GLOBALLY unary_ltl_formula {
          assert($2 != NULL);
	  $$ = new BasicLtl(BasicLtl::LTL_G,$2);
	  delete $2;
	}
        | MAGIC_FUTURE unary_ltl_formula {
          assert($2 != NULL);
	  $$ = new BasicLtl(BasicLtl::LTL_F,$2);
	  delete $2;
	}
        ;

and_ltl_formula 
        : unary_ltl_formula { $$ = $1; }
        | and_ltl_formula '&' unary_ltl_formula {
          assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicLtl(BasicLtl::LTL_AND,$1,$3);
	  delete $1; delete $3;
	}
        ;

or_ltl_formula 
        : and_ltl_formula { $$ = $1; }
        | or_ltl_formula '|' and_ltl_formula {
          assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicLtl(BasicLtl::LTL_OR,$1,$3);
	  delete $1; delete $3;
	}
        ;

until_ltl_formula 
        : or_ltl_formula { $$ = $1; }
        | until_ltl_formula MAGIC_UNTIL or_ltl_formula {
          assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicLtl(BasicLtl::LTL_U,$1,$3);
	  delete $1; delete $3;
	}
        ;

release_ltl_formula 
        : until_ltl_formula { $$ = $1; }
        | release_ltl_formula MAGIC_RELEASE until_ltl_formula {
          assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicLtl(BasicLtl::LTL_R,$1,$3);
	  delete $1; delete $3;
	}
        ;

implies_ltl_formula 
        : release_ltl_formula { $$ = $1; }
        | release_ltl_formula MAGIC_IMPLIES implies_ltl_formula {
          assert(($1 != NULL) && ($3 != NULL));
	  BasicLtl *x = new BasicLtl(BasicLtl::LTL_NOT,$1);
	  $$ = new BasicLtl(BasicLtl::LTL_OR,x,$3);
	  delete x; delete $1; delete $3;
	}
        ;

ltl_formula : implies_ltl_formula { $$ = $1; } ;

ltl_formula_def : upper_id ':' ltl_formula ';' {
          assert(($1 != NULL) && ($3 != NULL));
	  LtlManager::AddFormula(*$1,$3);
	  delete $1; delete $3;
	  $$ = NULL;
        }
        ;

/*********************************************************************/
//begin pattern definition productions
/*********************************************************************/

dummy_var_list 
        : dummy_var { 
          assert($1 != NULL);
	  $$ = new list<string>();
	  $$->push_back(*$1);
	  delete $1;
        } 
        | dummy_var_list ',' dummy_var {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = $1;
	  $$->push_back(*$3);
	  delete $3;
	}
        ;

atomic_regular 
        : '{' '}' { $$ = new OmegaExpr(set<int>()); } 
        | '{' dummy_var_list '}' {
	  assert($2 != NULL);
	  set<int> markers;
	  for(list<string>::const_iterator i = $2->begin();i != $2->end();++i) {
	    markers.insert(strtol(i->c_str() + 1,NULL,10));
	  }
	  $$ = new OmegaExpr(markers);
	  delete $2;
	} 
        | '(' regular_expr ')' { $$ = $2; }
        ;

unary_regular 
        : atomic_regular { $$ = $1; } 
        | '+' unary_regular { 
          assert($2 != NULL); 
	  $$ = new OmegaExpr(OmegaExpr::REGULAR_PLUS,$2); 
	  delete $2; 
	} 
        | '*' unary_regular { 
          assert($2 != NULL); 
	  $$ = new OmegaExpr(OmegaExpr::REGULAR_STAR,$2); 
	  delete $2; 
	} 
        | '!' unary_regular { 
          assert($2 != NULL); 
	  $$ = new OmegaExpr(OmegaExpr::REGULAR_NOT,$2); 
	  delete $2; 
	} 
        ;

dot_regular 
        : unary_regular { $$ = $1; } 
        | dot_regular unary_regular {
          assert(($1 != NULL) && ($2 != NULL));
          $$ = new OmegaExpr(OmegaExpr::REGULAR_DOT,$1,$2);
	  delete $1; delete $2;
	} 
        ;

and_regular 
        : dot_regular { $$ = $1; } 
        | and_regular '&' dot_regular {
          assert(($1 != NULL) && ($3 != NULL));
          $$ = new OmegaExpr(OmegaExpr::REGULAR_AND,$1,$3);
	  delete $1; delete $3;
	}
        ;

or_regular 
        : and_regular { $$ = $1; } 
        | or_regular '|' and_regular {
          assert(($1 != NULL) && ($3 != NULL));
          $$ = new OmegaExpr(OmegaExpr::REGULAR_OR,$1,$3);
	  delete $1; delete $3;
	}
        ;

implies_regular 
        : or_regular { $$ = $1; } 
        | or_regular MAGIC_IMPLIES implies_regular {
          assert(($1 != NULL) && ($3 != NULL));
	  OmegaExpr *neg = new OmegaExpr(OmegaExpr::REGULAR_NOT,$1);
          $$ = new OmegaExpr(OmegaExpr::REGULAR_OR,neg,$3);
	  delete neg; delete $1; delete $3;
	}
        ;

regular_expr : implies_regular { $$ = $1; };

atomic_omega 
        : dummy_var {
          assert($1 != NULL);
	  $$ = new OmegaExpr(strtol($1->c_str() + 1,NULL,10));
	  delete $1;  
        }
        | regular_expr '[' '%' regular_expr ']' {
          assert(($1 != NULL) && ($4 != NULL));
	  $$ = new OmegaExpr($1,$4);
	  delete $1; delete $4;
        } 
        | '[' '%' regular_expr ']' {
          assert($3 != NULL);
	  $$ = new OmegaExpr(static_cast<const OmegaExpr*>(NULL),$3);
	  delete $3;
	} 
        | '(' omega_expr ')' { $$ = $2; }
        ;

unary_omega 
        : atomic_omega { $$ = $1; } 
        | '[' '!' unary_omega ']' {
          assert($3 != NULL); 
	  $$ = new OmegaExpr(OmegaExpr::OMEGA_NOT,$3); 
	  delete $3; 	  
	}
        | MAGIC_NEXT_TIME unary_omega {
          assert($2 != NULL); 
	  $$ = new OmegaExpr(OmegaExpr::OMEGA_X,$2); 
	  delete $2;
	}
        | MAGIC_GLOBALLY unary_omega {
          assert($2 != NULL); 
	  $$ = new OmegaExpr(OmegaExpr::OMEGA_G,$2); 
	  delete $2;
	}
        | MAGIC_FUTURE unary_omega {
          assert($2 != NULL); 
	  $$ = new OmegaExpr(OmegaExpr::OMEGA_F,$2); 
	  delete $2;
	}
        ;

and_omega 
        : unary_omega { $$ = $1; } 
        | and_omega '&' unary_omega {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new OmegaExpr(OmegaExpr::OMEGA_AND,$1,$3);
	  delete $1; delete $3;
	}
        ;

or_omega 
        : and_omega { $$ = $1; } 
        | or_omega '|' and_omega {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new OmegaExpr(OmegaExpr::OMEGA_OR,$1,$3);
	  delete $1; delete $3;
	}
        ;

until_omega 
        : or_omega { $$ = $1; } 
        | until_omega MAGIC_UNTIL or_omega {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new OmegaExpr(OmegaExpr::OMEGA_U,$1,$3);
	  delete $1; delete $3;
	}
        ;

release_omega 
        : until_omega { $$ = $1; } 
        | release_omega MAGIC_RELEASE until_omega {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new OmegaExpr(OmegaExpr::OMEGA_R,$1,$3);
	  delete $1; delete $3;
	}
        ;

implies_omega 
        : release_omega { $$ = $1; } 
        | release_omega MAGIC_IMPLIES implies_omega {
          assert(($1 != NULL) && ($3 != NULL));
	  OmegaExpr *neg = new OmegaExpr(OmegaExpr::OMEGA_NOT,$1);
          $$ = new OmegaExpr(OmegaExpr::OMEGA_OR,neg,$3);
	  delete neg; delete $1; delete $3;	  
	}
        ;

omega_expr : implies_omega { $$ = $1; };

omega_op_def 
        : omega_op_name ':' omega_expr ';' {
          assert(($1 != NULL) && ($3 != NULL));
          if(*$1 == "OP::X") {
	    Stdcerror("operator name OP::X reserved for the next-time (X) operator");
	  }
	  if(*$1 == "OP::G") {
	    Stdcerror("operator name OP::G reserved for the globally (G) operator");
	  }
	  if(*$1 == "OP::F") {
	    Stdcerror("operator name OP::F reserved for the future (F) operator");
	  }
	  if(*$1 == "OP::U") {
	    Stdcerror("operator name OP::U reserved for the until (U) operator");
	  }
	  if(*$1 == "OP::R") {
	    Stdcerror("operator name OP::R reserved for the release (R) operator");
	  }
	  SeawManager::AddOmega(*$1,$3);
	  delete $1; delete $3;
	  $$ = NULL;
        }
        ;

/*********************************************************************/
//begin SE-AW formula definition productions
/*********************************************************************/

seaw_formula_list 
        : seaw_formula { 
          $$ = new list<BasicSeaw*>(); 
	  $$->push_back($1); 
        } 
        | seaw_formula_list ',' seaw_formula {
	  assert($1 != NULL);
	  $$ = $1;
	  $$->push_back($3);
	}
        ;

atomic_op_seaw 
        : MAGIC_ALL_PATHS omega_op_name '(' seaw_formula_list ')' {
          assert(($2 != NULL) && ($4 != NULL));
	  $$ = new BasicSeaw(*$2,*$4);
	  for(list<BasicSeaw*>::const_iterator i = $4->begin();i != $4->end();++i) delete *i;
	  delete $2; delete $4;
        } 
        | '(' op_seaw ')' { $$ = $2; }
        ;

and_op_seaw 
        : atomic_op_seaw { $$ = $1; } 
        | and_nop_seaw '&' atomic_op_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicSeaw(BasicSeaw::SEAW_AND,$1,$3);
	  delete $1; delete $3;
	} 
        | and_op_seaw '&' atomic_nop_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicSeaw(BasicSeaw::SEAW_AND,$1,$3);
	  delete $1; delete $3;
	} 
        | and_op_seaw '&' atomic_op_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicSeaw(BasicSeaw::SEAW_AND,$1,$3);
	  delete $1; delete $3;
	}
        ;

or_op_seaw 
        : and_op_seaw { $$ = $1; } 
        | or_nop_seaw '|' and_op_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicSeaw(BasicSeaw::SEAW_OR,$1,$3);
	  delete $1; delete $3;
	} 
        | or_op_seaw '|' and_nop_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicSeaw(BasicSeaw::SEAW_OR,$1,$3);
	  delete $1; delete $3;
	} 
        | or_op_seaw '|' and_op_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicSeaw(BasicSeaw::SEAW_OR,$1,$3);
	  delete $1; delete $3;
	}
        ;

implies_op_seaw 
        : or_op_seaw { $$ = $1; } 
        | or_nop_seaw MAGIC_IMPLIES implies_op_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  BasicSeaw *neg = new BasicSeaw($1);
	  $$ = new BasicSeaw(BasicSeaw::SEAW_OR,neg,$3);
	  delete neg; delete $1; delete $3;	  
	}
        ;

op_seaw : implies_op_seaw { $$ = $1; };

atomic_nop_seaw 
        : '[' conditional_expression ']' {
          assert($2 != NULL);
	  $$ = new BasicSeaw($2);
	  delete $2;
        } 
        | '{' action_label_list '}' {
          assert($2 != NULL);
	  $$ = new BasicSeaw(*$2);
	  delete $2;
	} 
        | '(' nop_seaw ')' { $$ = $2; }
        ;

unary_nop_seaw 
        : atomic_nop_seaw { $$ = $1; } 
        | '!' unary_nop_seaw {
          assert($2 != NULL);
	  $$ = new BasicSeaw($2);
	  delete $2;
	}
        ;

and_nop_seaw 
        : unary_nop_seaw { $$ = $1; }
        | and_nop_seaw '&' unary_nop_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicSeaw(BasicSeaw::SEAW_AND,$1,$3);
	  delete $1; delete $3;
	}
        ;

or_nop_seaw 
        : and_nop_seaw { $$ = $1; } 
        | or_nop_seaw '|' and_nop_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  $$ = new BasicSeaw(BasicSeaw::SEAW_OR,$1,$3);
	  delete $1; delete $3;
	}
        ;

implies_nop_seaw 
        : or_nop_seaw { $$ = $1; } 
        | or_nop_seaw MAGIC_IMPLIES implies_nop_seaw {
	  assert(($1 != NULL) && ($3 != NULL));
	  BasicSeaw *neg = new BasicSeaw($1);
	  $$ = new BasicSeaw(BasicSeaw::SEAW_OR,neg,$3);
	  delete neg; delete $1; delete $3;	  
	}
        ;

nop_seaw : implies_nop_seaw { $$ = $1; };

seaw_formula : nop_seaw { $$ = $1; } | op_seaw { $$ = $1; };

seaw_formula_def 
        : seaw_formula_name ':' seaw_formula ';' {
          assert(($1 != NULL) && ($3 != NULL));
	  SeawManager::AddSeaw(*$1,$3);
	  delete $1; delete $3;
	  $$ = NULL;
        }
        ;

/*********************************************************************/
//begin top-level specification productions
/*********************************************************************/

ext_def_list
        /*fsp*/
	: fsp_definition { 
          assert($1 == NULL);
	  $$ = NULL; 
        }
        /*list_fsp*/
	| ext_def_list fsp_definition {
          assert(($1 == NULL) && ($2 == NULL));
	  $$ = NULL; 
	}
        /*ltl*/
        | ltl_formula_def {
	  assert($1 == NULL);
	  $$ = NULL;
	}
        /*list_ltl*/
        | ext_def_list ltl_formula_def {
          assert(($1 == NULL) && ($2 == NULL));
	  $$ = NULL; 
	}
        /*omega_op*/
        | omega_op_def {
	  assert($1 == NULL);
	  $$ = NULL;
	}
        /*list_omega_op*/
        | ext_def_list omega_op_def {
          assert(($1 == NULL) && ($2 == NULL));
	  $$ = NULL; 
	}
        /*seaw*/
        | seaw_formula_def {}
        /*list_seaw*/
        | ext_def_list seaw_formula_def {}
        /*c_prog*/
	| c_prog_info { 
	  assert($1 != NULL);
	  if(Database::progInfos.count($1->GetName()) == 0) {
	    Database::progInfos[$1->GetName()] = *$1;
	  } else {
	    Database::progInfos[$1->GetName()].AddProgInfo(*$1);
	  }
	  delete $1;
	  $$ = NULL; 
	}
        /*list_c_prog*/
	| ext_def_list c_prog_info {
	  assert(($1 == NULL) && ($2 != NULL));
	  if(Database::progInfos.count($2->GetName()) == 0) {
	    Database::progInfos[$2->GetName()] = *$2;
	  } else {
	    Database::progInfos[$2->GetName()].AddProgInfo(*$2);
	  }
	  delete $2;
	  $$ = NULL; 
	}
        /*c_proc*/
	| c_proc_info { 
	  assert($1 != NULL);
	  if(Database::procAddInfos.count($1->GetName()) == 0) {
	    Database::procAddInfos[$1->GetName()] = *$1;
	  } else {
	    Database::procAddInfos[$1->GetName()].AddProcAddInfo(*$1);
	  }
	  delete $1;
	  $$ = NULL; 
	}
        /*list_c_proc*/
	| ext_def_list c_proc_info {
	  assert(($1 == NULL) && ($2 != NULL));
	  if(Database::procAddInfos.count($2->GetName()) == 0) {
	    Database::procAddInfos[$2->GetName()] = *$2;
	  } else {
	    Database::procAddInfos[$2->GetName()].AddProcAddInfo(*$2);
	  }
	  delete $2;
	  $$ = NULL; 
	}
        ;

spec_translation_unit 
        : ext_def_list {
          assert($1 == NULL);
	  $$ = NULL;
  	} 
        ;

/*********************************************************************/
//the common grammar part
/*********************************************************************/

translation_unit
        : stdc_translation_unit {
          assert($1 != NULL);
	  CSymbolTable::ast = $1;
	  $$ = NULL;
        }
        | spec_translation_unit {
          assert($1 == NULL);
          CSymbolTable::ast = NULL;
	  $$ = NULL;
  	}
	;
%%

void Stdcerror(char *s)
{
  fflush(stdout);
  if(Database::PARSE_ECHO) {
    Util::Message(2,"\n----------------------------------------------------------\n");
  }
  Util::Message(2,"ERROR: at line number %d column %d: %s ...\n",lineNum,column,s);
  if(!Database::PARSE_ECHO) {
    Util::Message(2,"try --echo for the token after which the error ocurred ...\n");
  }
  GlobalShutdown(-1);
}

list<LtsTrans> * magic::MakeTransList(FspTransSeq *seq,FspBody *body)
{
  list<LtsTrans> *res = new list<LtsTrans>();
  if(seq->end.empty()) {
    if(body->id == "") {
      string tvar = Util::NewTempVar();
      for(list<LtsTrans>::iterator i = body->tr.begin();i != body->tr.end();++i) {
	i->SetInitState(tvar);
	Database::fspInfo.AddTrans(*i);
      }
      for(list<LtsTrans>::iterator i = seq->begin.begin();i != seq->begin.end();++i) {
	i->SetFinalState(tvar);
	res->push_back(*i);
      }
    } else {
      for(list<LtsTrans>::iterator i = seq->begin.begin();i != seq->begin.end();++i) {
	i->SetFinalState(body->id);
	res->push_back(*i);
      }
    }
  } else {
    if(body->id == "") {
      string tvar = Util::NewTempVar();
      for(list<LtsTrans>::iterator i = body->tr.begin();i != body->tr.end();++i) {
	i->SetInitState(tvar);
	Database::fspInfo.AddTrans(*i);
      }
      for(list<LtsTrans>::iterator i = seq->end.begin();i != seq->end.end();++i) {
	i->SetFinalState(tvar);
	Database::fspInfo.AddTrans(*i);
      }
      res->insert(res->end(),seq->begin.begin(),seq->begin.end());
    } else {
      for(list<LtsTrans>::iterator i = seq->end.begin();i != seq->end.end();++i) {
	i->SetFinalState(body->id);
	Database::fspInfo.AddTrans(*i);
      }
      res->insert(res->end(),seq->begin.begin(),seq->begin.end());
    }
  }
  return res;
}

/*********************************************************************/
//shortcut evaluation of constant integer expressions
/*********************************************************************/
BasicExpr *magic::SimplifyExpr(BasicExpr *arg)
{
  //unary expression
  if(typeid(*arg) == typeid(UnaryExpr)) {
    UnaryExpr *a = static_cast<UnaryExpr*>(arg);
    if(typeid(*(a->expr)) == typeid(IntConstExpr)) {
      IntConstExpr *b = static_cast<IntConstExpr*>(a->expr);
      long long c = b->val;
      if(a->op == '-') {
	delete arg;
	return new IntConstExpr(-c);
      } else if(a->op == '+') {
	delete arg;
	return new IntConstExpr(c);
      } else if(a->op == '~') {
	delete arg;
	return new IntConstExpr(~c);
      } else if(a->op == '!') {
	delete arg;
	return new IntConstExpr((c == 0) ? 1 : 0);
      } else return arg;
    } else return arg;
  }
  //binary expression
  else if(typeid(*arg) == typeid(BinaryExpr)) {
    BinaryExpr *a = static_cast<BinaryExpr*>(arg);
    if((typeid(*(a->lhs)) == typeid(IntConstExpr)) && (typeid(*(a->rhs)) == typeid(IntConstExpr))) {
      long long b = (static_cast<IntConstExpr*>(a->lhs))->val;
      long long c = (static_cast<IntConstExpr*>(a->rhs))->val;
      if(a->op == '*') {
	delete arg;
	return new IntConstExpr(b * c);
      } else if(a->op == '/') {
	delete arg;
	return new IntConstExpr(b / c);
      } else if(a->op == '%') {
	delete arg;
	return new IntConstExpr(b % c);
      } else if(a->op == '+') {
	delete arg;
	return new IntConstExpr(b + c);
      } else if(a->op == '-') {
	delete arg;
	return new IntConstExpr(b - c);
      } else if(a->op == MAGIC_LEFT_OP) {
	delete arg;
	return new IntConstExpr(b << c);
      } else if(a->op == MAGIC_RIGHT_OP) {
	delete arg;
	return new IntConstExpr(b >> c);
      } else if(a->op == '<') {
	delete arg;
	return new IntConstExpr(b < c);
      } else if(a->op == '>') {
	delete arg;
	return new IntConstExpr(b > c);
      } else if(a->op == MAGIC_LE_OP) {
	delete arg;
	return new IntConstExpr(b <= c);
      } else if(a->op == MAGIC_GE_OP) {
	delete arg;
	return new IntConstExpr(b >= c);
      } else if(a->op == MAGIC_EQ_OP) {
	delete arg;
	return new IntConstExpr(b == c);
      } else if(a->op == MAGIC_NE_OP) {
	delete arg;
	return new IntConstExpr(b != c);
      } else if(a->op == '&') {
	delete arg;
	return new IntConstExpr(b & c);
      } else if(a->op == '^') {
	delete arg;
	return new IntConstExpr(b ^ c);
      } else if(a->op == '|') {
	delete arg;
	return new IntConstExpr(b | c);
      } else if(a->op == MAGIC_AND_OP) {
	delete arg;
	return new IntConstExpr(b && c);
      } else if(a->op == MAGIC_OR_OP) {
	delete arg;
	return new IntConstExpr(b || c);
      } else {
	assert(a->op == ',');
	return arg;
      }
    } else return arg;
  }
  //quest expression
  else if(typeid(*arg) == typeid(QuestExpr)) {
    QuestExpr *a = static_cast<QuestExpr*>(arg);
    if(typeid(*(a->cond)) == typeid(IntConstExpr)) {
      long b = (static_cast<IntConstExpr*>(a->cond))->val;
      BasicExpr *c = static_cast<BasicExpr*>((b == 0) ? a->ecase->Clone() : a->tcase->Clone());
      delete arg;
      return c;
    } else return arg;
  }
  //this should not happen
  else {
    assert(false);
    return NULL;
  }
}

/*********************************************************************/
//given the components of a for statement return an equivalent while
//statement
/*********************************************************************/
Stmt *magic::ForToWhile(Stmt *one,Stmt *two,BasicExpr *three,Stmt *body)
{
  assert(typeid(*two) == typeid(ExprStmt));
  
  //extract the expression from the for statement
  BasicExpr *expr2 = static_cast<BasicExpr*>((static_cast<ExprStmt*>(two))->expr->Clone());
  if(typeid(*expr2) == typeid(EmptyExpr)) {
    delete expr2;
    expr2 = new IntConstExpr(1);
  }

  //create the while statement
  list<Stmt*> wbody;
  wbody.push_back(body);
  Stmt *a = new ExprStmt(three);
  wbody.push_back(a);
  StmtList *b = new StmtList(wbody);
  delete a;
  Stmt *c = new CompStmt(b);
  delete b;
  Stmt *wstmt = new WhileStmt(expr2,c);
  delete expr2; delete c;

  //create the final compound statement
  list<Stmt*> f;
  f.push_back(one);
  f.push_back(wstmt);
  StmtList *g = new StmtList(f);
  delete wstmt;
  Stmt *res = new CompStmt(g);
  delete g;
  
  //done
  return res;
}

/*********************************************************************/
//end of StdcParser.cpp
/*********************************************************************/
