/**************************** CPPHeaderFile ***************************

* FileName [StdcAst.h]

* PackageName [parser]

* Synopsis [Header file for Ast classes for C file.]

* Description [These classes encapsulate top-level ast of a file.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __STDCAST_H__
#define __STDCAST_H__

namespace magic {

/*********************************************************************/
//declarator - a declarator is an identifier plus optionally a list of
//identifiers for procedure declarations
/*********************************************************************/
class Decl : public Node
{
 private:
  string id;
  list<string> idList;
 public:
  Decl() {}
  Decl(const string &n,const list<string> &p) { 
    id = n;
    idList = p;
  }
  Decl(const Decl &rhs) { *this = rhs; }
  ~Decl() {}
  const Decl &operator = (const Decl &rhs) {
    id = rhs.id;
    idList = rhs.idList;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new Decl(id,idList)); }
  void Cleanup() { assert(false); }

  string ToString() const {
    string res = id + " ( ";
    for(list<string>::const_iterator i = idList.begin();i != idList.end();++i) {
      res += *i + " ";
      if(*i != idList.back()) res += ", ";
    }
    return res + ") ";
  }
  void Apply(DFSAdapter &a) const;

  void SetId(const string &i) { id = i; }
  string &GetId() { return id; }
  void SetIdList(const list<string> &i) { idList = i; }
  void AddIdToList(const string &i) { idList.push_back(i); }
  list<string> &GetIdList() { return idList; }
  void ClearIdList() { idList.clear(); }
};

/*********************************************************************/
//procedure
/*********************************************************************/
class Proc : public Node
{
 private:
  string name;
  list<string> params;
  list<string> locals;
  CompStmt *body;
 public:
  Proc() { body = NULL; }
  Proc(const string &n,const list<string> &p,const list<string> &l,const CompStmt *b) { 
    name = n;
    params = p;
    locals = l;
    body = (b == NULL) ? NULL : static_cast<CompStmt*>(b->Clone());
  }
  Proc(const Proc &rhs) { body = NULL; *this = rhs; }
  ~Proc() { Cleanup(); }
  const Proc &operator = (const Proc &rhs) {
    Cleanup();
    name = rhs.name;
    params = rhs.params;
    locals = rhs.locals;
    body = (rhs.body == NULL) ? NULL : static_cast<CompStmt*>(rhs.body->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new Proc(name,params,locals,body)); }
  void Cleanup() {
    if(body != NULL) delete body;
    body = NULL;
  }

  string ToString() const {
    string res = name + " ( ";
    for(list<string>::const_iterator i = params.begin();i != params.end();++i) {
      res += *i + " ";
      if(*i != params.back()) res += ", ";
    }
    return res + ") " + body->ToString();
  }
  void Apply(DFSAdapter &a) const;

  friend class DFSAdapter;
  friend class ProcInfo;
  friend class Database; //to access name
};

/*********************************************************************/
//C file - a list of C procedures
/*********************************************************************/
class StdcFile : public Node
{
 private:
  list<Proc*> procs;
 public:
  StdcFile() {}
  StdcFile(const list<Proc*> &p) {
    for(list<Proc*>::const_iterator i = p.begin();i != p.end();++i) {
      procs.push_back(static_cast<Proc*>((*i)->Clone()));
    }
  }
  StdcFile(const StdcFile &rhs) { *this = rhs; }
  ~StdcFile() { Cleanup(); }
  const StdcFile &operator = (const StdcFile &rhs) {
    Cleanup();
    for(list<Proc*>::const_iterator i = rhs.procs.begin();i != rhs.procs.end();++i) {
      procs.push_back(static_cast<Proc*>((*i)->Clone()));
    }
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new StdcFile(procs)); }
  void Cleanup() {
    for(list<Proc*>::iterator i = procs.begin();i != procs.end();++i) {
      delete *i;
    }
    procs.clear();
  }

  string ToString() const {
    string res = "";
    for(list<Proc*>::const_iterator i = procs.begin();i != procs.end();++i) {
      res += (*i)->ToString() + "\n";
    }
    return res;
  }
  void Apply(DFSAdapter &a) const;

  //add a procedure at the end of the list
  void AddProc(const Proc *p) { procs.push_back(static_cast<Proc*>(p->Clone())); }

  friend class DFSAdapter;
  friend class Database; //to access procs
};

} //namespace magic

#endif //__STDCAST_H__

/*********************************************************************/
//end of StdcAst.h
/*********************************************************************/
