/******************************** CPPFile *****************************

* FileName [SeawManager.cpp]

* PackageName [main]

* Synopsis [Method definitions of SeawManager class.]

* SeeAlso [SeawManager.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <string>
#include <list>
#include <vector>
#include <set>
#include <map>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "Seaw.h"
#include "SeawManager.h"
using namespace magic;

/*********************************************************************/
//static fields of SeawManager
/*********************************************************************/
map<string,OmegaOp> SeawManager::omegaMap;
map<string,SeawFormula> SeawManager::seawMap;
map< string,set<Expr> > SeawManager::propMap;
map< string,set<Action> > SeawManager::actMap;

/*********************************************************************/
//initialise the manager
/*********************************************************************/
void SeawManager::Initialise()
{
  //add definitions for the LTL operators
  OmegaExpr *d1 = new OmegaExpr(1);
  OmegaExpr *d2 = new OmegaExpr(2);
  OmegaExpr *oeX = new OmegaExpr(OmegaExpr::OMEGA_X,d1);
  AddOmega("OP::X",oeX);
  delete oeX;
  OmegaExpr *oeG = new OmegaExpr(OmegaExpr::OMEGA_G,d1);
  AddOmega("OP::G",oeG);
  delete oeG;
  OmegaExpr *oeF = new OmegaExpr(OmegaExpr::OMEGA_F,d1);
  AddOmega("OP::F",oeF);
  delete oeF;
  OmegaExpr *oeU = new OmegaExpr(OmegaExpr::OMEGA_U,d1,d2);
  AddOmega("OP::U",oeU);
  delete oeU;
  OmegaExpr *oeR = new OmegaExpr(OmegaExpr::OMEGA_R,d1,d2);
  AddOmega("OP::R",oeR);
  delete oeR;
  delete d1; 
  delete d2;
}

/*********************************************************************/
//add a new omega operator
/*********************************************************************/
void SeawManager::AddOmega(const string &n,const OmegaOp &o)
{
  if(omegaMap.count(n) == 0) {
    set<int> markers;
    o.ComputeMarkers(markers);
    if(markers.empty()) {
      Util::Error("ERROR: no markers in omega expression " + n + " ...\n");
    }
    int max = *(markers.begin());
    for(set<int>::const_iterator i = markers.begin();i != markers.end();++i) {
      int marker = *i;
      if(*i <= 0) {
	Util::Error("ERROR: invalid index in omega expression " + n + " ...\n"
		    "COMMENT: marker indices must be greater than zero ...\n");
      }
      max = (max < marker) ? marker : max;
    }
    if(max != static_cast<int>(markers.size())) {
      Util::Error("ERROR: non-contiguous markers in omega expression " + n + " ...\n");
    }
    omegaMap[n] = o;
  } else Util::Error("ERROR: omega operator " + n + " multiply defined ...\n");
}

/*********************************************************************/
//add a new SE-AW formula
/*********************************************************************/
void SeawManager::AddSeaw(const string &n,const SeawFormula &s)
{
  if(seawMap.count(n) == 0) {
    seawMap[n] = s;
    s.ComputeProps(propMap[n]);
    s.ComputeActions(actMap[n]);
  } else Util::Error("ERROR: SE-AW formula " + n + " multiply defined ...\n");
}

/*********************************************************************/
//perform sanity check on various data structures : every omega
//operator referred to in any formula is defined and also has the same
//number of parameters as the number of arguments passed
/*********************************************************************/
void SeawManager::SanityCheck()
{
  for(map<string,SeawFormula>::const_iterator i = seawMap.begin();i != seawMap.end();++i) {
    list< pair<string,size_t> > opsToArgs;
    i->second.ComputeOpsToArgs(opsToArgs);
    for(list< pair<string,size_t> >::const_iterator j = opsToArgs.begin();j != opsToArgs.end();++j) {
      if(omegaMap.count(j->first) == 0) {
	Util::Error("ERROR: undefined omega operator " + j->first + " used in SE-AW formula " + i->first + " ...\n");
      }
      set<int> markers;
      omegaMap[j->first].ComputeMarkers(markers);
      if(markers.size() != j->second) {
	Util::Error("ERROR: incorrect number of arguments passed to omega operator " + 
		    j->first + " in SE-AW formula " + i->first + " ...\n");
      }
    }
  }
}

/*********************************************************************/
//given a formula name return its set of propositions
/*********************************************************************/
const set<Expr> &SeawManager::GetProps(const string &name)
{
  map< string,set<Expr> >::const_iterator i = propMap.find(name);
  if(i == propMap.end()) {
    Util::Error("ERROR: trying to find actions of undefined SE-AW formula " + name + " ...\n");
  } else return i->second;
}

/*********************************************************************/
//given a formula name return its set of actions
/*********************************************************************/
const set<Action> &SeawManager::GetActions(const string &name)
{
  map< string,set<Action> >::const_iterator i = actMap.find(name);
  if(i == actMap.end()) {
    Util::Error("ERROR: trying to find actions of undefined SE-AW formula " + name + " ...\n");
  } else return i->second;
}

/*********************************************************************/
//compute the set of assign actions in the SE-AW formula with given
//name
/*********************************************************************/
void SeawManager::ComputeAssignActions(const string &name,set<Action> &res)
{
  const set<Action> &actions = GetActions(name);
  for(set<Action>::const_iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsAssign()) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of return actions in the SE-AW formula with given
//name
/*********************************************************************/
void SeawManager::ComputeReturnActions(const string&name,set<Action> &res)
{
  const set<Action> &actions = GetActions(name);
  for(set<Action>::const_iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsReturn()) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of actions associated with a return. the arguments
//are (1) the name of the SE-AW formula, (2) the expression for the
//return value, (3) a list of arguments to the procedure, and (4) a
//list of expressions describing the state in which the return is
//executed.
/*********************************************************************/
void SeawManager::GetAssignActions(const string &name,const Expr &lhs,const Expr &rhs,
				   const list<string> &params,const set<Expr> &context,
				   set<Action> &res)
{
  set<Action> assigns; ComputeAssignActions(name,assigns);
  for(set<Action>::const_iterator i = assigns.begin();i != assigns.end();++i) {
    if(i->AssignActionPossible(lhs,rhs,params,context)) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of actions associated with a return. the arguments
//are the name of the SE-AW formula, the expression for the return
//value, a list of arguments to the procedure, and a list of
//expressions describing the state in which the return is executed.
/*********************************************************************/
void SeawManager::GetReturnActions(const string &name,const Expr &expr,const list<string> &params,
				   const set<Expr> &context,set<Action> &res)
{
  set<Action> returns; ComputeReturnActions(name,returns);
  for(set<Action>::const_iterator i = returns.begin();i != returns.end();++i) {
    if(i->ReturnActionPossible(expr,params,context)) res.insert(*i);
  }
}

/*********************************************************************/
//return the expressions associated with assign actions
/*********************************************************************/
void SeawManager::GetAssignActExprs(const string &name,set< pair<Expr,Expr> > &res)
{
  const set<Action> &actions = GetActions(name);
  for(set<Action>::iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsAssign()) {
      res.insert(pair<Expr,Expr>(i->GetLhsExpr(),i->GetRhsExpr()));
    }
  }
}

/*********************************************************************/
//return the expressions associated with return actions
/*********************************************************************/
void SeawManager::GetRetActExprs(const string &name,set<Expr> &res)
{
  const set<Action> &actions = GetActions(name);
  for(set<Action>::iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsReturn() && (!i->GetRetExpr().IsEmptyExpr())) res.insert(i->GetRetExpr());
  }
}

/*********************************************************************/
//return the BasicSeaw formula with given name
/*********************************************************************/
const BasicSeaw *SeawManager::GetBasicSeaw(const string &name)
{
  map<string,SeawFormula>::const_iterator i = seawMap.find(name);
  if(i == seawMap.end()) {
    Util::Error("ERROR: SE-AW formula " + name + " undefined ...\n");
  }
  return i->second.GetSeaw();
}

/*********************************************************************/
//end of SeawManager.cpp
/*********************************************************************/
