/**************************** CPPHeaderFile ***************************

* FileName [Seaw.h]

* PackageName [main]

* Synopsis [Header file for Seaw classes.]

* Description [These classes SE-AW operators and formulas.]

* SeeAlso [Seaw.cpp]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __SEAW_H__
#define __SEAW_H__

namespace magic {

//other classes needed
class Expr;
class Action;
class BasicLtl;
class BasicSeaw;

class OmegaExpr
{
 private:
  int type;
  OmegaExpr *sub1,*sub2;
  set<int> markers;

 public:
  static const int OMEGA_NONE;
  static const int OMEGA_X;
  static const int OMEGA_G;
  static const int OMEGA_F;
  static const int OMEGA_U;
  static const int OMEGA_R;
  static const int OMEGA_OR;
  static const int OMEGA_AND;
  static const int OMEGA_NOT;
  static const int OMEGA_DUMMY;
  static const int OMEGA_OMEGA;
  static const int REGULAR_OR;
  static const int REGULAR_AND;
  static const int REGULAR_DOT;
  static const int REGULAR_PLUS;
  static const int REGULAR_STAR;
  static const int REGULAR_NOT;
  static const int REGULAR_ATOMIC;

  OmegaExpr();
  OmegaExpr(const int t,const OmegaExpr *s1,const OmegaExpr *s2,const set<int> &m);
  //create an atomic regular expression
  OmegaExpr(const set<int> &m);
  //create atomic omega expression from a dummy variable
  OmegaExpr(const int m);
  //create atomic omega expression from regular expressions
  OmegaExpr(const OmegaExpr *s1,const OmegaExpr *s2);
  //create an unary regular or omega expression
  OmegaExpr(const int t,const OmegaExpr *s);
  //create an binary regular or omega expression
  OmegaExpr(const int t,const OmegaExpr *s1,const OmegaExpr *s2);
  OmegaExpr(const OmegaExpr &rhs);
  ~OmegaExpr();

  const OmegaExpr &operator = (const OmegaExpr &rhs);
  void Cleanup();
  OmegaExpr *Clone() const;

  void ComputeMarkers(set<int> &res) const;
  BasicLtl *ToLtl(const vector<BasicSeaw*> &args) const;

  friend class SeawChecker;
};

class OmegaOp
{
 private:
  OmegaExpr *omega;

 public:
  OmegaOp() : omega(NULL) {}
  OmegaOp(const OmegaExpr *o);
  OmegaOp(const OmegaOp &rhs);
  ~OmegaOp() { Cleanup(); }

  const OmegaOp &operator = (const OmegaOp &rhs);  
  void Cleanup();

  const OmegaExpr *GetOmega() const { return omega; }

  void ComputeMarkers(set<int> &res) const;
};

class BasicSeaw
{
 private:
  int type;
  Expr prop;
  set<Action> acts;
  string omega;
  vector<BasicSeaw*> subs;

 public:
  static const int SEAW_NONE;
  static const int SEAW_PROP;
  static const int SEAW_ACTS;
  static const int SEAW_OMEGA;
  static const int SEAW_AND;
  static const int SEAW_OR;
  static const int SEAW_NOT;

  BasicSeaw() { type = SEAW_NONE; }
  BasicSeaw(const int t,const Expr &p,const set<Action> &a,const string &o,const vector<BasicSeaw*> &s);
  //construct atomic propositional formula
  BasicSeaw(const Expr &p);  
  //construct atomic action-set formula
  BasicSeaw(const set<Action> &a);
  //construct unary (i.e. negation) formula
  BasicSeaw(const BasicSeaw *s);
  //construct omega operator formula
  BasicSeaw(const string &o,const list<BasicSeaw*> &s);
  //construct a binary formula
  BasicSeaw(const int t,const BasicSeaw *s1,const BasicSeaw *s2);
  //copy constructor
  BasicSeaw(const BasicSeaw &rhs);
  //destructor
  ~BasicSeaw() { Cleanup(); }

  const BasicSeaw &operator = (const BasicSeaw &rhs);
  void Cleanup();
  BasicSeaw *Clone() const;

  int GetType() const { return type; }
  const Expr &GetProp() const { return prop; }
  const set<Action> &GetActs() const { return acts; }
  const string &GetOmega() const { return omega; }
  const vector<BasicSeaw*> GetSubs() const { return subs; }

  string ToString() const;
  void ComputeOpsToArgs(list< pair<string,size_t> > &res) const;
  void ComputeProps(set<Expr> &res) const;
  void ComputeActions(set<Action> &res) const;
  void SanityCheck() const;
};

class SeawFormula
{
 private:
  BasicSeaw *seaw;

 public:
  SeawFormula() : seaw(NULL) {}
  SeawFormula(const BasicSeaw *s);
  SeawFormula(const SeawFormula &rhs);
  ~SeawFormula() { Cleanup(); }

  const SeawFormula &operator = (const SeawFormula &rhs);
  void Cleanup();
  
  const BasicSeaw *GetSeaw() const { return seaw; }
  
  void ComputeOpsToArgs(list< pair<string,size_t> > &res) const;
  void ComputeProps(set<Expr> &res) const;
  void ComputeActions(set<Action> &res) const;
};

} //namespace magic

#endif //__SEAW_H__

/*********************************************************************/
//end of Seaw.h
/*********************************************************************/
