/******************************** CPPFile *****************************

* FileName [Seaw.cpp]

* PackageName [main]

* Synopsis [Method definitions of Seaw classes.]

* SeeAlso [Seaw.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <string>
#include <list>
#include <vector>
#include <set>
#include <map>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Buchi.h"
#include "Seaw.h"
#include "SeawManager.h"
using namespace magic;

/*********************************************************************/
//static fields of OmegaExpr class
/*********************************************************************/
const int OmegaExpr::OMEGA_NONE = 10500;
const int OmegaExpr::OMEGA_X = 10510;
const int OmegaExpr::OMEGA_G = 10520;
const int OmegaExpr::OMEGA_F = 10530;
const int OmegaExpr::OMEGA_U = 10540;
const int OmegaExpr::OMEGA_R = 10550;
const int OmegaExpr::OMEGA_OR = 10560;
const int OmegaExpr::OMEGA_AND = 10570;
const int OmegaExpr::OMEGA_NOT = 10580;
const int OmegaExpr::OMEGA_DUMMY = 10590;
const int OmegaExpr::OMEGA_OMEGA = 10600;
const int OmegaExpr::REGULAR_OR = 10610;
const int OmegaExpr::REGULAR_AND = 10620;
const int OmegaExpr::REGULAR_DOT = 10630;
const int OmegaExpr::REGULAR_PLUS = 10640;
const int OmegaExpr::REGULAR_STAR = 10650;
const int OmegaExpr::REGULAR_NOT = 10660;
const int OmegaExpr::REGULAR_ATOMIC = 10670;

/*********************************************************************/
//methods of OmegaExpr class
/*********************************************************************/
OmegaExpr::OmegaExpr() : type(OMEGA_NONE),sub1(NULL),sub2(NULL) {}

OmegaExpr::OmegaExpr(const int t,const OmegaExpr *s1,const OmegaExpr *s2,const set<int> &m)
  : type(t),markers(m)
{
  sub1 = (s1 == NULL) ? NULL : s1->Clone();
  sub2 = (s2 == NULL) ? NULL : s2->Clone();
}

//create atomic regular expression
OmegaExpr::OmegaExpr(const set<int> &m)
{
  type = REGULAR_ATOMIC;
  sub1  = sub2 = NULL;
  markers = m;
}

//create atomic omega expression from a dummy variable
OmegaExpr::OmegaExpr(const int m)
{
  type = OMEGA_DUMMY;
  sub1  = sub2 = NULL;
  markers.insert(m);
}

//create atomic omega expression from regular expressions
OmegaExpr::OmegaExpr(const OmegaExpr *s1,const OmegaExpr *s2)
{
  type = OMEGA_OMEGA;
  sub1 = (s1 == NULL) ? NULL : s1->Clone();
  sub2 = s2->Clone();
}

//create unary regular or omega expression
OmegaExpr::OmegaExpr(const int t,const OmegaExpr *s)
{
  assert((t == REGULAR_PLUS) || (t == REGULAR_STAR) || 
	 (t == REGULAR_NOT) || (t == OMEGA_NOT) || 
	 (t == OMEGA_X) || (t == OMEGA_G) || (t == OMEGA_F));
  type = t;
  sub1 = s->Clone();
  sub2 = NULL;
}

//create binary regular or omega expression
OmegaExpr::OmegaExpr(const int t,const OmegaExpr *s1,const OmegaExpr *s2)
{
  assert((t == REGULAR_OR) || (t == REGULAR_AND) || 
	 (t == REGULAR_DOT) || (t == OMEGA_OR) || 
	 (t == OMEGA_AND) || (t == OMEGA_U) || (t == OMEGA_R));
  type = t;
  sub1 = s1->Clone();
  sub2 = s2->Clone();
}

OmegaExpr::OmegaExpr(const OmegaExpr &rhs)
{
  sub1 = sub2 = NULL;
  *this = rhs;
}

OmegaExpr::~OmegaExpr() { Cleanup(); }

const OmegaExpr &OmegaExpr::operator = (const OmegaExpr &rhs)
{
  Cleanup();
  type = rhs.type;
  sub1 = (rhs.sub1 == NULL) ? NULL : rhs.sub1->Clone();
  sub2 = (rhs.sub2 == NULL) ? NULL : rhs.sub2->Clone();
  markers = rhs.markers;
  return *this;
}

void OmegaExpr::Cleanup()
{
  if(sub1 != NULL) { delete sub1; sub1 = NULL; }
  if(sub2 != NULL) { delete sub2; sub2 = NULL; }
}

OmegaExpr *OmegaExpr::Clone() const
{
  return new OmegaExpr(type,sub1,sub2,markers);
}

//compute the set of markers appearing in an expression. if the dummy
//variables appearing in the expression is {$1, $2, $3} then the set
//of markers is {1, 2, 3}.
void OmegaExpr::ComputeMarkers(set<int> &res) const
{
  res.insert(markers.begin(),markers.end());
  if(sub1 != NULL) sub1->ComputeMarkers(res); 
  if(sub2 != NULL) sub2->ComputeMarkers(res); 
}

//construct an LTL formula equivalent to this operator
BasicLtl *OmegaExpr::ToLtl(const vector<BasicSeaw*> &args) const
{
  BasicLtl *res = NULL;
  if((type == OMEGA_X) || (type == OMEGA_G) || (type == OMEGA_F) || (type == OMEGA_NOT)) {
    BasicLtl *ltl = sub1->ToLtl(args);
    if(type == OMEGA_X) res = new BasicLtl(BasicLtl::LTL_X,ltl);
    if(type == OMEGA_G) res = new BasicLtl(BasicLtl::LTL_G,ltl);
    if(type == OMEGA_F) res = new BasicLtl(BasicLtl::LTL_F,ltl);
    if(type == OMEGA_NOT) res = new BasicLtl(BasicLtl::LTL_NOT,ltl);
    delete ltl;
  } else if((type == OMEGA_U) || (type == OMEGA_R) || (type == OMEGA_OR) || (type == OMEGA_AND)) {
    BasicLtl *ltl1 = sub1->ToLtl(args);
    BasicLtl *ltl2 = sub2->ToLtl(args);
    if(type == OMEGA_U) res = new BasicLtl(BasicLtl::LTL_U,ltl1,ltl2);
    if(type == OMEGA_R) res = new BasicLtl(BasicLtl::LTL_R,ltl1,ltl2);
    if(type == OMEGA_OR) res = new BasicLtl(BasicLtl::LTL_OR,ltl1,ltl2);
    if(type == OMEGA_AND) res = new BasicLtl(BasicLtl::LTL_AND,ltl1,ltl2);
    delete ltl1; delete ltl2;
  } else if(type == OMEGA_DUMMY) { 
    int marker = *(markers.begin());
    const BasicSeaw *seaw = args[marker - 1];
    if(seaw->GetType() == BasicSeaw::SEAW_ACTS) {
      res = new BasicLtl(*(seaw->GetActs().begin()));
    } else {
      assert(seaw->GetType() != BasicSeaw::SEAW_NONE);
      char buf[64]; snprintf(buf,64,"prop_%d",marker);
      res = new BasicLtl(ExprManager::GetIdExpr(buf));
    }
  } else assert(false);
  return res;
}

/*********************************************************************/
//methods of OmegaOp class
/*********************************************************************/
OmegaOp::OmegaOp(const OmegaExpr *o)
{
  omega = (o == NULL) ? NULL : o->Clone();
}

OmegaOp::OmegaOp(const OmegaOp &rhs) { omega = NULL; *this = rhs; }

const OmegaOp &OmegaOp::operator = (const OmegaOp &rhs)
{
  Cleanup();
  omega = (rhs.omega == NULL) ? NULL : rhs.omega->Clone();
  return *this;
}

void OmegaOp::Cleanup()
{
  if(omega != NULL) { delete omega; omega = NULL; }
}

void OmegaOp::ComputeMarkers(set<int> &res) const
{
  assert(omega != NULL);
  omega->ComputeMarkers(res);
}

/*********************************************************************/
//static fields of BasicSeaw class
/*********************************************************************/
const int BasicSeaw::SEAW_NONE = 11000;
const int BasicSeaw::SEAW_PROP = 11010;
const int BasicSeaw::SEAW_ACTS = 11020;
const int BasicSeaw::SEAW_OMEGA = 11300;
const int BasicSeaw::SEAW_AND = 11040;
const int BasicSeaw::SEAW_OR = 11050;
const int BasicSeaw::SEAW_NOT = 11060;

/*********************************************************************/
//methods of BasicSeaw class
/*********************************************************************/
BasicSeaw::BasicSeaw(const int t,const Expr &p,const set<Action> &a,const string &o,const vector<BasicSeaw*> &s)
  : type(t),prop(p),acts(a),omega(o)
{
  subs.resize(s.size());
  for(size_t i = 0;i < s.size();++i) subs[i] = s[i]->Clone();
}

//construct atomic propositional formula
BasicSeaw::BasicSeaw(const Expr &p) : type(SEAW_PROP),prop(p) {}

//construct atomic action-set formula
BasicSeaw::BasicSeaw(const set<Action> &a) : type(SEAW_ACTS),acts(a) {}

//construct unary (i.e. negation) formula
BasicSeaw::BasicSeaw(const BasicSeaw *s)
  : type(SEAW_NOT)
{
  subs.resize(1);
  subs[0] = s->Clone();
}

//construct omega operator formula
BasicSeaw::BasicSeaw(const string &o,const list<BasicSeaw*> &s)
  : type(SEAW_OMEGA),omega(o)
{
  subs.resize(s.size());
  size_t count = 0;
  for(list<BasicSeaw*>::const_iterator i = s.begin();i != s.end();++i,++count) {
    subs[count] = (*i)->Clone();
  }
}

//construct a binary formula
BasicSeaw::BasicSeaw(const int t,const BasicSeaw *s1,const BasicSeaw *s2)
  : type(t)
{
  assert((t == SEAW_AND) || (t == SEAW_OR));
  subs.resize(2);
  subs[0] = s1->Clone();
  subs[1] = s2->Clone();
}

//copy constructor
BasicSeaw::BasicSeaw(const BasicSeaw &rhs) { *this = rhs; }

//assignment operator
const BasicSeaw &BasicSeaw::operator = (const BasicSeaw &rhs)
{
  Cleanup();
  type = rhs.type;
  prop = rhs.prop;
  acts = rhs.acts;
  omega = rhs.omega;
  subs.resize(rhs.subs.size());
  for(size_t i = 0;i < rhs.subs.size();++i) subs[i] = rhs.subs[i]->Clone();
  return *this;
}

void BasicSeaw::Cleanup()
{
  for(vector<BasicSeaw*>::const_iterator i = subs.begin();i != subs.end();++i) delete *i;
  subs.clear();
}

BasicSeaw *BasicSeaw::Clone() const
{
  return new BasicSeaw(type,prop,acts,omega,subs);
}

string BasicSeaw::ToString() const
{
  string res;
  if(type == SEAW_PROP) res = "[ " + prop.ToString() + "]";
  else if(type == SEAW_ACTS) {
    res = "{";
    for(set<Action>::const_iterator i = acts.begin();i != acts.end();) {
      res += i->ToString(); ++i;
      if(i == acts.end()) res += "}";
      else res += ",";
    }
  } else if(type == SEAW_OMEGA) {
    res = "<A>" + omega + "(";
    for(vector<BasicSeaw*>::const_iterator i = subs.begin();i != subs.end();) {
      res += (*i)->ToString(); ++i;
      if(i == subs.end()) res += ")";
      else res += ",";
    }
  } else if(type == SEAW_AND) {
    res = subs[0]->ToString() + " & " + subs[1]->ToString();
  } else if(type == SEAW_OR) {
    res = subs[0]->ToString() + " | " + subs[1]->ToString();
  } else if(type == SEAW_NOT) {
    res = "! " + subs[0]->ToString();
  } else assert(false);
  return res;
}

void BasicSeaw::ComputeOpsToArgs(list< pair<string,size_t> > &res) const
{
  if(type == SEAW_OMEGA) res.push_back(pair<string,size_t>(omega,subs.size()));
  for(size_t i = 0;i < subs.size();++i) subs[i]->ComputeOpsToArgs(res);
}

void BasicSeaw::ComputeProps(set<Expr> &res) const
{
  if(type == SEAW_PROP) res.insert(prop);
  for(size_t i = 0;i < subs.size();++i) subs[i]->ComputeProps(res);  
}

void BasicSeaw::ComputeActions(set<Action> &res) const
{
  res.insert(acts.begin(),acts.end());
  for(size_t i = 0;i < subs.size();++i) subs[i]->ComputeActions(res);  
}

/*********************************************************************/
//methods of SeawFormula class
/*********************************************************************/
SeawFormula::SeawFormula(const BasicSeaw *s)
{
  seaw = (s == NULL) ? NULL : s->Clone();
}

SeawFormula::SeawFormula(const SeawFormula &rhs)
{
  seaw = NULL;
  *this = rhs;
}

const SeawFormula &SeawFormula::operator = (const SeawFormula &rhs)
{
  Cleanup();
  seaw = (rhs.seaw == NULL) ? NULL : rhs.seaw->Clone();
  return *this;
}

void SeawFormula::Cleanup()
{
  if(seaw != NULL) { delete seaw; seaw = NULL; }
}

void SeawFormula::ComputeOpsToArgs(list< pair<string,size_t> > &res) const
{
  assert(seaw != NULL);
  seaw->ComputeOpsToArgs(res);
}

void SeawFormula::ComputeProps(set<Expr> &res) const
{
  seaw->ComputeProps(res);
}

void SeawFormula::ComputeActions(set<Action> &res) const
{
  seaw->ComputeActions(res);
}

/*********************************************************************/
//end of Seaw.cpp
/*********************************************************************/
