/**************************** CPPHeaderFile ***************************

* FileName [LtlManager.h]

* PackageName [parser]

* Synopsis [Header file for LtlManager class.]

* Description [This class encapsulates a manager for LTL formulas. One
* of the things it tries to do is maintain canonical representations
* of LTL formulas encountered during the entire process.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __LTLMANAGER_H__
#define __LTLMANAGER_H__

namespace magic {

//other classes needed
class LtlFormula;
class BasicLtl;

class LtlManager
{
 private:
  //the map from names to formulas
  static map<string,const BasicLtl*> nameToLtl;

  //the cache of all formulas
  static set<const BasicLtl*> allLtls;

  //the cache of all propositional formulas
  static map<Expr,const BasicLtl*> propLtls;

  //the cache of all action formulas
  static map<Action,const BasicLtl*> actLtls;

  //the cache of all unary formulas
  static map< pair<const BasicLtl*,int>,const BasicLtl* > unaryLtls;

  //the cache of all binary formulas
  static map< pair< pair<const BasicLtl*,const BasicLtl*>,int >,const BasicLtl* > binaryLtls;

  //map from formulas to their propositiona
  static map< const BasicLtl*,set<Expr> > propCache;

  //map from formulas to their actions
  static map< const BasicLtl*,set<Action> > actCache;

 public:
  //register a basic ltl formula
  static const BasicLtl *Register(const BasicLtl *ltl);

  //get all kinds of formulas
  static LtlFormula GetPropLtl(const Expr &p);
  static LtlFormula GetActLtl(const Action &a);
  static LtlFormula GetUnaryLtl(const LtlFormula &l,int op);
  static LtlFormula GetBinaryLtl(const LtlFormula &l1,const LtlFormula &l2,int op);

  static void AddFormula(const string &n,const LtlFormula &f);
  static LtlFormula GetFormula(const string &n);
  static const set<Expr> &GetProps(const LtlFormula &arg);
  static const set<Action> &GetActions(const LtlFormula &arg);
  static LtlFormula NegateLtl(const LtlFormula &arg);
  static LtlFormula NormalizeLtl(const LtlFormula &arg);
  static LtlFormula RewriteLtl(const LtlFormula &arg);
  static LtlFormula SimplifyLtl(const LtlFormula &arg);
  static LtlFormula AddActionConstraints(const LtlFormula &arg);
  static string GetWringString(const LtlFormula &arg,map<Expr,string> &propToVar,
			       map<string,Expr> &varToProp,map<Action,string> &actToVar,
			       map<string,Action> &varToAct);

  friend class LtlToBuchi;
};

} //namespace magic

#endif //__LTLMANAGER_H__

/*********************************************************************/
//end of LtlManager.h
/*********************************************************************/
