/******************************** CPPFile *****************************

* FileName [LtlManager.cpp]

* PackageName [parser]

* Synopsis [Method definitions of LtlManager class.]

* SeeAlso [LtlManager.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <string>
#include <list>
#include <set>
#include <map>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "LtlFormula.h"
#include "LtlManager.h"
using namespace magic;

/*********************************************************************/
//define static members
/*********************************************************************/
map<string,const BasicLtl*> LtlManager::nameToLtl;
set<const BasicLtl*> LtlManager::allLtls;
map<Expr,const BasicLtl*> LtlManager::propLtls;
map<Action,const BasicLtl*> LtlManager::actLtls;
map< pair<const BasicLtl*,int>,const BasicLtl* > LtlManager::unaryLtls;
map< pair< pair<const BasicLtl*,const BasicLtl*>,int >,const BasicLtl* > LtlManager::binaryLtls;
map< const BasicLtl*,set<Expr> > LtlManager::propCache;
map< const BasicLtl*,set<Action> > LtlManager::actCache;

/*********************************************************************/
//register a basic ltl formula. registering involves recursively
//registering all subformulas.
/*********************************************************************/
const BasicLtl *LtlManager::Register(const BasicLtl *ltl)
{
  //first check if the expression has already been registered before
  if(allLtls.count(ltl) != 0) return ltl;
  
  //propositional formula
  if(ltl->type == BasicLtl::LTL_PROP) {
    return GetPropLtl(ltl->prop).ltl;
  }
  //action formula
  else if(ltl->type == BasicLtl::LTL_ACT) {
    return GetActLtl(ltl->action).ltl;
  }
  //unary formula
  else if((ltl->type == BasicLtl::LTL_NOT) || (ltl->type == BasicLtl::LTL_X) ||
	  (ltl->type == BasicLtl::LTL_G) || (ltl->type == BasicLtl::LTL_F)) {
    return GetUnaryLtl(ltl->left,ltl->type).ltl;
  }
  //binary formula
  else if((ltl->type == BasicLtl::LTL_AND) || (ltl->type == BasicLtl::LTL_OR) ||
	  (ltl->type == BasicLtl::LTL_U) || (ltl->type == BasicLtl::LTL_R)) {
    return GetBinaryLtl(ltl->left,ltl->right,ltl->type).ltl;
  }
  //illegal formula
  else assert(false);
}

/*********************************************************************/
//returns the propositional ltl formula corresponding to the
//proposition supplied as argument. create one if necessary.
/*********************************************************************/
LtlFormula LtlManager::GetPropLtl(const Expr &p)
{
  LtlFormula res;
  map<Expr,const BasicLtl*>::const_iterator i = propLtls.find(p);
  if(i == propLtls.end()) {
    BasicLtl *x = new BasicLtl(p);
    allLtls.insert(x);
    propLtls[p] = x;
    res.ltl = x;
  } else {
    res.ltl = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the action ltl formula corresponding to the action supplied
//as argument. create one if necessary.
/*********************************************************************/
LtlFormula LtlManager::GetActLtl(const Action &a)
{
  LtlFormula res;
  map<Action,const BasicLtl*>::const_iterator i = actLtls.find(a);
  if(i == actLtls.end()) {
    BasicLtl *x = new BasicLtl(a);
    allLtls.insert(x);
    actLtls[a] = x;
    res.ltl = x;
  } else {
    res.ltl = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the unary ltl formula corresponding to the operand and
//operator supplied as arguments. create one if necessary.
/*********************************************************************/
LtlFormula LtlManager::GetUnaryLtl(const LtlFormula &l,int op)
{
  LtlFormula res;
  pair<const BasicLtl*,int> x(l.ltl,op);
  map< pair<const BasicLtl*,int>,const BasicLtl* >::const_iterator i = unaryLtls.find(x);
  if(i == unaryLtls.end()) {
    const BasicLtl *y = NULL;
    //!!A <=> A
    if((op == BasicLtl::LTL_NOT) && (l.ltl->type == BasicLtl::LTL_NOT)) {      
      y = l.ltl->left;
    }
    //!true = false
    else if((op == BasicLtl::LTL_NOT) && (l.ltl->type == BasicLtl::LTL_PROP) &&
	    ExprManager::IsTrue(l.ltl->prop)) {
      y = GetPropLtl(ExprManager::GetIntConstExpr(0)).ltl;
    }
    //!false = true
    else if((op == BasicLtl::LTL_NOT) && (l.ltl->type == BasicLtl::LTL_PROP) &&
	    ExprManager::IsFalse(l.ltl->prop)) {
      y = GetPropLtl(ExprManager::GetIntConstExpr(1)).ltl;
    }
    //all checks done
    if(y == NULL) {
      y = new BasicLtl();
      ((BasicLtl*)(y))->type = x.second;
      ((BasicLtl*)(y))->left = x.first;
      allLtls.insert(y);
    } else assert(allLtls.count(y) != 0);
    unaryLtls[x] = y;
    res.ltl = y;
  } else {
    res.ltl = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the binary ltl formula corresponding to the operand and
//operators supplied as arguments. create one if necessary.
/*********************************************************************/
LtlFormula LtlManager::GetBinaryLtl(const LtlFormula &l1,const LtlFormula &l2,int op)
{
  LtlFormula res;
  pair<const BasicLtl*,const BasicLtl*> x(l1.ltl,l2.ltl);
  pair< pair<const BasicLtl*,const BasicLtl*>,int > y(x,op);
  map< pair< pair<const BasicLtl*,const BasicLtl*>,int >,const BasicLtl* >::const_iterator i = binaryLtls.find(y);
  if(i == binaryLtls.end()) {
    bool found = false;
    if((op == BasicLtl::LTL_AND) || (op == BasicLtl::LTL_OR)) {
      x = pair<const BasicLtl*,const BasicLtl*>(l2.ltl,l1.ltl);
      y = pair< pair<const BasicLtl*,const BasicLtl*>,int >(x,op);
      i = binaryLtls.find(y);
      found = (i != binaryLtls.end());
      if(found) res.ltl = i->second;
    }
    if(!found) {
      BasicLtl *z = new BasicLtl();
      z->type = y.second;
      z->left = x.first;
      z->right = x.second;
      allLtls.insert(z);
      binaryLtls[y] = z;
      res.ltl = z;
    }
  } else {
    res.ltl = i->second;
  }
  return res;
}

/*********************************************************************/
//add a new formula
/*********************************************************************/
void LtlManager::AddFormula(const string &n,const LtlFormula &f)
{
  if(nameToLtl.count(n) != 0) {
    Util::Error("ERROR: LTL formula with name " + n + " multiply defined ...\n");
  }
  nameToLtl[n] = f.ltl;
}

/*********************************************************************/
//return formula with given name
/*********************************************************************/
LtlFormula LtlManager::GetFormula(const string &n)
{
  map<string,const BasicLtl*>::const_iterator i = nameToLtl.find(n);
  if(i == nameToLtl.end()) {
    Util::Error("ERROR: undefined formula " + n + " ...\n");
  }
  return LtlFormula(i->second);
}

/*********************************************************************/
//return the set of propositionscontained in the ltl formula
/*********************************************************************/
const set<Expr> &LtlManager::GetProps(const LtlFormula &arg)
{
  const BasicLtl *ltl = arg.ltl;
  if(propCache.count(ltl) == 0) {
    if(ltl->type == BasicLtl::LTL_PROP) { propCache[ltl].insert(ltl->prop); }
    else if(ltl->type == BasicLtl::LTL_ACT) {}
    else if((ltl->type == BasicLtl::LTL_NOT) || (ltl->type == BasicLtl::LTL_X) || 
	    (ltl->type == BasicLtl::LTL_G) || (ltl->type == BasicLtl::LTL_F)) {
      const set<Expr> &x = GetProps(ltl->left);
      propCache[ltl].insert(x.begin(),x.end());
    } else if((ltl->type == BasicLtl::LTL_AND) || (ltl->type == BasicLtl::LTL_OR) || 
	      (ltl->type == BasicLtl::LTL_U) || (ltl->type == BasicLtl::LTL_R)) {
      const set<Expr> &x = GetProps(ltl->left);
      propCache[ltl].insert(x.begin(),x.end());
      const set<Expr> &y = GetProps(ltl->right);
      propCache[ltl].insert(y.begin(),y.end());
    } else assert(false);
  }
  return propCache[ltl];
}

/*********************************************************************/
//return the set of actions contained in the ltl formula
/*********************************************************************/
const set<Action> &LtlManager::GetActions(const LtlFormula &arg)
{
  const BasicLtl *ltl = arg.ltl;
  if(actCache.count(ltl) == 0) {
    if(ltl->type == BasicLtl::LTL_PROP) {}
    else if(ltl->type == BasicLtl::LTL_ACT) actCache[ltl].insert(ltl->action);
    else if((ltl->type == BasicLtl::LTL_NOT) || (ltl->type == BasicLtl::LTL_X) || 
	    (ltl->type == BasicLtl::LTL_G) || (ltl->type == BasicLtl::LTL_F)) {
      const set<Action> &x = GetActions(ltl->left);
      actCache[ltl].insert(x.begin(),x.end());
    } else if((ltl->type == BasicLtl::LTL_AND) || (ltl->type == BasicLtl::LTL_OR) || 
	      (ltl->type == BasicLtl::LTL_U) || (ltl->type == BasicLtl::LTL_R)) {
      const set<Action> &x = GetActions(ltl->left);
      actCache[ltl].insert(x.begin(),x.end());
      const set<Action> &y = GetActions(ltl->right);
      actCache[ltl].insert(y.begin(),y.end());
    } else assert(false);
  }
  return actCache[ltl];
}

/*********************************************************************/
//return the negation of the argument formula
/*********************************************************************/
LtlFormula LtlManager::NegateLtl(const LtlFormula &arg)
{
  return GetUnaryLtl(arg,BasicLtl::LTL_NOT);
}

/*********************************************************************/
//return the negation normal form of the argument formula
/*********************************************************************/
LtlFormula LtlManager::NormalizeLtl(const LtlFormula &arg)
{
  //proposition
  if(arg.ltl->type == BasicLtl::LTL_PROP) return arg;
  //action
  else if(arg.ltl->type == BasicLtl::LTL_ACT) return arg;
  //negation
  else if(arg.ltl->type == BasicLtl::LTL_NOT) {
    const BasicLtl *sub = arg.ltl->left;
    if(sub->type == BasicLtl::LTL_PROP) return arg;
    else if(sub->type == BasicLtl::LTL_ACT) return arg;
    else if(sub->type == BasicLtl::LTL_NOT) return NormalizeLtl(sub->left);
    else if(sub->type == BasicLtl::LTL_X) {
      LtlFormula a = GetUnaryLtl(sub->left,BasicLtl::LTL_NOT);
      LtlFormula b = NormalizeLtl(a);
      return GetUnaryLtl(b,BasicLtl::LTL_X);
    } else if(sub->type == BasicLtl::LTL_G) {
      LtlFormula a = GetUnaryLtl(sub->left,BasicLtl::LTL_NOT);
      LtlFormula b = NormalizeLtl(a);
      return GetUnaryLtl(b,BasicLtl::LTL_F);
    } else if(sub->type == BasicLtl::LTL_F) {
      LtlFormula a = GetUnaryLtl(sub->left,BasicLtl::LTL_NOT);
      LtlFormula b = NormalizeLtl(a);
      return GetUnaryLtl(b,BasicLtl::LTL_G);
    } else if(sub->type == BasicLtl::LTL_AND) {
      LtlFormula a = GetUnaryLtl(sub->left,BasicLtl::LTL_NOT);
      LtlFormula b = NormalizeLtl(a);
      LtlFormula c = GetUnaryLtl(sub->right,BasicLtl::LTL_NOT);
      LtlFormula d = NormalizeLtl(c);
      return GetBinaryLtl(b,d,BasicLtl::LTL_OR);
    } else if(sub->type == BasicLtl::LTL_OR) {
      LtlFormula a = GetUnaryLtl(sub->left,BasicLtl::LTL_NOT);
      LtlFormula b = NormalizeLtl(a);
      LtlFormula c = GetUnaryLtl(sub->right,BasicLtl::LTL_NOT);
      LtlFormula d = NormalizeLtl(c);
      return GetBinaryLtl(b,d,BasicLtl::LTL_AND);
    } else if(sub->type == BasicLtl::LTL_U) {
      LtlFormula a = GetUnaryLtl(sub->left,BasicLtl::LTL_NOT);
      LtlFormula b = NormalizeLtl(a);
      LtlFormula c = GetUnaryLtl(sub->right,BasicLtl::LTL_NOT);
      LtlFormula d = NormalizeLtl(c);
      return GetBinaryLtl(b,d,BasicLtl::LTL_R);
    } else if(sub->type == BasicLtl::LTL_R) {
      LtlFormula a = GetUnaryLtl(sub->left,BasicLtl::LTL_NOT);
      LtlFormula b = NormalizeLtl(a);
      LtlFormula c = GetUnaryLtl(sub->right,BasicLtl::LTL_NOT);
      LtlFormula d = NormalizeLtl(c);
      return GetBinaryLtl(b,d,BasicLtl::LTL_U);
    } else assert(false);
  }
  //next-time
  else if(arg.ltl->type == BasicLtl::LTL_X) {
    return GetUnaryLtl(NormalizeLtl(arg.ltl->left),BasicLtl::LTL_X);
  }
  //globally
  else if(arg.ltl->type == BasicLtl::LTL_G) {
    return GetUnaryLtl(NormalizeLtl(arg.ltl->left),BasicLtl::LTL_G);
  }
  //future
  else if(arg.ltl->type == BasicLtl::LTL_F) {
    return GetUnaryLtl(NormalizeLtl(arg.ltl->left),BasicLtl::LTL_F);
  }
  //conjunction
  else if(arg.ltl->type == BasicLtl::LTL_AND) {
    LtlFormula a = NormalizeLtl(arg.ltl->left);
    LtlFormula b = NormalizeLtl(arg.ltl->right);
    return GetBinaryLtl(a,b,BasicLtl::LTL_AND);
  }
  //disjunction
  else if(arg.ltl->type == BasicLtl::LTL_OR) {
    LtlFormula a = NormalizeLtl(arg.ltl->left);
    LtlFormula b = NormalizeLtl(arg.ltl->right);
    return GetBinaryLtl(a,b,BasicLtl::LTL_OR);
  }
  //until
  else if(arg.ltl->type == BasicLtl::LTL_U) {
    LtlFormula a = NormalizeLtl(arg.ltl->left);
    LtlFormula b = NormalizeLtl(arg.ltl->right);
    return GetBinaryLtl(a,b,BasicLtl::LTL_U);
  }
  //release
  else if(arg.ltl->type == BasicLtl::LTL_R) {
    LtlFormula a = NormalizeLtl(arg.ltl->left);
    LtlFormula b = NormalizeLtl(arg.ltl->right);
    return GetBinaryLtl(a,b,BasicLtl::LTL_R);
  }
  //illegal
  else assert(false);
}

/*********************************************************************/
//rewrite the argument formula according to specific rules
/*********************************************************************/
LtlFormula LtlManager::RewriteLtl(const LtlFormula &arg)
{
  //top-level proposition
  if(arg.ltl->type == BasicLtl::LTL_PROP) return arg;
  //top-level action
  else if(arg.ltl->type == BasicLtl::LTL_ACT) return arg;
  //top-level negation
  else if(arg.ltl->type == BasicLtl::LTL_NOT) {
    return GetUnaryLtl(RewriteLtl(arg.ltl->left),BasicLtl::LTL_NOT);
  }
  //top-level next-time
  else if(arg.ltl->type == BasicLtl::LTL_X) {
    LtlFormula left = RewriteLtl(arg.ltl->left);
    //X(true) <=> (true)
    if((left.ltl->type == BasicLtl::LTL_PROP) && ExprManager::IsTrue(left.ltl->prop)) {
      return GetPropLtl(ExprManager::GetIntConstExpr(1));
    }
    //X G F A <=> G F A
    else if((left.ltl->type == BasicLtl::LTL_G) && (left.ltl->left->type == BasicLtl::LTL_F)) {
      return GetUnaryLtl(GetUnaryLtl(left.ltl->left->left,BasicLtl::LTL_F),BasicLtl::LTL_G);
    }
    //X(A & G F B) <=> (X A) & (G F B)
    else if((left.ltl->type == BasicLtl::LTL_AND) && (left.ltl->right->type == BasicLtl::LTL_G) &&
	    (left.ltl->right->left->type == BasicLtl::LTL_F)) {
      LtlFormula a = RewriteLtl(GetUnaryLtl(left.ltl->left,BasicLtl::LTL_X));
      return GetBinaryLtl(a,left.ltl->right,BasicLtl::LTL_AND);
    }
    //X(A | G F B) <=> (X A) | (G F B)
    else if((left.ltl->type == BasicLtl::LTL_OR) && (left.ltl->right->type == BasicLtl::LTL_G) &&
	    (left.ltl->right->left->type == BasicLtl::LTL_F)) {
      LtlFormula a = RewriteLtl(GetUnaryLtl(left.ltl->left,BasicLtl::LTL_X));
      return GetBinaryLtl(a,left.ltl->right,BasicLtl::LTL_OR);
    }
    //no rewriting possible
    else return GetUnaryLtl(left,BasicLtl::LTL_X);
  }
  //top-level globally
  else if(arg.ltl->type == BasicLtl::LTL_G) {
    LtlFormula left = RewriteLtl(arg.ltl->left);
    //G G F A <=> G F A
    if((left.ltl->type == BasicLtl::LTL_G) && (left.ltl->left->type == BasicLtl::LTL_F)) {
      return LtlFormula(left.ltl->left);
    }
    //G(A | G F B) <=> (G A) | (G F B)
    else if((left.ltl->type == BasicLtl::LTL_OR) && (left.ltl->right->type == BasicLtl::LTL_G) &&
	    (left.ltl->right->left->type == BasicLtl::LTL_F)) {
      LtlFormula a = RewriteLtl(GetUnaryLtl(left.ltl->left,BasicLtl::LTL_G));
      return GetBinaryLtl(a,left.ltl->right,BasicLtl::LTL_OR);
    }
    //no rewriting possible
    else return GetUnaryLtl(left,BasicLtl::LTL_G);
  }
  //top-level future
  else if(arg.ltl->type == BasicLtl::LTL_F) {
    LtlFormula left = RewriteLtl(arg.ltl->left);
    //F X A <=> X F A
    if(left.ltl->type == BasicLtl::LTL_X) {
      LtlFormula a = RewriteLtl(GetUnaryLtl(left.ltl->left,BasicLtl::LTL_F));
      return RewriteLtl(GetUnaryLtl(a,BasicLtl::LTL_X));
    }
    //F G F A <=> G F A
    else if((left.ltl->type == BasicLtl::LTL_G) && (left.ltl->left->type == BasicLtl::LTL_F)) {
      return LtlFormula(left.ltl->left);
    }
    //F(A & G F B) <=> (F A) & (G F B)
    else if((left.ltl->type == BasicLtl::LTL_AND) && (left.ltl->right->type == BasicLtl::LTL_G) &&
	    (left.ltl->right->left->type == BasicLtl::LTL_F)) {
      LtlFormula a = RewriteLtl(GetUnaryLtl(left.ltl->left,BasicLtl::LTL_F));
      return GetBinaryLtl(a,left.ltl->right,BasicLtl::LTL_AND);
    }
    //no rewriting possible
    else return GetUnaryLtl(left,BasicLtl::LTL_F);
  }  
  //top-lebel and
  else if(arg.ltl->type == BasicLtl::LTL_AND) {
    LtlFormula left = RewriteLtl(arg.ltl->left);
    LtlFormula right = RewriteLtl(arg.ltl->right);
    //(A R B) & (A R C) <=> A R (B & C)
    if((left.ltl->type == BasicLtl::LTL_R) && (right.ltl->type == BasicLtl::LTL_R) && 
       (LtlFormula(left.ltl->left) == LtlFormula(right.ltl->left))) {
      LtlFormula a = RewriteLtl(GetBinaryLtl(left.ltl->right,right.ltl->right,BasicLtl::LTL_AND));
      return RewriteLtl(GetBinaryLtl(left.ltl->left,a,BasicLtl::LTL_R));
    }       
    //(A U B) & (C U B) <=> (A & C) U B
    else if((left.ltl->type == BasicLtl::LTL_U) && (right.ltl->type == BasicLtl::LTL_U) && 
       (LtlFormula(left.ltl->right) == LtlFormula(right.ltl->right))) {
      LtlFormula a = RewriteLtl(GetBinaryLtl(left.ltl->left,right.ltl->left,BasicLtl::LTL_AND));
      return RewriteLtl(GetBinaryLtl(a,left.ltl->right,BasicLtl::LTL_U));
    }
    //no rewriting possible
    else return GetBinaryLtl(left,right,BasicLtl::LTL_AND);
  }
  //top-lebel or
  else if(arg.ltl->type == BasicLtl::LTL_OR) {
    LtlFormula left = RewriteLtl(arg.ltl->left);
    LtlFormula right = RewriteLtl(arg.ltl->right);
    //(A R B) | (C R B) <=> (A | C) R B
    if((left.ltl->type == BasicLtl::LTL_R) && (right.ltl->type == BasicLtl::LTL_R) && 
       (LtlFormula(left.ltl->right) == LtlFormula(right.ltl->right))) {
      LtlFormula a = RewriteLtl(GetBinaryLtl(left.ltl->left,right.ltl->left,BasicLtl::LTL_OR));
      return RewriteLtl(GetBinaryLtl(a,left.ltl->right,BasicLtl::LTL_R));
    }
    //(A U B) | (A U C) <=> A U (B | C)
    else if((left.ltl->type == BasicLtl::LTL_U) && (right.ltl->type == BasicLtl::LTL_U) && 
       (LtlFormula(left.ltl->left) == LtlFormula(right.ltl->left))) {
      LtlFormula a = RewriteLtl(GetBinaryLtl(left.ltl->right,right.ltl->right,BasicLtl::LTL_OR));
      return RewriteLtl(GetBinaryLtl(left.ltl->left,a,BasicLtl::LTL_U));
    }
    //(G F A) | (G F B) <=> G F (A | B)
    else if((left.ltl->type == BasicLtl::LTL_G) && (left.ltl->left->type == BasicLtl::LTL_F) &&
	    (right.ltl->type == BasicLtl::LTL_G) && (right.ltl->left->type == BasicLtl::LTL_F)) {
      LtlFormula a = RewriteLtl(GetBinaryLtl(left.ltl->left->left,right.ltl->left->left,BasicLtl::LTL_OR));
      LtlFormula b = RewriteLtl(GetUnaryLtl(a,BasicLtl::LTL_F));
      return RewriteLtl(GetUnaryLtl(b,BasicLtl::LTL_G));
    }
    //no rewriting possible
    else return GetBinaryLtl(left,right,BasicLtl::LTL_OR);
  }
  //top-lebel until
  else if(arg.ltl->type == BasicLtl::LTL_U) {
    LtlFormula left = RewriteLtl(arg.ltl->left);
    LtlFormula right = RewriteLtl(arg.ltl->right);
    //(X A) U (X B) <=> X(A U B)
    if((left.ltl->type == BasicLtl::LTL_X) && (right.ltl->type == BasicLtl::LTL_X)) {
      LtlFormula a = RewriteLtl(GetBinaryLtl(left.ltl->left,right.ltl->left,BasicLtl::LTL_U));
      return RewriteLtl(GetUnaryLtl(a,BasicLtl::LTL_X));
    }
    //A U false <=> false
    else if((right.ltl->type == BasicLtl::LTL_PROP) && ExprManager::IsFalse(right.ltl->prop)) {
      return GetPropLtl(ExprManager::GetIntConstExpr(0));
    }
    //no rewriting possible
    else return GetBinaryLtl(left,right,BasicLtl::LTL_U);
  }
  //top-lebel release
  else if(arg.ltl->type == BasicLtl::LTL_R) {
    LtlFormula left = RewriteLtl(arg.ltl->left);
    LtlFormula right = RewriteLtl(arg.ltl->right);
    return GetBinaryLtl(left,right,BasicLtl::LTL_R);
  }
  //no rewriting possible
  else return arg;
}

/*********************************************************************/
//simplify the argument formula by replacing G and F formulae with
//equivalent R and U versions
/*********************************************************************/
LtlFormula LtlManager::SimplifyLtl(const LtlFormula &arg)
{
  if((arg.ltl->type == BasicLtl::LTL_PROP) || (arg.ltl->type == BasicLtl::LTL_ACT)) {
    return arg;
  } else if(arg.ltl->type == BasicLtl::LTL_NOT) {
    return GetUnaryLtl(SimplifyLtl(arg.ltl->left),BasicLtl::LTL_NOT);
  } else if(arg.ltl->type == BasicLtl::LTL_X) {
    return GetUnaryLtl(SimplifyLtl(arg.ltl->left),BasicLtl::LTL_X);
  } else if(arg.ltl->type == BasicLtl::LTL_G) {
    LtlFormula left = SimplifyLtl(arg.ltl->left);
    return GetBinaryLtl(GetPropLtl(ExprManager::GetIntConstExpr(0)),left,BasicLtl::LTL_R);
  } else if(arg.ltl->type == BasicLtl::LTL_F) {
    LtlFormula left = SimplifyLtl(arg.ltl->left);
    return GetBinaryLtl(GetPropLtl(ExprManager::GetIntConstExpr(1)),left,BasicLtl::LTL_U);
  } else if(arg.ltl->type == BasicLtl::LTL_AND) {
    LtlFormula left = SimplifyLtl(arg.ltl->left);
    LtlFormula right = SimplifyLtl(arg.ltl->right);
    return GetBinaryLtl(left,right,BasicLtl::LTL_AND);
  } else if(arg.ltl->type == BasicLtl::LTL_OR) {
    LtlFormula left = SimplifyLtl(arg.ltl->left);
    LtlFormula right = SimplifyLtl(arg.ltl->right);
    return GetBinaryLtl(left,right,BasicLtl::LTL_OR);
  } else if(arg.ltl->type == BasicLtl::LTL_U) {
    LtlFormula left = SimplifyLtl(arg.ltl->left);
    LtlFormula right = SimplifyLtl(arg.ltl->right);
    return GetBinaryLtl(left,right,BasicLtl::LTL_U);
  } else if(arg.ltl->type == BasicLtl::LTL_R) {
    LtlFormula left = SimplifyLtl(arg.ltl->left);
    LtlFormula right = SimplifyLtl(arg.ltl->right);
    return GetBinaryLtl(left,right,BasicLtl::LTL_R);
  } assert(false);
}

/*********************************************************************/
//create the string representation of the argument LTL formula that
//can be accepted by Wring. also return the mappings from propositions
//and actions to variables and vice-versa in the other arguments.
/*********************************************************************/
string LtlManager::GetWringString(const LtlFormula &arg,map<Expr,string> &propToVar,
				  map<string,Expr> &varToProp,map<Action,string> &actToVar,
				  map<string,Action> &varToAct)
{
  //proposition
  if(arg.ltl->type == BasicLtl::LTL_PROP) {
    if(ExprManager::IsTrue(arg.ltl->prop)) {
      return "(TRUE)";
    } else if(ExprManager::IsFalse(arg.ltl->prop)) {
      return "(FALSE)";
    } else {
      string var;
      if(propToVar.count(arg.ltl->prop) == 0) {
	var = Util::NewTempVar();
	propToVar[arg.ltl->prop] = var;
	varToProp[var] = arg.ltl->prop;
      } else var = propToVar[arg.ltl->prop];
      return "(" + var + "=1)";
    }
  } 
  //action
  else if(arg.ltl->type == BasicLtl::LTL_ACT) {
    string var;
    if(actToVar.count(arg.ltl->action) == 0) {
      var = Util::NewTempVar();
      actToVar[arg.ltl->action] = var;
      varToAct[var] = arg.ltl->action;
    } else var = actToVar[arg.ltl->action];
    return "(" + var + "=1)";
  }
  //negation
  else if(arg.ltl->type == BasicLtl::LTL_NOT) {
    const BasicLtl *sub = arg.ltl->left;
    //negation of proposition
    if(sub->type == BasicLtl::LTL_PROP) {
      if(ExprManager::IsTrue(sub->prop)) {
	return "(FALSE)";
      } else if(ExprManager::IsFalse(sub->prop)) {
	return "(TRUE)";
      } else {
	string var;
	if(propToVar.count(sub->prop) == 0) {
	  var = Util::NewTempVar();
	  propToVar[sub->prop] = var;
	  varToProp[var] = sub->prop;
	} else var = propToVar[sub->prop];
	return "(" + var + "=0)";
      }
    }
    //negation of action
    else if(sub->type == BasicLtl::LTL_ACT) {
      string var;
      if(actToVar.count(sub->action) == 0) {
	var = Util::NewTempVar();
	actToVar[sub->action] = var;
	varToAct[var] = sub->action;
      } else var = actToVar[sub->action];
      return "(" + var + "=0)";
    }
    //illegal
    else assert(false);
  }
  //next-time
  else if(arg.ltl->type == BasicLtl::LTL_X) {
    string x = GetWringString(arg.ltl->left,propToVar,varToProp,actToVar,varToAct);
    return "(X" + x + ")";
  }
  //globally
  else if(arg.ltl->type == BasicLtl::LTL_G) {
    string x = GetWringString(arg.ltl->left,propToVar,varToProp,actToVar,varToAct);
    return "(G" + x + ")";
  }
  //future
  else if(arg.ltl->type == BasicLtl::LTL_F) {
    string x = GetWringString(arg.ltl->left,propToVar,varToProp,actToVar,varToAct);
    return "(F" + x + ")";
  }
  //conjunction
  else if(arg.ltl->type == BasicLtl::LTL_AND) {
    string x = GetWringString(arg.ltl->left,propToVar,varToProp,actToVar,varToAct);
    string y = GetWringString(arg.ltl->right,propToVar,varToProp,actToVar,varToAct);
    return "(" + x + "*" + y + ")";
  }
  //disjunction
  else if(arg.ltl->type == BasicLtl::LTL_OR) {
    string x = GetWringString(arg.ltl->left,propToVar,varToProp,actToVar,varToAct);
    string y = GetWringString(arg.ltl->right,propToVar,varToProp,actToVar,varToAct);
    return "(" + x + "+" + y + ")";
  }
  //until
  else if(arg.ltl->type == BasicLtl::LTL_U) {
    string x = GetWringString(arg.ltl->left,propToVar,varToProp,actToVar,varToAct);
    string y = GetWringString(arg.ltl->right,propToVar,varToProp,actToVar,varToAct);
    return "(" + x + "U" + y + ")";
  }
  //release
  else if(arg.ltl->type == BasicLtl::LTL_R) {
    string x = GetWringString(arg.ltl->left,propToVar,varToProp,actToVar,varToAct);
    string y = GetWringString(arg.ltl->right,propToVar,varToProp,actToVar,varToAct);
    return "(" + x + "V" + y + ")";
  }
  //illegal
  else assert(false);
}

/*********************************************************************/
//add constraints to the argument ltl formula that globally prevents
//multiple actions to be true at any state
/*********************************************************************/
LtlFormula LtlManager::AddActionConstraints(const LtlFormula &arg)
{
  const set<Action> acts = GetActions(arg);
  LtlFormula res = arg;
  for(set<Action>::const_iterator i = acts.begin();i != acts.end();++i) {
    ++i; set<Action>::const_iterator j = i; --i;
    for(;j != acts.end();++j) {
      LtlFormula a = GetActLtl(*i);
      LtlFormula b = GetActLtl(*j);
      LtlFormula na = GetUnaryLtl(a,BasicLtl::LTL_NOT);
      LtlFormula nb = GetUnaryLtl(b,BasicLtl::LTL_NOT);
      LtlFormula c = GetBinaryLtl(na,nb,BasicLtl::LTL_OR);
      LtlFormula gc = GetUnaryLtl(c,BasicLtl::LTL_G);
      res = GetBinaryLtl(res,gc,BasicLtl::LTL_AND);
    }
  }
  return res;
}

/*********************************************************************/
//end of LtlManager.cpp
/*********************************************************************/
