/**************************** CPPHeaderFile ***************************

* FileName [LtlFormula.h]

* PackageName [parser]

* Synopsis [Header file for LtlFormula class.]

* Description [This class encapsulates an LTL formula.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __LTLFORMULA_H__
#define __LTLFORMULA_H__

namespace magic {

//other classes needed
class BasicLtl;
class Action;

/*********************************************************************/
//an LTL formula simply encapsulates a pointer to a basic LTL
//formula. this pointer is registered to the manager.
/*********************************************************************/
class LtlFormula
{
 private:
  const BasicLtl *ltl;

  void ComputeReturnActions(set<Action> &res) const;
  void ComputeAssignActions(set<Action> &res) const;

 public:
  LtlFormula() { ltl = NULL; }
  LtlFormula(const BasicLtl *l);
  LtlFormula(const LtlFormula &rhs) { *this = rhs; }
  const LtlFormula &operator = (const LtlFormula &rhs);
  bool operator == (const LtlFormula &rhs) const;
  bool operator < (const LtlFormula &rhs) const;  

  void GetAssignActExprs(set< pair<Expr,Expr> > &res) const;
  void GetRetActExprs(set<Expr> &res) const;
  void GetReturnActions(const Expr &expr,const list<string> &params,
			const set<Expr> &context,set<Action> &res) const;
  void GetAssignActions(const Expr &lhs,const Expr &rhs,
			const list<string> &params,
			const set<Expr> &context,set<Action> &res) const; 
  string ToString() const;

  friend class LtlManager;
  friend class Buchi;
};

/*********************************************************************/
//a basic LTL formula is maintainted by the LTL manager
/*********************************************************************/
class BasicLtl
{
 private:
  int type;
  Expr prop;
  Action action;
  const BasicLtl *left,*right;

 public:
  static const int LTL_NONE;
  static const int LTL_PROP;
  static const int LTL_ACT;
  static const int LTL_NOT;
  static const int LTL_AND;
  static const int LTL_OR;
  static const int LTL_X;
  static const int LTL_G;
  static const int LTL_F;
  static const int LTL_U;
  static const int LTL_R;

  BasicLtl() { type = LTL_NONE; left = right = NULL; }
  BasicLtl(const Expr &p) { type = LTL_PROP; prop = p; left = right = NULL; }
  BasicLtl(const Action &a) { type = LTL_ACT; action = a; left = right = NULL; }
  BasicLtl(int t,const BasicLtl *l);
  BasicLtl(int t,const BasicLtl *l,const BasicLtl *r);
  BasicLtl(int t,const Expr &p,const Action &a,const BasicLtl *l,const BasicLtl *r);
  BasicLtl(const BasicLtl &rhs) { left = right = NULL; *this = rhs; }
  ~BasicLtl() { Cleanup(); }

  const BasicLtl &operator = (const BasicLtl &rhs) {
    Cleanup();
    type = rhs.type;
    prop = rhs.prop;
    action = rhs.action;
    left = (rhs.left == NULL) ? NULL : rhs.left->Clone();
    right = (rhs.right == NULL) ? NULL : rhs.right->Clone();
    return *this;
  }
  BasicLtl *Clone() const { return new BasicLtl(type,prop,action,left,right); }
  void Cleanup() {
    if(left != NULL) delete left; left = NULL;
    if(right != NULL) delete right; right = NULL;
  }

  string ToString() const;

  friend class LtlManager;
  friend class Buchi;
};

} //namespace magic

#endif //__LTLFORMULA_H__

/*********************************************************************/
//end of LtlFormula.h
/*********************************************************************/
