/******************************** CPPFile *****************************

* FileName [LtlFormula.cpp]

* PackageName [parser]

* Synopsis [Method definitions of LtlFormula class.]

* SeeAlso [LtlFormula.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "ActionManager.h"
#include "LtlFormula.h"
#include "LtlManager.h"
using namespace magic;

/*********************************************************************/
//methods for class LtlFormula
/*********************************************************************/
LtlFormula::LtlFormula(const BasicLtl *l)
{
  assert(l != NULL);
  ltl = LtlManager::Register(l);  
}

const LtlFormula &LtlFormula::operator = (const LtlFormula &rhs)
{
  ltl = rhs.ltl;
  return *this;
}

bool LtlFormula::operator == (const LtlFormula &rhs) const
{
  return (ltl == rhs.ltl);
}

bool LtlFormula::operator < (const LtlFormula &rhs) const
{
  return (ltl < rhs.ltl);
}

string LtlFormula::ToString() const
{
  return ltl->ToString();
}

/*********************************************************************/
//return the expressions associated with assign actions
/*********************************************************************/
void LtlFormula::GetAssignActExprs(set< pair<Expr,Expr> > &res) const
{
  const set<Action> &actions = LtlManager::GetActions(*this);
  for(set<Action>::iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsAssign()) {
      res.insert(pair<Expr,Expr>(i->GetLhsExpr(),i->GetRhsExpr()));
    }
  }
}

/*********************************************************************/
//return the expressions associated with return actions
/*********************************************************************/
void LtlFormula::GetRetActExprs(set<Expr> &res) const
{
  const set<Action> &actions = LtlManager::GetActions(*this);
  for(set<Action>::iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsReturn() && (!i->GetRetExpr().IsEmptyExpr())) res.insert(i->GetRetExpr());
  }
}

/*********************************************************************/
//compute the set of actions associated with a return. the arguments
//are the expression for the return value, a list of arguments to the
//procedure, and a list of expressions describing the state in which
//the return is executed.
/*********************************************************************/
void LtlFormula::GetReturnActions(const Expr &expr,const list<string> &params,
				  const set<Expr> &context,set<Action> &res) const
{
  set<Action> returns; ComputeReturnActions(returns);
  for(set<Action>::const_iterator i = returns.begin();i != returns.end();++i) {
    if(i->ReturnActionPossible(expr,params,context)) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of actions associated with a return. the arguments
//are the expression for the return value, a list of arguments to the
//procedure, and a list of expressions describing the state in which
//the return is executed.
/*********************************************************************/
void LtlFormula::GetAssignActions(const Expr &lhs,const Expr &rhs,
				  const list<string> &params,
				  const set<Expr> &context,set<Action> &res) const
{
  set<Action> assigns; ComputeAssignActions(assigns);
  for(set<Action>::const_iterator i = assigns.begin();i != assigns.end();++i) {
    if(i->AssignActionPossible(lhs,rhs,params,context)) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of return actions in this LTS
/*********************************************************************/
void LtlFormula::ComputeReturnActions(set<Action> &res) const
{
  const set<Action> &actions = LtlManager::GetActions(*this);
  for(set<Action>::const_iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsReturn()) res.insert(*i);
  }
}

/*********************************************************************/
//compute the set of assign actions in this LTS
/*********************************************************************/
void LtlFormula::ComputeAssignActions(set<Action> &res) const
{
  const set<Action> &actions = LtlManager::GetActions(*this);
  for(set<Action>::const_iterator i = actions.begin();i != actions.end();++i) {
    if(i->IsAssign()) res.insert(*i);
  }
}

/*********************************************************************/
//define static members of class BasicLtl
/*********************************************************************/
const int BasicLtl::LTL_NONE = 8500;
const int BasicLtl::LTL_PROP = 8510;
const int BasicLtl::LTL_ACT = 8520;
const int BasicLtl::LTL_NOT = 8530;
const int BasicLtl::LTL_AND = 8540;
const int BasicLtl::LTL_OR = 8550;
const int BasicLtl::LTL_X = 8560;
const int BasicLtl::LTL_G = 8570;
const int BasicLtl::LTL_F = 8580;
const int BasicLtl::LTL_U = 8590;
const int BasicLtl::LTL_R = 8600;

/*********************************************************************/
//constructors and destructors
/*********************************************************************/
BasicLtl::BasicLtl(int t,const Expr &p,const Action &a,const BasicLtl *l,const BasicLtl *r)
{
  type = t;
  prop = p;
  action = a;
  left = (l == NULL) ? NULL : l->Clone();
  right = (r == NULL) ? NULL : r->Clone();
}

BasicLtl::BasicLtl(int t,const BasicLtl *l) 
{ 
  assert(((t == LTL_NOT) || (t == LTL_X) || (t == LTL_G) || (t == LTL_F)) && (l != NULL)); 
  type = t; 
  left = l->Clone(); 
  right = NULL; 
}

BasicLtl::BasicLtl(int t,const BasicLtl *l,const BasicLtl *r) 
{ 
  assert(((t == LTL_AND) || (t == LTL_OR) || (t == LTL_U) || (t == LTL_R)) && 
	 (l != NULL) && (r != NULL)); 
  type = t; 
  left = l->Clone(); 
  right = r->Clone();
} 

/*********************************************************************/
//get string representation
/*********************************************************************/
string BasicLtl::ToString() const
{
  if(type == LTL_PROP) {
    return Util::TrimString(prop.ToString());
  } else if(type == LTL_ACT) {
    return action.ToString();
  } else if((type == LTL_NOT) || (type == LTL_X) || (type == LTL_G) || (type == LTL_F)) {
    string x = left->ToString();
    if(type == LTL_NOT) return "! " + x;
    else if(type == LTL_X) return "X " + x;
    else if(type == LTL_G) return "G " + x;
    else return "F " + x;
  } else if((type == LTL_AND) || (type == LTL_OR) || (type == LTL_U) || (type == LTL_R)) {
    string x = left->ToString();
    string y = right->ToString();
    if(type == LTL_AND) return x + " & " + y;
    else if(type == LTL_OR) return x + " | " + y;
    else if(type == LTL_U) return x + " U " + y;
    else return x + " R " + y;
  } else assert(false);
}

/*********************************************************************/
//end of LtlFormula.cpp
/*********************************************************************/
