/******************************** CPPFile *****************************

* FileName [FspInfo.cpp]

* PackageName [parser]

* Synopsis [Method definitions of FspInfo class.]

* SeeAlso [FspInfo.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <string>
#include <list>
#include <set>
#include <vector>
#include <map>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "ActionManager.h"
#include "LtsTrans.h"
#include "LtsInfo.h"
#include "FspInfo.h"
#include "Database.h"
using namespace magic;

/*********************************************************************/
//constructor - we must add transtions for the EpsilonSpec,
//DefaultSpec and NullPtrDerefSpec LTSs
/*********************************************************************/
FspInfo::FspInfo()
{
  trans.insert(LtsTrans(Database::EPSILON_SPEC,Database::STOP_STATE,ActionManager::GetVoidRetAction()));
  trans.insert(LtsTrans(Database::DEFAULT_SPEC,Database::STOP_STATE,ActionManager::GetDefaultRetAction()));
  trans.insert(LtsTrans(Database::INVALID_PTR_DEREF_SPEC,Database::ERROR_STATE,ActionManager::GetInvalidPtrDerefAction()));
  trans.insert(LtsTrans(Database::ASSERTION_FAILURE_SPEC,Database::ERROR_STATE,ActionManager::GetAssertionFailureAction()));
}

/*********************************************************************/
//add a transition
/*********************************************************************/
void FspInfo::AddTrans(const LtsTrans &t)
{
  string is = t.GetInitState();
  string fs = t.GetFinalState();
  assert((is != "") && (fs != ""));
  assert((is != Database::EPSILON_SPEC) && (is != Database::DEFAULT_SPEC));
  trans.insert(t);
}

/*********************************************************************/
//add an alphabet extension
/*********************************************************************/
void FspInfo::AddExtAct(const string &n,const set<Action> &a)
{
  assert(extAct.count(n) == 0);
  extAct[n] = a;
}

/*********************************************************************/
//get the LtsInfo object for the named FSP process
/*********************************************************************/
LtsInfo FspInfo::GetLtsInfo(const string &name) const
{
  //compute the set of states,actions and transitions
  set<string> st;
  set<Action> ac;
  set<LtsTrans> tr;

  set<string> frontier,next;
  frontier.insert(name);
  while(!frontier.empty()) {
    //add the frontier to the set of states
    st.insert(frontier.begin(),frontier.end());
    
    //compute new neighbours of the frontier and store them in
    //next. also add the new transitions and actions to tr and ac.
    for(set<string>::iterator i = frontier.begin();i != frontier.end();++i) {
      for(set<LtsTrans>::const_iterator j = trans.begin();j != trans.end();++j) {
	string a = j->GetInitState();
	if(frontier.count(a) != 0) {
	  if(tr.count(*j) == 0) {
	    Util::Message(3,"adding transition : %s\n",j->ToString().c_str());
	  }
	  tr.insert(*j);
	  ac.insert(j->GetAction());
	  string b = j->GetFinalState();
	  if(st.count(b) == 0) next.insert(b);
	}
      }
    }

    //update the frontier
    frontier = next;
    next.clear();
  }

  //if no transitions were found then this FSP process is undefined
  if(tr.empty()) {
    Util::Error("ERROR: undefined FSP process or LTL formula " + name + " ...\n");
  }

  //add the set of extended actions
  for(map< string,set<Action> >::const_iterator i = extAct.begin();i != extAct.end();++i) {
    if(st.count(i->first) != 0) {
      ac.insert(i->second.begin(),i->second.end());
    }
  }

  //create the LTS info and check if it is useful
  LtsInfo res(name,st,ac,tr);
  res.ComputeUseful();
  return res;
}

/*********************************************************************/
//get the LtsInfo object for the negation of the named FSP process
/*********************************************************************/
LtsInfo FspInfo::GetSpecLtsNeg(const string &name)
{
  map< set<string>,string > stateMap;
  set<string> st,nonDetSt;
  set<Action> ac;
  set<LtsTrans> tr;
  map< string,set<Action> > trMap;

  set<string> isub;
  isub.insert(name);
  string istate = Util::NewTempVar();
  stateMap[isub] = istate;
  set< set<string> > frontier;
  frontier.insert(isub);

  //determinize
  while(!frontier.empty()) {
    set< set<string> > newFront = frontier;
    frontier.clear();

    for(set< set<string> >::const_iterator i = newFront.begin();i != newFront.end();++i) {
      st.insert(stateMap[*i]);
      nonDetSt.insert(i->begin(),i->end());
      map< Action,set<string> > succMap;
      //consider epsilon self-loops if necessary
      for(set<string>::const_iterator j = i->begin();j != i->end();++j) {
	if(((*j) != Database::STOP_STATE) && ((*j) != Database::ERROR_STATE)) {
	  succMap[ActionManager::GetSpecEpsilonAction()].insert(*j);
	}
      }
      //now consider actual transitions
      for(set<LtsTrans>::const_iterator j = trans.begin();j != trans.end();++j) {
	if(i->count(j->GetInitState()) != 0) {
	  succMap[j->GetAction()].insert(j->GetFinalState());
	}
      }
      for(map< Action,set<string> >::const_iterator j = succMap.begin();j != succMap.end();++j) {
	string finalst;
	if(stateMap.count(j->second) == 0) {
	  finalst = Util::NewTempVar();
	  stateMap[j->second] = finalst;
	  frontier.insert(j->second);
	} else {
	  finalst = stateMap[j->second];
	}
	tr.insert(LtsTrans(stateMap[*i],finalst,j->first));
	trMap[stateMap[*i]].insert(j->first);
	ac.insert(j->first);
      }
    }
  }

  //if no transitions were found then this FSP process is undefined
  if(tr.empty()) {
    Util::Error("undefined FSP process " + name + " ...\n");
  }

  //add the set of extended actions
  for(map< string,set<Action> >::const_iterator i = extAct.begin();i != extAct.end();++i) {
    if((nonDetSt.count(i->first) != 0) || (st.count(i->first) != 0)) {
      ac.insert(i->second.begin(),i->second.end());
    }
  }

  //negate
  for(set<string>::const_iterator i = st.begin();i != st.end();++i) {
    for(set<Action>::const_iterator j = ac.begin();j != ac.end();++j) {
      if((trMap.count(*i) == 0) || (trMap[*i].count(*j) == 0)) {
	tr.insert(LtsTrans(*i,Database::ERROR_STATE,*j));
      }
    }
    tr.insert(LtsTrans(*i,Database::ERROR_STATE,ActionManager::GetDefaultRetAction()));
  }
  
  //update FspInfo
  trans.insert(tr.begin(),tr.end());
  
  //create the LTS info and check if it is useful
  st.insert(Database::ERROR_STATE);
  LtsInfo res(istate,st,ac,tr);
  res.ComputeUseful();
  return res;
}

/*********************************************************************/
//end of FspInfo.cpp
/*********************************************************************/
