/**************************** CPPHeaderFile ***************************

* FileName [ExprManager.h]

* PackageName [parser]

* Synopsis [Header file for ExprManager class.]

* Description [This class encapsulates the expression manager.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __EXPRMANAGER_H__
#define __EXPRMANAGER_H__

namespace magic {

//other classes needed
class ProgInfo;
class ProcAddInfo;
class Action;
class LtsTrans;
class FspBody;
class FspTransSeq;
class BasicLtl;

class ExprManager
{
 private:
  //the cache of all expressions
  static set<const BasicExpr*> allExprs;

  //the cache of id expressions
  static map<string,const IdExpr*> idExprs;

  //the cache of integer constant expressions
  static map<long long,const IntConstExpr*> intConstExprs;

  //the cache of constant expressions
  static map<string,const ConstExpr*> constExprs;

  //the cache of string expressions
  static map<string,const StrExpr*> strExprs;

  //the cache of brack expressions
  static map< pair<const BasicExpr*,const BasicExpr*>,const BrackExpr* > brackExprs;

  //the cache of par expressions
  static map< pair< const BasicExpr*,list<const BasicExpr*> >,const ParExpr* > parExprs;

  //the cache of dot expressions
  static map< pair<const BasicExpr*,string>,const DotExpr* > dotExprs;

  //the cache of arrow expressions
  static map< pair<const BasicExpr*,string>,const ArrowExpr* > arrowExprs;

  //the cache of unary expressions
  static map< pair<const BasicExpr*,char>,const BasicExpr* > unaryExprs;

  //the cache of binary expressions
  static map< pair< pair<const BasicExpr*,const BasicExpr*>,short >,const BasicExpr* > binaryExprs;

  //the cache of the empty expression
  static const BasicExpr *emptyExpr;

  //the cache of relations between expressions
  static map< Expr,set<Expr> > equivCache;
  static map< Expr,set<Expr> > negCache;
  static map< Expr,set<Expr> > implyCache;
  static map< Expr,set<Expr> > disjCache;

  //the theorem prover cache
  static set< pair< set<Expr>,set<Expr> > > trueCache;
  static set< pair< set<Expr>,set<Expr> > > falseCache;

  //the replacement cache and weakest precondition caches
  static map< pair< pair< list<Expr>,list<Expr> >,Expr >,Expr > replaceCache;
  static map< pair< pair< list<Expr>,list<Expr> >,Expr >,Expr > wpCache;

  //cache for syntactic value type checks
  static map<const BasicExpr*,int> synValCache;

  static pair<bool,bool> SyntacticCheck(const set<Expr> &ante,const set<Expr> &cons);
  static int GetTypeSyntactic(const Expr &arg);
  static int EquivalentToSyntactic(const Expr &expr1,const Expr &expr2);
  static int NegationOfSyntactic(const Expr &expr1,const Expr &expr2);
  static int DisjointWithSyntactic(const Expr &expr1,const Expr &expr2);
  static int ImpliesSyntactic(const Expr &expr1,const Expr &expr2);
  static bool ProveImpliesNoSyntactic(const set<Expr> &ante,const set<Expr> &cons);

  //determine if the disjunction of a set of expressions is true
  static bool IsTrue(const set<Expr> &arg);

  //determine if the conjunction of a set of expressions is false
  static bool IsFalse(const set<Expr> &arg);

  //split up a set of expressions into sets that refer to disjoint
  //sets of lvalues
  static void SplitExprSet(const set<Expr> &arg,set< set<Expr> > &res);

  //replace dummy variables
  static Expr ReplaceDummyVars(const Expr &expr,const list<Expr> &actuals,size_t startIndex);

  //some types needed for expression analysis
  typedef pair<const BasicExpr*,const BasicExpr*> Ops;
  typedef pair<Ops,short> Bin;

  //simplify various types of binary expressions
  static int AnalyseExpr(const Expr &e,Bin &res);
  static Expr SimplifyMult(const Expr &e1,const Expr &e2);
  static Expr SimplifyAdd(const Expr &e1,const Expr &e2);
  static Expr SimplifySub(const Expr &e1,const Expr &e2);
  static Expr SimplifyLT(const Expr &e1,const Expr &e2);
  static Expr SimplifyGT(const Expr &e1,const Expr &e2);
  static Expr SimplifyLE(const Expr &e1,const Expr &e2);
  static Expr SimplifyGE(const Expr &e1,const Expr &e2);
  static Expr SimplifyEQ(const Expr &e1,const Expr &e2);
  static Expr SimplifyNE(const Expr &e1,const Expr &e2);

 public:
  //register a basic expression
  static const BasicExpr *Register(const BasicExpr *expr);

  //get all kinds of expressions
  static Expr GetIdExpr(const string &id);
  static Expr GetIntConstExpr(const long long &val);
  static Expr GetConstExpr(const string &id);
  static Expr GetStrExpr(const string &id);
  static Expr GetBrackExpr(const Expr &e1,const Expr &e2);
  static Expr GetParExpr(const Expr &e,const list<Expr> &l);
  static Expr GetDotExpr(const Expr &e,const string &id);
  static Expr GetArrowExpr(const Expr &e,const string &id);
  static Expr GetUnaryExpr(const Expr &e,char op);
  static Expr GetEmptyExpr();
  static Expr GetBinaryExpr(const Expr &e1,const Expr &e2,short op);

  //determine if an expression is true or false
  static bool IsTrue(const Expr &arg);
  static bool IsFalse(const Expr &arg);

  //determine relationships between expressions
  static bool EquivalentTo(const Expr &lhs,const Expr &rhs);
  static bool NegationOf(const Expr &lhs,const Expr &rhs);
  static bool DisjointWith(const Expr &lhs,const Expr &rhs);
  static bool Implies(const Expr &lhs,const Expr &rhs);
  static bool LvalueCompatible(const Expr &lhs,const Expr &rhs);

  //manipulate expressions
  static Expr NegateExpr(const Expr &e);
  static Expr ReplaceDummyVarsZero(const Expr &expr,const list<Expr> &actuals);
  static Expr ReplaceDummyVarsOne(const Expr &expr,const list<Expr> &actuals);
  static Expr ReplaceDummyVarsNonZero(const Expr &expr,const list<Expr> &actuals);
  static bool ProveImplies(const set<Expr> &ante,const set<Expr> &cons);
  static Expr ComputeWP(const Expr &lhs,const Expr &rhs,const Expr &expr);
  static Expr ComputeWP(const list<Expr> &lhsList,const list<Expr> &rhsList,const Expr &expr);
  static Expr Replace(const list<Expr> &lhsList,const list<Expr> &rhsList,const Expr &expr);
  static void GetCalledProcDetails(const Expr &node,pair< Expr,list<Expr> > &res);
  static void StringListToExprList(const list<string> &arg,list<Expr> &res);
  static string ExprSetToString(const set<Expr> &arg);
  static void NegateExprSet(const set<Expr> &arg,set<Expr> &res);
  static Expr ConjunctExprSet(const set<Expr> &arg);
  static Expr DisjunctExprSet(const set<Expr> &arg);
  static void ClearCaches();
};

} //namespace magic


#endif //__EXPRMANAGER_H__

/*********************************************************************/
//end of ExprManager.h
/*********************************************************************/
