/******************************** CPPFile *****************************

* FileName [ExprManager.cpp]

* PackageName [parser]

* Synopsis [Method definitions of ExprManager class.]

* SeeAlso [ExprManager.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
#include <typeinfo>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "TheoremProver.h"
#include "WPComputer.h"
#define YYSTYPE int
#include "StdcParser.h"
#include "Database.h"
using namespace magic;

/*********************************************************************/
//static members
/*********************************************************************/
set<const BasicExpr*> ExprManager::allExprs;
map<string,const IdExpr*> ExprManager::idExprs;
map<long long,const IntConstExpr*> ExprManager::intConstExprs;
map<string,const ConstExpr*> ExprManager::constExprs;
map<string,const StrExpr*> ExprManager::strExprs;
map< pair<const BasicExpr*,const BasicExpr*>,const BrackExpr* > ExprManager::brackExprs;
map< pair< const BasicExpr*,list<const BasicExpr*> >,const ParExpr* > ExprManager::parExprs;
map< pair<const BasicExpr*,string>,const DotExpr* > ExprManager::dotExprs;
map< pair<const BasicExpr*,string>,const ArrowExpr* > ExprManager::arrowExprs;
map< pair<const BasicExpr*,char>,const BasicExpr* > ExprManager::unaryExprs;
const BasicExpr *ExprManager::emptyExpr = NULL;
map< pair< pair<const BasicExpr*,const BasicExpr*>,short >,const BasicExpr* > ExprManager::binaryExprs;
map< Expr,set<Expr> > ExprManager::equivCache;
map< Expr,set<Expr> > ExprManager::negCache;
map< Expr,set<Expr> > ExprManager::implyCache;
map< Expr,set<Expr> > ExprManager::disjCache;
set< pair< set<Expr>,set<Expr> > > ExprManager::trueCache;
set< pair< set<Expr>,set<Expr> > > ExprManager::falseCache;
map< pair< pair< list<Expr>,list<Expr> >,Expr >,Expr > ExprManager::replaceCache;
map< pair< pair< list<Expr>,list<Expr> >,Expr >,Expr > ExprManager::wpCache;
map<const BasicExpr*,int> ExprManager::synValCache;

/*********************************************************************/
//register a basic expression. registering involves recursively
//registering all subexpressions.
/*********************************************************************/
const BasicExpr *ExprManager::Register(const BasicExpr *expr)
{
  //first check if the expression has already been registered before
  if(allExprs.count(expr) != 0) return expr;
  
  //id_expression
  if(typeid(*expr) == typeid(IdExpr)) {
    const IdExpr *x = static_cast<const IdExpr*>(expr);
    return GetIdExpr(x->id).expr;
  }
  //int_const_expression
  else if(typeid(*expr) == typeid(IntConstExpr)) {
    const IntConstExpr *x = static_cast<const IntConstExpr*>(expr);
    return GetIntConstExpr(x->val).expr;
  }
  //const_expression
  else if(typeid(*expr) == typeid(ConstExpr)) {
    const ConstExpr *x = static_cast<const ConstExpr*>(expr);
    return GetConstExpr(x->id).expr;
  }
  //str_expression
  else if(typeid(*expr) == typeid(StrExpr)) {
    const StrExpr *x = static_cast<const StrExpr*>(expr);
    return GetStrExpr(x->id).expr;
  }
  //str_str_expression
  else if(typeid(*expr) == typeid(StrStrExpr)) {
    Util::Error("ERROR: illegal use of expression %s: check your source and specification files ...\n",expr->ToString().c_str());
  } 
  //statement_expression
  else if(typeid(*expr) == typeid(StmtExpr)) {
    Util::Error("ERROR: illegal use of expression %s: check your source and specification files ...\n",expr->ToString().c_str());
  }
  //brack_expression
  else if(typeid(*expr) == typeid(BrackExpr)) {
    const BrackExpr *x = static_cast<const BrackExpr*>(expr);
    return GetBrackExpr(Expr(x->array),Expr(x->index)).expr;
  }
  //par_expression
  else if(typeid(*expr) == typeid(ParExpr)) {
    const ParExpr *x = static_cast<const ParExpr*>(expr);
    list<BasicExpr*> &args = x->args->data;
    list<Expr> nargs;
    for(list<BasicExpr*>::iterator i = args.begin();i != args.end();++i) {
      nargs.push_back(Expr(*i));
    }
    return GetParExpr(Expr(x->proc),nargs).expr;
  }
  //dot_expression
  else if(typeid(*expr) == typeid(DotExpr)) {
    const DotExpr *x = static_cast<const DotExpr*>(expr);
    return GetDotExpr(Expr(x->expr),x->id).expr;
  }
  //arrow_expression
  else if(typeid(*expr) == typeid(ArrowExpr)) {
    const ArrowExpr *x = static_cast<const ArrowExpr*>(expr);
    return GetArrowExpr(Expr(x->expr),x->id).expr;
  }
  //inc_dec_expression
  else if(typeid(*expr) == typeid(IncExpr)) {
    Util::Error("ERROR: illegal use of expression %s: check your source and specification files ...\n",expr->ToString().c_str());
  }
  //unary_expression
  else if(typeid(*expr) == typeid(UnaryExpr)) {
    const UnaryExpr *x = static_cast<const UnaryExpr*>(expr);
    return GetUnaryExpr(Expr(x->expr),x->op).expr;
  }
  //empty_expression
  else if(typeid(*expr) == typeid(EmptyExpr)) {
    return GetEmptyExpr().expr;
  }
  //binary_expression
  else if(typeid(*expr) == typeid(BinaryExpr)) {
    const BinaryExpr *x = static_cast<const BinaryExpr*>(expr);
    return GetBinaryExpr(Expr(x->lhs),Expr(x->rhs),x->op).expr;
  }
  //quest_expression
  else if(typeid(*expr) == typeid(QuestExpr)) {
    Util::Error("ERROR: illegal use of expression %s: check your source and specification files ...\n",expr->ToString().c_str());
  }
  //assign_expression
  else if(typeid(*expr) == typeid(AssignExpr)) {
    Util::Error("ERROR: illegal use of expression %s: check your source and specification files ...\n",expr->ToString().c_str());
  }
  //illegal expression
  else {
    Util::Error("ERROR: illegal use of expression %s: check your source and specification files ...\n",expr->ToString().c_str());
  }
  return NULL;
}

/*********************************************************************/
//returns the id expr corresponding to the string supplied as
//argument. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetIdExpr(const string &id)
{
  Expr res;
  map<string,const IdExpr*>::const_iterator i = idExprs.find(id);
  if(i == idExprs.end()) {
    IdExpr *x = new IdExpr(id);
    allExprs.insert(x);
    idExprs[id] = x;
    res.expr = x;
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the integer constant expr corresponding to the integer
//supplied as argument. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetIntConstExpr(const long long &val)
{
  Expr res;
  map<long long,const IntConstExpr*>::const_iterator i = intConstExprs.find(val);
  if(i == intConstExprs.end()) {
    IntConstExpr *x = new IntConstExpr(val);
    allExprs.insert(x);
    intConstExprs[val] = x;
    res.expr = x;
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the constant expr corresponding to the string supplied as
//argument. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetConstExpr(const string &id)
{
  Expr res;
  map<string,const ConstExpr*>::const_iterator i = constExprs.find(id);
  if(i == constExprs.end()) {
    ConstExpr *x = new ConstExpr(id);
    allExprs.insert(x);
    constExprs[id] = x;
    res.expr = x;
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the string expr corresponding to the string supplied as
//argument. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetStrExpr(const string &id)
{
  Expr res;
  map<string,const StrExpr*>::const_iterator i = strExprs.find(id);
  if(i == strExprs.end()) {
    StrExpr *x = new StrExpr(id);
    allExprs.insert(x);
    strExprs[id] = x;
    res.expr = x;
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the brack expr corresponding to the expressions supplied as
//arguments. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetBrackExpr(const Expr &e1,const Expr &e2)
{
  Expr res;
  pair<const BasicExpr*,const BasicExpr*> x(e1.expr,e2.expr);
  map< pair<const BasicExpr*,const BasicExpr*>,const BrackExpr* >::const_iterator i = brackExprs.find(x);
  if(i == brackExprs.end()) {
    BrackExpr *y = new BrackExpr();
    y->array = (BasicExpr*)(x.first);
    y->index = (BasicExpr*)(x.second);
    allExprs.insert(y);
    brackExprs[x] = y;
    res.expr = y;
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the par expr corresponding to the expressions supplied as
//arguments. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetParExpr(const Expr &e,const list<Expr> &l)
{
  Expr res;
  list<const BasicExpr*> x;
  for(list<Expr>::const_iterator i = l.begin();i != l.end();++i) {
    x.push_back(i->expr);
  }
  pair< const BasicExpr*,list<const BasicExpr*> > y(e.expr,x);
  map< pair< const BasicExpr*,list<const BasicExpr*> >,const ParExpr* >::const_iterator i = parExprs.find(y);
  if(i == parExprs.end()) {
    ParExpr *z = new ParExpr();
    z->proc = (BasicExpr*)(y.first);
    z->args = new ExprList();
    for(list<const BasicExpr*>::iterator j = y.second.begin();j != y.second.end();++j) {
      z->args->data.push_back((BasicExpr*)(*j));
    }
    allExprs.insert(z);
    parExprs[y] = z;
    res.expr = z;
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the dot expr corresponding to the expression and id supplied
//as arguments. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetDotExpr(const Expr &e,const string &id)
{
  Expr res;
  pair<const BasicExpr*,string> x(e.expr,id);
  map< pair<const BasicExpr*,string>,const DotExpr* >::const_iterator i = dotExprs.find(x);
  if(i == dotExprs.end()) {
    DotExpr *y = new DotExpr();
    y->expr = (BasicExpr*)(x.first);
    y->id = x.second;
    allExprs.insert(y);
    dotExprs[x] = y;
    res.expr = y;
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the arrow expr corresponding to the expression and id
//supplied as arguments. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetArrowExpr(const Expr &e,const string &id)
{
  Expr res;
  pair<const BasicExpr*,string> x(e.expr,id);
  map< pair<const BasicExpr*,string>,const ArrowExpr* >::const_iterator i = arrowExprs.find(x);
  if(i == arrowExprs.end()) {
    ArrowExpr *y = new ArrowExpr();
    y->expr = (BasicExpr*)(x.first);
    y->id = x.second;
    allExprs.insert(y);
    arrowExprs[x] = y;
    res.expr = y;
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns the binary expr corresponding to the operand and operator
//supplied as arguments. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetUnaryExpr(const Expr &e,char op)
{
  Expr res;
  pair<const BasicExpr*,char> x(e.expr,op);
  map< pair<const BasicExpr*,char>,const BasicExpr* >::const_iterator i = unaryExprs.find(x);
  if(i == unaryExprs.end()) {
    const BasicExpr *y = NULL;
    //address dereference
    if(op == '&') {
      if(typeid(*(e.expr)) == typeid(UnaryExpr)) {
	const UnaryExpr *z = static_cast<const UnaryExpr*>(e.expr);
	if(z->op == '*') y = z->expr;
      }      
    }
    //dereference address
    else if(op == '*') {
      if(typeid(*(e.expr)) == typeid(UnaryExpr)) {
	const UnaryExpr *z = static_cast<const UnaryExpr*>(e.expr);
	if(z->op == '&') y = z->expr;
      }      
    }
    //unary plus operator
    else if(op == '+') y = e.expr;
    //unary minus operator
    else if(op == '-') {
      if(typeid(*(e.expr)) == typeid(IntConstExpr)) {
	const IntConstExpr *z = static_cast<const IntConstExpr*>(e.expr);
	y = GetIntConstExpr(-(z->val)).expr;
      }
    }
    //push negations inside
    else if(op == '!') {  
      //int_const_expression
      if(typeid(*(e.expr)) == typeid(IntConstExpr)) {
	const IntConstExpr *z = static_cast<const IntConstExpr*>(e.expr);
	if(z->val == 0) y = GetIntConstExpr(1).expr;
	else y = GetIntConstExpr(0).expr;
      }
      //unary_expression
      else if(typeid(*(e.expr)) == typeid(UnaryExpr)) {
	const UnaryExpr *z = static_cast<const UnaryExpr*>(e.expr);
	if(z->op == '!') {
	  y = z->expr;
	}
      }
      //binary_expression
      else if(typeid(*(e.expr)) == typeid(BinaryExpr)) {
	const BinaryExpr *z = static_cast<const BinaryExpr*>(e.expr);
	if(z->op == '<') y = GetBinaryExpr(Expr(z->rhs),Expr(z->lhs),MAGIC_LE_OP).expr;
	else if(z->op == '>') y = GetBinaryExpr(Expr(z->lhs),Expr(z->rhs),MAGIC_LE_OP).expr;
	else if(z->op == MAGIC_LE_OP) y = GetBinaryExpr(Expr(z->rhs),Expr(z->lhs),'<').expr;
	else if(z->op == MAGIC_GE_OP) y = GetBinaryExpr(Expr(z->lhs),Expr(z->rhs),'<').expr;
	else if(z->op == MAGIC_EQ_OP) y = GetBinaryExpr(Expr(z->lhs),Expr(z->rhs),MAGIC_NE_OP).expr;
	else if(z->op == MAGIC_NE_OP) y = GetBinaryExpr(Expr(z->lhs),Expr(z->rhs),MAGIC_EQ_OP).expr;
	else if(z->op == MAGIC_AND_OP) {
	  y = GetBinaryExpr(GetUnaryExpr(Expr(z->lhs),'!'),GetUnaryExpr(Expr(z->rhs),'!'),MAGIC_OR_OP).expr;
	} else if(z->op == MAGIC_OR_OP) {
	  y = GetBinaryExpr(GetUnaryExpr(Expr(z->lhs),'!'),GetUnaryExpr(Expr(z->rhs),'!'),MAGIC_AND_OP).expr;
	}
      }
    }
    //all checks done
    if(y == NULL) {
      UnaryExpr *z = new UnaryExpr();
      z->expr = (BasicExpr*)(x.first);
      z->op = x.second;
      allExprs.insert(z);
      unaryExprs[x] = z;
      res.expr = z;
    } else {
      assert(allExprs.count(y) != 0);
      unaryExprs[x] = y;
      res.expr = y;
    }
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//returns an empty expression. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetEmptyExpr()
{
  Expr res;
  if(emptyExpr == NULL) {
    emptyExpr = new EmptyExpr();
    allExprs.insert(emptyExpr);
  }
  res.expr = emptyExpr;
  return res;
}

/*********************************************************************/
//returns the binary expr corresponding to the lhs, rhs and operator
//supplied as arguments. create one if necessary.
/*********************************************************************/
Expr ExprManager::GetBinaryExpr(const Expr &e1,const Expr &e2,short op)
{
  Expr res;
  pair<const BasicExpr*,const BasicExpr*> x(e1.expr,e2.expr);
  pair< pair<const BasicExpr*,const BasicExpr*>,short > y(x,op);
  map< pair< pair<const BasicExpr*,const BasicExpr*>,short >,const BasicExpr* >::const_iterator i = binaryExprs.find(y);
  if(i == binaryExprs.end()) {
    const BasicExpr *w = NULL;
    if(op == '*') {
      w = SimplifyMult(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyMult(e2,e1).GetExpr() : w;
    } else if(op == '+') {
      w = SimplifyAdd(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyAdd(e2,e1).GetExpr() : w;
    } else if(op == '-') {
      w = SimplifySub(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyAdd(GetUnaryExpr(e2,'-'),e1).GetExpr() : w;
    } else if(op == '<') {
      w = SimplifyLT(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyGT(e2,e1).GetExpr() : w;
    } else if(op == '>') {
      w = SimplifyGT(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyLT(e2,e1).GetExpr() : w;
    } else if(op == MAGIC_LE_OP) {
      w = SimplifyLE(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyGE(e2,e1).GetExpr() : w;
    } else if(op == MAGIC_GE_OP) {
      w = SimplifyGE(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyLE(e2,e1).GetExpr() : w;
    } else if(op == MAGIC_EQ_OP) {
      w = SimplifyEQ(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyEQ(e2,e1).GetExpr() : w;
    } else if(op == MAGIC_NE_OP) {
      w = SimplifyNE(e1,e2).GetExpr();
      w = (w == NULL) ? SimplifyNE(e2,e1).GetExpr() : w;
    }
    //all checks done
    if(w == NULL) {
      BinaryExpr *z = new BinaryExpr();
      z->lhs = (BasicExpr*)(x.first);
      z->rhs = (BasicExpr*)(x.second);
      z->op = y.second;
      allExprs.insert(z);
      binaryExprs[y] = z;
      res.expr = z;
    } else {
      assert(allExprs.count(w) != 0);
      binaryExprs[y] = w;
      res.expr = w;
    }
  } else {
    res.expr = i->second;
  }
  return res;
}

/*********************************************************************/
//get the type of the valuation of an expression by a purely syntactic
//analysis. return -1 if nothing can be determined, 0 if the
//expression always evaluates to 0, 1 if the expression can take only
//non-zero values and 2 if the expression can take both 0 and non-zero
//values and 3 if the expression can take any value.
/*********************************************************************/
int ExprManager::GetTypeSyntactic(const Expr &arg)
{
  //see if we should do syntactic checks
  if(!Database::USE_SYNTACTIC_CHECKS) return -1;

  //check the cache
  map<const BasicExpr*,int>::const_iterator it = synValCache.find(arg.expr);
  if(it != synValCache.end()) return it->second;

  //if this is not a constant expression or a binary expression or a
  //unary expression with ! or ~ then this cannot be true
  int res = -1;
  if(typeid(*(arg.expr)) == typeid(IdExpr)) {
    res = 3;
  } else if(typeid(*(arg.expr)) == typeid(IntConstExpr)) {
    const IntConstExpr *a = static_cast<const IntConstExpr*>(arg.expr);
    res = (a->val == 0) ? 0 : 1;
  } else if(typeid(*(arg.expr)) == typeid(ConstExpr)) {
    res = -1;
  } else if(typeid(*(arg.expr)) == typeid(StrExpr)) {
    res = 1;
  } else if(typeid(*(arg.expr)) == typeid(StrStrExpr)) {
    res = 1;
  } else if(typeid(*(arg.expr)) == typeid(DotExpr)) {
    res = 3;
  } else if(typeid(*(arg.expr)) == typeid(ArrowExpr)) {
    res = 3;
  } else if(typeid(*(arg.expr)) == typeid(BrackExpr)) {
    res = 3;
  } else if(typeid(*(arg.expr)) == typeid(ParExpr)) {
    if(Database::CHECK_INVALID_PTR_DEREF) {
      const ParExpr *a = static_cast<const ParExpr*>(arg.expr);
      if(typeid(*(a->proc)) == typeid(IdExpr)) {
	const IdExpr *b = static_cast<const IdExpr*>(a->proc);
	res = (b->id == Database::VALID_PTR_PRED) ? -1 : 3;
      } else res = 3;
    } else res = 3;
  } else if(typeid(*(arg.expr)) == typeid(UnaryExpr)) {
    const UnaryExpr *a = static_cast<const UnaryExpr*>(arg.expr);
    if(a->op == '&') res = 3;
    else if(a->op == '*') res = 3;
    else if((a->op == '+') || (a->op == '-')) {
      res = GetTypeSyntactic(Expr(a->expr));
    } else if(a->op == '!') {
      int b = GetTypeSyntactic(Expr(a->expr));
      if(b == -1) res = -1;
      else if(b == 0) res = 1;
      else if(b == 1) res = 0;
      else if(b == 2) res = 2;
      else if(b == 3) res = 2;
      else assert(false);
    } else if(a->op == '~') {
      int b = GetTypeSyntactic(Expr(a->expr));
      if(b == -1) res = -1;
      else if(b == 0) res = 1;
      else if(b == 1) res = -1;
      else if(b == 2) res = -1;
      else if(b == 3) res = 3;
      else assert(false);
    } else assert(false);
  } else if(typeid(*(arg.expr)) == typeid(EmptyExpr)) {
    res = 3;
  } else if(typeid(*(arg.expr)) == typeid(BinaryExpr)) {
    const BinaryExpr *a = static_cast<const BinaryExpr*>(arg.expr);
    if((typeid(*(a->lhs)) == typeid(IntConstExpr)) && (typeid(*(a->rhs)) == typeid(IntConstExpr))) {
      long long b = static_cast<const IntConstExpr*>(a->lhs)->val;
      long long c = static_cast<const IntConstExpr*>(a->rhs)->val;
      if(a->op == '*') res = (b * c) ? 1 : 0;
      else if(a->op == '/') res = (b / c) ? 1 : 0;
      else if(a->op == '%') res = (b % c) ? 1 : 0;
      else if(a->op == '+') res = (b + c) ? 1 : 0;
      else if(a->op == '-') res = (b - c) ? 1 : 0;
      else if(a->op == MAGIC_LEFT_OP) res = (b << c) ? 1 : 0;
      else if(a->op == MAGIC_RIGHT_OP) res = (b >> c) ? 1 : 0;
      else if(a->op == '<') res = (b < c) ? 1 : 0;
      else if(a->op == '>') res = (b > c) ? 1 : 0;
      else if(a->op == MAGIC_LE_OP) res = (b <= c) ? 1 : 0;
      else if(a->op == MAGIC_GE_OP) res = (b >= c) ? 1 : 0;
      else if(a->op == MAGIC_EQ_OP) res = (b == c) ? 1 : 0;
      else if(a->op == MAGIC_NE_OP) res = (b != c) ? 1 : 0;
      else if(a->op == '&') res = (b & c) ? 1 : 0;
      else if(a->op == '^') res = (b ^ c) ? 1 : 0;
      else if(a->op == '|') res = (b | c) ? 1 : 0;
      else if(a->op == MAGIC_AND_OP) res = (b && c) ? 1 : 0;
      else if(a->op == MAGIC_OR_OP) res = (b || c) ? 1 : 0;
      else if(a->op == ',') res = c ? 1 : 0;
      else assert(false);
    } else if(a->op == MAGIC_EQ_OP) {
      //quick check for equality
      if(Expr(a->lhs) == Expr(a->rhs)) return 1;
      int b = GetTypeSyntactic(Expr(a->lhs));
      int c = GetTypeSyntactic(Expr(a->rhs));
      if(b == -1) {
	set<string> idlvs = Util::ComputeIdLvalues(a->lhs);
	if(a->rhs->ContainsLvalue(idlvs)) res = -1;
	else res = (c == 3) ? 2 : -1;
      } else if(b == 0) {
	if(c == 0) res = 1;
	else if(c == 1) res = 0;
	else if(c == 2) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 1) {
	if(c == 0) res = 0;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 2) {
	if(c == 0) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 3) {
	res = (c == 3) ? -1 : 2;
      } else assert(false);
    } else if(a->op == MAGIC_NE_OP) {
      //quick check for equality
      if(Expr(a->lhs) == Expr(a->rhs)) return 0;
      int b = GetTypeSyntactic(Expr(a->lhs));
      int c = GetTypeSyntactic(Expr(a->rhs));
      if(b == -1) {
	set<string> idlvs = Util::ComputeIdLvalues(a->lhs);
	if(a->rhs->ContainsLvalue(idlvs)) res = -1;
	else res = (c == 3) ? 2 : -1;
      } else if(b == 0) {
	if(c == 0) res = 0;
	else if(c == 1) res = 1;
	else if(c == 2) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 1) {
	if(c == 0) res = 1;
	else if(c == 3) res =2;
	else res = -1;
      } else if(b == 2) {
	if(c == 0) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 3) {
	res = (c == 3) ? -1 : 2;
      } else assert(false);
    } else if(a->op == MAGIC_AND_OP) {
      int b = GetTypeSyntactic(Expr(a->lhs));
      int c = GetTypeSyntactic(Expr(a->rhs));
      if(b == -1) {
	if(c == 0) res = 0;
	else res = -1;
      } else if(b == 0) {
	res = 0;
      } else if(b == 1) {
	if(c == 0) res = 0;
	else if(c == 1) res = 1;
	else if(c == 2) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 2) {
	if(c == 0) res = 0;
	else if(c == 1) res = 2;
	else if(c == 2) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 3) {
	if(c == 0) res = 0;
	else if(c == 1) res = 2;
	else if(c == 2) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else assert(false);
    } else if(a->op == MAGIC_OR_OP) {
      int b = GetTypeSyntactic(Expr(a->lhs));
      int c = GetTypeSyntactic(Expr(a->rhs));
      if(b == -1) {
	if(c == 1) res = 1;
	else res = -1;
      } else if(b == 0) {
	if(c == 0) res = 0;
	else if(c == 1) res = 1;
	else if(c == 2) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 1) {
	res = 1;
      } else if(b == 2) {
	if(c == 0) res = 2;
	else if(c == 1) res = 1;
	else if(c == 2) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else if(b == 3) {
	if(c == 0) res = 2;
	else if(c == 1) res = 1;
	else if(c == 2) res = 2;
	else if(c == 3) res = 2;
	else res = -1;
      } else assert(false);
    } else res = -1;
  } else res = -1;

  synValCache[arg.expr] = res;
  return res;
}

/*********************************************************************/
//return true if the argument expression is logically true and false
//otherwise
/*********************************************************************/
bool ExprManager::IsTrue(const Expr &arg)
{
  int syn = GetTypeSyntactic(arg);
  if((syn == 0) || (syn == 2) || (syn == 3)) return false;
  else if(syn == 1) return true;

  set<Expr> x; x.insert(arg);
  return ProveImpliesNoSyntactic(set<Expr>(),x);
}

/*********************************************************************/
//return true if the disjunction of the argument expression set is
//logically true and false otherwise
/*********************************************************************/
bool ExprManager::IsTrue(const set<Expr> &arg)
{
  set<Expr> dummy;
  pair< set<Expr>,set<Expr> > x(dummy,arg);
  if(trueCache.count(x) != 0) return true;
  else if(falseCache.count(x) != 0) return false;

  for(set<Expr>::const_iterator i = arg.begin();i != arg.end();++i) {
    if(IsTrue(*i)) {
      trueCache.insert(x);
      return true;
    }
  }

  bool res = false;
  set< set<Expr> > split; SplitExprSet(arg,split);
  list<const BasicExpr*> a;
  for(set< set<Expr> >::const_iterator i = split.begin();i != split.end();++i) {
    if(ProveImpliesNoSyntactic(dummy,*i)) {
      res = true;
      break;
    } 
  }
  if(res) trueCache.insert(x);
  else falseCache.insert(x);
  return res;
}

/*********************************************************************/
//return true if the argument expression is logically false and false
//otherwise
/*********************************************************************/
bool ExprManager::IsFalse(const Expr &arg)
{
  int syn = GetTypeSyntactic(arg);
  if((syn == 1) || (syn == 2) || (syn == 3)) return false;
  else if(syn == 0) return true;

  //check the cache
  set<Expr> x;
  x.insert(arg);
  return ProveImpliesNoSyntactic(x,set<Expr>());
}

/*********************************************************************/
//return true if the conjunction of the argument expression set is
//logically false and false otherwise
/*********************************************************************/
bool ExprManager::IsFalse(const set<Expr> &arg)
{
  pair< set<Expr>,set<Expr> > x(arg,set<Expr>());
  if(trueCache.count(x) != 0) return true;
  else if(falseCache.count(x) != 0) return false;

  for(set<Expr>::const_iterator i = arg.begin();i != arg.end();++i) {
    if(IsFalse(*i)) {
      trueCache.insert(x);
      return true;
    }
  }

  bool res = false;
  set< set<Expr> > split; SplitExprSet(arg,split);
  list<const BasicExpr*> a;
  for(set< set<Expr> >::const_iterator i = split.begin();i != split.end();++i) {
    if(ProveImpliesNoSyntactic(*i,set<Expr>())) {
      res = true;
      break;
    } 
  }
  if(res) trueCache.insert(x);
  else falseCache.insert(x);
  return res;
}

/*********************************************************************/
//check syntactically if the first argument is logically equivalent to
//the second argument. return 1 if this is the case, 0 if this is not
//the case and -1 if nothing definite can be determined.
/*********************************************************************/
int ExprManager::EquivalentToSyntactic(const Expr &lhs,const Expr &rhs)
{
  //quick check for equality
  if(lhs == rhs) return 1;

  const set<Expr> &a = equivCache[lhs];
  if(a.count(rhs) != 0) return 1;

  const set<Expr> &b = negCache[lhs];
  if(b.count(rhs) != 0) return 0;

  const set<Expr> &c = disjCache[lhs];
  if(c.count(rhs) != 0) return 0;

  return -1;
}

/*********************************************************************/
//returns true if the first argument is logically equivalent to the
//second argument and false otherwise. 
/*********************************************************************/
bool ExprManager::EquivalentTo(const Expr &lhs,const Expr &rhs)
{
  int syn = EquivalentToSyntactic(lhs,rhs);
  if(syn == 0) return false;
  else if(syn == 1) return true;

  set<Expr> lset; lset.insert(lhs);
  set<Expr> rset; rset.insert(rhs);
  bool res = ProveImplies(lset,rset) && ProveImplies(rset,lset);
  if(res) {
    equivCache[lhs].insert(rhs);
    equivCache[rhs].insert(lhs);
  }

  return res;
}

/*********************************************************************/
//check syntactically if the first argument is logically the negation
//of the second argument. return 1 if this is the case, 0 if this is
//not the case and -1 if nothing definite can be determined.
/*********************************************************************/
int ExprManager::NegationOfSyntactic(const Expr &lhs,const Expr &rhs)
{
  //quick check for equality
  if(lhs == rhs) return 0;

  set<Expr> &a = negCache[lhs];
  if(a.count(rhs) != 0) return 1;

  set<Expr> &b = equivCache[lhs];
  if(b.count(rhs) != 0) return 0;

  return -1;
}

/*********************************************************************/
//returns true if the first argument is logically negation of the
//second argument and false otherwise
/*********************************************************************/
bool ExprManager::NegationOf(const Expr &lhs,const Expr &rhs)
{
  int syn = NegationOfSyntactic(lhs,rhs);
  if(syn == 0) return false;
  else if(syn == 1) return true;

  bool res = EquivalentTo(lhs,NegateExpr(rhs));
  if(res) {
    negCache[lhs].insert(rhs);
    negCache[rhs].insert(lhs);
  }

  return res;
}

/*********************************************************************/
//check syntactically if the first argument is logically disjoint with
//the second argument. return 1 if this is the case, 0 if this is not
//the case and -1 if nothing definite can be determined.
/*********************************************************************/
int ExprManager::DisjointWithSyntactic(const Expr &lhs,const Expr &rhs)
{
  //quick check if the lhs or rhs is false
  int x = GetTypeSyntactic(lhs);
  if(x == 0) return 1;
  int y = GetTypeSyntactic(rhs);
  if(y == 0) return 1;

  //quick check for equality
  if(((x == 1) || (x == 2) || (x == 3)) && (lhs == rhs)) return 0;

  set<Expr> &a = disjCache[lhs];
  if(a.count(rhs) != 0) return 1;

  set<Expr> &b = negCache[lhs];
  if(b.count(rhs) != 0) return 1;

  set<Expr> &c = equivCache[lhs];
  if(c.count(rhs) != 0) return 0;

  //otherwise we can only compare binary expressions
  if(typeid(*(lhs.expr)) != typeid(BinaryExpr)) return -1;
  const BinaryExpr *bin1 = static_cast<const BinaryExpr*>(lhs.expr);
  if(typeid(*(rhs.expr)) != typeid(BinaryExpr)) return -1;
  const BinaryExpr *bin2 = static_cast<const BinaryExpr*>(rhs.expr);

  if((typeid(*(bin1->lhs)) != typeid(IntConstExpr)) && (bin1->lhs == bin2->lhs)) {
    if(typeid(*(bin1->rhs)) == typeid(*(bin2->rhs))) {
      if(bin1->op == MAGIC_EQ_OP) {
	if(bin2->op == MAGIC_EQ_OP) {
	  if((typeid(*(bin1->rhs)) == typeid(IdExpr)) || (typeid(*(bin1->rhs)) == typeid(IntConstExpr))) {
	    if(bin1->rhs != bin2->rhs) {
	      disjCache[lhs].insert(rhs);
	      return 1;
	    } else return 0;
	  } else return -1;
	} else if(bin2->op == MAGIC_NE_OP) {
	  if(bin1->rhs == bin2->rhs) {
	    disjCache[lhs].insert(rhs);
	    return 1;
	  } else return 0;	  
	} else return -1;
      } else if(bin1->op == MAGIC_NE_OP) {
	if(bin2->op == MAGIC_EQ_OP) {
	  if(bin1->rhs == bin2->rhs) {
	    disjCache[lhs].insert(rhs);
	    return 1;
	  } else return 0;
	} else return -1;
      } else return -1;
    } else return -1;
  } else if((typeid(*(bin1->lhs)) != typeid(IntConstExpr)) && (bin1->lhs == bin2->rhs)) {
    if(typeid(*(bin1->rhs)) == typeid(*(bin2->lhs))) {
      if(bin1->op == MAGIC_EQ_OP) {
	if(bin2->op == MAGIC_EQ_OP) {
	  if((typeid(*(bin1->rhs)) == typeid(IdExpr)) || (typeid(*(bin1->rhs)) == typeid(IntConstExpr))) {
	    if(bin1->rhs != bin2->lhs) {
	      disjCache[lhs].insert(rhs);
	      return 1;
	    } else return 0;
	  } else return -1;
	} else if(bin2->op == MAGIC_NE_OP) {
	  if(bin1->rhs == bin2->lhs) {
	    disjCache[lhs].insert(rhs);
	    return 1;
	  } else return 0;
	} else return -1;
      } else if(bin1->op == MAGIC_NE_OP) {
	if(bin2->op == MAGIC_EQ_OP) {
	  if(bin1->rhs == bin2->lhs) {
	    disjCache[lhs].insert(rhs);
	    return 1;
	  } else return 0;
	} else return -1;
      } else return -1;
    } else return -1;    
  } else if((typeid(*(bin1->rhs)) != typeid(IntConstExpr)) && (bin1->rhs == bin2->lhs)) {
    if(typeid(*(bin1->lhs)) == typeid(*(bin2->rhs))) {
      if(bin1->op == MAGIC_EQ_OP) {
	if(bin2->op == MAGIC_EQ_OP) {
	  if((typeid(*(bin1->lhs)) == typeid(IdExpr)) || (typeid(*(bin1->lhs)) == typeid(IntConstExpr))) {
	    if(bin1->lhs != bin2->rhs) {
	      disjCache[lhs].insert(rhs);
	      return 1;
	    } else return 0;
	  } else return -1;
	} else if(bin2->op == MAGIC_NE_OP) {
	  if(bin1->lhs == bin2->rhs) {
	    disjCache[lhs].insert(rhs);
	    return 1;
	  } else return 0;
	} else return -1;
      } else if(bin1->op == MAGIC_NE_OP) {
	if(bin2->op == MAGIC_EQ_OP) {
	  if(bin1->lhs == bin2->rhs) {
	    disjCache[lhs].insert(rhs);
	    return 1;
	  } else return 0;
	} else return -1;
      } else return -1;
    } else return -1;
  } else if((typeid(*(bin1->rhs)) != typeid(IntConstExpr)) && (bin1->rhs == bin2->rhs)) {
    if(typeid(*(bin1->lhs)) == typeid(*(bin2->lhs))) {
      if(bin1->op == MAGIC_EQ_OP) {
	if(bin2->op == MAGIC_EQ_OP) {
	  if((typeid(*(bin1->lhs)) == typeid(IdExpr)) || (typeid(*(bin1->lhs)) == typeid(IntConstExpr))) {
	    if(bin1->lhs != bin2->lhs) {
	      disjCache[lhs].insert(rhs);
	      return 1;
	    } else return 0;
	  } else return -1;
	} else if(bin2->op == MAGIC_NE_OP) {
	  if(bin1->lhs == bin2->lhs) {
	    disjCache[lhs].insert(rhs);
	    return 1;
	  } else return 0;
	} else return -1;
      } else if(bin1->op == MAGIC_NE_OP) {
	if(bin2->op == MAGIC_EQ_OP) {
	  if(bin1->lhs == bin2->lhs) {
	    disjCache[lhs].insert(rhs);
	    return 1;
	  } else return 0;
	} else return -1;
      } else return -1;
    } else return -1;
  } 

  return -1;
}

/*********************************************************************/
//returns true if the first argument is logically disjoint with the
//second argument and false otherwise
/*********************************************************************/
bool ExprManager::DisjointWith(const Expr &lhs,const Expr &rhs)
{
  int syn = DisjointWithSyntactic(lhs,rhs);
  if(syn == 0) return false;
  else if(syn == 1) return true;

  set<Expr> lset; lset.insert(lhs);
  set<Expr> rset; rset.insert(NegateExpr(rhs));
  bool res = ProveImplies(lset,rset);
  if(res) {
    disjCache[lhs].insert(rhs);
    disjCache[rhs].insert(lhs);
  }

  return res;
}

/*********************************************************************/
//check syntactically if the first argument logically implies the
//second argument. return 1 if this is the case, 0 if this is not the
//case and -1 if nothing definite can be determined.
/*********************************************************************/
int ExprManager::ImpliesSyntactic(const Expr &lhs,const Expr &rhs)
{
  //quick check for equality
  if(lhs == rhs) return 1;
  set<Expr> &a = implyCache[lhs];
  if(a.count(rhs) != 0) return 1;
  set<Expr> &b = equivCache[lhs];
  if(b.count(rhs) != 0) return 1;
  if(IsTrue(rhs)) return 1;
  //check disjointness with negation
  return DisjointWithSyntactic(lhs,NegateExpr(rhs));
}

/*********************************************************************/
//returns true if the first argument logically implies the second
//argument and false otherwise
/*********************************************************************/
bool ExprManager::Implies(const Expr &lhs,const Expr &rhs)
{
  int syn = ImpliesSyntactic(lhs,rhs);
  if(syn == 0) return false;
  else if(syn == 1) return true;
  
  set<Expr> lset; lset.insert(lhs);
  set<Expr> rset; rset.insert(rhs);
  bool res = ProveImplies(lset,rset);
  if(res) {
    implyCache[lhs].insert(rhs);
  }

  return res;
}

/*********************************************************************/
//return true if the two argument expressions could possibly have the
//same lvalues
/*********************************************************************/
bool ExprManager::LvalueCompatible(const Expr &lhs,const Expr &rhs)
{
  const BasicExpr *lexpr = lhs.GetExpr();
  const BasicExpr *rexpr = rhs.GetExpr();
  if(typeid(*lexpr) == typeid(IdExpr)) {
    if(typeid(*rexpr) == typeid(IdExpr)) {
      const IdExpr *lid = static_cast<const IdExpr*>(lexpr);
      const IdExpr *rid = static_cast<const IdExpr*>(rexpr);
      return (lid->id == rid->id);
    } else if(typeid(*rexpr) == typeid(UnaryExpr)) {
      const UnaryExpr *runary = static_cast<const UnaryExpr*>(rexpr);
      return (runary->op == '*');
    } else return false;
  } else if(typeid(*lexpr) == typeid(IntConstExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(ConstExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(StrExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(StrStrExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(StmtExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(BrackExpr)) {
    if(typeid(*rexpr) == typeid(BrackExpr)) {
      const BrackExpr *lbrack = static_cast<const BrackExpr*>(lexpr);
      const BrackExpr *rbrack = static_cast<const BrackExpr*>(rexpr);
      return LvalueCompatible(Expr(lbrack->array),Expr(rbrack->array));
    } else if(typeid(*rexpr) == typeid(UnaryExpr)) {
      const UnaryExpr *x = static_cast<const UnaryExpr*>(rexpr);
      return (x->op == '*');
    } else return false;
  } else if(typeid(*lexpr) == typeid(ParExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(DotExpr)) {
    if(typeid(*rexpr) == typeid(DotExpr)) {
      const DotExpr *ldot = static_cast<const DotExpr*>(lexpr);
      const DotExpr *rdot = static_cast<const DotExpr*>(rexpr);
      return ((ldot->id == rdot->id) && LvalueCompatible(Expr(ldot->expr),Expr(rdot->expr)));
    } else if(typeid(*rexpr) == typeid(ArrowExpr)) {
      const DotExpr *ldot = static_cast<const DotExpr*>(lexpr);
      const ArrowExpr *rarrow = static_cast<const ArrowExpr*>(rexpr);
      return (ldot->id == rarrow->id);
    } else return false;
  } else if(typeid(*lexpr) == typeid(ArrowExpr)) {
    if(typeid(*rexpr) == typeid(DotExpr)) {
      const ArrowExpr *larrow = static_cast<const ArrowExpr*>(lexpr);
      const DotExpr *rdot = static_cast<const DotExpr*>(rexpr);
      return (larrow->id == rdot->id);
    } else if(typeid(*rexpr) == typeid(ArrowExpr)) {
      const ArrowExpr *larrow = static_cast<const ArrowExpr*>(lexpr);
      const ArrowExpr *rarrow = static_cast<const ArrowExpr*>(rexpr);
      return (larrow->id == rarrow->id);
    } else return false;
  } else if(typeid(*lexpr) == typeid(IncExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(UnaryExpr)) {
    const UnaryExpr *lunary = static_cast<const UnaryExpr*>(lexpr);
    if(lunary->op == '*') {      
      if(typeid(*rexpr) == typeid(UnaryExpr)) {
	const UnaryExpr *runary = static_cast<const UnaryExpr*>(rexpr);
	if(runary->op == '*') return LvalueCompatible(Expr(lunary->expr),Expr(runary->expr));
	else return false;
      } else return true;
    } else return false;
  } else if(typeid(*lexpr) == typeid(EmptyExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(BinaryExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(QuestExpr)) {
    return false;
  } else if(typeid(*lexpr) == typeid(AssignExpr)) {
    return false;
  } else assert(false);
}

/*********************************************************************/
//negate an expression
/*********************************************************************/
Expr ExprManager::NegateExpr(const Expr &e)
{
  //check the negCache first
  set<Expr> &a = negCache[e];
  if(a.empty()) {
    Expr z = GetUnaryExpr(e,'!');
    negCache[e].insert(z);
    return z;
  } else {
    return *(a.begin());
  }
}

/*********************************************************************/
//replace all dummy variable starting from "$0". no dummy variables
//should remain after the replacement.
/*********************************************************************/
Expr ExprManager::ReplaceDummyVarsZero(const Expr &expr,const list<Expr> &actuals)
{
  Expr res = ExprManager::ReplaceDummyVars(expr,actuals,0);
  set<string> a = Util::ComputeIds(res.GetExpr());
  for(set<string>::iterator i = a.begin();i != a.end();++i) {
    if((i->at(0) == '$') && (!Util::IsPredicate(*i))) {
      Util::Error("dummy variable after substitution in " + res.ToString() + "...\n");
    }      
  }
  return res;
}

/*********************************************************************/
//replace all dummy variable starting from "$1". no dummy variables
//should remain after the replacement.
/*********************************************************************/
Expr ExprManager::ReplaceDummyVarsOne(const Expr &expr,const list<Expr> &actuals)
{
  Expr res = ExprManager::ReplaceDummyVars(expr,actuals,1);
  set<string> a = Util::ComputeIds(res.GetExpr());
  for(set<string>::iterator i = a.begin();i != a.end();++i) {
    if((i->at(0) == '$') && (!Util::IsPredicate(*i))) {
      Util::Error("dummy variable after substitution in " + res.ToString() + "...\n");
    }      
  }
  return res;
}

/*********************************************************************/
//replace all dummy variable starting from "$1". only "$0" may remain
//after the replacement.
/*********************************************************************/
Expr ExprManager::ReplaceDummyVarsNonZero(const Expr &expr,const list<Expr> &actuals)
{
  Expr res = ExprManager::ReplaceDummyVars(expr,actuals,1);
  set<string> a = Util::ComputeIds(res.GetExpr());
  for(set<string>::iterator i = a.begin();i != a.end();++i) {
    if((i->at(0) == '$') && (!Util::IsPredicate(*i)) && ((*i) != "$0")) {
      Util::Error("dummy variable after substitution in " + res.ToString() + "...\n");
    }      
  }
  return res;
}

/*********************************************************************/
//replace the dummy variables in an expression with actual
//expressions. the first argument is the target expression. the second
//argument is the list of actual expressions to be substituted and the
//third argument is the starting index of the dummy variables to be
//replaced. if the fourth argument is true then a sanity check is made
//to ensure that no dummy variable remain after replacement.
/*********************************************************************/
Expr ExprManager::ReplaceDummyVars(const Expr &expr,const list<Expr> &actuals,size_t startIndex)
{
  //create the list of arguments and the list of "$0", "$1", "$2"
  //etc. the initial element will depend on the start index.
  list<Expr> dummys;
  char buf[50];
  for(size_t count = 0;count < actuals.size();++count) {
    snprintf(buf,50,"%d",count + startIndex);
    Expr y = GetIdExpr(string("$") + buf);
    dummys.push_back(y);
  }
  return Replace(dummys,actuals,expr);
}

/*********************************************************************/
//analyse an expression. return its components in the second
//argument. 0 if its an integer constant, 1 if it is an unary
//expression, 2 if it is a binary expression with an integral lhs
//constant, 3 if it is a binary expression with an integral rhs. -1 if
//none of the above.
/*********************************************************************/
int ExprManager::AnalyseExpr(const Expr &e,Bin &res)
{
  if(typeid(*(e.expr)) == typeid(IntConstExpr)) {
    res.first = Ops(e.expr,NULL);
    return 0;
  } else if(typeid(*(e.expr)) == typeid(UnaryExpr)) {
    const UnaryExpr *a = static_cast<const UnaryExpr*>(e.expr);
    res.first = Ops(a->expr,NULL);
    res.second = a->op;
    return 1;
  } else if(typeid(*(e.expr)) == typeid(BinaryExpr)) {
    const BinaryExpr *a = static_cast<const BinaryExpr*>(e.expr);
    if(typeid(*(a->lhs)) == typeid(IntConstExpr)) {
      assert(typeid(*(a->rhs)) != typeid(IntConstExpr));
      res.first = Ops(a->lhs,a->rhs);
      res.second = a->op;
      return 2;      
    } else if(typeid(*(a->rhs)) == typeid(IntConstExpr)) {
      assert(typeid(*(a->lhs)) != typeid(IntConstExpr));
      res.first = Ops(a->lhs,a->rhs);
      res.second = a->op;
      return 3;      
    }
  }
  return -1;
}

/*********************************************************************/
//simplify a multiplicative expression
/*********************************************************************/
Expr ExprManager::SimplifyMult(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    if(int1 == 0) return e1;
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return GetIntConstExpr(int1 * int2);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(e1,bin2.first.first,'*');
      else if(bin2.second == '-')  return GetBinaryExpr(GetIntConstExpr(-int1),bin2.first.first,'*');
    } else if(type2 == 2) {
      if(bin2.second == '*') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 * int2),bin2.first.second,'*');
      }
    } else if(type2 == 3) {
      if(bin2.second == '*') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 * int2),bin2.first.first,'*');
      }
    }
  }
  return Expr();
}

/*********************************************************************/
//simplify an additive expression
/*********************************************************************/
Expr ExprManager::SimplifyAdd(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    if(int1 == 0) return e2;
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return GetIntConstExpr(int1 + int2);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(e1,bin2.first.first,'+');
      else if(bin2.second == '-')  return GetBinaryExpr(e1,bin2.first.first,'-');
    } else if(type2 == 2) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 + int2),bin2.first.second,'+');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 + int2),bin2.first.second,'-');
      }
    } else if(type2 == 3) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 + int2),bin2.first.first,'+');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.first,'+');
      }
    }
  }
  return Expr();
}

/*********************************************************************/
//simplify a difference expression
/*********************************************************************/
Expr ExprManager::SimplifySub(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    if(int1 == 0) return GetUnaryExpr(e2,'-');
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return GetIntConstExpr(int1 - int2);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(e1,bin2.first.first,'-');
      else if(bin2.second == '-')  return GetBinaryExpr(e1,bin2.first.first,'+');
    } else if(type2 == 2) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.second,'-');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.second,'+');
      }
    } else if(type2 == 3) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.first,'-');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 + int2),bin2.first.first,'-');
      }
    }
  }
  return Expr();
}

/*********************************************************************/
//simplify a less-than expression
/*********************************************************************/
Expr ExprManager::SimplifyLT(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return (int1 < int2) ? GetIntConstExpr(1) : GetIntConstExpr(0);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(e1,bin2.first.first,'<');
      else if(bin2.second == '-')  return GetBinaryExpr(bin2.first.first,GetIntConstExpr(-int1),'<');
    } else if(type2 == 2) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.second,'<');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(bin2.first.second,GetIntConstExpr(int2 - int1),'<');
      }
    } else if(type2 == 3) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.first,'<');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 + int2),bin2.first.first,'<');
      }
    }
  }
  return Expr();
}

/*********************************************************************/
//simplify a greater-than expression
/*********************************************************************/
Expr ExprManager::SimplifyGT(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return (int1 > int2) ? GetIntConstExpr(1) : GetIntConstExpr(0);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(bin2.first.first,e1,'<');
      else if(bin2.second == '-')  return GetBinaryExpr(GetIntConstExpr(-int1),bin2.first.first,'<');
    } else if(type2 == 2) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(bin2.first.second,GetIntConstExpr(int1 - int2),'<');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int2 - int1),bin2.first.second,'<');
      }
    } else if(type2 == 3) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(bin2.first.first,GetIntConstExpr(int1 - int2),'<');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(bin2.first.first,GetIntConstExpr(int1 + int2),'<');
      }
    }
  }
  return Expr();
}

/*********************************************************************/
//simplify a less-equal expression
/*********************************************************************/
Expr ExprManager::SimplifyLE(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return (int1 <= int2) ? GetIntConstExpr(1) : GetIntConstExpr(0);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(GetIntConstExpr(int1 - 1),bin2.first.first,'<');
      else if(bin2.second == '-')  return GetBinaryExpr(bin2.first.first,GetIntConstExpr(-int1+1),'<');
    } else if(type2 == 2) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2 - 1),bin2.first.second,'<');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(bin2.first.second,GetIntConstExpr(int2 - int1 + 1),'<');
      }
    } else if(type2 == 3) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2 - 1),bin2.first.first,'<');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 + int2 - 1),bin2.first.first,'<');
      }
    } else return GetBinaryExpr(GetIntConstExpr(int1 - 1),e2,'<');
  }
  return Expr();
}

/*********************************************************************/
//simplify a greater-equal expression
/*********************************************************************/
Expr ExprManager::SimplifyGE(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return (int1 >= int2) ? GetIntConstExpr(1) : GetIntConstExpr(0);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(bin2.first.first,GetIntConstExpr(int1 + 1),'<');
      else if(bin2.second == '-')  return GetBinaryExpr(GetIntConstExpr(-int1 - 1),bin2.first.first,'<');
    } else if(type2 == 2) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(bin2.first.second,GetIntConstExpr(int1 - int2 + 1),'<');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int2 - int1 - 1),bin2.first.second,'<');
      }
    } else if(type2 == 3) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(bin2.first.first,GetIntConstExpr(int1 - int2 + 1),'<');
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(bin2.first.first,GetIntConstExpr(int1 + int2 + 1),'<');
      }
    } else return GetBinaryExpr(e2,GetIntConstExpr(int1 + 1),'<');
  }
  return Expr();
}

/*********************************************************************/
//simplify an equality expression
/*********************************************************************/
Expr ExprManager::SimplifyEQ(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return (int1 == int2) ? GetIntConstExpr(1) : GetIntConstExpr(0);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(e1,bin2.first.first,MAGIC_EQ_OP);
      else if(bin2.second == '-')  return GetBinaryExpr(bin2.first.first,GetIntConstExpr(-int1),MAGIC_EQ_OP);
    } else if(type2 == 2) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.second,MAGIC_EQ_OP);
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(bin2.first.second,GetIntConstExpr(int2 - int1),MAGIC_EQ_OP);
      }
    } else if(type2 == 3) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.first,MAGIC_EQ_OP);
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 + int2),bin2.first.first,MAGIC_EQ_OP);
      }
    }
  }
  return Expr();
}

/*********************************************************************/
//simplify a not-equal expression
/*********************************************************************/
Expr ExprManager::SimplifyNE(const Expr &e1,const Expr &e2)
{
  Bin bin1; int type1 = AnalyseExpr(e1,bin1);
  if(type1 == 0) {
    long long int1 = static_cast<const IntConstExpr*>(bin1.first.first)->val;
    Bin bin2; int type2 = AnalyseExpr(e2,bin2);
    if(type2 == 0) {
      long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
      return (int1 != int2) ? GetIntConstExpr(1) : GetIntConstExpr(0);
    } else if(type2 == 1) {
      if(bin2.second == '+') return GetBinaryExpr(e1,bin2.first.first,MAGIC_NE_OP);
      else if(bin2.second == '-')  return GetBinaryExpr(bin2.first.first,GetIntConstExpr(-int1),MAGIC_NE_OP);
    } else if(type2 == 2) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.second,MAGIC_NE_OP);
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.first)->val;
	return GetBinaryExpr(bin2.first.second,GetIntConstExpr(int2 - int1),MAGIC_NE_OP);
      }
    } else if(type2 == 3) {
      if(bin2.second == '+') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 - int2),bin2.first.first,MAGIC_NE_OP);
      } else if(bin2.second == '-') {
	long long int2 = static_cast<const IntConstExpr*>(bin2.first.second)->val;
	return GetBinaryExpr(GetIntConstExpr(int1 + int2),bin2.first.first,MAGIC_NE_OP);
      }
    }
  }
  return Expr();
}

/*********************************************************************/
//prove that an antecedent implies a consequent. both the antecedent
//and the consequent are provided as a set of formulas. the result is
//true if the cojunction of all of the antecedents implies the
//disjunction of all of the consequents. return true if it can be
//proved and false otherwise. make no syntactic checks.
/*********************************************************************/
bool ExprManager::ProveImpliesNoSyntactic(const set<Expr> &ante,const set<Expr> &cons)
{
  pair< set<Expr>,set<Expr> > x(ante,cons);
  if(trueCache.count(x) != 0) return true;
  else if(falseCache.count(x) != 0) return false;

  list<const BasicExpr*> newAnte;
  for(set<Expr>::const_iterator i = ante.begin();i != ante.end();++i) {
    newAnte.push_back(i->expr);
  }
  list<const BasicExpr*> newCons;
  for(set<Expr>::const_iterator i = cons.begin();i != cons.end();++i) {
    newCons.push_back(i->expr);
  }

  bool res = TheoremProver::ProveImplies(newAnte,newCons);
  if(res) {
    trueCache.insert(x);
  } else {
    falseCache.insert(x);
  }
  return res;
}

/*********************************************************************/
//prove that an antecedent implies a consequent. both the antecedent
//and the consequent are provided as a set of formulas. the result is
//true if the cojunction of all of the antecedents implies the
//disjunction of all of the consequents. return true if it can be
//proved and false otherwise.
/*********************************************************************/
bool ExprManager::ProveImplies(const set<Expr> &ante,const set<Expr> &cons)
{
  pair< set<Expr>,set<Expr> > x(ante,cons);
  if(trueCache.count(x) != 0) return true;
  else if(falseCache.count(x) != 0) return false;

  bool res = false;

  //first do a syntactic check
  pair<bool,bool> syn = SyntacticCheck(ante,cons);

  if(syn.first) {
    res = syn.second;
  } else {
    list<const BasicExpr*> newAnte;
    for(set<Expr>::const_iterator i = ante.begin();i != ante.end();++i) {
      newAnte.push_back(i->expr);
    }
    list<const BasicExpr*> newCons;
    for(set<Expr>::const_iterator i = cons.begin();i != cons.end();++i) {
      newCons.push_back(i->expr);
    }
    res = TheoremProver::ProveImplies(newAnte,newCons);
  }
  
  if(res) {
    trueCache.insert(x);
  } else {
    falseCache.insert(x);
  }

  return res;
}

/*********************************************************************/
//try to prove the implication by a purely syntactic check. if we get
//a definite answer we return a pair where the first element is true
//and the second element is the answer we got. otherwise we return a
//pair where both elements are false.
/*********************************************************************/
pair<bool,bool> ExprManager::SyntacticCheck(const set<Expr> &ante,const set<Expr> &cons)
{
  //see if syntactic checks are disabled
  if(!Database::USE_SYNTACTIC_CHECKS) return pair<bool,bool>(false,false);

  //cannot handle empty sets as arguments
  if(ante.empty() && cons.empty()) return pair<bool,bool>(true,false);

  if(ante.empty()) {
    return pair<bool,bool>(true,IsTrue(cons));
  }

  if(cons.empty()) {
    return pair<bool,bool>(true,IsFalse(ante));
  }

  //the count of the number of elements X in cons such that there is
  //an element Y in ante with X and Y being disjoint
  size_t count1 = 0;

  //the count of the number of elements X in cons such that there is
  //an element Y in ante and X implies Y
  size_t count2 = 0;

  if(ante.size() == 1) {
    if(cons.size() == 1) {
      for(set<Expr>::const_iterator i = cons.begin();i != cons.end();++i) {    
	for(set<Expr>::const_iterator j = ante.begin();j != ante.end();++j) {
	  int x = ImpliesSyntactic(*j,*i);
	  if(x == 1) {
	    return pair<bool,bool>(true,true);  
	  }
	  int gt = GetTypeSyntactic(*j);
	  if((gt == 1) || (gt == 2) || (gt == 3)) {
	    if(DisjointWithSyntactic(*j,*i) == 1) ++count1;
	    if((x == 0) && (ImpliesSyntactic(*i,*j) == 1)) ++count2;
	  }
	}
      }
      if((count1 == 1) || (count2 == 1)) {
	return pair<bool,bool>(true,false);
      } else {
	return pair<bool,bool>(false,false);
      }
    } else {
      for(set<Expr>::const_iterator i = cons.begin();i != cons.end();++i) {    
	for(set<Expr>::const_iterator j = ante.begin();j != ante.end();++j) {
	  if(ImpliesSyntactic(*j,*i) == 1) {
	    return pair<bool,bool>(true,true);  
	  }
	  int gt = GetTypeSyntactic(*j);
	  if((gt == 1) || (gt == 2) || (gt == 3)) {
	    if(DisjointWithSyntactic(*j,*i) == 1) ++count1;
	  }
	}
      }
      if(count1 == 1) {
	return pair<bool,bool>(true,false);
      } else {
	return pair<bool,bool>(false,false);
      }
    }
  } else {
    for(set<Expr>::const_iterator i = cons.begin();i != cons.end();++i) {    
      bool flag = false;
      for(set<Expr>::const_iterator j = ante.begin();j != ante.end();++j) {
	if(ImpliesSyntactic(*j,*i) == 1) {
	  return pair<bool,bool>(true,true);  
	}
	if(DisjointWithSyntactic(*i,*j) == 1) {
	  flag = true;
	}
      }
      if(flag) ++count1;
    }

    if(IsFalse(ante)) return pair<bool,bool>(true,true);
    if(count1 == cons.size()) return pair<bool,bool>(true,false);

    return pair<bool,bool>(false,false);
  }
}

/*********************************************************************/
//compute the weakest precondition of the third argument when the
//first two arguments provide the lhs and rhs of the assignment
/*********************************************************************/
Expr ExprManager::ComputeWP(const Expr &lhs,const Expr &rhs,const Expr &expr)
{
  list<Expr> lhsList; lhsList.push_back(lhs);
  list<Expr> rhsList; rhsList.push_back(rhs);
  return ComputeWP(lhsList,rhsList,expr);
}

/*********************************************************************/
//compute the weakest precondition of the third argument when the
//first two arguments provide the lhs-list and rhs-list of the
//parallel assignment
/*********************************************************************/
Expr ExprManager::ComputeWP(const list<Expr> &lhsList,const list<Expr> &rhsList,const Expr &expr)
{
  pair< list<Expr>,list<Expr> > a(lhsList,rhsList);
  pair< pair< list<Expr>,list<Expr> >,Expr > b(a,expr);
  map< pair< pair< list<Expr>,list<Expr> >,Expr >,Expr >::iterator i = wpCache.find(b);
  if(i != wpCache.end()) return (i->second);

  list<const BasicExpr*> newLhsList;
  for(list<Expr>::const_iterator i = lhsList.begin();i != lhsList.end();++i) {
    newLhsList.push_back(i->expr);
  }
  list<const BasicExpr*> newRhsList;
  for(list<Expr>::const_iterator i = rhsList.begin();i != rhsList.end();++i) {
    newRhsList.push_back(i->expr);
  }
  const BasicExpr *x = WPComputer::ComputeWP(newLhsList,newRhsList,expr.expr);
  Expr res(x);
  delete x;
  wpCache[b] = res;
  return res;
}

/*********************************************************************/
//compute the expression obtained from the third argument when all
//ocurrences of an element of the first argument is replaced by the
//corresponding element of the second argument
/*********************************************************************/
Expr ExprManager::Replace(const list<Expr> &lhsList,const list<Expr> &rhsList,const Expr &expr)
{
  pair< list<Expr>,list<Expr> > a(lhsList,rhsList);
  pair< pair< list<Expr>,list<Expr> >,Expr > b(a,expr);
  map< pair< pair< list<Expr>,list<Expr> >,Expr >,Expr >::iterator i = replaceCache.find(b);
  if(i != replaceCache.end()) return (i->second);

  list<const BasicExpr*> newLhsList;
  for(list<Expr>::const_iterator i = lhsList.begin();i != lhsList.end();++i) {
    newLhsList.push_back(i->expr);
  }
  list<const BasicExpr*> newRhsList;
  for(list<Expr>::const_iterator i = rhsList.begin();i != rhsList.end();++i) {
    newRhsList.push_back(i->expr);
  }
  const BasicExpr *x = WPComputer::Replace(newLhsList,newRhsList,expr.expr,false);
  Expr res(x);
  delete x;
  replaceCache[b] = res;
  return res;
}

/*********************************************************************/
//given an expression that represents a procedure call return the
//called procedure name and the list of arguments passed
/*********************************************************************/
void ExprManager::GetCalledProcDetails(const Expr &node,pair< Expr,list<Expr> > &res)
{
  const BasicExpr *expr = node.GetExpr();
  assert(typeid(*expr) == typeid(ParExpr));
  const ParExpr *a = static_cast<const ParExpr*>(expr);
  res.first = Expr(a->proc);
  const list<BasicExpr*> &args = a->args->data;
  for(list<BasicExpr*>::const_iterator i = args.begin();i != args.end();++i) {
    res.second.push_back(Expr(*i));
  }
}

/*********************************************************************/
//convert a list of strings to a list of expressions. the strings are
//treated as identifiers.
/*********************************************************************/
void ExprManager::StringListToExprList(const list<string> &arg,list<Expr> &res)
{
  for(list<string>::const_iterator i = arg.begin();i != arg.end();++i) {
    res.push_back(GetIdExpr(*i));
  }
}

/*********************************************************************/
//convert a list of expressions to a string
/*********************************************************************/
string ExprManager::ExprSetToString(const set<Expr> &arg)
{
  string res = "[";
  for(set<Expr>::const_iterator i = arg.begin();i != arg.end();) {
    res += Util::TrimString(i->ToString());
    ++i;
    if(i != arg.end()) res += ",";
  }
  return res + "]";
}

/*********************************************************************/
//negate a set of expressions
/*********************************************************************/
void ExprManager::NegateExprSet(const set<Expr> &arg,set<Expr> &res)
{
  for(set<Expr>::const_iterator i = arg.begin();i != arg.end();++i) {
    res.insert(NegateExpr(*i));
  }
}

/*********************************************************************/
//given a list of expressions return a single expression that is the
//conjunction of all the elements of the list.
/*********************************************************************/
Expr ExprManager::ConjunctExprSet(const set<Expr> &arg)
{
  if(arg.empty()) return GetIntConstExpr(1);
  Expr res = *(arg.begin());
  set<Expr>::const_iterator i = arg.begin(); ++i;
  for(;i != arg.end();++i) {
    res = GetBinaryExpr(res,*i,MAGIC_AND_OP);
  }
  return res;
}

/*********************************************************************/
//given a list of expressions return a single expression that is the
//disjunction of all the elements of the list.
/*********************************************************************/
Expr ExprManager::DisjunctExprSet(const set<Expr> &arg)
{
  if(arg.empty()) return GetIntConstExpr(0);
  Expr res = *(arg.begin());
  set<Expr>::const_iterator i = arg.begin(); ++i;
  for(;i != arg.end();++i) {
    res = GetBinaryExpr(res,*i,MAGIC_OR_OP);
  }
  return res;
}

/*********************************************************************/
//split a set of expressions into disjoint groups. each group consists
//of expressions related by common lvalues. expressions in different
//groups are unrelated in terms of lvalues.
/*********************************************************************/
void ExprManager::SplitExprSet(const set<Expr> &arg,set< set<Expr> > &res)
{
  //the disjoint groups
  map< set<Expr>,set<string> > groups;
  //initialise the groups
  for(set<Expr>::const_iterator i = arg.begin();i != arg.end();++i) {
    set<Expr> x; x.insert(*i);
    groups[x] = Util::ComputeStrLvalues(i->expr);
  }
  //iterate till no more change
  bool flag = true;
  while(flag) {
    flag = false;
    for(map< set<Expr>,set<string> >::iterator i = groups.begin();(!flag) && (i != groups.end());++i) {
      ++i;
      map< set<Expr>,set<string> >::iterator j = i;
      --i;
      if(j == groups.end()) break;
      for(;j != groups.end();++j) {
	set<string> x = i->second;
	set<string> y = j->second;
	size_t xsize = x.size();
	x.insert(y.begin(),y.end());
	if(x.size() != (xsize + y.size())) {
	  set<Expr> z = i->first;
	  z.insert(j->first.begin(),j->first.end());
	  groups.erase(i);
	  groups.erase(j);
	  groups[z] = x;
	  flag = true;
	  break;
	}
      }
    }
  }
  //compute the result
  for(map< set<Expr>,set<string> >::const_iterator i = groups.begin();i != groups.end();++i) {
    res.insert(i->first);
  }
}

/*********************************************************************/
//clear caches
/*********************************************************************/
void ExprManager::ClearCaches()
{
  equivCache.clear();
  negCache.clear();
  implyCache.clear();
  disjCache.clear();
  trueCache.clear();
  falseCache.clear();
  replaceCache.clear();
  wpCache.clear();
  synValCache.clear();
}

/*********************************************************************/
//end of ExprManager.cpp
/*********************************************************************/
