/**************************** CPPHeaderFile ***************************

* FileName [Expr.h]

* PackageName [parser]

* Synopsis [Header file for Expr classes.]

* Description [These classes encapsulate expression nodes in the AST.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __EXPR_H__
#define __EXPR_H__

namespace magic {

//other classes needed
class CompStmt;
class BasicExpr;
class DeclStmt;
class DeclStmtList;
class ProcAbs;
class BasicLtl;

/*********************************************************************/
//the Expr class used by the rest of the tool. this essentially
//encapsulates a constant pointer to a BasicExpr
/*********************************************************************/
class Expr
{
 private:
  const BasicExpr *expr;

 public:
  Expr() { expr = NULL; }
  Expr(const BasicExpr *e);
  Expr(const Expr &rhs) { *this = rhs; }
  const Expr &operator = (const Expr &rhs);
  bool operator == (const Expr &rhs) const;
  bool operator < (const Expr &rhs) const;
  bool IsNull() const { return (expr == NULL); }
  string ToString() const;
  bool IsEmptyExpr() const; 
  const BasicExpr *GetExpr() const { return expr; }
  bool ContainsLvalue(const set<string> &arg) const;
  bool IsIdExpr() const;

  friend class ExprManager;
};

/*********************************************************************/
//basic expression
/*********************************************************************/
class BasicExpr : public Node 
{
 protected:
  int pamKey;
  
 public:
  BasicExpr() { pamKey = -1; }
  BasicExpr(const int pk) { pamKey = pk; }

  void SetPamKey(const int i) { pamKey = i; }
  int GetPamKey() const { return pamKey; }

  BasicExpr *Negate() const;
  bool ContainsLvalue(const set<string> &arg) const;
};

/*********************************************************************/
//list of expressions
/*********************************************************************/
class ExprList : public Node
{
 private:
  list<BasicExpr*> data;
 public:
  ExprList() {}
  ExprList(const list<BasicExpr*> &d) { 
    for(list<BasicExpr*>::const_iterator i = d.begin();i != d.end();++i) {
      data.push_back(static_cast<BasicExpr*>((*i)->Clone()));
    }
  }
  ExprList(const ExprList &rhs) { *this = rhs; }
  ~ExprList() { Cleanup(); }
  const ExprList &operator = (const ExprList &rhs) {
    Cleanup();
    for(list<BasicExpr*>::const_iterator i = rhs.data.begin();i != rhs.data.end();++i) {
      data.push_back(static_cast<BasicExpr*>((*i)->Clone()));
    }
    return *this;    
  }
  Node *Clone() const { 
    return static_cast<Node*>(new ExprList(data)); 
  }
  void Cleanup() {
    for(list<BasicExpr*>::iterator i = data.begin();i != data.end();++i) {
      delete *i;
    }
    data.clear();
  }

  string ToString() const {
    string res;
    for(list<BasicExpr*>::const_iterator i = data.begin();i != data.end();++i) {
      res += (*i)->ToString();
      if(*i != data.back()) res += ", ";
    } 
    return res + " "; 
  }
  void Apply(DFSAdapter &a) const;

  //add a new expression at the end of the list
  void AddExpr(const BasicExpr *e) { 
    data.push_back(static_cast<BasicExpr*>(e->Clone())); 
  }

  //add a list of expressions at the end
  void AddExprList(const ExprList &arg) {
    for(list<BasicExpr*>::const_iterator i = arg.data.begin();i != arg.data.end();++i) {
      data.push_back(static_cast<BasicExpr*>((*i)->Clone()));
    }
  }
  
  //convert to a list of strings
  list<string> ToStringList() const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
  friend class Alias;
  friend class ExprManager;
  friend class ProgAbs;
  friend class Action;
};

/*********************************************************************/
//identifier expression
/*********************************************************************/
class IdExpr : public BasicExpr
{
 private:
  string id;
 public:
  IdExpr() { assert(false); }
  IdExpr(const string &s,const int pk = -1) : BasicExpr(pk) { id = s; }
  IdExpr(const IdExpr &rhs) { *this = rhs; }
  ~IdExpr() {}
  const IdExpr &operator = (const IdExpr &rhs) {
    pamKey = rhs.pamKey;
    id = rhs.id;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new IdExpr(id,pamKey)); }
  void Cleanup() { assert(false); }

  string ToString() const { return id + " "; }
  void Apply(DFSAdapter &a) const;

  string GetId() const { return id; }

  //AST walkers are made friends
  friend class TokenCollector;
  friend class ExprManager;
  friend class ProcInfo;
};

/*********************************************************************/
//integer constant expression
/*********************************************************************/
class IntConstExpr : public BasicExpr
{
 private:
  long long val;
 public:
  IntConstExpr() { assert(false); }
  IntConstExpr(const long long v,const int pk = -1) : BasicExpr(pk) { val = v; }
  IntConstExpr(const IntConstExpr &rhs) { *this = rhs; }
  ~IntConstExpr() {}
  const IntConstExpr &operator = (const IntConstExpr &rhs) {
    pamKey = rhs.pamKey;
    val = rhs.val;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new IntConstExpr(val,pamKey)); }
  void Cleanup() { assert(false); }

  string ToString() const;
  void Apply(DFSAdapter &a) const;

  friend BasicExpr *SimplifyExpr(BasicExpr *arg);
  friend class ExprManager;
  friend class Predicate;
};

/*********************************************************************/
//constant expression
/*********************************************************************/
class ConstExpr : public BasicExpr
{
 private:
  string id;
 public:
  ConstExpr() { assert(false); }
  ConstExpr(const string &s,const int pk = -1) : BasicExpr(pk) { id = s; }
  ConstExpr(const ConstExpr &rhs) { *this = rhs; }
  ~ConstExpr() {}
  const ConstExpr &operator = (const ConstExpr &rhs) {
    pamKey = rhs.pamKey;
    id = rhs.id;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new ConstExpr(id,pamKey)); }
  void Cleanup() { assert(false); }

  string ToString() const { return id + " "; }
  void Apply(DFSAdapter &a) const;

  friend class ExprManager;
};

/*********************************************************************/
//string literal expression
/*********************************************************************/
class StrExpr : public BasicExpr
{
 private:
  string id;
 public:
  StrExpr() { assert(false); }
  StrExpr(const string &s,const int pk = -1) : BasicExpr(pk) { id = s; }
  StrExpr(const StrExpr &rhs) { *this = rhs; }
  ~StrExpr() {}
  const StrExpr &operator = (const StrExpr &rhs) {
    pamKey = rhs.pamKey;
    id = rhs.id;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new StrExpr(id,pamKey)); }
  void Cleanup() { assert(false); }

  string ToString() const { return id + " "; }
  void Apply(DFSAdapter &a) const;

  friend class ExprManager;
};

/*********************************************************************/
//double string literal expression
/*********************************************************************/
class StrStrExpr : public BasicExpr
{
 private:
  string id1,id2;
 public:
  StrStrExpr() { assert(false); }
  StrStrExpr(const string &s1,const string &s2,const int pk = -1) : BasicExpr(pk) { 
    id1 = s1; 
    id2 = s2;    
  }
  StrStrExpr(const StrStrExpr &rhs) { *this = rhs; }
  ~StrStrExpr() {}
  const StrStrExpr &operator = (const StrStrExpr &rhs) {
    pamKey = rhs.pamKey;
    id1 = rhs.id1;
    id2 = rhs.id2;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new StrStrExpr(id1,id2,pamKey)); }
  void Cleanup() { assert(false); }

  string ToString() const { return id1 + " " + id2 + " "; }
  void Apply(DFSAdapter &a) const;
};

/*********************************************************************/
//compound statement expression
/*********************************************************************/
class StmtExpr : public BasicExpr
{
 private:
  CompStmt *stmt;
 public:
  StmtExpr() { stmt = NULL; }
  StmtExpr(const CompStmt *rhs,const int pk = -1);
  StmtExpr(const StmtExpr &rhs) { stmt = NULL; *this = rhs; }
  ~StmtExpr() { Cleanup(); }
  const StmtExpr &operator = (const StmtExpr &rhs);
  Node *Clone() const { return static_cast<Node*>(new StmtExpr(stmt,pamKey)); }
  void Cleanup();

  string ToString() const;
  void Apply(DFSAdapter &a) const;

  CompStmt *GetStmt() { return stmt; }

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
};

/*********************************************************************/
//array expression expression
/*********************************************************************/
class BrackExpr : public BasicExpr
{
 private:
  BasicExpr *array,*index;
 public:
  BrackExpr() { array = index = NULL; }
  BrackExpr(const BasicExpr *a,const BasicExpr *i,const int pk = -1) : BasicExpr(pk) { 
    array = (a == NULL) ? NULL : static_cast<BasicExpr*>(a->Clone());
    index = (i == NULL) ? NULL : static_cast<BasicExpr*>(i->Clone());
  }
  BrackExpr(const BrackExpr &rhs) { 
    array = index = NULL;
    *this = rhs; 
  }
  ~BrackExpr() { Cleanup(); }
  const BrackExpr &operator = (const BrackExpr &rhs) {
    Cleanup();
    pamKey = rhs.pamKey;
    array = (rhs.array == NULL) ? NULL : static_cast<BasicExpr*>(rhs.array->Clone());
    index = (rhs.index == NULL) ? NULL : static_cast<BasicExpr*>(rhs.index->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new BrackExpr(array,index,pamKey)); }
  void Cleanup() {
    if(array != NULL) delete array;
    array = NULL;
    if(index != NULL) delete index;
    index = NULL;
  }

  string ToString() const { 
    return array->ToString() + "[ " + index->ToString() + "] "; 
  }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend class TokenCollector;
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
  friend class ExprManager;
};

/*********************************************************************/
//procedure call expression
/*********************************************************************/
class ParExpr : public BasicExpr
{
 private:
  BasicExpr *proc;
  ExprList *args;
 public:
  ParExpr() { proc = NULL; args = NULL; }
  ParExpr(const BasicExpr *p,const ExprList *a,const int pk = -1) : BasicExpr(pk) { 
    proc = (p == NULL) ? NULL : static_cast<BasicExpr*>(p->Clone());
    args = (a == NULL) ? NULL : static_cast<ExprList*>(a->Clone());
  }
  ParExpr(const ParExpr &rhs) { 
    proc = NULL;
    args = NULL;
    *this = rhs; 
  }
  ~ParExpr() { Cleanup(); }
  const ParExpr &operator = (const ParExpr &rhs) {
    Cleanup();
    pamKey = rhs.pamKey;
    proc = (rhs.proc == NULL) ? NULL : static_cast<BasicExpr*>(rhs.proc->Clone());
    args = (rhs.args == NULL) ? NULL : static_cast<ExprList*>(rhs.args->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new ParExpr(proc,args,pamKey)); }
  void Cleanup() {
    if(proc != NULL) delete proc;
    proc = NULL;
    if(args != NULL) delete args;
    args = NULL;
  }

  string ToString() const {
    return proc->ToString() + "( " + args->ToString() + ") ";
  }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class CallGraphCreator;
  friend class ContLocExtractor;
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
  friend class ExprManager;
  friend class Util;
};

/*********************************************************************/
//field access expression
/*********************************************************************/
class DotExpr : public BasicExpr
{
 private:
  BasicExpr *expr;
  string id;
 public:
  DotExpr() { expr = NULL; }
  DotExpr(const BasicExpr *e,const string &s,const int pk = -1) : BasicExpr(pk) { 
    expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
    id = s; 
  }
  DotExpr(const DotExpr &rhs) { expr = NULL; *this = rhs; }
  ~DotExpr() { Cleanup(); }
  const DotExpr &operator = (const DotExpr &rhs) {
    Cleanup();
    pamKey = rhs.pamKey;
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    id = rhs.id;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new DotExpr(expr,id,pamKey)); }
  void Cleanup() { if(expr != NULL) delete expr; expr = NULL; }

  string ToString() const { return expr->ToString() + ". " + id + " "; }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend class TokenCollector;
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
  friend class ExprManager;
};

/*********************************************************************/
//field dereference expression
/*********************************************************************/
class ArrowExpr : public BasicExpr
{
 private:
  BasicExpr *expr;
  string id;
 public:
  ArrowExpr() { expr = NULL; }
  ArrowExpr(const BasicExpr *e,const string &s,const int pk = -1) : BasicExpr(pk) { 
    expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
    id = s; 
  }
  ArrowExpr(const ArrowExpr &rhs) { expr = NULL; *this = rhs; }
  ~ArrowExpr() { Cleanup(); }
  const ArrowExpr &operator = (const ArrowExpr &rhs) {
    Cleanup();
    pamKey = rhs.pamKey;
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    id = rhs.id;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new ArrowExpr(expr,id,pamKey)); }
  void Cleanup() { if(expr != NULL) delete expr; expr = NULL; }

  string ToString() const { return expr->ToString() + "-> " + id + " "; }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend class TokenCollector;
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
  friend class ExprManager;
};

/*********************************************************************/
//increment decrement expression
/*********************************************************************/
class IncExpr : public BasicExpr
{
 private:
  BasicExpr *expr;
  short int type;
 public:
  IncExpr() { expr = NULL; type = 0; }
  IncExpr(const BasicExpr *e,const int t,const int pk = -1);
  IncExpr(const IncExpr &rhs) { expr = NULL; *this = rhs; }
  ~IncExpr() { Cleanup(); }
  const IncExpr &operator = (const IncExpr &rhs) {
    Cleanup();
    pamKey = rhs.pamKey;
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    type = rhs.type;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new IncExpr(expr,type,pamKey)); }
  void Cleanup() { if(expr != NULL) delete expr; expr = NULL; }

  string ToString() const;
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend class WPComputer;
};

/*********************************************************************/
//unary expression
/*********************************************************************/
class UnaryExpr : public BasicExpr
{
 private:
  BasicExpr *expr;
  char op;
 public:
  UnaryExpr() { expr = NULL; op = 0; }
  UnaryExpr(const BasicExpr *e,const char o,const int pk = -1) : BasicExpr(pk) {
    assert((o == '&') || (o == '*') || (o == '+') ||
	   (o == '-') || (o == '~') || (o == '!'));
    expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
    op = o;
  }
  UnaryExpr(const UnaryExpr &rhs) { expr = NULL; *this = rhs; }
  ~UnaryExpr() { Cleanup(); }
  const UnaryExpr &operator = (const UnaryExpr &rhs) {
    Cleanup();
    pamKey = rhs.pamKey;
    expr = (rhs.expr == NULL) ? NULL : static_cast<BasicExpr*>(rhs.expr->Clone());
    op = rhs.op;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new UnaryExpr(expr,op,pamKey)); }
  void Cleanup() { if(expr != NULL) delete expr; expr = NULL; }

  string ToString() const { 
    string res = expr->ToString();
    char x[3]; x[0] = op; x[1] = ' '; x[2] = '\0'; 
    return x + res;
  }
  void Apply(DFSAdapter &a) const;

  const BasicExpr *GetExpr() const { return expr;}
  const char GetOp() const { return op; }

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend class TokenCollector;
  friend BasicExpr *SimplifyExpr(BasicExpr *arg);
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
  friend class ImplState;
  friend class ProcInfo;
  friend class ExprManager;
  friend class Util;
  friend class ProcInliner;
};

/*********************************************************************/
//empty expression - just a placeholder
/*********************************************************************/
class EmptyExpr : public BasicExpr
{
 public:
  EmptyExpr(const int pk = -1) : BasicExpr(pk) {}
  EmptyExpr(const EmptyExpr &rhs) { *this = rhs; }
  ~EmptyExpr() {}
  const EmptyExpr &operator = (const EmptyExpr &rhs) {
    pamKey = rhs.pamKey;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new EmptyExpr(pamKey)); }
  void Cleanup() { assert(false); }

  string ToString() const { return "() "; }
  void Apply(DFSAdapter &a) const;
};

/*********************************************************************/
//binary expression
/*********************************************************************/
class BinaryExpr : public BasicExpr
{
 private:
  BasicExpr *lhs;
  BasicExpr *rhs;
  short int op;
 public:
  BinaryExpr() { lhs = rhs = NULL; op = 0; }
  BinaryExpr(const BasicExpr *l,const BasicExpr *r,const short int o,const int pk = -1);
  BinaryExpr(const BinaryExpr &arg) { lhs = rhs = NULL; *this = arg; }
  ~BinaryExpr() { Cleanup(); }
  const BinaryExpr &operator = (const BinaryExpr &arg) {
    Cleanup();
    pamKey = arg.pamKey;
    lhs = (arg.lhs == NULL) ? NULL : static_cast<BasicExpr*>(arg.lhs->Clone());
    rhs = (arg.rhs == NULL) ? NULL : static_cast<BasicExpr*>(arg.rhs->Clone());
    op = arg.op;
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new BinaryExpr(lhs,rhs,op,pamKey)); }
  void Cleanup() { 
    if(lhs != NULL) delete lhs; lhs = NULL;
    if(rhs != NULL) delete rhs; rhs = NULL;
  }

  string ToString() const;
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend BasicExpr *SimplifyExpr(BasicExpr *arg);
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
  friend class ExprManager;
  friend class Predicate;
  friend class CEDagVerifier;
};

/*********************************************************************/
//conditional expression - question mark colon style
/*********************************************************************/
class QuestExpr : public BasicExpr
{
 private:
  BasicExpr *cond;
  BasicExpr *tcase; //if cond is true
  BasicExpr *ecase; //if cond is false
 public:
  QuestExpr() { cond = tcase = ecase = NULL; }
  QuestExpr(const BasicExpr *c,const BasicExpr *t,const BasicExpr *e,const int pk = -1) : BasicExpr(pk) {
    cond = (c == NULL) ? NULL : static_cast<BasicExpr*>(c->Clone());
    tcase = (t == NULL) ? NULL : static_cast<BasicExpr*>(t->Clone());
    ecase = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
  }
  QuestExpr(const QuestExpr &rhs) { cond = tcase = ecase = NULL; *this = rhs; }
  ~QuestExpr() { Cleanup(); }
  const QuestExpr &operator = (const QuestExpr &rhs) {
    Cleanup();
    pamKey = rhs.pamKey;
    cond = (rhs.cond == NULL) ? NULL : static_cast<BasicExpr*>(rhs.cond->Clone());
    tcase = (rhs.tcase == NULL) ? NULL : static_cast<BasicExpr*>(rhs.tcase->Clone());
    ecase = (rhs.ecase == NULL) ? NULL : static_cast<BasicExpr*>(rhs.ecase->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new QuestExpr(cond,tcase,ecase,pamKey)); }
  void Cleanup() { 
    if(cond != NULL) delete cond; cond = NULL;
    if(tcase != NULL) delete tcase; tcase = NULL;
    if(ecase != NULL) delete ecase; ecase = NULL;
  }

  string ToString() const { 
    return cond->ToString() + "? " + tcase->ToString() + ": " + ecase->ToString();
  }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend BasicExpr *SimplifyExpr(BasicExpr *arg);
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
};

/*********************************************************************/
//assignment expression
/*********************************************************************/
class AssignExpr : public BasicExpr
{
 private:
  BasicExpr *lhs;
  BasicExpr *rhs;
 public:
  AssignExpr() { lhs = rhs = NULL; }
  AssignExpr(const BasicExpr *l,const BasicExpr *r,const int pk = -1) : BasicExpr(pk) {
    lhs = (l == NULL) ? NULL : static_cast<BasicExpr*>(l->Clone());
    rhs = (r == NULL) ? NULL : static_cast<BasicExpr*>(r->Clone());
  }
  AssignExpr(const AssignExpr &arg) { lhs = rhs = NULL; *this = arg; }
  ~AssignExpr() { Cleanup(); }
  const AssignExpr &operator = (const AssignExpr &arg) {
    Cleanup();
    pamKey = arg.pamKey;
    lhs = (arg.lhs == NULL) ? NULL : static_cast<BasicExpr*>(arg.lhs->Clone());
    rhs = (arg.rhs == NULL) ? NULL : static_cast<BasicExpr*>(arg.rhs->Clone());
    return *this;
  }
  Node *Clone() const { return static_cast<Node*>(new AssignExpr(lhs,rhs,pamKey)); }
  void Cleanup() { 
    if(lhs != NULL) delete lhs; lhs = NULL;
    if(rhs != NULL) delete rhs; rhs = NULL;
  }

  string ToString() const { 
    return lhs->ToString() + "= " + rhs->ToString();
  }
  void Apply(DFSAdapter &a) const;

  //AST walkers are made friends
  friend class DFSAdapter;
  friend class ContLocExtractor;
  friend class WPComputer;
  friend class SimplifyTP;
  friend class CproverTP;
  friend class CvcTP;
  friend class IcsTP;
  friend class SvcTP;
  friend class CvclTP;
};

} //namespace magic

#endif //__EXPR_H__

/*********************************************************************/
//end of Expr.h
/*********************************************************************/
