/******************************** CPPFile *****************************

* FileName [Expr.cpp]

* PackageName [parser]

* Synopsis [Method definitions of various expression AST node class.]

* SeeAlso [Expr.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <string>
#include <list>
#include <map>
#include <vector>
#include <set>
#include <typeinfo>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "DFSAdapter.h"
#include "Database.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//global shutdown method defined in arc.cpp needed
/*********************************************************************/
__attribute__ ((noreturn)) void GlobalShutdown(int sig);

/*********************************************************************/
//methods for class Expr
/*********************************************************************/
Expr::Expr(const BasicExpr *e)
{
  assert(e != NULL);
  expr = ExprManager::Register(e);
}

const Expr &Expr::operator = (const Expr &rhs)
{
  expr = rhs.expr;
  return *this;
}

bool Expr::operator == (const Expr &rhs) const
{
  return (expr == rhs.expr);
}

bool Expr::operator < (const Expr &rhs) const
{
  return (expr < rhs.expr);
}

string Expr::ToString() const
{
  return expr->ToString();
}

bool Expr::IsEmptyExpr() const
{
  return (typeid(*expr) == typeid(EmptyExpr));
}


bool Expr::ContainsLvalue(const set<string> &arg) const
{
  return expr->ContainsLvalue(arg);
}

bool Expr::IsIdExpr() const
{
  return (typeid(*expr) == typeid(IdExpr));
}

/*********************************************************************/
//methods for class BasicExpr
/*********************************************************************/
BasicExpr *BasicExpr::Negate() const
{
  //quick check - if this expression is of the type !A then return A
  if(typeid(*this) == typeid(UnaryExpr)) {
    const UnaryExpr *a = static_cast<const UnaryExpr*>(this);
    if(a->GetOp() == '!') {
      return static_cast<BasicExpr*>(a->GetExpr()->Clone());
    } else {
      return new UnaryExpr(this,'!');
    }
  } else {
    return new UnaryExpr(this,'!');
  }
}

bool BasicExpr::ContainsLvalue(const set<string> &arg) const
{
  set<string> a = Util::ComputeStrLvalues(this);
  for(set<string>::const_iterator i = arg.begin();i != arg.end();++i) {
    if(a.count(*i) != 0) return true;
  }
  return false;
}

/*********************************************************************/
//methods for class ExprList
/*********************************************************************/
void ExprList::Apply(DFSAdapter &a) const
{ 
  a.caseExprList(*this); 
}
  
//convert to a list of strings
list<string> ExprList::ToStringList() const {
  list<string> res;
  for(list<BasicExpr*>::const_iterator i = data.begin();i != data.end();++i) {
    res.push_back(Util::TrimString((*i)->ToString()));
  }
  return res;
}

/*********************************************************************/
//methods for class IdExpr
/*********************************************************************/
void IdExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseIdExpr(*this); 
}

/*********************************************************************/
//methods for class IntConstExpr
/*********************************************************************/
string IntConstExpr::ToString() const 
{ 
  char x[50];
  snprintf(x,50,"%lld ",val);
  return x;
}

void IntConstExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseIntConstExpr(*this); 
}

/*********************************************************************/
//methods for class ConstExpr
/*********************************************************************/
void ConstExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseConstExpr(*this); 
}

/*********************************************************************/
//methods for class StrExpr
/*********************************************************************/
void StrExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseStrExpr(*this); 
}

/*********************************************************************/
//methods for class StrStrExpr
/*********************************************************************/
void StrStrExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseStrStrExpr(*this); 
}

/*********************************************************************/
//methods for class StmtExpr
/*********************************************************************/
StmtExpr::StmtExpr(const CompStmt *rhs,const int pk) : BasicExpr(pk) 
{ 
  stmt = (rhs == NULL) ? NULL : static_cast<CompStmt*>(rhs->Clone()); 
}

void StmtExpr::Cleanup() { if(stmt != NULL) delete stmt; stmt = NULL; }

const StmtExpr &StmtExpr::operator = (const StmtExpr &rhs) 
{
  Cleanup();
  pamKey = rhs.pamKey;
  stmt = (rhs.stmt == NULL) ? NULL : static_cast<CompStmt*>(rhs.stmt->Clone());
  return *this;
}

string StmtExpr::ToString() const { return stmt->ToString() + " "; }

void StmtExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseStmtExpr(*this); 
}

/*********************************************************************/
//methods for class BrackExpr
/*********************************************************************/
void BrackExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseBrackExpr(*this); 
}

/*********************************************************************/
//methods for class ParExpr
/*********************************************************************/
void ParExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseParExpr(*this); 
}

/*********************************************************************/
//methods for class DotExpr
/*********************************************************************/
void DotExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseDotExpr(*this); 
}

/*********************************************************************/
//methods for class ArrowExpr
/*********************************************************************/
void ArrowExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseArrowExpr(*this); 
}

/*********************************************************************/
//methods for class IncExpr
/*********************************************************************/
IncExpr::IncExpr(const BasicExpr *e,const int t,const int pk) : BasicExpr(pk) 
{
  assert((t == Database::PRE_INC) || (t == Database::PRE_DEC) ||
	 (t == Database::POST_INC) || (t == Database::POST_DEC));
  expr = (e == NULL) ? NULL : static_cast<BasicExpr*>(e->Clone());
  type = t;
}

string IncExpr::ToString() const { 
  string res = expr->ToString();
  if(type == Database::PRE_INC) return "++ " + res;
  else if(type == Database::PRE_DEC) return "-- " + res;
  else if(type == Database::POST_INC) return res + "++ ";
  else if(type == Database::POST_DEC) return res + "-- ";
  else assert(false);
  GlobalShutdown(-1);
}

void IncExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseIncExpr(*this); 
}

/*********************************************************************/
//methods for class UnaryExpr
/*********************************************************************/
void UnaryExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseUnaryExpr(*this); 
}

/*********************************************************************/
//methods for class EmptyExpr
/*********************************************************************/
void EmptyExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseEmptyExpr(*this); 
}

/*********************************************************************/
//methods for class BinaryExpr
/*********************************************************************/
void BinaryExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseBinaryExpr(*this); 
}

BinaryExpr::BinaryExpr(const BasicExpr *l,const BasicExpr *r,const short int o,const int pk) : BasicExpr(pk)
{
  assert((o == '*') || (o == '/') || (o == '%') ||
	 (o == '+') || (o == '-') || (o == MAGIC_LEFT_OP) ||
	 (o == MAGIC_RIGHT_OP) || (o == '<') || (o == '>') ||
	 (o == MAGIC_LE_OP) || (o == MAGIC_GE_OP) || (o == MAGIC_EQ_OP) ||
	 (o == MAGIC_NE_OP) || (o == '&') || (o == '^') ||
	 (o == '|') || (o == MAGIC_AND_OP) || (o == MAGIC_OR_OP) ||
	 (o == ','));
  lhs = (l == NULL) ? NULL : static_cast<BasicExpr*>(l->Clone());
  rhs = (r == NULL) ? NULL : static_cast<BasicExpr*>(r->Clone());
  op = o;
}

string BinaryExpr::ToString() const 
{ 
  string ls = lhs->ToString();
  string rs = rhs->ToString();
  string x;
  if(op == '*') x = "* ";
  else if(op == '/') x = "/ ";
  else if(op == '%') x = "% ";
  else if(op == '+') x = "+ ";
  else if(op == '-') x = "- ";
  else if(op == MAGIC_LEFT_OP) x = "<< ";
  else if(op == MAGIC_RIGHT_OP) x = ">> ";
  else if(op == '<') x = "< ";
  else if(op == '>') x = "> ";
  else if(op == MAGIC_LE_OP) x = "<= ";
  else if(op == MAGIC_GE_OP) x = ">= ";
  else if(op == MAGIC_EQ_OP) x = "== ";
  else if(op == MAGIC_NE_OP) x = "!= ";
  else if(op == '&') x = "& ";
  else if(op == '^') x = "^ ";
  else if(op == '|') x = "| ";
  else if(op == MAGIC_AND_OP) x = "&& ";
  else if(op == MAGIC_OR_OP) x = "|| ";
  else if(op == ',') x = ", ";
  else assert(false);
  return ls + x + rs;
}

/*********************************************************************/
//methods for class QuestExpr
/*********************************************************************/
void QuestExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseQuestExpr(*this); 
}

/*********************************************************************/
//methods for class AssignExpr
/*********************************************************************/
void AssignExpr::Apply(DFSAdapter &a) const 
{ 
  a.caseAssignExpr(*this); 
}

/*********************************************************************/
//end of Expr.cpp
/*********************************************************************/
