/**************************** CPPHeaderFile ***************************

* FileName [ActionManager.h]

* PackageName [parser]

* Synopsis [Header file for ActionManager class.]

* Description [This class encapsulates the action manager.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __ACTIONMANAGER_H__
#define __ACTIONMANAGER_H__

namespace magic {

/*********************************************************************/
//the basic action class
/*********************************************************************/
class BasicAction
{
 private:
  //the type of the action
  int type;

  //the action name if this is a basic action and name of the channel
  //if this is a channel action.
  string name;

  //this is expression represents the return value of a return action
  //and the lhs of an assign action. the convention the
  //non-simulatable default return action has the expression "!" and
  //the void return action has the empty expression.
  Expr expr1;

  //this expression represents the rhs of an assign action. 
  Expr expr2;

  //the list of expressions for arguments of a channel action
  list<Expr> chanArgs;

 public:
  //the action types
  static const int ACTION_BASIC;
  static const int ACTION_RETURN;
  static const int ACTION_ASSIGN;
  static const int ACTION_BCAST;
  static const int ACTION_SEND;
  static const int ACTION_RECV;

  //default constructor
  BasicAction() { assert(false); }
  //for creating a basic action
  BasicAction(const string &n);
  //for creating a return action
  BasicAction(const Expr &e);
  //for creating an assign action
  BasicAction(const Expr &e1,const Expr &e2);
  //for creating a bcast/send/recv action
  BasicAction(const int t,const string &n,const list<Expr> &ca);
  //copy constructor
  BasicAction(const BasicAction &rhs) { *this = rhs; }
  //destructor
  ~BasicAction() {}
  const BasicAction &operator = (const BasicAction &rhs);
  bool operator == (const BasicAction &rhs) const;

  const string &GetName() const { return name; }
  const Expr &GetRetExpr() const { return expr1; }
  const Expr &GetLhsExpr() const { return expr1; }
  const Expr &GetRhsExpr() const { return expr2; }
  const string &GetChannel() const { return name; }
  const list<Expr> &GetMessage() const { return chanArgs; }
  const list<Expr> &GetStore() const { return chanArgs; }

  string ToString() const;

  bool IsBasic() const { return (type == ACTION_BASIC); }
  bool IsReturn() const { return (type == ACTION_RETURN); }
  bool IsAssign() const { return (type == ACTION_ASSIGN); }
  bool IsInvalidPtrDeref() const;
  bool IsAssertionFailure() const;
  bool IsBcast() const { return (type == ACTION_BCAST); }
  bool IsSend() const { return (type == ACTION_SEND); }
  bool IsRecv() const { return (type == ACTION_RECV); }

  bool ReturnActionPossible(const Expr &retExpr,const list<string> &params,
			    const set<Expr> &context) const;
  bool AssignActionPossible(const Expr &lhs,const Expr &rhs,
			    const list<string> &params,const set<Expr> &context) const;
  Expr GetConcreteReturnExpr(const list<Expr> &args) const;
  pair<Expr,Expr> GetConcreteAssignExprs(const list<Expr> &args) const;
};

/*********************************************************************/
//the action manager class
/*********************************************************************/
class ActionManager
{
 private:
  //the set of all actions
  static vector<BasicAction> actions;

  //indices of the epsilon actions. -1 maps to the spec epsilon action
  //and i maps to the epsilon action for the i-th proc.
  static map<int,short> epsilonIndices;

  //index of the invalid pointer dereference
  static short ipdIndex;

  //index of the assertion failure
  static short afIndex;

  //index of the default return
  static short defRetIndex;

 public:
  static short RegisterBasicAction(const string &n);
  static short RegisterReturnAction(const Expr &e);
  static short RegisterAssignAction(const Expr &e1,const Expr &e2);
  static short RegisterChannelAction(const int t,const string &n,const list<Expr> &ca);

  static const vector<BasicAction> &GetActions() { return actions; }
  static const map<int,short> &GetEpsilonIndices() { return epsilonIndices; }
  static short GetDefRetIndex() { return defRetIndex; }
  static Action GetSpecEpsilonAction();
  static Action GetImplEpsilonAction(size_t id);
  static Action GetVoidRetAction();
  static Action GetDefaultRetAction();
  static Action GetInvalidPtrDerefAction();
  static Action GetAssertionFailureAction();
};

} //namespace magic

#endif //__ACTIONMANAGER_H__

/*********************************************************************/
//end of ActionManager.h
/*********************************************************************/
