/******************************** CPPFile *****************************

* FileName [ActionManager.cpp]

* PackageName [parser]

* Synopsis [Method definitions of ActionManager class.]

* SeeAlso [ActionManager.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "Node.h"
#include "Action.h"
#include "ActionManager.h"
#include "Database.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//static members of BasicAction class
/*********************************************************************/
const int BasicAction::ACTION_BASIC = 6310;
const int BasicAction::ACTION_RETURN = 6320;
const int BasicAction::ACTION_ASSIGN = 6330;
const int BasicAction::ACTION_BCAST = 6340;
const int BasicAction::ACTION_SEND = 6350;
const int BasicAction::ACTION_RECV = 6360;

/*********************************************************************/
//methods of BasicAction class
/*********************************************************************/

/*********************************************************************/
//constructors
/*********************************************************************/
//for creating a basic action
BasicAction::BasicAction(const string &n) 
{
  type = ACTION_BASIC;
  name = n;
}

//for creating a return action
BasicAction::BasicAction(const Expr &e) 
{
  assert(!e.IsNull());
  type = ACTION_RETURN;
  expr1 = e;
}

//for creating an assign action
BasicAction::BasicAction(const Expr &e1,const Expr &e2) 
{
  assert((!e1.IsNull()) && (!e2.IsNull()));
  type = ACTION_ASSIGN;
  expr1 = e1;
  expr2 = e2;
}

//for creating a bcast/send/recv action
BasicAction::BasicAction(const int t,const string &n,const list<Expr> &ca)
{
  assert((t == ACTION_BCAST) || (t == ACTION_SEND) || (t == ACTION_RECV));
  type = t;
  name = n;
  chanArgs = ca;
}

/*********************************************************************/
//operators
/*********************************************************************/
const BasicAction &BasicAction::operator = (const BasicAction &rhs) 
{
  type = rhs.type;
  name = rhs.name;
  expr1 = rhs.expr1;
  expr2 = rhs.expr2;
  chanArgs = rhs.chanArgs;
  return *this;
}

bool BasicAction::operator == (const BasicAction &rhs) const
{
  return ((type == rhs.type) && (ToString() == rhs.ToString()));
}

/*********************************************************************/
//convert to string format
/*********************************************************************/
string BasicAction::ToString() const
{
  if(type == ACTION_BASIC) return name;
  else if(type == ACTION_RETURN) return "return { " + expr1.ToString() + "}";
  else if(type == ACTION_ASSIGN) return "{" + expr1.ToString() + "= [ " + expr2.ToString() + "]}";
  else {
    string caStr = "";
    for(list<Expr>::const_iterator i = chanArgs.begin();i != chanArgs.end();) {
      caStr += i->ToString();
      ++i;
      if(i != chanArgs.end()) caStr += ", ";
    }
    if(type == ACTION_BCAST) return name + "!! [ " + caStr + "]";
    else if(type == ACTION_SEND) return name + "! [ " + caStr + "]";
    else if(type == ACTION_RECV) return name + "? [ " + caStr + "]";
    assert(false);
  }
  return "";
}

/*********************************************************************/
//return true if this is an invalid pointer dereference action and
//false otherwise
/*********************************************************************/
bool BasicAction::IsInvalidPtrDeref() const
{
  return ((type == ACTION_BASIC) && (name == Database::INVALID_PTR_DEREF));
}

/*********************************************************************/
//return true if this is an assertion failure action and false
//otherwise
/*********************************************************************/
bool BasicAction::IsAssertionFailure() const
{
  return ((type == ACTION_BASIC) && (name == Database::ASSERTION_FAILURE));
}

/*********************************************************************/
//given return expressions, the parameter list of a procedure and a
//context, return true if the return statement can cause the return
//action and false otherwise
/*********************************************************************/
bool BasicAction::ReturnActionPossible(const Expr &retExpr,const list<string> &params,
				       const set<Expr> &context) const
{
  //sanity check
  assert(type == ACTION_RETURN);
  //check for void return
  if(retExpr.IsEmptyExpr()) return expr1.IsEmptyExpr();
  else if(expr1.IsEmptyExpr()) return false;
  //do the dummy variable replacement
  list<Expr> repList;
  repList.push_back(retExpr);
  for(list<string>::const_iterator i = params.begin();i != params.end();++i) {
    repList.push_back(ExprManager::GetIdExpr(*i));
  }  
  Expr newRet = ExprManager::ReplaceDummyVarsZero(expr1,repList);
  //finally check if the return expression can be equal to the return
  //value
  Expr neg = ExprManager::GetUnaryExpr(newRet,'!');
  set<Expr> elist;
  elist.insert(neg);
  bool res = ExprManager::ProveImplies(context,elist);
  return !res;
}

/*********************************************************************/
//given expressions for the lhs and rhs, the parameter list of a
//procedure and a context, return true if the assignment statement can
//cause the assignment action and false otherwise
/*********************************************************************/
bool BasicAction::AssignActionPossible(const Expr &lhs,const Expr &rhs,
				       const list<string> &params,const set<Expr> &context) const
{
  //sanity check
  assert(type == ACTION_ASSIGN);
  //do the dummy variable replacement
  list<Expr> repList;
  for(list<string>::const_iterator i = params.begin();i != params.end();++i) {
    repList.push_back(ExprManager::GetIdExpr(*i));
  }
  Expr newLhs = ExprManager::ReplaceDummyVarsOne(expr1,repList);
  //first we check if the addresses of the left hand sides can be the
  //same    
  if(!(ExprManager::LvalueCompatible(lhs,newLhs) || ExprManager::LvalueCompatible(newLhs,lhs))) {
    return false;
  }
  Expr lhs1 = ExprManager::GetUnaryExpr(lhs,'&');
  Expr newLhs1 = ExprManager::GetUnaryExpr(newLhs,'&');
  Expr eq1 = ExprManager::GetBinaryExpr(lhs1,newLhs1,MAGIC_NE_OP);
  set<Expr> elist1; elist1.insert(eq1);
  bool res1 = ExprManager::ProveImplies(context,elist1);
  if(res1) {
    return false;
  }
  //if the assigned expression is a wildcard then the rhs can always
  //be equal
  if(rhs.ToString() == "* ") {
    return true;
  }
  //otherwise check if the right hand side satisfies the condition
  repList.push_front(rhs);
  Expr newRhs = ExprManager::ReplaceDummyVarsZero(expr2,repList);
  Expr neg = ExprManager::NegateExpr(newRhs);
  set<Expr> elist2; elist2.insert(neg);
  bool res2 = ExprManager::ProveImplies(context,elist2);
  return !res2;
}

/*********************************************************************/
//return the concrete expression for this action by replacing "$1",
//"$2" etc with the list of arguments provided as argument.
/*********************************************************************/
Expr BasicAction::GetConcreteReturnExpr(const list<Expr> &args) const
{
  //sanity check
  assert(type == ACTION_RETURN);
  //quick check for void returns
  if(expr1.IsEmptyExpr()) return ExprManager::GetEmptyExpr();
  //quick check for wildcard and default returns
  if((expr1.ToString() == "* ") || (expr1.ToString() == "! ")) {
    return expr1;
  }
  //replace the arguments
  return ExprManager::ReplaceDummyVarsZero(expr1,args);
}

/*********************************************************************/
//return the pair of concrete expression for the lhs and rhs of this
//action by replacing "$1", "$2" etc with the list of arguments
//provided as argument.
/*********************************************************************/
pair<Expr,Expr> BasicAction::GetConcreteAssignExprs(const list<Expr> &args) const
{
  //sanity check
  assert(type == ACTION_ASSIGN);
  //replace the arguments
  return pair<Expr,Expr>(ExprManager::ReplaceDummyVarsZero(expr1,args),ExprManager::ReplaceDummyVarsZero(expr2,args));
}

/*********************************************************************/
//static members of ActionManager class
/*********************************************************************/
vector<BasicAction> ActionManager::actions;
map<int,short> ActionManager::epsilonIndices;
short ActionManager::ipdIndex = -1;
short ActionManager::afIndex = -1;
short ActionManager::defRetIndex = -1;

/*********************************************************************/
//methods of ActionManager class
/*********************************************************************/

/*********************************************************************/
//register a basic action and return its index
/*********************************************************************/
short ActionManager::RegisterBasicAction(const string &n)
{
  //make sure the action name does not begin with "epsilon_" or is not
  //the invalid pointer dereference action
  assert(n.find(Database::EPSILON + "_") != 0);
  assert(n != Database::INVALID_PTR_DEREF);

  BasicAction act(n);
  for(size_t i = 0;i < actions.size();++i) {
    if(actions[i] == act) return i;
  }
  short res = actions.size();
  if(n == Database::EPSILON) epsilonIndices[-1] = res;
  actions.push_back(act);
  return res;
}

/*********************************************************************/
//register a return action and return its index
/*********************************************************************/
short ActionManager::RegisterReturnAction(const Expr &e)
{
  BasicAction act(e);
  for(size_t i = 0;i < actions.size();++i) {
    if(actions[i] == act) return i;
  }
  short res = actions.size();
  if(e.ToString() == "! ") defRetIndex = res;
  actions.push_back(act);
  return res;
}

/*********************************************************************/
//register an assign action and return its index
/*********************************************************************/
short ActionManager::RegisterAssignAction(const Expr &e1,const Expr &e2)
{
  BasicAction act(e1,e2);
  for(size_t i = 0;i < actions.size();++i) {
    if(actions[i] == act) return i;
  }
  actions.push_back(act);
  return (actions.size() - 1);
}

/*********************************************************************/
//register a bcast action and return its index
/*********************************************************************/
short ActionManager::RegisterChannelAction(const int t,const string &n,const list<Expr> &ca)
{
  BasicAction act(t,n,ca);
  for(size_t i = 0;i < actions.size();++i) {
    if(actions[i] == act) return i;
  }
  actions.push_back(act);
  return (actions.size() - 1);
}

/*********************************************************************/
//create a specification epsilon action
/*********************************************************************/
Action ActionManager::GetSpecEpsilonAction()
{
  if(epsilonIndices.count(-1) == 0) {
    epsilonIndices[-1] = actions.size();
    actions.push_back(BasicAction(Database::EPSILON));
  }  
  return Action(epsilonIndices[-1] + 1);
}

/*********************************************************************/
//create a implementation epsilon action for i-th proc
/*********************************************************************/
Action ActionManager::GetImplEpsilonAction(size_t id)
{
  if(epsilonIndices.count(id) == 0) {
    epsilonIndices[id] = actions.size();
    char prefix[32];
    snprintf(prefix,32,"P%d::",id);
    actions.push_back(BasicAction(prefix + Database::EPSILON));
  }
  return Action(epsilonIndices[id] + 1);
}

/*********************************************************************/
//create a void return action
/*********************************************************************/
Action ActionManager::GetVoidRetAction()
{
  return Action(ExprManager::GetEmptyExpr());
}

/*********************************************************************/
//create a default return action
/*********************************************************************/
Action ActionManager::GetDefaultRetAction()
{
  if(defRetIndex == -1) {
    defRetIndex = actions.size();
    actions.push_back(BasicAction(ExprManager::GetIdExpr("!")));
  }
  return Action(defRetIndex + 1);
}

/*********************************************************************/
//create an invalid pointer dereference action
/*********************************************************************/
Action ActionManager::GetInvalidPtrDerefAction()
{
  if(ipdIndex == -1) {
    ipdIndex = actions.size();
    actions.push_back(BasicAction(Database::INVALID_PTR_DEREF));
  }
  return Action(ipdIndex + 1);
}

/*********************************************************************/
//create an assertion failure action
/*********************************************************************/
Action ActionManager::GetAssertionFailureAction()
{
  if(afIndex == -1) {
    afIndex = actions.size();
    actions.push_back(BasicAction(Database::ASSERTION_FAILURE));
  }
  return Action(afIndex + 1);
}

/*********************************************************************/
//end of ActionManager.cpp
/*********************************************************************/
