/******************************** CPPFile *****************************

* FileName [Action.cpp]

* PackageName [parser]

* Synopsis [Method definitions of Action class.]

* SeeAlso [Action.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <string>
#include <list>
#include <map>
#include <set>
#include <vector>
using namespace std;

#include "Node.h"
#include "Action.h"
#include "ActionManager.h"
#include "Database.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//define static members
/*********************************************************************/
const int Action::ACTION_NONE = 0;

/*********************************************************************/
//constructors
/*********************************************************************/
//creating from an id
Action::Action(short i)
{
  assert(i > 0);
  index = i;
}

//for creating a basic action
Action::Action(const string &n) 
{
  index = ActionManager::RegisterBasicAction(n) + 1;
}

//for creating a return action
Action::Action(const Expr &e) 
{
  index = ActionManager::RegisterReturnAction(e) + 1;
}

//for creating an assign action
Action::Action(const Expr &e1,const Expr &e2) 
{
  index = ActionManager::RegisterAssignAction(e1,e2) + 1;
}

//for creating a channel action
Action::Action(const int t,const string &n,const list<Expr> &ca)
{
  index = ActionManager::RegisterChannelAction(t,n,ca) + 1;
}

//for creating a channel action
Action::Action(const int t,const string &n,const ExprList &ca)
{
  list<Expr> elist;
  for(list<BasicExpr*>::const_iterator i = ca.data.begin();i != ca.data.end();++i) {
    elist.push_back(Expr(*i));
  }
  index = ActionManager::RegisterChannelAction(t,n,elist) + 1;
}

/*********************************************************************/
//operators
/*********************************************************************/
const Action &Action::operator = (const Action &rhs) 
{
  index = rhs.index;
  return *this;
}

bool Action::operator == (const Action &rhs) const
{
  return (index == rhs.index);
}

//for the less than operator we first compare the types and then the
//string representations
bool Action::operator < (const Action &rhs) const
{
  return (index < rhs.index);
}

/*********************************************************************/
//accessors
/*********************************************************************/
const string &Action::GetName() const
{
  int i = (index < 0) ? (-index) : index;
  return ActionManager::GetActions()[i - 1].GetName();
}

const Expr &Action::GetRetExpr() const
{
  int i = (index < 0) ? (-index) : index;
  return ActionManager::GetActions()[i - 1].GetRetExpr();
}

const Expr &Action::GetLhsExpr() const
{
  int i = (index < 0) ? (-index) : index;
  return ActionManager::GetActions()[i - 1].GetLhsExpr();
}

const Expr &Action::GetRhsExpr() const
{
  int i = (index < 0) ? (-index) : index;
  return ActionManager::GetActions()[i - 1].GetRhsExpr();
}

const string &Action::GetChannel() const
{
  int i = (index < 0) ? (-index) : index;
  return ActionManager::GetActions()[i - 1].GetChannel();
}

const list<Expr> &Action::GetMessage() const
{
  int i = (index < 0) ? (-index) : index;
  return ActionManager::GetActions()[i - 1].GetMessage();
}

const list<Expr> &Action::GetStore() const
{
  int i = (index < 0) ? (-index) : index;
  return ActionManager::GetActions()[i - 1].GetStore();
}

/*********************************************************************/
//convert to string format
/*********************************************************************/
string Action::ToString() const
{
  int i = (index < 0) ? (-index) : index;
  string s = ActionManager::GetActions()[i - 1].ToString();
  return (index < 0) ? ("~" + s) : s;
}

/*********************************************************************/
//returns true if this is an spec epsilon action and false otherwise
/*********************************************************************/
bool Action::IsSpecEpsilon() const
{
  if(index < 0) return false;
  const map<int,short>::const_iterator i = ActionManager::GetEpsilonIndices().find(-1);
  if(i == ActionManager::GetEpsilonIndices().end()) return false;
  else return (index == (i->second + 1));
}

/*********************************************************************/
//returns true if this is a default return action and false otherwise
/*********************************************************************/
bool Action::IsDefaultRet() const
{
  if(index < 0) return false;
  else return (index == (ActionManager::GetDefRetIndex() + 1));
}

/*********************************************************************/
//returns true if this is a basic action and false otherwise
/*********************************************************************/
bool Action::IsBasic() const
{
  if(index < 0) return false;
  else return ActionManager::GetActions()[index - 1].IsBasic();
}

/*********************************************************************/
//returns true if this is a return action and false otherwise
/*********************************************************************/
bool Action::IsReturn() const
{
  if(index < 0) return false;
  else return ActionManager::GetActions()[index - 1].IsReturn();
}

/*********************************************************************/
//returns true if this is an assign action and false otherwise
/*********************************************************************/
bool Action::IsAssign() const
{
  if(index < 0) return false;
  else return ActionManager::GetActions()[index - 1].IsAssign();
}

/*********************************************************************/
//return true if this is an invalid pointer dereference action and
//false otherwise
/*********************************************************************/
bool Action::IsInvalidPtrDeref() const
{
  if(index < 0) return false;
  else return ActionManager::GetActions()[index - 1].IsInvalidPtrDeref();
}

/*********************************************************************/
//return true if this is an assertion failure action and false
//otherwise
/*********************************************************************/
bool Action::IsAssertionFailure() const
{
  if(index < 0) return false;
  else return ActionManager::GetActions()[index - 1].IsAssertionFailure();
}

/*********************************************************************/
//return true if this is a broadcast action and false otherwise
/*********************************************************************/
bool Action::IsBcast() const
{
  if(index < 0) return false;
  else return ActionManager::GetActions()[index - 1].IsBcast();
}

/*********************************************************************/
//return true if this is a send action and false otherwise
/*********************************************************************/
bool Action::IsSend() const
{
  if(index < 0) return false;
  else return ActionManager::GetActions()[index - 1].IsSend();
}

/*********************************************************************/
//return true if this is a receive action and false otherwise
/*********************************************************************/
bool Action::IsRecv() const
{
  if(index < 0) return false;
  else return ActionManager::GetActions()[index - 1].IsRecv();
}

/*********************************************************************/
//given return expressions, the parameter list of a procedure and a
//context, return true if the return statement can cause the return
//action and false otherwise
/*********************************************************************/
bool Action::ReturnActionPossible(const Expr &retExpr,const list<string> &params,
				  const set<Expr> &context) const
{
  assert(index > 0);
  return ActionManager::GetActions()[index - 1].ReturnActionPossible(retExpr,params,context);
}

/*********************************************************************/
//given expressions for the lhs and rhs, the parameter list of a
//procedure and a context, return true if the assignment statement can
//cause the assignment action and false otherwise
/*********************************************************************/
bool Action::AssignActionPossible(const Expr &lhs,const Expr &rhs,
				  const list<string> &params,const set<Expr> &context) const
{
  assert(index > 0);
  return ActionManager::GetActions()[index - 1].AssignActionPossible(lhs,rhs,params,context);
}

/*********************************************************************/
//return the concrete expression for this action by replacing "$1",
//"$2" etc with the list of arguments provided as argument.
/*********************************************************************/
Expr Action::GetConcreteReturnExpr(const list<Expr> &args) const
{
  assert(index > 0);
  return ActionManager::GetActions()[index - 1].GetConcreteReturnExpr(args);
}

/*********************************************************************/
//return the pair of concrete expression for the lhs and rhs of this
//action by replacing "$1", "$2" etc with the list of arguments
//provided as argument.
/*********************************************************************/
pair<Expr,Expr> Action::GetConcreteAssignExprs(const list<Expr> &args) const
{
  assert(index > 0);
  return ActionManager::GetActions()[index - 1].GetConcreteAssignExprs(args);
}

/*********************************************************************/
//return the complement receive action. this must be a send or
//broadcast action.
/*********************************************************************/
Action Action::Complement() const
{
  assert((index < 0) || IsBcast() || IsSend() || IsAssign());
  Action res; res.index = -index;
  return res;
}

/*********************************************************************/
//end of Action.cpp
/*********************************************************************/
