/******************************** CPPFile *****************************

* FileName [PaccProcess.cpp]

* PackageName [pacc]

* Synopsis [Method definitions of PaccProcess class.]

* SeeAlso [PaccAPI.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "Component.h"
#include "PaccAPI.h"
using namespace magic;

/*********************************************************************/
//constructors and destructors
/*********************************************************************/
PaccProcess::PaccProcess() { assert(false); }

PaccProcess::PaccProcess(const size_t i,const string &n) { index = i; name = n; }

PaccProcess::PaccProcess(const PaccProcess &rhs) { *this = rhs; }

/*********************************************************************/
//operators
/*********************************************************************/
const PaccProcess &PaccProcess::operator = (const PaccProcess &rhs)
{
  index = rhs.index;
  name = rhs.name;
  stateNum = rhs.stateNum;
  states = rhs.states;
  init = rhs.init;
  props = rhs.props;
  actions = rhs.actions;
  labels = rhs.labels;
  trans = rhs.trans;
  return *this;
}

bool PaccProcess::operator == (const PaccProcess &rhs) const
{
  return (name == rhs.name);
}

bool PaccProcess::operator < (const PaccProcess &rhs) const
{
  return (name < rhs.name);
}

/*********************************************************************/
//initialise
/*********************************************************************/
bool PaccProcess::Initialise()
{
  FILE *in = fopen("pacc_proc","r");
  if(in == NULL) {
    Util::Error("could not open pacc_proc file to read PACC process ...\n");
  }
  char buf[512];
  //get to the beginning of the process
  while(true) {
    fscanf(in,"%s",buf);
    if(!strcmp(buf,"proc")) {
      fscanf(in,"%s",buf);
      if(!strcmp(buf,name.c_str())) break;
    }
  }
  set<string> istr;
  typedef pair< string,pair<string,string> > Trans;
  set<Trans> tstr;
  map< string,set<string> > lstr; 
  set<string> pstr,astr;
  //get components
  while(true) {
    fscanf(in,"%s",buf);
    if((strlen(buf) > 1) && (buf[0] == '/') && (buf[1] == '/')) {
      fgets(buf,512,in);
      continue;
    }
    if(!strcmp(buf,"end")) break;
    else if(!strcmp(buf,"init")) {
      fscanf(in,"%s",buf);
      istr.insert(string(buf));
    } else if(!strcmp(buf,"label")) {
      fscanf(in,"%s",buf);
      string x(buf);
      fscanf(in,"%s",buf);
      string y(buf);
      lstr[x].insert(y);
    } else if(!strcmp(buf,"prop")) {
      fscanf(in,"%s",buf);
      pstr.insert(string(buf));
    } else if(!strcmp(buf,"action")) {
      fscanf(in,"%s",buf);
      astr.insert(string(buf));
    } else {
      string x(buf);
      fscanf(in,"%s",buf);
      fscanf(in,"%s",buf);
      string y(buf);
      fscanf(in,"%s",buf);
      fscanf(in,"%s",buf);
      string z(buf);
      tstr.insert(Trans(x,pair<string,string>(y,z)));
    }
  }
  //close file
  fclose(in);
  //process initial states
  map<string,int> idMap;
  for(set<string>::const_iterator i = istr.begin();i != istr.end();++i) {
    if(idMap.count(*i) == 0) {
      int x = idMap.size();
      idMap[*i] = x;
      init.insert(x);
    }
  }
  //process propositions
  for(set<string>::const_iterator i = pstr.begin();i != pstr.end();++i) {
    Expr e = ExprManager::GetIdExpr(*i);
    PaccProp p = PaccProp::ToPacc(e);
    props.insert(p);
  }
  //process actions
  for(set<string>::const_iterator i = astr.begin();i != astr.end();++i) {
    PaccAct a = PaccAct::ToPacc(Action(*i));
    actions.insert(a);
  }
  for(set<Trans>::const_iterator i = tstr.begin();i != tstr.end();++i) {
    string x = i->first;
    string y = i->second.first;
    string z = i->second.second;
    if(idMap.count(x) == 0) {
      int a = idMap.size();
      idMap[x] = a;
    }
    if(idMap.count(z) == 0) {
      int a = idMap.size();
      idMap[z] = a;
    }
    PaccAct a = PaccAct::ToPacc(Action(y));
    actions.insert(a);
    trans[idMap[x]][a].insert(idMap[z]);
  }
  for(map< string,set<string> >::const_iterator i = lstr.begin();i != lstr.end();++i) {
    string x = i->first;
    if(idMap.count(x) == 0) {
      int a = idMap.size();
      idMap[x] = a;
    }
    for(set<string>::const_iterator j = i->second.begin();j != i->second.end();++j) {
      Expr e = ExprManager::GetIdExpr(*j);
      PaccProp p = PaccProp::ToPacc(e);
      props.insert(p);
      labels[idMap[x]].insert(p);
    }
  }
  //set stateNum and create state vector
  stateNum = idMap.size();
  states.resize(stateNum);
  for(map<string,int>::const_iterator i = idMap.begin();i != idMap.end();++i) {
    states[i->second] = i->first;
  }
  Display();
  //success
  return true;
}

/*********************************************************************/
//cleanup
/*********************************************************************/
void PaccProcess::Cleanup()
{
  name = "";
  stateNum = 0;
  states.clear();
  init.clear();
  props.clear();
  actions.clear();
  labels.clear();
  trans.clear();
}

/*********************************************************************/
//compute set of initial level-1 abstract states
/*********************************************************************/
void PaccProcess::GetL1AbsInit(L1StSet &res) const 
{ 
  set<L1AbsState*> x;
  for(set<int>::const_iterator i = init.begin();i != init.end();++i) {
    x.insert(new PaccL1AbsState(*i));
  }
  CreateL1StSet(x,res);
  for(set<L1AbsState*>::const_iterator i = x.begin();i != x.end();++i) delete *i;
}

/*********************************************************************/
//return the set of level-2 abstract initial states
/*********************************************************************/
void PaccProcess::GetL2AbsInit(set<L2AbsState> &res) { res = init; }

/*********************************************************************/
//return the set of level-2 abstract initial states from the previous
//iteration.
/*********************************************************************/
void PaccProcess::GetPrevL2AbsInit(set<L2AbsState> &res) { assert(false); }

/*********************************************************************/
//return the number of  level-1 abstract states
/*********************************************************************/
int PaccProcess::GetL1AbsStateNum() { return stateNum; }

/*********************************************************************/
//return the number of level-2 abstract states
/*********************************************************************/
int PaccProcess::GetL2AbsStateNum() { return stateNum; }

/*********************************************************************/
//return the set of successors of a given level-2 abstract state on a
//given action
/*********************************************************************/
void PaccProcess::GetL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res)
{
  PaccAct pact = PaccAct::ToPacc(act);
  map< int,map< PaccAct,set<int> > >::const_iterator i = trans.find(state);
  if(i != trans.end()) {
    map< PaccAct,set<int> >::const_iterator j = i->second.find(pact);
    if(j != i->second.end()) res = j->second;
  } 
}

/*********************************************************************/
//check validity of a level-2 abstract transition
/*********************************************************************/
bool PaccProcess::IsL2AbsTrans(const L2AbsState &from,const Action &act,const L2AbsState &to)
{
  PaccAct pact = PaccAct::ToPacc(act);
  map< int,map< PaccAct,set<int> > >::const_iterator i = trans.find(from);
  if(i == trans.end()) return false;
  map< PaccAct,set<int> >::const_iterator j = i->second.find(pact);
  if(j == i->second.end()) return false;
  return (j->second.count(to) != 0);
}

/*********************************************************************/
//return the set of predecessors of a given level-2 abstract state on
//a given action
/*********************************************************************/
void PaccProcess::GetL2AbsPredsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res) 
{
  PaccAct pact = PaccAct::ToPacc(act);
  for(map< int,map< PaccAct,set<int> > >::const_iterator i = trans.begin();i != trans.end();++i) {
    for(map< PaccAct,set<int> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
      if(j->first == pact) {
	if(j->second.count(state) != 0) res.insert(i->first);	  
      }
    }
  }
}

/*********************************************************************/
//checks if the argument action is a level-2 abstract action
/*********************************************************************/
bool PaccProcess::IsL2AbsAct(const Action &act) const
{ 
  for(set<PaccAct>::const_iterator i = actions.begin();i != actions.end();++i) {
    if(PaccAct::ToMagic(*i) == act) return true;
  }
  return false;
}

/*********************************************************************/
//return the set of level-2 abstract actions of the process
/*********************************************************************/
void PaccProcess::GetL2AbsActs(set<Action> &res) const
{ 
  for(set<PaccAct>::const_iterator i = actions.begin();i != actions.end();++i) {
    res.insert(PaccAct::ToMagic(*i));
  }
}

/*********************************************************************/
//return the set of level-2 abstract actions of the process enabled
//from a given state
/*********************************************************************/
void PaccProcess::GetL2AbsActsOut(const L2AbsState &state,set<Action> &res) const
{ 
  map< int,map< PaccAct,set<int> > >::const_iterator i = trans.find(state);
  if(i != trans.end()) {
    for(map< PaccAct,set<int> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
      res.insert(PaccAct::ToMagic(j->first));
    }
  }
}

/*********************************************************************/
//return the set of level-2 abstract actions of the process labeling
//incoming transitions into a level-2 abstract state
/*********************************************************************/
void PaccProcess::GetL2AbsActsIn(const L2AbsState &state,set<Action> &res) const
{ 
  assert(false);
}

/*********************************************************************/
//get the number of level-2 abstract states from the previous
//iteration
/*********************************************************************/
int PaccProcess::GetPrevL2AbsStateNum() { assert(false); }

/*********************************************************************/
//get the set of level-2 abstract states whose successors have changed
//due to action-guided abstraction refinement
/*********************************************************************/
void PaccProcess::GetL2AbsSuccsChanged(set<L2AbsState> &res) { assert(false); }

/*********************************************************************/
//get the successors of a level-2 abstract state on an action from the
//previous iteration
/*********************************************************************/
void PaccProcess::GetPrevL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res)
{
  assert(false);
}

/*********************************************************************/
//return the set of propositions labeling a level-2 abstract state
/*********************************************************************/
void PaccProcess::GetL2AbsProps(const L2AbsState &state,set<Expr> &res)
{ 
  map< int,set<PaccProp> >::const_iterator i = labels.find(state);
  if(i != labels.end()) {
    for(set<PaccProp>::const_iterator j = i->second.begin();j != i->second.end();++j) {
      res.insert(PaccProp::ToMagic(*j));
    }
  }
}

/*********************************************************************/
//return true if the two sets of level-2 abstract states correspond to
//different sets of level-1 abstract states. virtual method of
//Component.
/*********************************************************************/
bool PaccProcess::DistinctL2AbsStates(const set<L2AbsState> &arg1,const set<L2AbsState> &arg2)
{
  return (arg1 != arg2);
}

/*********************************************************************/
//compute the set of all level-1 abstract states
/*********************************************************************/
void PaccProcess::GetCompleteL1StSet(L1StSet &res) const
{
  res = L1StSet::GetComplete();
}

/*********************************************************************/
//compute intersection of two sets of level-1 abstract states. return
//the result in the first argument.
/*********************************************************************/
void PaccProcess::IntersectL1StSets(L1StSet &lhs,const L1StSet &rhs) const
{
  lhs.Intersect(rhs);
}

/*********************************************************************/
//compute predecessors of a set of level-1 abstract states
/*********************************************************************/
L1StSet PaccProcess::GetL1AbsPredsOnAction(const L1StSet &states,const Action &act) const
{
  PaccAct pact = PaccAct::ToPacc(act);
  set<int> x;
  if(states.GetAll()) {
    for(map< int,map< PaccAct,set<int> > >::const_iterator i = trans.begin();i != trans.end();++i) {
      for(map< PaccAct,set<int> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
	if(j->first == pact) {
	  x.insert(i->first);
	  break;
	}
      }
    }
  } else {
    const set<L1AbsState*> &y = states.GetEle();
    if(!y.empty()) {
      for(map< int,map< PaccAct,set<int> > >::const_iterator i = trans.begin();i != trans.end();++i) {
	for(map< PaccAct,set<int> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
	  if(j->first == pact) {
	    for(set<int>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	      if((*(y.begin()))->SetContains(y,PaccL1AbsState(*k))) {
		x.insert(i->first);
		break;
	      }
	    }
	    break;
	  }
	}
      }
    }
  }
  L1StSet res;
  if(x.size() == static_cast<size_t>(stateNum)) GetCompleteL1StSet(res);
  else {
    set<L1AbsState*> y;
    for(set<int>::const_iterator i = x.begin();i != x.end();++i) {
      y.insert(new PaccL1AbsState(*i));
    }
    res = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
  return res;
}

/*********************************************************************/
//compute successors of a level-1 abstract state
/*********************************************************************/
void PaccProcess::GetL1AbsSuccsOnAction(const L1AbsState &state,const Action &act,L1StSet &res) const
{
  PaccAct pact = PaccAct::ToPacc(act);
  set<int> x;
  for(map< int,map< PaccAct,set<int> > >::const_iterator i = trans.begin();i != trans.end();++i) {
    for(map< PaccAct,set<int> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
      if(j->first == pact) {
	if(static_cast<const PaccL1AbsState&>(state) == PaccL1AbsState(i->first)) {
	  x.insert(j->second.begin(),j->second.end());
	}
	break;
      }
    }
  }
  if(x.size() == static_cast<size_t>(stateNum)) GetCompleteL1StSet(res);
  else {
    set<L1AbsState*> y;
    for(set<int>::const_iterator i = x.begin();i != x.end();++i) {
      y.insert(new PaccL1AbsState(*i));
    }
    res = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
}

/*********************************************************************/
//compute successors of a set of level-1 abstract states
/*********************************************************************/
L1StSet PaccProcess::GetL1AbsSuccsOnAction(const L1StSet &states,const Action &act) const
{
  PaccAct pact = PaccAct::ToPacc(act);
  set<int> x;
  if(states.GetAll()) {
    for(map< int,map< PaccAct,set<int> > >::const_iterator i = trans.begin();i != trans.end();++i) {
      for(map< PaccAct,set<int> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
	if(j->first == pact) {
	  x.insert(j->second.begin(),j->second.end());
	  break;
	}
      }
    }
  } else {
    const set<L1AbsState*> &y = states.GetEle();
    if(!y.empty()) {
      for(map< int,map< PaccAct,set<int> > >::const_iterator i = trans.begin();i != trans.end();++i) {
	for(map< PaccAct,set<int> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
	  if(j->first == pact) {
	    if((*(y.begin()))->SetContains(y,PaccL1AbsState(i->first))) {
	      x.insert(j->second.begin(),j->second.end());
	    }
	    break;
	  }
	}
      }
    }
  }
  L1StSet res;
  if(x.size() == static_cast<size_t>(stateNum)) GetCompleteL1StSet(res);
  else {
    set<L1AbsState*> y;
    for(set<int>::const_iterator i = x.begin();i != x.end();++i) {
      y.insert(new PaccL1AbsState(*i));
    }
    res = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
  return res;
}

/*********************************************************************/
//project a set of level-1 abstract states on a level-2 abstract state
/*********************************************************************/
L1StSet PaccProcess::ProjectL1StSetOnL2AbsState(const L1StSet &one,const L2AbsState &two) const
{  
  set<int> res;
  if(one.GetAll()) {
    res.insert(two);
  } else {
    const set<L1AbsState*> a = one.GetEle();
    if(!a.empty()) {
      if((*(a.begin()))->SetContains(a,PaccL1AbsState(two))) res.insert(two);
    }
  }
  L1StSet x;
  if(res.size() == static_cast<size_t>(stateNum)) GetCompleteL1StSet(x);
  else {
    set<L1AbsState*> y;
    for(set<int>::const_iterator i = res.begin();i != res.end();++i) {
      y.insert(new PaccL1AbsState(*i));
    }
    x = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
  return x;
}

/*********************************************************************/
//create a map from level-2 to level-1 abstract states
/*********************************************************************/
void PaccProcess::CreateL2ToL1AbsStateMap(map<L2AbsState,const L1AbsState*> &res) const
{
  for(int i = 0;i < stateNum;++i) res[i] = new PaccL1AbsState(i);
}

/*********************************************************************/
//pick an element from a set of level-1 abstract states
/*********************************************************************/
L1AbsState *PaccProcess::PickL1AbsState(const L1StSet &arg) const
{  
  if(arg.GetAll()) return new PaccL1AbsState(0);
  else {
    const set<L1AbsState*> &x = arg.GetEle();
    assert(!x.empty());
    return (*(x.begin()))->Clone();
  }
}

/*********************************************************************/
//check if a level-2 abstract state belongs to a set
/*********************************************************************/
bool PaccProcess::L1AbsContains(const L1StSet &arg1,const L1AbsState &arg2) const
{
  if(arg1.GetAll()) return true;
  else {
    const set<L1AbsState*> &x = arg1.GetEle();
    if(x.empty()) return false;
    else return (*(x.begin()))->SetContains(x,arg2);
  }
}

/*********************************************************************/
//return the set of all level-1 abstract states
/*********************************************************************/
void PaccProcess::GetAllL1AbsStates(set<L1AbsState*> &res) const
{
  for(int i = 0;i < stateNum;++i) {
    res.insert(new PaccL1AbsState(i));
  }
}

/*********************************************************************/
//compute the level-2 abstract state corresponding to a level-1
//abstract state
/*********************************************************************/
void PaccProcess::L1ToL2AbsState(const L1AbsState &arg,L2AbsState &res) const
{
  const PaccL1AbsState &x = static_cast<const PaccL1AbsState &>(arg);
  res = x.GetState();
}

/*********************************************************************/
//create a L1StSet object corresponding to a set of level-1 abstract
//states
/*********************************************************************/
void PaccProcess::CreateL1StSet(const set<L1AbsState*> &arg,L1StSet &res) const
{
  if(arg.size() == static_cast<size_t>(stateNum)) GetCompleteL1StSet(res);
  else res = L1StSet(false,arg);
}

/*********************************************************************/
//reset the info for action guided abstraction refinement
/*********************************************************************/
void PaccProcess::ClearL2AbsRefineInfo()
{
  assert(false);
}

/*********************************************************************/
//update the info for action guided abstraction refinement
/*********************************************************************/
bool PaccProcess::UpdateL2AbsRefineInfo(const L2AbsState &l2s,const L1StSet &l1s)
{
  assert(false);
}

/*********************************************************************/
//level-1 abstract state to string
/*********************************************************************/
void PaccProcess::L1AbsStateToString(const L1AbsState &arg,string &res) const
{
  assert(false);
}

/*********************************************************************/
//set of level-1 abstract states to string
/*********************************************************************/
void PaccProcess::L1StSetToString(const L1StSet &arg,string &res) const
{
  assert(false);
}

/*********************************************************************/
//string representation of a level-1 abstract state
/*********************************************************************/
void PaccProcess::L1AbsStateToString(const L1AbsState &curr,const L1AbsState *next,const bool stutter,string &res) const
{ 
  const PaccL1AbsState &x = static_cast<const PaccL1AbsState&>(curr);
  assert(static_cast<size_t>(x.GetState()) < states.size());
  res = stutter ? (states[x.GetState()] + " : STUTTER") : states[x.GetState()];
}

/*********************************************************************/
//get the paths corresponding to a level-1 abstract CE dag
/*********************************************************************/
void PaccProcess::L1AbsDagToPaths(const vector<L1AbsState*> &stateLabel,const vector<Action> &actLabel,const vector< set<int> > &trans,const vector<int> &topo,const set<Action> &absActs,set<ConcPath> &res) const
{
  assert(false);
}

/*********************************************************************/
//get the set of fair loop actions
/*********************************************************************/
void PaccProcess::GetFairLoopActs(set< pair<Action,Action> > &res) const {}

/*********************************************************************/
//return true if the expression is a concrete global variable
/*********************************************************************/
bool PaccProcess::IsConcGlobalVar(const Expr &arg) const { return false; }

/*********************************************************************/
//return the refusal set of a level-2 abstract state
/*********************************************************************/
void PaccProcess::GetL2AbsRefusal(const L2AbsState &arg,set<Action> &res) const
{
  assert(false);
}

/*********************************************************************/
//check if the second argument is the refusal set of the first argument
/*********************************************************************/
bool PaccProcess::IsL1AbsRefusal(const L1AbsState &state,const set<Action> &refusal) const
{
  assert(false);
}
/*********************************************************************/
//output the current model as an FSP process. the name of the initial
//state is INIT_x where x is the component index
/*********************************************************************/
void PaccProcess::OutputFsp() const {}

/*********************************************************************/
//display the process
/*********************************************************************/
void PaccProcess::Display() const
{
  Util::Message(2,"************ begin PACC process ************\n");
  Util::Message(2,"name : " + name + "\n");
  Util::Message(2,"props: ");
  for(set<PaccProp>::const_iterator i = props.begin();i != props.end();++i) {
    Util::Message(2,PaccProp::ToMagic(*i).ToString());
  }
  Util::Message(2,"\n");
  Util::Message(2,"actions: ");
  for(set<PaccAct>::const_iterator i = actions.begin();i != actions.end();++i) {
    Util::Message(2,PaccAct::ToMagic(*i).ToString() + " ");
  }
  Util::Message(2,"\n");
  Util::Message(2,"states :\n");
  for(size_t i = 0;i < states.size();++i) {
    Util::Message(2,"\t" + states[i] + ": [ ");
    map< int,set<PaccProp> >::const_iterator j = labels.find(i);
    if(j != labels.end()) {
      for(set<PaccProp>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	Util::Message(2,PaccProp::ToMagic(*k).ToString());
      }
    }
    Util::Message(2,"]\n");
  }
  Util::Message(2,"init: ");
  for(set<int>::const_iterator i = init.begin();i != init.end();++i) {
    Util::Message(2,states[*i] + " ");
  }
  Util::Message(2,"\n");
  Util::Message(2,"trans:\n");
  for(map< int,map< PaccAct,set<int> > >::const_iterator i = trans.begin();i != trans.end();++i) {
    string x = states[i->first];
    for(map< PaccAct,set<int> >::const_iterator j = i->second.begin();j != i->second.end();++j) {
      string y = PaccAct::ToMagic(j->first).ToString();
      for(set<int>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	string z = states[*k];
	Util::Message(2,"\t" + x + " -> " + y + " -> " + z + "\n");
      }
    }
  }
  Util::Message(2,"************* end PACC process *************\n");
}

/*********************************************************************/
//end of PaccProcess.cpp
/*********************************************************************/
