/**************************** CPPHeaderFile ***************************

* FileName [PaccAPI.h]

* PackageName [pacc]

* Synopsis [Header file for PaccProp, PaccAct and PaccProcess
* classes.]

* Description [This is the API that MAGIC requires in order to
* interface with the PACC project. It contains classes representing
* PACC's actions, propositions and processes.]

* SeeAlso [PaccProp.cpp PaccAct.cpp PaccProcess.cpp ../main/Component.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __PACCAPI_H__
#define __PACCAPI_H__

namespace magic {

//other classes needed  
class Action;  
class Expr;

/*********************************************************************/
//An action of a PACC process. Actions are used to label transitions.
/*********************************************************************/
class PaccAct 
{
 private:
  Action action;

 public:
  PaccAct();
  PaccAct(const PaccAct &rhs);
  ~PaccAct();
  const PaccAct &operator = (const PaccAct &rhs);
  bool operator == (const PaccAct &rhs) const;
  bool operator < (const PaccAct &rhs) const;

  //convert a MAGIC action to a PACC action and vice-versa. we assume
  //a bijection between MAGIC's and PACC's actions.
  static Action ToMagic(const PaccAct &arg);
  static PaccAct ToPacc(const Action &arg);
};

/*********************************************************************/
//A proposition of a PACC process. Propositions label states.
/*********************************************************************/
class PaccProp 
{
 private:
  Expr prop;

 public:
  PaccProp();
  PaccProp(const PaccProp &rhs);
  ~PaccProp();
  const PaccProp &operator = (const PaccProp &rhs);
  bool operator == (const PaccProp &rhs) const;
  bool operator < (const PaccProp &rhs) const;

  //convert a MAGIC proposition to a PACC proposition and vice
  //versa. we assume a bijection between MAGIC's and PACC's
  //propositions.
  static Expr ToMagic(const PaccProp &arg);
  static PaccProp ToPacc(const Expr &arg);
};

/*********************************************************************/
//level-1 abstract state of a PACC process
/*********************************************************************/
class PaccL1AbsState : public L1AbsState
{
 private:
  int state;

  void ConvertSet(const set<L1AbsState*> &arg,set<PaccL1AbsState> &res) const {
    for(set<L1AbsState*>::const_iterator i = arg.begin();i != arg.end();++i) {
      res.insert(static_cast<PaccL1AbsState&>(**i));
    }
  }
  void ConvertVec(const vector<L1AbsState*> &arg,vector<PaccL1AbsState> &res) const {
    for(vector<L1AbsState*>::const_iterator i = arg.begin();i != arg.end();++i) {
      res.push_back(static_cast<PaccL1AbsState&>(**i));
    }
  }

 public:
  PaccL1AbsState() { assert(false); }
  PaccL1AbsState(const int s) : state(s) {}
  PaccL1AbsState(const PaccL1AbsState &rhs) { *this = rhs; }
  ~PaccL1AbsState() { Cleanup(); }
  const PaccL1AbsState &operator = (const PaccL1AbsState &rhs) {
    Cleanup();
    state = rhs.state;
    return *this;
  }
  bool operator == (const PaccL1AbsState &rhs) const {
    return (state == rhs.state);
  }
  bool operator < (const PaccL1AbsState &rhs) const {
    return (state < rhs.state);
  }

  int GetState() const { return state; }

  L1AbsState *Clone() const {
    return new PaccL1AbsState(state);
  }
  void Cleanup() {}
  bool SetEq(const set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const {
    set<PaccL1AbsState> one; ConvertSet(lhs,one);
    set<PaccL1AbsState> two; ConvertSet(rhs,two);
    return (one == two);    
  }
  bool SetLt(const set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const {
    set<PaccL1AbsState> one; ConvertSet(lhs,one);
    set<PaccL1AbsState> two; ConvertSet(rhs,two);
    return (one < two);    
  }
  bool SetContains(const set<L1AbsState*> &lhs,const L1AbsState &rhs) const {  
    set<PaccL1AbsState> one; ConvertSet(lhs,one);
    return (one.count(static_cast<const PaccL1AbsState&>(rhs)) != 0);
  }
  void SetIntersect(set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const {
    set<PaccL1AbsState> one; ConvertSet(rhs,one);
    set<L1AbsState*> newLhs;
    for(set<L1AbsState*>::const_iterator i = lhs.begin();i != lhs.end();++i) {
      if(one.count(static_cast<const PaccL1AbsState&>(**i)) == 0) delete *i;
      else newLhs.insert(*i);
    }
    lhs = newLhs;
  }
  bool VecEq(const vector<L1AbsState*> &lhs,const vector<L1AbsState*> &rhs) const {
    vector<PaccL1AbsState> one; ConvertVec(lhs,one);
    vector<PaccL1AbsState> two; ConvertVec(rhs,two);
    return (one == two);    
  }
  bool VecLt(const vector<L1AbsState*> &lhs,const vector<L1AbsState*> &rhs) const {
    vector<PaccL1AbsState> one; ConvertVec(lhs,one);
    vector<PaccL1AbsState> two; ConvertVec(rhs,two);
    return (one < two);    
  }
};

/*********************************************************************/
//The PaccProcess class encapsulates a PACC system that has been
//compiled from a CCL description. It is essentially a finite labeled
//Kripke structure. MAGIC's model checker uses an integer as the
//representation of a state. the integers must range from 0 to (N - 1)
//where N is the total number of states. the PACC interface will also
//use integers as a state's representation.
/*********************************************************************/

class  PaccProcess : public Component
{
 private:
  int stateNum;
  vector<string> states;
  set<int> init;
  set<PaccProp> props;
  set<PaccAct> actions;
  map< int,set<PaccProp> > labels;
  map< int,map< PaccAct,set<int> > > trans;

 public:
  PaccProcess();
  PaccProcess(const size_t i,const string &n);
  PaccProcess(const PaccProcess &rhs);
  virtual ~PaccProcess() {}
  const PaccProcess &operator = (const PaccProcess &rhs);
  bool operator == (const PaccProcess &rhs) const;
  bool operator < (const PaccProcess &rhs) const;

  //Initialise the system from a CCL component with same name. We
  //assume that every CCL component can be identified uniquely by a
  //name. Return true on success and false otherwise.
  bool Initialise();

  //Cleanup the system. 
  void Cleanup();

  void GetL1AbsInit(L1StSet &res) const;

  //get the set of level-2 abstract initial states
  void GetL2AbsInit(set<L2AbsState> &res);

  //get the set of level-2 abstract initial states from the previous
  //iteration
  void GetPrevL2AbsInit(set<L2AbsState> &res);

  //get the number of level-1 abstract states
  int GetL1AbsStateNum();

  //get the number of level-2 abstract states
  int GetL2AbsStateNum();

  //returns the set of successors of a level-2 abstract state on a
  //given action
  void GetL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res);

  //check validity of a level-2 abstract transition
  bool IsL2AbsTrans(const L2AbsState &from,const Action &act,const L2AbsState &to);

  //returns the set of predecessors of a level-2 abstract state on a
  //given action
  void GetL2AbsPredsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res);

  //checks if the argument action is a level-2 abstract action
  bool IsL2AbsAct(const Action &act) const;

  //get the set of level-2 abstract actions - either all actions or
  //only actions enabled from a given state or only incoming actions
  //into a given state
  void GetL2AbsActs(set<Action> &res) const;
  void GetL2AbsActsOut(const L2AbsState &state,set<Action> &res) const;
  void GetL2AbsActsIn(const L2AbsState &state,set<Action> &res) const;

  //get the number of level-2 abstract states from the previous
  //iteration
  int GetPrevL2AbsStateNum();

  //get the set of level-2 abstract states whose successors have
  //changed due to action-guided abstraction refinement
  void GetL2AbsSuccsChanged(set<L2AbsState> &res);

  //get the successors of a level-2 abstract state on an action from
  //the previous iteration
  void GetPrevL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res);

  //get the set of propositions labeling a level-2 abstract state
  void GetL2AbsProps(const L2AbsState &state,set<Expr> &res);

  bool DistinctL2AbsStates(const set<L2AbsState> &arg1,const set<L2AbsState> &arg2);

  //return the complete set of all level-1 abstract states
  void GetCompleteL1StSet(L1StSet &res) const;

  //return the intersection of two sets of level-1 abstract
  //states. change the first set to return the result.
  void IntersectL1StSets(L1StSet &lhs,const L1StSet &rhs) const;

  //return the set of predecessors of a set of level-1 abstract states
  L1StSet GetL1AbsPredsOnAction(const L1StSet &states,const Action &act) const;

  //return the successors of a level-1 abstract state
  void GetL1AbsSuccsOnAction(const L1AbsState &state,const Action &act,L1StSet &res) const;
  L1StSet GetL1AbsSuccsOnAction(const L1StSet &states,const Action &act) const;

  //project a set of level-1 abstract states on a level-2 abstract state
  L1StSet ProjectL1StSetOnL2AbsState(const L1StSet &one,const L2AbsState &two) const;

  //compute the map from level-2 to level-1 abstract states. this only
  //makes sense when we are using no action-guided abstraction.
  void CreateL2ToL1AbsStateMap(map<L2AbsState,const L1AbsState*> &res) const;

  //pick an arbitrary level-1 abstract state from a set
  L1AbsState *PickL1AbsState(const L1StSet &arg) const;
  
  //check if a set of level-1 abstract states contains a level-1
  //abstract state
  bool L1AbsContains(const L1StSet &arg1,const L1AbsState &arg2) const;

  //get the set of all level-1 abstract states
  void GetAllL1AbsStates(set<L1AbsState*> &res) const;

  //get the level-2 abstract state corresponding to a level-1 abstract state
  void L1ToL2AbsState(const L1AbsState &arg,L2AbsState &res) const;

  //other virtual methods of base class Component
  void CreateL1StSet(const set<L1AbsState*> &arg,L1StSet &res) const;
  void ClearL2AbsRefineInfo();
  bool UpdateL2AbsRefineInfo(const L2AbsState &l2s,const L1StSet &l1s);
  void L1AbsStateToString(const L1AbsState &arg,string &res) const;
  void L1StSetToString(const L1StSet &arg,string &res) const;
  void L1AbsStateToString(const L1AbsState &curr,const L1AbsState *next,const bool stutter,string &res) const;
  void L1AbsDagToPaths(const vector<L1AbsState*> &stateLabel,const vector<Action> &actLabel,const vector< set<int> > &trans,const vector<int> &topo,const set<Action> &absActs,set<ConcPath> &res) const;
  void GetFairLoopActs(set< pair<Action,Action> > &res) const;
  bool IsConcGlobalVar(const Expr &arg) const;
  void GetL2AbsRefusal(const L2AbsState &arg,set<Action> &res) const;
  bool IsL1AbsRefusal(const L1AbsState &state,const set<Action> &refusal) const;
  void OutputFsp() const;

  //display the process
  void Display() const;
};

} //namespace magic

#endif //__PACCAPI_H__

/*********************************************************************/
// End of PaccAPI.h
/*********************************************************************/
