/******************************** CPPFile *****************************

* FileName [magic.cpp]

* PackageName [main]

* Synopsis [The file containing the main procedure.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <signal.h>
#include <time.h>
#include <gmp.h>
#include <string>
#include <list>
#include <map>
#include <set>
#include <vector>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Timer.h"
#include "Statistics.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "TheoremProver.h"
#include "SimChecker.h"
#include "InputProcessor.h"
#include "ModelExtractor.h"
#include "ProofGen.h"
#include "CEGen.h"
#include "ConcCEDag.h"
#include "AbsRefiner.h"
#include "PredAbsRefiner.h"
#include "LocalAbsLts.h"
#include "ContLoc.h"
#include "Component.h"
#include "ProcManager.h"
#include "GlobalAbsLts.h"
#ifdef MAGIC_FULL
#include "PamAPI.h"
#include "PaccAPI.h"
using namespace ctool;
#endif //MAGIC_FULL
#include "LtlChecker.h"
#include "DeadlockChecker.h"
#include "SeawChecker.h"
#include "NFA.h"
using namespace magic;

/*********************************************************************/
//procedure declarations
/*********************************************************************/
void InstallHandlers();
__attribute__ ((noreturn)) void GlobalShutdown(int sig);
void CegarC(bool &res);
#ifdef MAGIC_FULL
void CegarPacc(bool &res);
#endif //MAGIC_FULL
bool CegarCore(bool &res);
void Backup();

namespace magic {
  void LearnAngluin(NFA*, NFA*);
}

/*********************************************************************/
//this is where it all begins
/*********************************************************************/
int main(int argc,char *argv[])
{
  //install the signal handlers
  InstallHandlers();

  //start the global timer
  Statistics::globalTimer.Start();

  //set random number generator seed
  srand(time(NULL));

  //copy the command line to the database and also print the command
  //line arguments
  Util::Message(2,"MAGIC version %s ...\n",Database::MAGIC_VERSION.c_str());
  Util::Message(2,"Command Line: %s ",argv[0]);
  for(int i = 1;i < argc;++i) {
    Util::Message(2,"%s ",argv[i]);
    Database::commandLine.push_back(argv[i]);
  }
  Util::Message(2,"\n");

  //run the input processor
  Statistics::ipTimer.Start();
  bool ipRes = InputProcessor::Run();
  Statistics::ipTimer.Stop();
  Util::Message(2,"inputs processed ...\n");

  //if there is any abstraction to be verified then proceed with the
  //abstract-verify-refine cycle
  bool cegarRes = false;
  if(ipRes) {
    if(Database::CONF_TYPE == Database::CONF_SIMUL) {
      Util::Message(2,"checking " + Database::specLtsName + " {simul} " + Database::progName + "\n");
    } else if(Database::CONF_TYPE == Database::CONF_REACH) {
      Util::Message(2,"checking " + Database::specLtsName + " {reach} " + Database::progName + "\n");
    } else if(Database::CONF_TYPE == Database::CONF_LTL) {
      Util::Message(2,"checking " + Database::specLtsName + " {LTL} " + Database::progName + "\n");
    } else if(Database::CONF_TYPE == Database::CONF_DLOCK) {
      Util::Message(2,"checking " + Database::specLtsName + " {Deadlock} " + Database::progName + "\n");
    } else if(Database::CONF_TYPE == Database::CONF_SEAW) {
      Util::Message(2,"checking " + Database::specLtsName + " {SE-AW} " + Database::progName + "\n");
    } else if(Database::CONF_TYPE == Database::CONF_ANGLUIN) {
      Util::Message(2,"checking " + Database::specLtsName + " {substitutability} " + Database::progName + "\n");
    } else assert(false);

    //if the implementation is expressed in C
    if(Database::IMPL_TYPE == Database::IMPL_C) CegarC(cegarRes);
#ifdef MAGIC_FULL
    //if the implementation is expressed in PACC
    else if(Database::IMPL_TYPE == Database::IMPL_PACC) CegarPacc(cegarRes);
#endif //MAGIC_FULL
    //impossible
    else assert(false);
  } 
  //all done
  if(Database::ABSTRACTION_NAME == "") {
    Util::Message(1,"no abstraction specified ...\n");
  } else if(ipRes) {
    if(cegarRes) {
      Util::Message(1,"abstraction " + Database::ABSTRACTION_NAME + " is valid ...\n");
    } else {
      Util::Message(1,"abstraction " + Database::ABSTRACTION_NAME + " is invalid ...\n");
    }
  } else {
    Util::Message(1,"abstraction " + Database::ABSTRACTION_NAME + " not found ...\n");
  }
  //cleanup
  GlobalShutdown(-1);

  return 0;
}

/*********************************************************************/
//install signal handlers
/*********************************************************************/
void InstallHandlers()
{
#ifdef WIN32
  signal(SIGABRT,GlobalShutdown);
  signal(SIGFPE,GlobalShutdown);
  signal(SIGILL,GlobalShutdown);
  signal(SIGINT,GlobalShutdown);
  signal(SIGSEGV,GlobalShutdown);
  signal(SIGTERM,GlobalShutdown);
#else
  struct sigaction sa;
  sa.sa_handler = GlobalShutdown;
  sigaction(SIGINT,&sa,NULL);
  sigaction(SIGSEGV,&sa,NULL);
  sigaction(SIGILL,&sa,NULL);
  sigaction(SIGFPE,&sa,NULL);
  sigaction(SIGBUS,&sa,NULL);
  sigaction(SIGTERM,&sa,NULL);
  sigaction(SIGABRT,&sa,NULL);
  sigaction(SIGSYS,&sa,NULL);
#endif //WIN32
}

/*********************************************************************/
//cleanup
/*********************************************************************/
__attribute__ ((noreturn)) void GlobalShutdown(int sig)
{
  //stop all running timers
  Statistics::ShutdownTimers();
  //display statistics
  Statistics::Display();
  //shutdown the theorem prover
  TheoremProver::Shutdown();

#ifdef MAGIC_FULL
  //cleanup the PAM interface
  PamAPI::Cleanup();
#endif //MAGIC_FULL

  //close the output dot file if any
  if(Database::dotFilePtr != NULL) {
    fprintf(Database::dotFilePtr,"}\n");
    fclose(Database::dotFilePtr);
    string command = "dot -Tps " + Database::DOT_FILE_NAME + " -o " + Database::drawFileName;
    system(command.c_str());
    remove(Database::DOT_FILE_NAME.c_str());
  }
  //close the output FSP file if any
  if(Database::modelFilePtr != NULL) {
    fclose(Database::modelFilePtr);
  }
  //backup state if this is a clean shutdown
  if(Database::BACKUP && (sig == -1)) {
    Backup();
  }
  //cleanup the components
  for(vector<Component*>::const_iterator i = Database::components.begin();i != Database::components.end();++i) {
    (*i)->Cleanup();
    delete (*i);
  }
  Database::components.clear();
  //display a message
  if(sig == -1) {
    Util::Message(0,"terminating normally ...\n");
  } else {
    Util::Message(0,"received signal %d ...\n",sig);
  }
  //all done
  exit(0);
}

/*********************************************************************/
//do CEGAR for C programs
/*********************************************************************/
void CegarC(bool &res)
{
  //preoptimize
  if (Database::PREOPTIMIZE) {
    for(size_t i = 0;i < Database::components.size();++i) {
      (PredAbsRefiner(*(static_cast<ProcManager*>(Database::components[i])))).PreOptimize();
    }
  }
  //if we are checking for the invalid pointer dereferences or
  //assertion failures
  if(Database::CHECK_INVALID_PTR_DEREF || Database::CHECK_ASSERTION_FAILURE) {
    int iterNum = 0;
    for(vector<Component*>::iterator it1 = Database::components.begin();(iterNum < Database::END_ITER) && (it1 != Database::components.end());++it1) {
      ProcManager *pmgr = static_cast<ProcManager*>(*it1);
      for(vector<ContLoc*>::const_iterator it2 = pmgr->GetAllLocs().begin();(iterNum < Database::END_ITER) && (it2 != pmgr->GetAllLocs().end());++it2) {
	//if checking invalid pointer dereference
	if(Database::CHECK_INVALID_PTR_DEREF) {
	  set<Expr> deref = (*it2)->GetDerefPtrExprs();
	  for(set<Expr>::const_iterator it3 = deref.begin();(iterNum < Database::END_ITER) && (it3 != deref.end());++it3,++iterNum) {
	    //reset seed branches
	    for(vector<Component*>::iterator i = Database::components.begin();i != Database::components.end();++i) {
	      static_cast<ProcManager*>(*i)->Cleanup();
	      static_cast<ProcManager*>(*i)->SetSeedBranches(set<ContLoc*>());
	    }
	    //set the location and expression to be checked for invalid
	    //pointer access
	    pmgr->SetMarkedLoc(*it2);
	    (*it2)->SetIpdExpr(*it3);
	    Util::Message(2,"valid deref check #%d of %s at %s\n",iterNum + 1,it3->ToString().c_str(),(*it2)->ToString().c_str());
	    //the main CEGAR loop
	    while(((iterNum + 1) >= Database::START_ITER) && CegarCore(res)) {}
	  }
	}
	//else we are checking for assertion failures
	else {
	  //if this is a call-site
	  if((*it2)->GetLocType() == ContLoc::ASSIGN_CALL) {
	    //if this is a call to assert
	    pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails((*it2)->GetLocRhs(),comps);
	    string procName = Util::TrimString(comps.first.ToString());
	    if(procName == Database::ASSERT_C_PROC) {
	      for(vector<Component*>::iterator i = Database::components.begin();i != Database::components.end();++i) {
		static_cast<ProcManager*>(*i)->Cleanup();
		static_cast<ProcManager*>(*i)->SetSeedBranches(set<ContLoc*>());
	      }
	      //set the location and expression to be checked for invalid
	      //pointer access
	      pmgr->SetMarkedLoc(*it2);
	      Util::Message(2,"assertion failure check #%d at %s\n",iterNum + 1,(*it2)->ToString().c_str());
	      //the main CEGAR loop
	      while(((iterNum + 1) >= Database::START_ITER) && CegarCore(res)) {}
	      ++iterNum;
	    }
	  }
	}
      }
    }
  }
  //otherwise we are doing standard cegar
  else { while(CegarCore(res)) {} }
}

/*********************************************************************/
//do CEGAR for PACC processes
/*********************************************************************/
#ifdef MAGIC_FULL
void CegarPacc(bool &res)
{
  //the main CEGAR loop
  while(CegarCore(res)) {}
}
#endif //MAGIC_FULL

/*********************************************************************/
//the core CEGAR loop
/*********************************************************************/
bool CegarCore(bool &res)
{
  Util::Message(2,"starting iteration number %d ...\n",++(Statistics::iterNum));	  
  //update iteration numbers and seed branches
  if(Database::IMPL_TYPE == Database::IMPL_C) {
    //update the number of predicate or lts iterations
    if((Statistics::predIterNum == 0) && (Statistics::ltsIterNum == 0)) {
      ++Statistics::predIterNum;
    } else {
      bool predIter = false;
      for(size_t i = 0;i < Database::components.size();++i) {
	if(static_cast<ProcManager*>(Database::components[i])->GetNewSeeds()) {
	  ++Statistics::predIterNum;
	  predIter = true;
	  break;
	}
      }
      if(!predIter) ++Statistics::ltsIterNum;
    }	  
    //update the number of seed branches
    Statistics::seedNum = 0;
    for(size_t i = 0;i < Database::components.size();++i) {
      Statistics::seedNum += static_cast<ProcManager*>(Database::components[i])->GetSeedBranches().size();
    }	  
  }
  //extract the model
  Timer imeTimer; imeTimer.Start();
  bool meRes = ModelExtractor::Run(false);
  imeTimer.Stop(); Statistics::imeTimer.Forward(imeTimer);
  Util::Message(2,"implementation machine extracted in %.1f milliseconds ...\n",imeTimer.Read() * 1000);
  if(!meRes) Util::Error("error while extracting model ...\n");	  
  //update the maximum number of global states
  Statistics::stateNum = (Statistics::stateNum < GlobalAbsLts::GetStateNum()) ? GlobalAbsLts::GetStateNum() : Statistics::stateNum;	  
  Util::Message(2,"global states : ( ");
  for(size_t i = 0;i < Database::components.size();++i) {
    Util::Message(2,"%d ", Database::components[i]->GetL2AbsStateNum());
  }
  Util::Message(2,") = %s\n",GlobalAbsLts::GetStateNum().ToString().c_str());
  //check if time limit has been exceeded
  Statistics::globalTimer.Stop();
  if(Statistics::globalTimer.Read() > Database::MAX_TIME_LIMIT) {
    Util::Error("time limit exceeded ...\n");
  }
  Statistics::globalTimer.Start();
  //verify simulation
  if(Database::CONF_TYPE == Database::CONF_SIMUL) {
    Timer verTimer; verTimer.Start();
    res = SimChecker::CheckSimulationProperty();
    verTimer.Stop(); Statistics::verTimer.Forward(verTimer);
    Util::Message(2,"simulation checked in %.1f milliseconds ...\n",verTimer.Read() * 1000);
  }
  //verify error state reachability
  else if(Database::CONF_TYPE == Database::CONF_REACH) {
    Timer verTimer; verTimer.Start();
    res = SimChecker::CheckReachability();
    verTimer.Stop(); Statistics::verTimer.Forward(verTimer);
    Util::Message(2,"reachability checked in %.1f milliseconds ...\n",verTimer.Read() * 1000);
  }
  //verify LTL formula
  else if(Database::CONF_TYPE == Database::CONF_LTL) {
    Timer verTimer; verTimer.Start();
    res = LtlChecker::CheckFormula();
    verTimer.Stop(); Statistics::verTimer.Forward(verTimer);
    Util::Message(2,"LTL Formula checked in %.1f milliseconds ...\n",verTimer.Read() * 1000);
  }
  //verify deadlock freedom
  else if(Database::CONF_TYPE == Database::CONF_DLOCK) {
    Timer verTimer; verTimer.Start();
    res = DeadlockChecker::CheckDeadlock();
    verTimer.Stop(); Statistics::verTimer.Forward(verTimer);
    Util::Message(2,"Deadlock freedom checked in %.1f milliseconds ...\n",verTimer.Read() * 1000);
  }
  //verify SE-AW formula
  else if(Database::CONF_TYPE == Database::CONF_SEAW) {
    Timer verTimer; verTimer.Start();
    res = SeawChecker::CheckSeaw();
    verTimer.Stop(); Statistics::verTimer.Forward(verTimer);
    Util::Message(2,"SE-AW formula checked in %.1f milliseconds ...\n",verTimer.Read() * 1000);
  }
  //component-substitutability learning
  else if(Database::CONF_TYPE == Database::CONF_ANGLUIN) {
    Timer verTimer; verTimer.Start();
    vector<Component*>::iterator i = Database::components.begin();
    NFA* toLearn = new NFA();
    NFA* start = NULL;
    static_cast<ProcManager*>(*i)->GetNFA(toLearn);
    cout << "toLearn aut is\n"; toLearn->show();
    i++;
    if( i != Database::components.end() ) {
      start = new NFA();
      static_cast<ProcManager*>(*i)->GetNFA(start);
      cout<<"start aut is"<<endl; start->show();
    }
    LearnAngluin(start,toLearn);
    verTimer.Stop(); Statistics::verTimer.Forward(verTimer);
    Util::Message(2,"Learning completed in %.1f milliseconds ...\n",verTimer.Read() * 1000);
  } 
  //impossible
  else assert(false);	  
  //if the abstraction holds generate the proof and exit
  if(Database::CONF_TYPE == Database::CONF_ANGLUIN) return false;
  
  if(res) {
    ofstream ceFile("ce.out");
    ceFile<<"1";
    ceFile.close();
    Util::Message(1,"conformance relation exists !!\n");
    ProofGen::Run();
    return false;
  }
  //if the abstraction does not hold, generate counter-example,
  //verify it and refine if necessary.
  else {
    if(!Database::CEGAR) {
      CEGen::Run();
      Util::Message(1,"conformance relation does not exist !!\n");
      if(Database::CONF_CHECK_MECHANISM == Database::CONF_CHECK_LTSA) {
	fprintf(stderr,"COUNTEREXAMPLE_VALID\n");
      }
      return false;
    } else {
      //refine the abstraction on the basis of the
      //counter-example. if the counter example is valid the
      //simulation does not hold. otherwise we repeat the loop if
      //new predicates were discovered.
      Timer arTimer; arTimer.Start();
      int arRes = AbsRefiner::Run();
      arTimer.Stop(); Statistics::arTimer.Forward(arTimer);
      if(arRes == AbsRefiner::CE_VALID) {
	Util::Message(1,"conformance relation does not exist !!\n");
	if(Database::CONF_CHECK_MECHANISM == Database::CONF_CHECK_LTSA) {
	  system("echo VALID_CE > magic.out");
	  fprintf(stderr,"COUNTEREXAMPLE_VALID\n");
	}
	return false;
      } else {
	Util::Message(1,"spurious counter-example detected !!\n");
	if(arRes == AbsRefiner::CE_INVALID_NOT_REFINED) {
	  Util::Message(1,"abstraction not refined !!\n");
	  return false;
	} else {
	  assert(arRes == AbsRefiner::CE_INVALID_REFINED);
	  Util::Message(1,"abstraction refined !!\n");
	  return true;
	}
      }
    }
  }
}

/*********************************************************************/
//backup state
/*********************************************************************/
void Backup()
{
  //open the backup file
  FILE *out = fopen(Database::backupFileName.c_str(),"w");
  assert(out != NULL);
  char indent[] = "    ";
  //save options
  fprintf(out,"options {\n");  
  fprintf(out,"%sABSTRACTION_NAME = %s\n",indent,Database::ABSTRACTION_NAME.c_str());
  fprintf(out,"%sSIM_REL_FILE = %s\n",indent,Database::SIM_REL_FILE.c_str());
  fprintf(out,"%sPARSE_ONLY = %s\n",indent,Database::PARSE_ONLY ? "true" : "false");
  fprintf(out,"%sPARSE_ECHO = %s\n",indent,Database::PARSE_ECHO ? "true" : "false");
  fprintf(out,"%sDISPLAY_STATISTICS = %s\n",indent,Database::DISPLAY_STATISTICS ? "true" : "false");

  if(Database::IMPL_TYPE == Database::IMPL_C) fprintf(out,"%sIMPL_TYPE = IMPL_C\n",indent);
  else if(Database::IMPL_TYPE == Database::IMPL_PACC) fprintf(out,"%sIMPL_TYPE = IMPL_PACC\n",indent);
  else assert(false);

  if(Database::CONF_TYPE == Database::CONF_SIMUL) fprintf(out,"%sCONF_TYPE = CONF_SIMUL\n",indent);
  else if(Database::CONF_TYPE == Database::CONF_REACH) fprintf(out,"%sCONF_TYPE = CONF_REACH\n",indent);
  else if(Database::CONF_TYPE == Database::CONF_TRACE) fprintf(out,"%sCONF_TYPE = CONF_TRACE\n",indent);
  else if(Database::CONF_TYPE == Database::CONF_LTL) fprintf(out,"%sCONF_TYPE = CONF_LTL\n",indent);
  else if(Database::CONF_TYPE == Database::CONF_DLOCK) fprintf(out,"%sCONF_TYPE = CONF_DLOCK\n",indent);
  else if(Database::CONF_TYPE == Database::CONF_SEAW) fprintf(out,"%sCONF_TYPE = CONF_SEAW\n",indent);
  else assert(false);

  if(Database::CONF_CHECK_MECHANISM == Database::CONF_CHECK_EXPLICIT) fprintf(out,"%sCONF_CHECK_MECHANISM = CONF_CHECK_EXPLICIT\n",indent);
  else if(Database::CONF_CHECK_MECHANISM == Database::CONF_CHECK_SAT) fprintf(out,"%sCONF_CHECK_MECHANISM = CONF_CHECK_SAT\n",indent);
  else if(Database::CONF_CHECK_MECHANISM == Database::CONF_CHECK_LTSA) fprintf(out,"%sCONF_CHECK_MECHANISM = CONF_CHECK_LTSA\n",indent);
  else assert(false);

  if(Database::C_PARSER_TYPE == Database::C_PARSER_DEFAULT) fprintf(out,"%sC_PARSER_TYPE = C_PARSER_DEFAULT\n",indent);
  else if(Database::C_PARSER_TYPE == Database::C_PARSER_CTOOL) fprintf(out,"%sC_PARSER_TYPE = C_PARSER_CTOOL\n",indent);
  else assert(false);

  if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_SIMPLIFY) fprintf(out,"%sTHEOREM_PROVER_USED = THEOREM_PROVER_SIMPLIFY\n",indent);
  else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_CPROVER) fprintf(out,"%sTHEOREM_PROVER_USED = THEOREM_PROVER_CPROVER\n",indent);
  else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_CVC) fprintf(out,"%sTHEOREM_PROVER_USED = THEOREM_PROVER_CVC\n",indent);
  else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_ICS) fprintf(out,"%sTHEOREM_PROVER_USED = THEOREM_PROVER_ICS\n",indent);
  else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_SVC) fprintf(out,"%sTHEOREM_PROVER_USED = THEOREM_PROVER_SVC\n",indent);
  else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_CVCL) fprintf(out,"%sTHEOREM_PROVER_USED = THEOREM_PROVER_CVCL\n",indent);
  else assert(false);

  if(Database::SAT_SOLVER_USED == Database::SAT_SOLVER_CHAFF) fprintf(out,"%sSAT_SOLVER_USED = SAT_SOLVER_CHAFF\n",indent);
  else if(Database::SAT_SOLVER_USED == Database::SAT_SOLVER_SATO) fprintf(out,"%sSAT_SOLVER_USED = SAT_SOLVER_SATO\n",indent);
  else if(Database::SAT_SOLVER_USED == Database::SAT_SOLVER_GRASP) fprintf(out,"%sSAT_SOLVER_USED = SAT_SOLVER_GRASP\n",indent);
  else if(Database::SAT_SOLVER_USED == Database::SAT_SOLVER_HORN_AI) fprintf(out,"%sSAT_SOLVER_USED = SAT_SOLVER_HORN_AI\n",indent);
  else assert(false);

  if(Database::PATTERN_MATCHER_USED == Database::PATTERN_MATCHER_NAIVE) fprintf(out,"%sPATTERN_MATCHER_USED = PATTERN_MATCHER_NAIVE\n",indent);
  else assert(false);

  fprintf(out,"%sIGNORE_POINTERS = %s\n",indent,Database::IGNORE_POINTERS ? "true" : "false");
  fprintf(out,"%sUSE_SYNTACTIC_CHECKS = %s\n",indent,Database::USE_SYNTACTIC_CHECKS ? "true" : "false");
  fprintf(out,"%sTHEOREM_PROVER_CACHE = %s\n",indent,Database::THEOREM_PROVER_CACHE ? "true" : "false");
  fprintf(out,"%sTHEOREM_PROVER_CACHE_SIZE = %d\n",indent,Database::THEOREM_PROVER_CACHE_SIZE);
  fprintf(out,"%sUSE_PAM = %s\n",indent,Database::USE_PAM ? "true" : "false");
  fprintf(out,"%sPREDS_FROM_SPEC = %s\n",indent,Database::PREDS_FROM_SPEC ? "true" : "false");
  fprintf(out,"%sINIT_SEEDS_FROM_SPEC = %s\n",indent,Database::INIT_SEEDS_FROM_SPEC ? "true" : "false");
  fprintf(out,"%sPREDS_FROM_CE = %s\n",indent,Database::PREDS_FROM_CE ? "true" : "false");
  fprintf(out,"%sVERBOSITY_LEVEL = %d\n",indent,Database::VERBOSITY_LEVEL);
  fprintf(out,"%sHANDLE_DATA_COMM = %s\n",indent,Database::HANDLE_DATA_COMM ? "true" : "false");

  fprintf(out,"%sUSE_LIVE_VARS = %s\n",indent,Database::USE_LIVE_VARS ? "true" : "false");
  fprintf(out,"%sINLINE_LIBS = %s\n",indent,Database::INLINE_LIBS ? "true" : "false");
  fprintf(out,"%sCHECK_INVALID_PTR_DEREF = %s\n",indent,Database::CHECK_INVALID_PTR_DEREF ? "true" : "false");
  fprintf(out,"%sCHECK_ASSERTION_FAILURE = %s\n",indent,Database::CHECK_ASSERTION_FAILURE ? "true" : "false");
  fprintf(out,"%sSTART_ITER = %d\n",indent,Database::START_ITER);
  fprintf(out,"%sEND_ITER = %d\n",indent,Database::END_ITER);
  fprintf(out,"%sUSE_REACH_IMPL = %s\n",indent,Database::USE_REACH_IMPL ? "true" : "false");
  fprintf(out,"%sPRECOMPUTE_IMPL_TRANS = %s\n",indent,Database::PRECOMPUTE_IMPL_TRANS ? "true" : "false");
  fprintf(out,"%sCACHE_IMPL_TRANS = %s\n",indent,Database::CACHE_IMPL_TRANS ? "true" : "false");
  fprintf(out,"%sSIM_REL_FROM_SAT = %s\n",indent,Database::SIM_REL_FROM_SAT ? "true" : "false");
  fprintf(out,"%sPREOPTIMIZE = %s\n",indent,Database::PREOPTIMIZE ? "true" : "false");
  fprintf(out,"%sPREOPTVISIT = %s\n",indent,Database::PREOPTVISIT ? "true" : "false");
  fprintf(out,"%sPREOPTVISITS = %d\n",indent,Database::PREOPTVISITS);
  fprintf(out,"%sPREOPTMINDEPTH = %d\n",indent,Database::PREOPTMINDEPTH);
  fprintf(out,"%sPREOPTMINCES = %d\n",indent,Database::PREOPTMINCES);
  fprintf(out,"%sPREOPTMAXDEPTH = %d\n",indent,Database::PREOPTMAXDEPTH);
  fprintf(out,"%sPREOPTMAXCES = %d\n",indent,Database::PREOPTMAXCES);
  fprintf(out,"%sCOVERAGE = %s\n",indent,Database::COVERAGE ? "true" : "false");
  fprintf(out,"%sCOVERCONT = %s\n",indent,Database::COVERCONT ? "true" : "false");
  fprintf(out,"%sSHOWCOVER = %s\n",indent,Database::SHOWCOVER ? "true" : "false");
  fprintf(out,"%sCEGAR = %s\n",indent,Database::CEGAR ? "true" : "false");

  if(Database::CEGAR_TYPE == Database::CEGAR_ELIMINATE_PRED) fprintf(out,"%sCEGAR_TYPE = CEGAR_ELIMINATE_PRED\n",indent);
  else if(Database::CEGAR_TYPE == Database::CEGAR_USEFUL_PRED) fprintf(out,"%sCEGAR_TYPE = CEGAR_USEFUL_PRED\n",indent);
  else if(Database::CEGAR_TYPE == Database::CEGAR_OPTIMIZE_PRED) fprintf(out,"%sCEGAR_TYPE = CEGAR_OPTIMIZE_PRED\n",indent);
  else if(Database::CEGAR_TYPE == Database::CEGAR_GREEDY_PRED) fprintf(out,"%sCEGAR_TYPE = CEGAR_GREEDY_PRED\n",indent);
  else assert(false);

  fprintf(out,"%sINC_VERIFY = %s\n",indent,Database::INC_VERIFY ? "true" : "false");

  if(Database::PATH_CHECK_TYPE == Database::PATH_CHECK_FORWARD) fprintf(out,"%sPATH_CHECK_TYPE = PATH_CHECK_FORWARD\n",indent);
  else if(Database::PATH_CHECK_TYPE == Database::PATH_CHECK_BACKWARD) fprintf(out,"%sPATH_CHECK_TYPE = PATH_CHECK_BACKWARD\n",indent);
  else assert(false);

  if(Database::CE_DISPLAY_TYPE == Database::CE_LOC_ONLY) fprintf(out,"%sCE_DISPLAY_TYPE = CE_LOC_ONLY\n",indent);
  else if(Database::CE_DISPLAY_TYPE == Database::CE_LOC_ACT) fprintf(out,"%sCE_DISPLAY_TYPE = CE_LOC_ACT\n",indent);
  else if(Database::CE_DISPLAY_TYPE == Database::CE_LOC_CONS) fprintf(out,"%sCE_DISPLAY_TYPE = CE_LOC_CONS\n",indent);
  else if(Database::CE_DISPLAY_TYPE == Database::CE_LOC_CONS_ACT) fprintf(out,"%sCE_DISPLAY_TYPE = CE_LOC_CONS_ACT\n",indent);
  else assert(false);

  fprintf(out,"%sMAX_ADD_PRED = %d\n",indent,Database::MAX_ADD_PRED);
  fprintf(out,"%sCHECK_CE_BRANCHING = %s\n",indent,Database::CHECK_CE_BRANCHING ? "true" : "false");
  fprintf(out,"%sCHECK_CE_VALIDITY = %s\n",indent,Database::CHECK_CE_VALIDITY ? "true" : "false");
  fprintf(out,"%sUSE_PATTERN_MATCHER = %s\n",indent,Database::USE_PATTERN_MATCHER ? "true" : "false");
  fprintf(out,"%sMAX_GLOBAL_CE = %d\n",indent,Database::MAX_GLOBAL_CE);
  fprintf(out,"%sMAX_CE_LOOP = %d\n",indent,Database::MAX_CE_LOOP);
  fprintf(out,"%sCNF_NO_NEW_VARS = %s\n",indent,Database::CNF_NO_NEW_VARS ? "true" : "false");
  fprintf(out,"%sLARGEST_SUBSET = %d\n",indent,Database::LARGEST_SUBSET);
  fprintf(out,"%sSMALLEST_SUBSET = %d\n",indent,Database::SMALLEST_SUBSET);
  fprintf(out,"%sSUBSET_CUTOFF = %d\n",indent,Database::SUBSET_CUTOFF);
  fprintf(out,"%sCONSTRAINT_CUTOFF = %d\n",indent,Database::CONSTRAINT_CUTOFF);
  fprintf(out,"%sSTOP_OPT_AFTER_ELIM = %s\n",indent,Database::STOP_OPT_AFTER_ELIM ? "true" : "false");
  fprintf(out,"%sUSE_PRED_ABSTRACTION = %s\n",indent,Database::USE_PRED_ABSTRACTION ? "true" : "false");
  fprintf(out,"%sUSE_LTS_ABSTRACTION = %s\n",indent,Database::USE_LTS_ABSTRACTION ? "true" : "false");
  fprintf(out,"%sNO_USELESS_LOCS = %s\n",indent,Database::NO_USELESS_LOCS ? "true" : "false");
  fprintf(out,"%sNO_SILENT_TRANS = %s\n",indent,Database::NO_SILENT_TRANS ? "true" : "false");
  fprintf(out,"%sMAX_PRED_INFER_LOOP = %d\n",indent,Database::MAX_PRED_INFER_LOOP);
  fprintf(out,"%sMAX_TIME_LIMIT = %d\n",indent,Database::MAX_TIME_LIMIT);
  fprintf(out,"%sLTL_TO_BUCHI = %s\n",indent,Database::LTL_TO_BUCHI ? "true" : "false");
  fprintf(out,"%sCREATE_DOT_FILE = %s\n",indent,Database::CREATE_DOT_FILE ? "true" : "false");
  fprintf(out,"%sDRAW_LTS = %s\n",indent,Database::DRAW_LTS ? "true" : "false");
  fprintf(out,"%sDRAW_CFG = %s\n",indent,Database::DRAW_CFG ? "true" : "false");
  fprintf(out,"%sDRAW_INFERRED_PREDS = %s\n",indent,Database::DRAW_INFERRED_PREDS ? "true" : "false");
  fprintf(out,"%sDRAW_PRED_ABS_LTS = %s\n",indent,Database::DRAW_PRED_ABS_LTS ? "true" : "false");
  fprintf(out,"%sDRAW_LTS_ABS_LTS = %s\n",indent,Database::DRAW_LTS_ABS_LTS ? "true" : "false");
  fprintf(out,"%sDRAW_COMPOSED_LTS = %s\n",indent,Database::DRAW_COMPOSED_LTS ? "true" : "false");
  fprintf(out,"%sDRAW_CE_DAG = %s\n",indent,Database::DRAW_CE_DAG ? "true" : "false");
  fprintf(out,"%sDRAW_CE_PROJ_LTS = %s\n",indent,Database::DRAW_CE_PROJ_LTS ? "true" : "false");
  fprintf(out,"%sDRAW_CE_PROJ_PRED = %s\n",indent,Database::DRAW_CE_PROJ_PRED ? "true" : "false");
  fprintf(out,"%sOUTPUT_MODELS = %s\n",indent,Database::OUTPUT_MODELS ? "true" : "false");

  if(Database::OUTPUT_MODEL_TYPE == Database::OUTPUT_MODEL_FSP) fprintf(out,"%sOUTPUT_MODEL_TYPE = OUTPUT_MODEL_FSP\n",indent);
  else assert(false);

  fprintf(out,"%sEXPLAIN_CE = %s\n",indent,Database::EXPLAIN_CE ? "true" : "false");
  fprintf(out,"%sADDITIONAL_UNWIND = %d\n",indent,Database::ADDITIONAL_UNWIND);
  fprintf(out,"%sMAX_UNWIND = %d\n",indent,Database::MAX_UNWIND);
  fprintf(out,"%sMORE_UNWIND = %s\n",indent,Database::MORE_UNWIND ? "true" : "false");
  fprintf(out,"%sSPURIOUS_TRIES = %d\n",indent,Database::SPURIOUS_TRIES);
  fprintf(out,"%sACTION_WEIGHT = %d\n",indent,Database::ACTION_WEIGHT);
  fprintf(out,"%sPRED_WEIGHT = %d\n",indent,Database::PRED_WEIGHT);
  fprintf(out,"%sLTL_NEG = %s\n",indent,Database::LTL_NEG ? "true" : "false");
  fprintf(out,"}\n\n");
  //save files
  fprintf(out,"input_files {\n");
  for(list<string>::const_iterator i = Database::FILE_NAMES.begin();i != Database::FILE_NAMES.end();++i) {
    fprintf(out,"%s%s\n",indent,i->c_str());
  }
  fprintf(out,"}\n");
  //close backup file
  fclose(out);
}

/*********************************************************************/
//end of magic.cpp
/*********************************************************************/
