/******************************** CPPFile *****************************

* FileName [WPComputer.cpp]

* PackageName [main]

* Synopsis [Method definitions of WPComputer class.]

* SeeAlso [WPComputer.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <string>
#include <list>
#include <set>
#include <typeinfo>
#include <map>
#include <vector>
using namespace std;

#include "Util.h"
#include "Node.h"
#include "WPComputer.h"
using namespace magic;

/*********************************************************************/
//compute the weakest liberal precondition of an expression wrt an
//assignment. the arguments are the lhs and the rhs of the assignment
//and the expression. rigth now this is done by simply replacing all
//occurrences of the lhs with the rhs in the expression.
/*********************************************************************/
BasicExpr *WPComputer::ComputeWP(const BasicExpr *lhs,const BasicExpr *rhs,const BasicExpr *expr)
{
  return Replace(lhs,rhs,expr,true);
}

/*********************************************************************/
//compute the weakest liberal precondition of an expression wrt a
//parallel assignment. the arguments are the lhs-list and the rhs-list
//of the assignment and the expression. rigth now this is done by
//simply replacing all occurrences of every lhs with the corresponding
//rhs in the expression.
/*********************************************************************/
BasicExpr *WPComputer::ComputeWP(const list<const BasicExpr*> &lhsList,const list<const BasicExpr*> &rhsList,const BasicExpr *expr)
{
  return Replace(lhsList,rhsList,expr,true);
}

/*********************************************************************/
//replace all ocurrences of the first argument with the second
//argument in the third argument
/*********************************************************************/
BasicExpr *WPComputer::Replace(const BasicExpr *lhs,const BasicExpr *rhs,const BasicExpr *expr,bool ignoreAddress)
{
  list<const BasicExpr*> lhsList,rhsList;
  lhsList.push_back(lhs);
  rhsList.push_back(rhs);
  return Replace(lhsList,rhsList,expr,ignoreAddress);
}

/*********************************************************************/
//replace all ocurrences of any element of the first argument with
//the corresponding element of the second argument in the third
//argument
/*********************************************************************/
BasicExpr *WPComputer::Replace(const list<const BasicExpr*> &lhsList,const list<const BasicExpr*> &rhsList,const BasicExpr *expr,bool ignoreAddress)
{
  //we will always first check if the some element of lhs and
  //expr are identical. in that case we know that the
  //corresponding element of rhs is what we want.
  string exprStr = Util::TrimString(expr->ToString());
  int count = 0;
  for(list<const BasicExpr*>::const_iterator i = lhsList.begin();i != lhsList.end();++i,++count) {
    string lhsStr = Util::TrimString((*i)->ToString());
    if(exprStr == lhsStr) {
      int count1 = 0;
      for(list<const BasicExpr*>::const_iterator j = rhsList.begin();j != rhsList.end();++j,++count1) {
	if(count == count1) return static_cast<BasicExpr*>((*j)->Clone());
      }
      assert(false);
    }
  }
  
  //id_expression, int_const_expression, const_expression,
  //string_expression, str_str_expression
  if((typeid(*expr) == typeid(IdExpr)) ||
     (typeid(*expr) == typeid(IntConstExpr)) ||
     (typeid(*expr) == typeid(ConstExpr)) ||
     (typeid(*expr) == typeid(StrExpr)) ||
     (typeid(*expr) == typeid(StrStrExpr))) {
    return static_cast<BasicExpr*>(expr->Clone());
  } 
  //statement_expression
  else if(typeid(*expr) == typeid(StmtExpr)) {
    Util::Error("cannot do substitution in " + expr->ToString() + "\n");
  }
  //brack_expression
  else if(typeid(*expr) == typeid(BrackExpr)) {
    const BrackExpr *x = static_cast<const BrackExpr*>(expr);
    BasicExpr *y = Replace(lhsList,rhsList,x->array,ignoreAddress);
    BasicExpr *z = Replace(lhsList,rhsList,x->index,ignoreAddress);
    BrackExpr *res = new BrackExpr(y,z);
    delete y; delete z;
    return res;
  }
  //par_expression
  else if(typeid(*expr) == typeid(ParExpr)) {
    const ParExpr *x = static_cast<const ParExpr*>(expr);
    BasicExpr *y = Replace(lhsList,rhsList,x->proc,ignoreAddress);
    list<BasicExpr*> &args = x->args->data;
    list<BasicExpr*> nargs;
    for(list<BasicExpr*>::iterator i = args.begin();i != args.end();++i) {
      nargs.push_back(Replace(lhsList,rhsList,*i,ignoreAddress));
    }
    ExprList *z = new ExprList(nargs);
    for(list<BasicExpr*>::iterator i = nargs.begin();i != nargs.end();++i) {
      delete *i;
    }
    ParExpr *res = new ParExpr(y,z);
    delete y; delete z;
    return res;
  }
  //dot_expression
  else if(typeid(*expr) == typeid(DotExpr)) {
    const DotExpr *x = static_cast<const DotExpr*>(expr);
    BasicExpr *y = Replace(lhsList,rhsList,x->expr,ignoreAddress);
    DotExpr *res = new DotExpr(y,x->id);
    delete y;
    return res;
  }
  //arrow_expression
  else if(typeid(*expr) == typeid(ArrowExpr)) {
    const ArrowExpr *x = static_cast<const ArrowExpr*>(expr);
    BasicExpr *y = Replace(lhsList,rhsList,x->expr,ignoreAddress);
    BasicExpr *res = NULL;
    if(typeid(*y) == typeid(UnaryExpr)) {
      UnaryExpr *z = static_cast<UnaryExpr*>(y);
      if(z->op == '&') {
	res = new DotExpr(z->expr,x->id);
      } else {
	res = new ArrowExpr(y,x->id);
      }
    } else {
      res = new ArrowExpr(y,x->id);
    }
    delete y;
    return res;
  }
  //inc_dec_expression
  else if(typeid(*expr) == typeid(IncExpr)) {
    const IncExpr *x = static_cast<const IncExpr*>(expr);
    BasicExpr *y = Replace(lhsList,rhsList,x->expr,ignoreAddress);
    IncExpr *res = new IncExpr(y,x->type);
    delete y;
    return res;
  }
  //unary_expression
  else if(typeid(*expr) == typeid(UnaryExpr)) {
    const UnaryExpr *x = static_cast<const UnaryExpr*>(expr);
    if(ignoreAddress && (x->op == '&')) return static_cast<BasicExpr*>(x->Clone());
    BasicExpr *y = Replace(lhsList,rhsList,x->expr,ignoreAddress);    
    BasicExpr *res = NULL;
    if(typeid(*y) == typeid(UnaryExpr)) {
      UnaryExpr *z = static_cast<UnaryExpr*>(y);
      if(((x->op == '*') && (z->op == '&')) || ((x->op == '&') && (z->op == '*'))) {
	res = static_cast<BasicExpr*>(z->expr->Clone());
      } else {
	res = new UnaryExpr(y,x->op);
      }
    } else {
      res = new UnaryExpr(y,x->op);
    }
    delete y;
    return res;
  }
  //empty_expression
  else if(typeid(*expr) == typeid(EmptyExpr)) {
    return new EmptyExpr();
  }
  //binary_expression
  else if(typeid(*expr) == typeid(BinaryExpr)) {
    const BinaryExpr *x = static_cast<const BinaryExpr*>(expr);
    BasicExpr *y = Replace(lhsList,rhsList,x->lhs,ignoreAddress);
    BasicExpr *z = Replace(lhsList,rhsList,x->rhs,ignoreAddress);
    BinaryExpr *res = new BinaryExpr(y,z,x->op);
    delete y; delete z;
    return res;
  }
  //quest_expression
  else if(typeid(*expr) == typeid(QuestExpr)) {
    const QuestExpr *x = static_cast<const QuestExpr*>(expr);
    BasicExpr *a = Replace(lhsList,rhsList,x->cond,ignoreAddress);
    BasicExpr *b = Replace(lhsList,rhsList,x->tcase,ignoreAddress);
    BasicExpr *c = Replace(lhsList,rhsList,x->ecase,ignoreAddress);
    QuestExpr *res = new QuestExpr(a,b,c);
    delete a; delete b; delete c;
    return res;
  }
  //assign_expression
  else if(typeid(*expr) == typeid(AssignExpr)) {
    const AssignExpr *x = static_cast<const AssignExpr*>(expr);
    BasicExpr *y = Replace(lhsList,rhsList,x->lhs,ignoreAddress);
    BasicExpr *z = Replace(lhsList,rhsList,x->rhs,ignoreAddress);
    AssignExpr *res = new AssignExpr(y,z);
    delete y; delete z;
    return res;
  }
  //illegal expression
  else assert(false);
}

/*********************************************************************/
//end of WPComputer.cpp
/*********************************************************************/
