/******************************** CPPFile *****************************

* FileName [Util.cpp]

* PackageName [main]

* Synopsis [The file containing various utility routines.]

* SeeAlso []

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <cstdarg>
#include <gmp.h>
#include <string>
#include <list>
#include <map>
#include <set>
#include <vector>
#include <typeinfo>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "DFSAdapter.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ImplState.h"
#include "ContLoc.h"
#include "TokenCollector.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//define static variables
/*********************************************************************/
int Util::tempVarCounter = 0;

/*********************************************************************/
//global shutdown method defined in arc.cpp needed
/*********************************************************************/
__attribute__ ((noreturn)) void GlobalShutdown(int sig);

/*********************************************************************/
//prints a message and exits.
/*********************************************************************/
__attribute__ ((noreturn)) void Util::Error(char *format,...)
{
  va_list argp;
  va_start(argp,format);
  vprintf(format,argp);
  va_end(argp);
  GlobalShutdown(-1);
}

__attribute__ ((noreturn)) void Util::Error(string message)
{
  printf("%s",message.c_str());
  GlobalShutdown(-1);
}

/*********************************************************************/
//prints a message. the first argument is the verbosity level of the
//message.
/*********************************************************************/
void Util::Message(int vl,char *format,...)
{
  if(vl <= Database::VERBOSITY_LEVEL) {
    va_list argp;
    va_start(argp,format);
    vprintf(format,argp);
    va_end(argp);
    fflush(stdout);
  }
}

void Util::Message(int vl,string message)
{
  if(vl <= Database::VERBOSITY_LEVEL) {
    printf("%s",message.c_str());
    fflush(stdout);
  }
}

/*********************************************************************/
//trim a string - remove intial and trailing whitespace
/*********************************************************************/
string Util::TrimString(const string &s)
{
  string::size_type a = s.find_first_not_of(" ");
  string::size_type b = s.find_last_of(" ");
  if((a == string::npos) && (b == string::npos)) return s;
  else if(a == string::npos) return s.substr(0,b);
  else if(b == string::npos) return s.substr(a,s.length() - a);
  else return s.substr(a, b - a);
}

/*********************************************************************/
//generate new temporary variable
/*********************************************************************/
string Util::NewTempVar()
{
  ++tempVarCounter;
  char x[50];
  snprintf(x,50,"temp_var_%d",tempVarCounter);
  return x;
}

/*********************************************************************/
//check if the argument string is a temporary variable
/*********************************************************************/
bool Util::IsTempVar(const string &arg)
{
  const char *ptr = arg.c_str();
  return (strstr(ptr,"temp_var_") == ptr);
}

/*********************************************************************/
//check if an expression is a procedure call. if so, return true in
//first component. further if the expression is an indirect call of
//the form (*X)(...) then return the PAM key associated with X in the
//second. otherwise second component is NULL.
/*********************************************************************/
pair<bool,int> Util::IsProcCall(const BasicExpr *arg)
{
  pair<bool,int> res(false,-1);
  if(typeid(*arg) == typeid(ParExpr)) {
    res.first = true;
    const BasicExpr *x = (static_cast<const ParExpr*>(arg))->proc;
    if(typeid(*x) == typeid(UnaryExpr)) {
      const UnaryExpr *y = static_cast<const UnaryExpr*>(x);
      assert(y->op == '*');
      res.second = y->expr->GetPamKey();
    }
  } 
  return res;
}

/*********************************************************************/
//compute the set of lvalues in an expression and return them as a set
//of expressions
/*********************************************************************/
set<Expr> Util::ComputeExprLvalues(const BasicExpr *expr)
{
  TokenCollector tc(TokenCollector::ALL_LVS);
  expr->Apply(tc);
  return tc.GetTokens();
}

/*********************************************************************/
//compute the set of lvalues in an expression and return them as a set
//of strings
/*********************************************************************/
set<string> Util::ComputeStrLvalues(const BasicExpr *expr)
{
  set<Expr> le = Util::ComputeExprLvalues(expr);
  set<string> res;
  for(set<Expr>::const_iterator i = le.begin();i != le.end();++i) {
    res.insert(Util::TrimString(i->ToString()));
  }
  return res;
}

/*********************************************************************/
//compute the set of lvalues dereferenced in an expression and return
//them as a set of expressions
/*********************************************************************/
set<Expr> Util::ComputeExprDerefLvalues(const BasicExpr *expr)
{
  TokenCollector tc(TokenCollector::DEREF_LVS);
  expr->Apply(tc);
  return tc.GetTokens();
}

/*********************************************************************/
//return the set of identifier lvalues in an expression
/*********************************************************************/
set<string> Util::ComputeIdLvalues(const BasicExpr *expr)
{
  TokenCollector tc(TokenCollector::ID_LVS);
  expr->Apply(tc);
  set<string> res;
  for(set<Expr>::const_iterator i = tc.GetTokens().begin();i != tc.GetTokens().end();++i) {
    res.insert(Util::TrimString(i->ToString()));
  }
  return res;
}

/*********************************************************************/
//return the set of identifiers in an expression
/*********************************************************************/
set<string> Util::ComputeIds(const BasicExpr *expr)
{
  TokenCollector tc(TokenCollector::ALL_IDS);
  expr->Apply(tc);
  set<string> res;
  for(set<Expr>::const_iterator i = tc.GetTokens().begin();i != tc.GetTokens().end();++i) {
    res.insert(Util::TrimString(i->ToString()));
  }
  return res;
}

/*********************************************************************/
//convert a list of expressions to a string
/*********************************************************************/
string Util::ExprListToString(const list<const BasicExpr*> &arg)
{
  string res = "[";
  for(list<const BasicExpr*>::const_iterator i = arg.begin();i != arg.end();) {
    res += TrimString((*i)->ToString());
    ++i;
    if(i != arg.end()) res += ",";
  }
  return res + "]";
}

/*********************************************************************/
//escape all occurences of ch in str with a backslash
/*********************************************************************/
string Util::EscapeCharacter(const string &str,char ch)
{
  char *buf = new char[2 * str.size() + 1];
  const char *old = str.c_str();
  size_t pos = 0;
  for(size_t i = 0;i < str.size();++i) {
    if(old[i] == ch) buf[pos++] = '\\';
    buf[pos++] = old[i];
  }
  buf[pos] = '\0';       
  string res(buf);
  delete [] buf;
  return res;
}

/*********************************************************************/
//returns true if the argument is a special predicate
/*********************************************************************/
bool Util::IsPredicate(const string &arg)
{
  return (arg == Database::VALID_PTR_PRED);
}

/*********************************************************************/
//compute the elements of the second argument that are related to the
//first argument via lvalues
/*********************************************************************/
void Util::LvalueClosure(const Expr &expr,const set<Expr> &eset,set<Expr> &res)
{
  res.insert(expr); return;

  map< Expr,set<string> > lvmap;
  set<Expr> left = eset;
  for(set<Expr>::const_iterator i = eset.begin();i != eset.end();++i) {
    lvmap[*i] = ComputeStrLvalues(i->GetExpr());
  }
  set<string> lvs = ComputeStrLvalues(expr.GetExpr());
  res.insert(expr);
  bool flag = true;
  while(flag) {
    flag = false;
    for(set<Expr>::iterator i = left.begin();i != left.end();++i) {
      const set<string> &x = lvmap[*i];
      bool b = false;
      for(set<string>::const_iterator j = x.begin();j != x.end();++j) {
	if(lvs.count(*j) != 0) {
	  b = true;
	  break;
	}
      }
      if(b) {
	lvs.insert(x.begin(),x.end());
	res.insert(*i);
	left.erase(i);
	flag = true;
	break;
      }
    }
  }
}

/*********************************************************************/
//end of Util.cpp
/*********************************************************************/
