/******************************** CPPFile *****************************

* FileName [TheoremProver.cpp]

* PackageName [main]

* Synopsis [Method definitions of TheoremProver class.]

* SeeAlso [TheoremProver.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
#include <ext/hash_set>
using namespace std;
using namespace __gnu_cxx;

#include "Util.h"
#include "GenericTP.h"
#ifdef MAGIC_FULL
#include "expr_util.h"
#include "sat.h"
#include "SvcTP.h"
#include "IcsTP.h"
#include "CvcTP.h"
#include "CvclTP.h"
#include "CproverTP.h"
#endif //MAGIC_FULL
#include "SimplifyTP.h"
#ifdef MAGIC_CPROVER
#include "expr_util.h"
#include "CproverTP.h"
#endif //MAGIC_CPROVER
#include "TheoremProver.h"
#include "Database.h"
using namespace magic;

/*********************************************************************/
//static variable definition
/*********************************************************************/
GenericTP *TheoremProver::genericTP = NULL;

/*********************************************************************/
//initialise the theorem prover
/*********************************************************************/
void TheoremProver::Initialise() 
{
  if(genericTP != NULL) {
    Util::Error("theorem prover already initialised ...");
  } else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_SIMPLIFY) {
    genericTP = new SimplifyTP();
  } 
#ifdef MAGIC_CPROVER
  else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_CPROVER) {
    genericTP = new CproverTP();
  }
#endif //MAGIC_CPROVER
#ifdef MAGIC_FULL
  else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_CPROVER) {
    genericTP = new CproverTP();
  } else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_CVC) {
    genericTP = new CvcTP();
  } else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_ICS) {
    genericTP = new IcsTP();
  } else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_SVC) {
    genericTP = new SvcTP();
  } else if(Database::THEOREM_PROVER_USED == Database::THEOREM_PROVER_CVCL) {
    genericTP = new CvclTP();
  }
#endif //MAGIC_FULL
  else {
    Util::Error("unknown theorem prover type being initialised ...");
  }
  genericTP->Initialise();
}

/*********************************************************************/
//shutdown the theorem prover
/*********************************************************************/
void TheoremProver::Shutdown() 
{
  //shutdown the generic theorem prover
  if(genericTP != NULL) {
    genericTP->Shutdown();
    genericTP = NULL;
  }

  //sometimes cvc leaves a debug file. remove it.
  remove(".svc2__debug_info");
}

/*********************************************************************/
//add a constraint to the theorem prover environment. this constraint
//is provided as a C expression.
/*********************************************************************/
void TheoremProver::AddConstraint(const BasicExpr *expr) 
{
  if(genericTP == NULL) {
    Util::Error("theorem prover not initialised ...");
  }
  genericTP->AddConstraint(expr);
}

/*********************************************************************/
//prove that an antecedent implies a consequent. both the antecedent
//and the consequent are provided as a list of formulas. the result is
//true if the cojunction of all of the antecedents implies the
//disjunction of all of the consequents. return true if it can be
//proved and false otherwise.
/*********************************************************************/
bool TheoremProver::ProveImplies(const list<const BasicExpr*> &ante,const list<const BasicExpr*> &cons) 
{
  if(genericTP == NULL) {
    Util::Error("theorem prover not initialised ...");
  }
  return genericTP->ProveImplies(ante,cons);
}

/*********************************************************************/
//end of TheoremProver.cpp
/*********************************************************************/
