/******************************** CPPFile *****************************

* FileName [Statistics.cpp]

* PackageName [main]

* Synopsis [Method definitions of Statistics class.]

* SeeAlso [Statistics.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <sys/time.h>
#ifndef WIN32
#include <sys/resource.h>
#endif //WIN32
#include <unistd.h>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <vector>
#include <map>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Timer.h"
#include "Statistics.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "LocalAbsLts.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#include "ContLoc.h"
#include "LtsInfo.h"
using namespace magic;

/*********************************************************************/
//static members
/*********************************************************************/
Timer Statistics::globalTimer;
Timer Statistics::imeTimer;
Timer Statistics::bacTimer;
Timer Statistics::ipTimer;
Timer Statistics::verTimer;
Timer Statistics::arTimer;
Timer Statistics::cegTimer;
Timer Statistics::cevTimer;
Timer Statistics::parTimer;
Timer Statistics::larTimer;
Timer Statistics::exUnwindTimer;
Timer Statistics::exSearchTimer;
size_t Statistics::elimCombNum = 0;
size_t Statistics::maxElimCombNum = 0;
size_t Statistics::maxElimCombSize = 0;
size_t Statistics::maxTriedCombSize = 0;
int Statistics::seedNum = 0;
BigInt Statistics::stateNum = 0;
int Statistics::iterNum = 0;
int Statistics::predIterNum = 0;
int Statistics::ltsIterNum = 0;
int Statistics::explainIterDepthNum = 0;
int Statistics::explainIterSpurNum = 0;
int Statistics::tpCalls = 0;
int Statistics::tpInCache = 0;
int Statistics::tpCacheSize = 0;

/*********************************************************************/
//shutdown all running timers
/*********************************************************************/
void Statistics::ShutdownTimers()
{
  if(globalTimer.Running()) globalTimer.Stop();
  if(imeTimer.Running()) imeTimer.Stop();
  if(bacTimer.Running()) bacTimer.Stop();
  if(ipTimer.Running()) ipTimer.Stop();
  if(verTimer.Running()) verTimer.Stop();
  if(arTimer.Running()) arTimer.Stop();
  if(cegTimer.Running()) cegTimer.Stop();
  if(cevTimer.Running()) cevTimer.Stop();
  if(parTimer.Running()) parTimer.Stop();
  if(larTimer.Running()) larTimer.Stop();
}

/*********************************************************************/
//display statistics
/*********************************************************************/
void Statistics::Display()
{
  if(Database::DISPLAY_STATISTICS) {
#ifdef WIN32
    double cpuTime = static_cast<double>(clock()) / CLOCKS_PER_SEC;
#else
    struct rusage usage;
    getrusage(RUSAGE_SELF,&usage);
    double cpuTime = usage.ru_utime.tv_sec + usage.ru_utime.tv_usec / 1000000.0 +
      usage.ru_stime.tv_sec + usage.ru_stime.tv_usec / 1000000.0;
#endif //WIN32
    
    Util::Message(1,"total global time = %.1f milliseconds\n",globalTimer.Read() * 1000);
    Util::Message(1,"total cpu time = %.1f milliseconds\n",cpuTime * 1000);
    Util::Message(1,"total input processing time = %.1f milliseconds\n",ipTimer.Read() * 1000);
    Util::Message(1,"total Buchi automaton construction time = %.1f milliseconds\n",bacTimer.Read() * 1000);
    Util::Message(1,"total implementation machine extraction time = %.1f milliseconds\n",imeTimer.Read() * 1000);
    Util::Message(1,"total verification time = %.1f milliseconds\n",verTimer.Read() * 1000);
    Util::Message(1,"total abstraction refinement time = %.1f milliseconds\n",arTimer.Read() * 1000);
    Util::Message(1,"total CE generation time = %.1f milliseconds\n",cegTimer.Read() * 1000);
    Util::Message(1,"total CE verification time = %.1f milliseconds\n",cevTimer.Read() * 1000);
    Util::Message(1,"total predicate abstraction refinement time = %.1f milliseconds\n",parTimer.Read() * 1000);
    Util::Message(1,"total LTS abstraction refinement time = %.1f milliseconds\n",larTimer.Read() * 1000);
    if (Database::EXPLAIN_CE) {
      Util::Message(1,"total explanation unwinding time = %.1f milliseconds\n",exUnwindTimer.Read() * 1000);
      Util::Message(1,"total explanation search time = %.1f milliseconds\n",exSearchTimer.Read() * 1000);
      Util::Message(1,"number of depth increase iterations = %d\n",explainIterDepthNum);
      Util::Message(1,"number of spurious check iterations = %d\n",explainIterSpurNum);
    }
    Util::Message(1,"total number of eliminating combinations = %d\n",elimCombNum);
    Util::Message(1,"max number of eliminating combinations for a CE = %d\n",maxElimCombNum);
    Util::Message(1,"max size of eliminating combination = %d\n",maxElimCombSize);
    Util::Message(1,"max size of tried combination = %d\n",maxTriedCombSize);
    Util::Message(1,"number of iterations = %d\n",iterNum);
    Util::Message(1,"number of predicate iterations = %d\n",predIterNum);
    Util::Message(1,"number of lts iterations = %d\n",ltsIterNum);
    Util::Message(1,"number of seed branches : %d\n",seedNum);
    if((Database::CONF_TYPE == Database::CONF_SIMUL) || (Database::CONF_TYPE == Database::CONF_REACH)) {
      const LtsInfo &specInfo = Database::GetLtsInfo(Database::specLtsName);
      Util::Message(1,"specification details : %d states %d transitions\n",
		    specInfo.GetStates().size(),specInfo.GetTrans().size());
    }
    Util::Message(1,"number of implementation states : %s\n",stateNum.ToString().c_str());
    Util::Message(1,"number of theorem prover calls = %d\n",tpCalls); 
    Util::Message(1,"number of formulas found in cache = %d\n",tpInCache);
    Util::Message(1,"number of formulas stored in cache = %d\n",tpCacheSize);
    
    //print expressions and predicates used
    for(size_t k = 0;k < Database::components.size();++k) {
      const vector < ContLoc*> &allLocs = static_cast<ProcManager*>(Database::components[k])->GetUsefulLocs();
      set <Expr> es;
      PredSet ap;
      
      for(vector < ContLoc* >::const_iterator i = allLocs.begin ();
	  i != allLocs.end (); ++i) {
	const PredSet &p = (*i)->GetPreds ();
	ap.Merge (p);
	const vector <Predicate> &pv = p.GetPreds ();
	for (vector <Predicate>::const_iterator j = pv.begin ();
	     j != pv.end (); ++j) {
	  const vector <Expr> &pev = (*j).GetValToExpr ();
	  es.insert(pev.begin(),pev.end());
	}
      }
      
      Util::Message(1,"Using %i expressions:\n",es.size());
      for (set<Expr>::const_iterator i = es.begin();i != es.end();++i) {
	Util::Message(1,"  %s\n",(*i).ToString().c_str());
      }
      Util::Message(1,"Using %i predicates:\n",ap.GetPreds().size());
      Util::Message(1,"%s\n",ap.ToString().c_str());
    }
  }
}

/*********************************************************************/
//end of Statistics.cpp
/*********************************************************************/
