/******************************** CPPFile *****************************

* FileName [SeawChecker.cpp]

* PackageName [main]

* Synopsis [Method definitions of SeawChecker class.]

* SeeAlso [SeawChecker.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <assert.h>
#include <gmp.h>
#include <string>
#include <list>
#include <vector>
#include <set>
#include <map>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "Database.h"
#include "GlobalAbsLts.h"
#include "Component.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Buchi.h"
#include "Seaw.h"
#include "SeawManager.h"
#include "LtlChecker.h"
#include "SeawChecker.h"
using namespace magic;

/*********************************************************************/
//define static fields
/*********************************************************************/
set<const BasicSeaw*> SeawChecker::labeled;
map< const BasicSeaw*,set<BigInt> > SeawChecker::label;
pair<const BasicSeaw*,int> SeawChecker::opSeaw(NULL,Buchi::LTL_NON_TRIVIAL);

/*********************************************************************/
//model check for a SE-AW formula
/*********************************************************************/
bool SeawChecker::CheckSeaw()
{
  labeled.clear(); label.clear();
  opSeaw = pair<const BasicSeaw*,int>(NULL,Buchi::LTL_NON_TRIVIAL);;
  set<BigInt> &reach = GlobalAbsLts::reach; reach.clear();
  set<BigInt> front; GlobalAbsLts::GetInitStates(front);
  while(!front.empty()) {
    reach.insert(front.begin(),front.end());
    set<BigInt> newFront = front;
    front.clear();
    for(set<BigInt>::const_iterator i = newFront.begin();i != newFront.end();++i) {
      map< Action,set<BigInt> > saa;
      GlobalAbsLts::GetSuccsAndActions(*i,saa);
      for(map< Action,set<BigInt> >::const_iterator j = saa.begin();j != saa.end();++j) {
	for(set<BigInt>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	  if(reach.count(*k) == 0) front.insert(*k);
	}
      }
    }
  }
  Util::Message(2,"number of reachable states = %d ...\n",reach.size());
  //label the initial state
  const BasicSeaw *bs = SeawManager::GetBasicSeaw(Database::specLtsName);
  set<BigInt> inits; GlobalAbsLts::GetInitStates(inits);
  for(set<BigInt>::const_iterator i = inits.begin();i != inits.end();++i) {
    if(!LabelState(*i,bs)) {
      labeled.clear(); label.clear();
      opSeaw = pair<const BasicSeaw*,int>(NULL,Buchi::LTL_NON_TRIVIAL);;
      return false;
    }
  }
  labeled.clear(); label.clear();
  opSeaw = pair<const BasicSeaw*,int>(NULL,Buchi::LTL_NON_TRIVIAL);;
  GlobalAbsLts::reach.clear();
  return true;
}

/*********************************************************************/
//label a states with a SE-AW formula
/*********************************************************************/
bool SeawChecker::LabelState(const BigInt &state,const BasicSeaw *seaw)
{
  //check the cache
  if(labeled.count(seaw) != 0) return (label[seaw].count(state) != 0);
  //if not in cache, model check
  bool res = false;
  if(seaw->GetType() == BasicSeaw::SEAW_PROP) {
    set<Expr> implProps; GlobalAbsLts::GetPropositions(state,implProps);
    res = (implProps.count(seaw->GetProp()) != 0);
  } else if(seaw->GetType() == BasicSeaw::SEAW_ACTS) {
  } else if(seaw->GetType() == BasicSeaw::SEAW_OMEGA) {
    for(size_t i = 0;i < seaw->GetSubs().size();++i) LabelAllStates(seaw->GetSubs()[i]);
    CreateBuchi(seaw);
    if(opSeaw.second == Buchi::LTL_TAUTOLOGY) res = true;
    else if(opSeaw.second == Buchi::LTL_UNSATISFIABLE) res = false;
    else res = LtlChecker::CheckFormula(state);
  } else if(seaw->GetType() == BasicSeaw::SEAW_AND) {
    res = LabelState(state,seaw->GetSubs()[0]) && LabelState(state,seaw->GetSubs()[1]);
  } else if(seaw->GetType() == BasicSeaw::SEAW_OR) {
    res = LabelState(state,seaw->GetSubs()[0]) || LabelState(state,seaw->GetSubs()[1]);
  } else if(seaw->GetType() == BasicSeaw::SEAW_NOT) {
    res = !LabelState(state,seaw->GetSubs()[0]);
  } else assert(false);
  //update the cache
  if(res) label[seaw].insert(state);  
  return res;
}

/*********************************************************************/
//label all states with the given formula
/*********************************************************************/
void SeawChecker::LabelAllStates(const BasicSeaw *seaw)
{
  if(labeled.count(seaw) != 0) return;

  //check for simple LTL patterns
  if(seaw->GetType() == BasicSeaw::SEAW_OMEGA) {
    map<string,OmegaOp>::const_iterator i = SeawManager::GetOmegaMap().find(seaw->GetOmega());
    const OmegaExpr *oe = i->second.GetOmega();
    //<A>(p)
    if(oe->type == OmegaExpr::OMEGA_DUMMY) {
      int marker = *(oe->markers.begin());
      const BasicSeaw *subf = seaw->GetSubs()[marker - 1];
      //action
      if(subf->GetType() == BasicSeaw::SEAW_ACTS) {
	const Action &act = *(seaw->GetActs().begin());
	for(set<BigInt>::const_iterator i = GlobalAbsLts::reach.begin();i != GlobalAbsLts::reach.end();++i) {
	  map< Action,set<BigInt> > saa; GlobalAbsLts::GetSuccsAndActions(*i, saa);
	  bool flag = true;
	  for(map< Action,set<BigInt> >::const_iterator j = saa.begin();j != saa.end();++j) {
	    if((!(j->first == act)) && (!j->second.empty())) {
	      flag = false;
	      break;
	    }
	  }
	  if(flag) label[seaw].insert(*i);
	}
      }
      //atomic proposition
      else {
	LabelAllStates(subf);
	label[seaw] = label[subf];
      }
      labeled.insert(seaw); return;
    }
    //<A> G (p)
    else if(oe->type == OmegaExpr::OMEGA_G) {
      OmegaExpr *sube = oe->sub1;
      if(sube->type == OmegaExpr::OMEGA_DUMMY) {
	int marker = *(sube->markers.begin());
	const BasicSeaw *subf = seaw->GetSubs()[marker - 1];
	//action
	if(subf->GetType() == BasicSeaw::SEAW_ACTS) {
	  AGAct(seaw,*(seaw->GetActs().begin())); 
	} 
	//atomic proposition
	else AGProp(seaw,subf); 
	labeled.insert(seaw); return;
      }
    } 
    //<A> F (p)
    else if(oe->type == OmegaExpr::OMEGA_F) {
      OmegaExpr *sube = oe->sub1;
      if(sube->type == OmegaExpr::OMEGA_DUMMY) {
	int marker = *(sube->markers.begin());
	const BasicSeaw *subf = seaw->GetSubs()[marker - 1];
	//action
	if(subf->GetType() == BasicSeaw::SEAW_ACTS) {
	  AFAct(seaw,*(seaw->GetActs().begin())); 
	} 
	//atomic proposition
	else AFProp(seaw,subf); 
	labeled.insert(seaw); return;
      }
    }
  }

  int count = 0;
  for(set<BigInt>::const_iterator i = GlobalAbsLts::reach.begin();i != GlobalAbsLts::reach.end();++i) {
    LabelState(*i,seaw);
    ++count;
    if((count % 10000) == 0) {
      Util::Message(2,"%d states labeled with %s ...\n",count,seaw->ToString().c_str());
    }
  }
  labeled.insert(seaw);
}

/*********************************************************************/
//label all states with AG[act]
/*********************************************************************/
void SeawChecker::AGAct(const BasicSeaw *seaw,const Action &act)
{
  set<BigInt> res,left = GlobalAbsLts::reach;
  while(true) {
    size_t oldSize = res.size();
    set<BigInt> newLeft;
    for(set<BigInt>::const_iterator i = left.begin();i != left.end();++i) {
      map< Action,set<BigInt> > saa; GlobalAbsLts::GetSuccsAndActions(*i, saa);
      bool flag = false;
      for(map< Action,set<BigInt> >::const_iterator j = saa.begin();(!flag) && (j != saa.end());++j) {
	if(j->first == act) {
	  for(set<BigInt>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	    if(res.count(*k) != 0) {
	      flag = true;
	      break;
	    }
	  }
	} else {
	  if(!j->second.empty()) {
	    flag = true;
	    break;
	  }
	}
      }
      if(flag) {
	res.insert(*i);
	if((res.size() % 10000) == 0) {
	  Util::Message(2,"%d states labeled with negation of %s\n",res.size(),seaw->ToString().c_str());
	}
      } else newLeft.insert(*i);
    }
    if(res.size() == oldSize) break;
    left = newLeft;
  }
  for(set<BigInt>::const_iterator i = GlobalAbsLts::reach.begin();i != GlobalAbsLts::reach.end();++i) {
    if(res.count(*i) == 0) label[seaw].insert(*i);
  }
}

/*********************************************************************/
//label all states with AG[prop]
/*********************************************************************/
void SeawChecker::AGProp(const BasicSeaw *seaw,const BasicSeaw *prop)
{
  LabelAllStates(prop);
  set<BigInt> res,left;
  for(set<BigInt>::const_iterator i = GlobalAbsLts::reach.begin();i != GlobalAbsLts::reach.end();++i) {
    if(label[prop].count(*i) == 0) res.insert(*i);
    else left.insert(*i);
  }
  while(true) {
    size_t oldSize = res.size();
    set<BigInt> newLeft;
    for(set<BigInt>::const_iterator i = left.begin();i != left.end();++i) {
      map< Action,set<BigInt> > saa; GlobalAbsLts::GetSuccsAndActions(*i, saa);
      bool flag = false;
      for(map< Action,set<BigInt> >::const_iterator j = saa.begin();(!flag) && (j != saa.end());++j) {
	for(set<BigInt>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	  if(res.count(*k) != 0) {
	    flag = true;
	    break;
	  }
	}
      }
      if(flag) {
	res.insert(*i);
	if((res.size() % 10000) == 0) {
	  Util::Message(2,"%d states labeled with negation of %s\n",res.size(),seaw->ToString().c_str());
	}
      } else newLeft.insert(*i);
    }
    if(res.size() == oldSize) break;
    left = newLeft;
  }
  for(set<BigInt>::const_iterator i = GlobalAbsLts::reach.begin();i != GlobalAbsLts::reach.end();++i) {
    if(res.count(*i) == 0) label[seaw].insert(*i);
  }
}

/*********************************************************************/
//label all states with AF[act]
/*********************************************************************/
void SeawChecker::AFAct(const BasicSeaw *seaw,const Action &act)
{
  set<BigInt> left = GlobalAbsLts::reach;
  while(true) {
    size_t oldSize = label[seaw].size();
    set<BigInt> newLeft;
    for(set<BigInt>::const_iterator i = left.begin();i != left.end();++i) {
      map< Action,set<BigInt> > saa; GlobalAbsLts::GetSuccsAndActions(*i, saa);
      bool flag = true;
      for(map< Action,set<BigInt> >::const_iterator j = saa.begin();flag && (j != saa.end());++j) {
	if(j->first == act) continue;
	for(set<BigInt>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	  if(label[seaw].count(*k) == 0) {
	    flag = false;
	    break;
	  }
	}
      }
      if(flag) {
	label[seaw].insert(*i);
	if((label[seaw].size() % 10000) == 0) {
	  Util::Message(2,"%d states labeled with %s\n",label[seaw].size(),seaw->ToString().c_str());
	}
      } else newLeft.insert(*i);
    }
    if(label[seaw].size() == oldSize) break;
    left = newLeft;
  }
}

/*********************************************************************/
//label all states with AF[prop]
/*********************************************************************/
void SeawChecker::AFProp(const BasicSeaw *seaw,const BasicSeaw *prop)
{
  LabelAllStates(prop);
  set<BigInt> left;
  for(set<BigInt>::const_iterator i = GlobalAbsLts::reach.begin();i != GlobalAbsLts::reach.end();++i) {
    if(label[prop].count(*i) == 0) left.insert(*i);
    else label[seaw].insert(*i);
  }
  while(true) {
    size_t oldSize = label[seaw].size();
    set<BigInt> newLeft;
    for(set<BigInt>::const_iterator i = left.begin();i != left.end();++i) {
      map< Action,set<BigInt> > saa; GlobalAbsLts::GetSuccsAndActions(*i, saa);
      bool flag = true;
      for(map< Action,set<BigInt> >::const_iterator j = saa.begin();flag && (j != saa.end());++j) {
	for(set<BigInt>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	  if(label[seaw].count(*k) == 0) {
	    flag = false;
	    break;
	  }
	}
      }
      if(flag) {
	label[seaw].insert(*i);
	if((label[seaw].size() % 10000) == 0) {
	  Util::Message(2,"%d states labeled with %s\n",label[seaw].size(),seaw->ToString().c_str());
	}
      } else newLeft.insert(*i);
    }
    if(label[seaw].size() == oldSize) break;
    left = newLeft;
  }
}

/*********************************************************************/
//create the Buchi automaton corresponding to a SE-AW formula
/*********************************************************************/
void SeawChecker::CreateBuchi(const BasicSeaw *seaw)
{
  if(opSeaw.first == seaw) return ;
  map<string,OmegaOp>::const_iterator i = SeawManager::GetOmegaMap().find(seaw->GetOmega());
  const OmegaExpr *oe = i->second.GetOmega();
  BasicLtl *bl = oe->ToLtl(seaw->GetSubs()); LtlFormula specFormula(bl); delete bl;
  set< pair<Action,Action> > fairLoopActs;
  for(vector<Component*>::const_iterator i = Database::components.begin();i != Database::components.end();++i) {
    set< pair<Action,Action> > x;
    (*i)->GetFairLoopActs(x);
    fairLoopActs.insert(x.begin(),x.end());
  }
  for(set< pair<Action,Action> >::const_iterator i = fairLoopActs.begin();i != fairLoopActs.end();++i) {
    //then and else
    LtlFormula tform = LtlManager::GetActLtl(i->first);
    LtlFormula eform = LtlManager::GetActLtl(i->second);
    //(F then) and (F else)
    LtlFormula ftform = LtlManager::GetUnaryLtl(tform,BasicLtl::LTL_F);
    LtlFormula feform = LtlManager::GetUnaryLtl(eform,BasicLtl::LTL_F);
    //(X F then) and (X F else)
    LtlFormula xftform = LtlManager::GetUnaryLtl(ftform,BasicLtl::LTL_X);
    LtlFormula xfeform = LtlManager::GetUnaryLtl(feform,BasicLtl::LTL_X);
    //(then & X F then) and (else & X F else)
    LtlFormula txft = LtlManager::GetBinaryLtl(tform,xftform,BasicLtl::LTL_AND);
    LtlFormula exfe = LtlManager::GetBinaryLtl(eform,xfeform,BasicLtl::LTL_AND);
    //(then & X F then) => (F else)
    LtlFormula ntxft = LtlManager::GetUnaryLtl(txft,BasicLtl::LTL_NOT);
    LtlFormula timpl = LtlManager::GetBinaryLtl(ntxft,feform,BasicLtl::LTL_OR);
    //(else & X F else) => (F then)
    LtlFormula nexfe = LtlManager::GetUnaryLtl(exfe,BasicLtl::LTL_NOT);
    LtlFormula eimpl = LtlManager::GetBinaryLtl(nexfe,ftform,BasicLtl::LTL_OR);
    //G((then & X F then) => (F else))
    LtlFormula tglob = LtlManager::GetUnaryLtl(timpl,BasicLtl::LTL_G);
    //G((else & X F else) => (F then))
    LtlFormula eglob = LtlManager::GetUnaryLtl(eimpl,BasicLtl::LTL_G);
    //G((then & X F then) => (F else)) & G((else & X F else) => (F then))
    LtlFormula andf = LtlManager::GetBinaryLtl(tglob,eglob,BasicLtl::LTL_AND);
    //implication
    LtlFormula negf = LtlManager::GetUnaryLtl(andf,BasicLtl::LTL_NOT);
    specFormula = LtlManager::GetBinaryLtl(negf,specFormula,BasicLtl::LTL_OR);
  }
  opSeaw = pair<const BasicSeaw*,int>(seaw,Buchi::Initialize(specFormula));
}

/*********************************************************************/
//end of SeawChecker.cpp
/*********************************************************************/
