/**************************** CPPHeaderFile ***************************

* FileName [ProcManager.h]

* PackageName [main]

* Synopsis [This class encapsulates a procedure manager.]

* Description [A procedure manager represents an unit of sequential
* behavior in a concurrent C program. In MAGIC, such an unit
* comprises of a single procedure.]

* SeeAlso [magic.cpp Component.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __PROCMANAGER_H__
#define __PROCMANAGER_H__

//needed to interface with the model parser
extern FILE *Modelin;
int Modelparse();
extern magic::ProcManager *mpProcManager;
extern set<const magic::ImplState*> *mpReachImpl;
extern set<magic::Action> *mpAbsActs;
extern set<magic::Action> *mpNewChanActs;

namespace magic {

//other classes needed
class Component;
class ImplState;
class ImplTrans;
class ImplPath;
class ContLoc;
class CallGraph;
class ConcCEDag;
class L1AbsState;
class NFA;

/*********************************************************************/
//a class encapsulating a level-1 abstract state for C
//programs. contains a ImplState and the corresponding control
//locations and data constraints induced by the predicate valuation
/*********************************************************************/
class CL1AbsState : public L1AbsState
{
 private:
  const ImplState *implState;
  const ContLoc *contLoc;
  set<Expr> cons;
  short specStateId;
  char specInfoId;

  void ConvertSet(const set<L1AbsState*> &arg,set<CL1AbsState> &res) const;
  void ConvertVec(const vector<L1AbsState*> &arg,vector<const ContLoc*> &res) const;

 public:
  CL1AbsState() { assert(false); }
  CL1AbsState(const ImplState *is,const ContLoc *cl,const set<Expr> &co,const short ssi,const char sii);
  CL1AbsState(const ImplState *is);
  CL1AbsState(const CL1AbsState &rhs) { *this = rhs; }
  ~CL1AbsState() { Cleanup(); }
  const CL1AbsState &operator = (const CL1AbsState &rhs);
  bool operator == (const CL1AbsState &rhs) const { return (implState == rhs.implState); }
  bool operator < (const CL1AbsState &rhs) const { return (implState < rhs.implState); }

  const ImplState *GetImplState() const { return implState; }
  const ContLoc *GetContLoc() const { return contLoc; }
  const set<Expr> &GetCons() const { return cons; }
  short GetSpecStateId() const { return specStateId; }
  char GetSpecInfoId() const { return specInfoId; }

  L1AbsState *Clone() const;
  void Cleanup() {}
  bool SetEq(const set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const;
  bool SetLt(const set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const;
  bool SetContains(const set<L1AbsState*> &lhs,const L1AbsState &rhs) const;
  void SetIntersect(set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const;
  bool VecEq(const vector<L1AbsState*> &lhs,const vector<L1AbsState*> &rhs) const;
  bool VecLt(const vector<L1AbsState*> &lhs,const vector<L1AbsState*> &rhs) const;
};

/*********************************************************************/
//the procedure manager
/*********************************************************************/
class ProcManager : public Component
{
 private:
  //the list of parameters of the procedure
  list<string> params;

  //the list of local variables
  list<string> locals;

  //the abstraction guard
  Expr guard;

  //the array of all control locations after eliminating skip
  //locations
  vector<ContLoc*> allLocs;

  //the set of all branch locations
  set<ContLoc*> branchLocs;

  //the array of useful control locations according to the predicates
  //being used in a particular refinement cycle
  vector<ContLoc*> usefulLocs;

  //the initial control location
  ContLoc *initLoc;

  //the list of initially skipped control locations. these locations
  //are eliminated since they are useless w.r.t. the set of predicates
  //being considered
  list<ContLoc*> initSkipped;

  //the final control location
  ContLoc *finalLoc;

  //the set of relevant formula propositions. a proposition is
  //relevant if all its variables are in the scope of this procedure.
  set<Expr> props;

  //the control location being currently tracked for invalid pointer
  //dereferences or assertion failures
  ContLoc *markedLoc;

  //the set of reachable implementation states
  set<const ImplState*> reachImpl;

  //the map from transitions to sequence of skipped transitions
  map<ImplTrans,ImplPath> skipPathMap;  

  //the traces over which we are optimizing
  set<ConcCEDag> optDags;

  //the map from branch locations to predicates to install
  map< ContLoc*,map<ContLoc*,PredSet> > locToPreds;

  //the map from branch locations to useful locations
  map< ContLoc*,set<ContLoc*> > locToUseful;

  //the map from control locations to necessary predicates
  map<ContLoc*,PredSet> necPreds;

  //set of useful locations due to necessary predicates
  set<ContLoc*> necUseful;

  //the map from assignment locations to sets of branch
  //locations. location X is mapped to Y if some predicate inferred
  //from Y is related to the constraint created by X
  map< Expr,set<ContLoc*> > assignToBranch;

  //the set of branch locations whose branch conditions are used for
  //predicate seeding
  set<ContLoc*> seedBranches;

  //a flag indicating if the seed branches have been modified
  bool newSeeds;

  //the local abstract lts
  LocalAbsLts absLts;

  //the map of branches to indices
  map<ContLoc*, int> varMap;

  //the map of indices to branches
  map<int, ContLoc*> mapVar;

  //the set of all constraints in CNF form
  set< set<ContLoc*> > allCons;

  //map from spurious dags discovered to control location sets that
  //can possibly rule it out
  map< ConcCEDag,set<ContLoc*> > dagToLocs;

  //the number of clauses generated and stored in the constraint file
  size_t clauseNum;

  //the name of the constraints file. this contains just a list of the
  //disjunctive clauses. this gets updated in every iteration. the
  //dimacs file is constructed from this file.
  string constraintFile;

  //the name of the dimacs file
  string dimacsFile;

  //the name of the pb file
  string pbFile;

  //the name of the output file
  string outputFile;

  //the map of seeds cached to their indices
  map< set<ContLoc*>,int > seedMap;

  //the state of the abstract local lts that should be split and the
  //action on which it should be split
  map< L2AbsState,set<const ImplState*> > absLtsRefine;

  void AddAssignActPreds() const;
  void AddCallActPreds() const;
  void AddRetActPreds() const;
  void AddLogicPreds() const;

  void ComputeInferredPreds(ContLoc *seedLoc);
  void SetupLocs();
  void InitialiseLocs() const;
  void ComputeRelevantPreds();
  void RemoveUselessContLocs();
  void ComputeNecessaryPredicates() const;

  bool LoadImplStates(set<Action> &newChanActs);
  void SaveImplStates();
  void ComputeTrueProps(const ContLoc *loc,const PVal &pv,set<Expr> &res) const;
  int RestoreSilentWithRoot(const int root,map<int,int> &oldToNew,const vector<L1AbsState*> &stateLabel,
			    const vector<Action> &actLabel,const vector< set<int> > &trans,
			    ConcCEDag &concDag) const;
  void DrawCFG() const;

 public:
  ProcManager();
  ProcManager(size_t i,const string &n,const list<string> &p,const list<string> &l,const Expr &g);
  ProcManager(const ProcManager &rhs);
  virtual ~ProcManager();

  const ProcManager &operator = (const ProcManager &rhs);

  const list<string> &GetParams() const { return params; }
  const list<string> &GetLocals() const { return locals; }
  const Expr &GetGuard() const { return guard; }
  const vector<ContLoc*> &GetAllLocs() const { return allLocs; }
  const vector<ContLoc*> &GetUsefulLocs() const { return usefulLocs; }
  ContLoc *GetInitLoc() const { return initLoc; }
  list<ContLoc*> &GetInitSkipped() { return initSkipped; }
  ContLoc *GetFinalLoc() const { return finalLoc; }
  const set<Expr> &GetProps() const { return props; }
  ContLoc *GetMarkedLoc() const { return markedLoc; }
  const set<const ImplState*> &GetReachImpl() const { return reachImpl; }
  const map<ImplTrans,ImplPath> &GetSkipPathMap() const { return skipPathMap; }
  const set<ContLoc*> &GetSeedBranches() const { return seedBranches; }
  const LocalAbsLts &GetAbsLts() const { return absLts; }
  LocalAbsLts &GetAbsLts() { return absLts; }
  bool GetNewSeeds() const { return newSeeds; }

  void SetProps(const set<Expr> &p);
  void SetMarkedLoc(ContLoc *i);
  void SetSeedBranches(const set<ContLoc*> &s);
  void SetNewSeeds(const bool n) { newSeeds = n; }

  void CreateAllContLocs();
  void DoLiveVariableAnalysis() const;

  void ClearPredSets();
  void InitializeImplStates(set<Action> &newChanActs);
  void UpdateImplStates(const set<Action> &pendChanActs,set<Action> &newChanActs);
  bool IsSilentAct(const Action &act) const;
  bool IsSilentTrans(const ImplTrans &trans,const map< const ImplState*,set<ImplTrans> > &succMap) const;
  bool IsSilentLoop(const ImplTrans &trans,const map< const ImplState*,set<ImplTrans> > &succMap) const;
  void ShortcutTrans(const ImplTrans &trans1,const ImplTrans &trans2,map< const ImplState*,set<ImplTrans> > &succMap,set<ImplTrans> &allTrans);
  void RemoveTrans(const ImplTrans &trans,map< const ImplState*,set<ImplTrans> > &succMap,set<ImplTrans> &allTrans);  
  void EliminateSilentTrans();
  void RestoreSilentTrans(ConcCEDag &concDag) const;
  void CreateInitialAbstractStates();
  void ComputeAbsLtsInit();
  void ComputeAbsLtsTrans();

  //implementation of virtual methods of super-class Component
  bool Initialise();
  void Cleanup();
  void GetL1AbsInit(L1StSet &res) const;
  void GetL2AbsInit(set<L2AbsState> &res);
  void GetPrevL2AbsInit(set<L2AbsState> &res);
  int GetL1AbsStateNum();
  int GetL2AbsStateNum();
  void GetL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res);
  bool IsL2AbsTrans(const L2AbsState &from,const Action &act,const L2AbsState &to);
  void GetL2AbsPredsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res);
  bool IsL2AbsAct(const Action &act) const;
  void GetL2AbsActs(set<Action> &res) const;
  void GetL2AbsActsOut(const L2AbsState &state,set<Action> &res) const;
  void GetL2AbsActsIn(const L2AbsState &state,set<Action> &res) const;
  int GetPrevL2AbsStateNum();
  void GetL2AbsSuccsChanged(set<L2AbsState> &res);
  void GetPrevL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res);
  void GetL2AbsProps(const L2AbsState &state,set<Expr> &res);
  bool DistinctL2AbsStates(const set<L2AbsState> &arg1,const set<L2AbsState> &arg2);
  void GetCompleteL1StSet(L1StSet &res) const;
  void IntersectL1StSets(L1StSet &lhs,const L1StSet &rhs) const;
  L1StSet GetL1AbsPredsOnAction(const L1StSet &states,const Action &act) const;
  void GetL1AbsSuccsOnAction(const L1AbsState &state,const Action &act,L1StSet &res) const;
  L1StSet GetL1AbsSuccsOnAction(const L1StSet &states,const Action &act) const;
  L1StSet ProjectL1StSetOnL2AbsState(const L1StSet &one,const L2AbsState &two) const;
  void CreateL2ToL1AbsStateMap(map<L2AbsState,const L1AbsState*> &res) const;
  L1AbsState *PickL1AbsState(const L1StSet &arg) const;
  bool L1AbsContains(const L1StSet &arg1,const L1AbsState &arg2) const;
  void GetAllL1AbsStates(set<L1AbsState*> &res) const;
  void L1ToL2AbsState(const L1AbsState &arg,L2AbsState &res) const;
  void CreateL1StSet(const set<L1AbsState*> &arg,L1StSet &res) const;
  void ClearL2AbsRefineInfo();
  bool UpdateL2AbsRefineInfo(const L2AbsState &l2s,const L1StSet &l1s);
  void L1AbsStateToString(const L1AbsState &arg,string &res) const;
  void L1StSetToString(const L1StSet &arg,string &res) const;
  void L1AbsStateToString(const L1AbsState &curr,const L1AbsState *next,const bool stutter,string &res) const;
  void L1AbsDagToPaths(const vector<L1AbsState*> &stateLabel,const vector<Action> &actLabel,const vector< set<int> > &trans,const vector<int> &topo,const set<Action> &absActs,set<ConcPath> &res) const;
  void GetFairLoopActs(set< pair<Action,Action> > &res) const;
  bool IsConcGlobalVar(const Expr &arg) const;
  void GetL2AbsRefusal(const L2AbsState &arg,set<Action> &res) const;
  bool IsL1AbsRefusal(const L1AbsState &state,const set<Action> &refusal) const;
  void OutputFsp() const;

  void DrawInferredPreds() const;
  void DrawPredAbsLts() const;
  void OutputModel() const;
  
  //converts absLts to NFA and returns the result in the argument
  //pointer
  void ProcManager::GetNFA(NFA *nfa) ;
  
  friend class CEDagVerifier;
  friend class AbsRefiner;
  friend class PredAbsRefiner;
  friend class LtsAbsRefiner;
  friend class CEGen;
  friend class ConcCEDag;
};

} //namespace magic

#endif //__PROCMANAGER_H__

/*********************************************************************/
//end of ProcManager.h
/*********************************************************************/
