/******************************** CPPFile *****************************

* FileName [ProcManager.cpp]

* PackageName [main]

* Synopsis [Definitions for ProcManager class.]

* SeeAlso [ProcManager.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gmp.h>
#include <string>
#include <vector>
#include <list>
#include <set>
#include <map>
using namespace std;

#include "BigInt.h"
#include "Timer.h"
#include "Util.h"
#include "Statistics.h"
#include "Node.h"
#include "LtsInfo.h"
#include "Action.h"
#include "ActionManager.h"
#include "ProcInfo.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ImplState.h"
#include "AbsRefiner.h"
#include "LocalAbsLts.h"
#include "GlobalCEDag.h"
#include "ConcCEDag.h"
#include "Component.h"
#include "ProcManager.h"
#define YYSTYPE int
#include "StdcParser.h"

#include "NFA.h"
using namespace magic;

/*********************************************************************/
//constructors and destructors
/*********************************************************************/
CL1AbsState::CL1AbsState(const ImplState *is,const ContLoc *cl,const set<Expr> &co,const short ssi,const char sii)
{
  implState = is;
  contLoc = cl;
  cons = co;
  specStateId = ssi;
  specInfoId = sii;
}

CL1AbsState::CL1AbsState(const ImplState *is)
{
  implState = is;
  contLoc = is->GetContLoc();
  const PredSet &preds = contLoc->GetPreds();
  const PredVal &pval = preds.PValToPredVal(is->GetPredVal());
  preds.ToExpr(pval,cons);
  specStateId = is->GetSpecStateId();
  specInfoId = is->GetSpecInfoId();
}

/*********************************************************************/
//operators
/*********************************************************************/
const CL1AbsState &CL1AbsState::operator = (const CL1AbsState &rhs)
{
  Cleanup();
  implState = rhs.implState;
  contLoc = rhs.contLoc;
  cons = rhs.cons;
  specStateId = rhs.specStateId;
  specInfoId = rhs.specInfoId;
  return *this;
}

/*********************************************************************/
//convert a set
/*********************************************************************/
void CL1AbsState::ConvertSet(const set<L1AbsState*> &arg,set<CL1AbsState> &res) const
{
  for(set<L1AbsState*>::const_iterator i = arg.begin();i != arg.end();++i) {
    res.insert(static_cast<CL1AbsState&>(**i));
  }
}

/*********************************************************************/
//convert a set. we retain only the control locations since this is
//used to compare counterexamples.
/*********************************************************************/
void CL1AbsState::ConvertVec(const vector<L1AbsState*> &arg,vector<const ContLoc*> &res) const
{
  for(vector<L1AbsState*>::const_iterator i = arg.begin();i != arg.end();++i) {
    res.push_back(static_cast<const CL1AbsState*>(*i)->GetContLoc());
  }
}

/*********************************************************************/
//clone
/*********************************************************************/
L1AbsState *CL1AbsState::Clone() const
{
  return new CL1AbsState(implState,contLoc,cons,specStateId,specInfoId);
}

/*********************************************************************/
//compare sets
/*********************************************************************/
bool CL1AbsState::SetEq(const set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const
{
  set<CL1AbsState> one; ConvertSet(lhs,one);
  set<CL1AbsState> two; ConvertSet(rhs,two);
  return (one == two);
}

bool CL1AbsState::SetLt(const set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const
{
  set<CL1AbsState> one; ConvertSet(lhs,one);
  set<CL1AbsState> two; ConvertSet(rhs,two);
  return (one < two);
}

bool CL1AbsState::SetContains(const set<L1AbsState*> &lhs,const L1AbsState &rhs) const
{  
  set<CL1AbsState> one; ConvertSet(lhs,one);
  return (one.count(static_cast<const CL1AbsState&>(rhs)) != 0);
}

void CL1AbsState::SetIntersect(set<L1AbsState*> &lhs,const set<L1AbsState*> &rhs) const
{
  set<CL1AbsState> one; ConvertSet(rhs,one);
  set<L1AbsState*> newLhs;
  for(set<L1AbsState*>::const_iterator i = lhs.begin();i != lhs.end();++i) {
    if(one.count(static_cast<const CL1AbsState&>(**i)) == 0) delete *i;
    else newLhs.insert(*i);
  }
  lhs = newLhs;
}

/*********************************************************************/
//compare vectors
/*********************************************************************/
bool CL1AbsState::VecEq(const vector<L1AbsState*> &lhs,const vector<L1AbsState*> &rhs) const
{
  vector<const ContLoc*> one; ConvertVec(lhs,one);
  vector<const ContLoc*> two; ConvertVec(rhs,two);
  return (one == two);
}

bool CL1AbsState::VecLt(const vector<L1AbsState*> &lhs,const vector<L1AbsState*> &rhs) const
{
  vector<const ContLoc*> one; ConvertVec(lhs,one);
  vector<const ContLoc*> two; ConvertVec(rhs,two);
  return (one < two);
}

/*********************************************************************/
//constructors and destructors
/*********************************************************************/
ProcManager::ProcManager() { assert(false); }

ProcManager::ProcManager(size_t i,const string &n,const list<string> &p,const list<string> &l,const Expr &g) 
  : params(p),locals(l),guard(g),absLts(i)
{ 
  index = i;
  name = n;
  initLoc = finalLoc = markedLoc = NULL;
  newSeeds = true; 
  clauseNum = 0; 

  //create the file names
  char prefix[20];
  snprintf(prefix,20,"pbs-%d-%d",getpid(),index);
  constraintFile = string(prefix) + ".con";
  dimacsFile = string(prefix) + ".cnf";
  pbFile = string(prefix) + ".cnf.pb";
  outputFile = string(prefix) + ".out";
}

ProcManager::ProcManager(const ProcManager &rhs) { *this = rhs; }

ProcManager::~ProcManager() {}

/*********************************************************************/
//operators
/*********************************************************************/
const ProcManager &ProcManager::operator = (const ProcManager &rhs)
{
  index = rhs.index;
  name = rhs.name;
  params = rhs.params;
  locals = rhs.locals;
  guard = rhs.guard;
  allLocs = rhs.allLocs;
  branchLocs = rhs.branchLocs;
  usefulLocs = rhs.usefulLocs;
  initLoc = rhs.initLoc;
  initSkipped = rhs.initSkipped;
  finalLoc = rhs.finalLoc;
  props = rhs.props;
  markedLoc = rhs.markedLoc;
  reachImpl = rhs.reachImpl;
  skipPathMap = rhs.skipPathMap;
  optDags = rhs.optDags;
  locToPreds = rhs.locToPreds;
  locToUseful = rhs.locToUseful;
  necPreds = rhs.necPreds;
  necUseful = rhs.necUseful;
  assignToBranch = rhs.assignToBranch;
  seedBranches = rhs.seedBranches;
  newSeeds = rhs.newSeeds;
  absLts = rhs.absLts;
  varMap = rhs.varMap;
  mapVar = rhs.mapVar;
  allCons = rhs.allCons;
  dagToLocs = rhs.dagToLocs;
  clauseNum = rhs.clauseNum;
  constraintFile = rhs.constraintFile;
  dimacsFile = rhs.dimacsFile;
  pbFile = rhs.pbFile;
  outputFile = rhs.outputFile;
  seedMap = rhs.seedMap;
  absLtsRefine = rhs.absLtsRefine;
  return *this;
}

/*********************************************************************/
//initialise the abstraction refiner
/*********************************************************************/
bool ProcManager::Initialise()
{
  //remove any old copies of the data files
  remove(constraintFile.c_str());
  remove(dimacsFile.c_str());
  remove(pbFile.c_str());
  remove(outputFile.c_str());
  return true;
}

/*********************************************************************/
//cleanup the abstraction refiner
/*********************************************************************/
void ProcManager::Cleanup()
{
  //remove any current copies of the data files
  Initialise();
  //remove any impl state and transition files
  for(map< set<ContLoc*>,int >::const_iterator i = seedMap.begin();i != seedMap.end();++i) {
    char buf[32];
    snprintf(buf,32,"implstate-%d-%d-%d",getpid(),index,i->second);
    remove(buf);
    snprintf(buf,32,"impltrans-%d-%d-%d",getpid(),index,i->second);
    remove(buf);
  }
  seedMap.clear();
  //clear predicate abstraction refinement info
  varMap.clear();
  mapVar.clear();
  allCons.clear();
}

/*********************************************************************/
//set the locations being tracked for invalid pointer dereferences
/*********************************************************************/
void ProcManager::SetMarkedLoc(ContLoc *i) { markedLoc = i; }

/*********************************************************************/
//set the seed branches
/*********************************************************************/
void ProcManager::SetSeedBranches(const set<ContLoc*> &s) 
{ 
  seedBranches = s; 
  newSeeds = true; 
}

/*********************************************************************/
//set the relevant propositions
/*********************************************************************/
void ProcManager::SetProps(const set<Expr> &p) { props = p; }

/*********************************************************************/
//create the vector of all control locations. also backup fields and
//compute predecessors.
/*********************************************************************/
void ProcManager::CreateAllContLocs()
{
  //make a copy - rename the variable appearing in expressions by
  //prepending this proc manager's index followed by "::"
  map<ContLoc*,ContLoc*> oldToNew;
  const list<ContLoc*> &oldLocs = Database::procInfos[name].GetContLocs();
  for(list<ContLoc*>::const_iterator i = oldLocs.begin();i != oldLocs.end();++i) {
    ContLoc *x = new ContLoc(**i);
    x->AddNamespaceTag(*this);
    allLocs.push_back(x);    
    oldToNew[*i] = x;
  }
  //change the successors
  for(vector<ContLoc*>::const_iterator i = allLocs.begin();i != allLocs.end();++i) {
    int locType = (*i)->GetLocType();
    if(locType == ContLoc::FINAL) {
    } else if((locType == ContLoc::RETURN) || (locType == ContLoc::ASSIGN_EXPR)) {
      (*i)->SetTSucc(oldToNew[(*i)->GetTSucc()]);
    } else if(locType == ContLoc::ASSIGN_CALL) {
      assert((*i)->GetCallSucc() == NULL);
      (*i)->SetTSucc(oldToNew[(*i)->GetTSucc()]);
    } else if(locType == ContLoc::BRANCH) {
      (*i)->SetTSucc(oldToNew[(*i)->GetTSucc()]);
      (*i)->SetESucc(oldToNew[(*i)->GetESucc()]);
    } else assert(false);
  }
  //set the initial and final locations
  initLoc = oldToNew[Database::procInfos[name].GetInitLoc()];
  finalLoc = oldToNew[Database::procInfos[name].GetFinalLoc()];
  //backup and compute predecessors. create the set of all branch
  //locations. assign fairness actions to fair loop branches.
  for(vector<ContLoc*>::iterator i = allLocs.begin();i != allLocs.end();++i) {
    if((*i)->GetLocType() == ContLoc::BRANCH) {
      branchLocs.insert(*i);
      if((Database::CONF_TYPE == Database::CONF_LTL) && ((*i)->IsFairLoop())) {
	(*i)->SetFairLoopActs(pair<Action,Action>(Action("fair_loop::true"),Action("fair_loop::false")));
      }
    }
    (*i)->Backup();
    (*i)->MakePred();
  }
  //draw the control flow graph if necessary
  if(Database::DRAW_CFG) DrawCFG();
}

/*********************************************************************/
//perform live variable analysis
/*********************************************************************/
void ProcManager::DoLiveVariableAnalysis() const
{
  //initialise the analysis and create the working set
  set<ContLoc*> wset;
  for(size_t i = 0;i < allLocs.size();++i) {
    allLocs[i]->InitLiveVarAnalysis(*this);
    wset.insert(allLocs[i]);
  }  
  //then do the analysis
  while(!wset.empty()) {
    ContLoc *a = *(wset.begin());
    wset.erase(wset.begin());
    if(a->UpdateLiveVars()) {
      const set<ContLoc*> &x = a->GetPredecessors();
      wset.insert(x.begin(),x.end());
    }
  }
}

/*********************************************************************/
//seed the control locations with necessary predicates
/*********************************************************************/
void ProcManager::ComputeNecessaryPredicates() const
{
  //add the predicate qualifying the initial state to the initial
  //location
  if(initLoc->GetPreds().AddPred(Predicate(guard,Predicate::REQUIRED,pair<const ContLoc*,int>(initLoc,1)))) {
    Util::Message(3,"predicate " + guard.ToString() + " added to control location " 
		  + initLoc->ToString() + "\n");
  }
  //add predicates associated with assign actions to assignment
  //locations
  AddAssignActPreds();
  //add predicates associated with call actions to call locations
  AddCallActPreds();
  //add predicates associated with return actions to return locations
  AddRetActPreds();
  //add predicate from ltl formulas
  if((Database::CONF_TYPE == Database::CONF_LTL) || (Database::CONF_TYPE == Database::CONF_SEAW)) AddLogicPreds();
}

/*********************************************************************/
//add the predicates associated with assign actions to assignment
//locations
/*********************************************************************/
void ProcManager::AddAssignActPreds() const
{
  //get the set of expresions associated with assign actions
  set< pair<Expr,Expr> > preds; Database::GetSpecAssignActExprs(preds);
  ProcInfo &procInfo = Database::procInfos[name];
  for(size_t i = 0;i < allLocs.size();++i) {
    ContLoc *a = allLocs[i];
    ProcInfo *b = a->GetProcInfo();
    //if this is not the info of the main procedure, there are
    //no return action predicates to add
    if((&procInfo) != b) continue;
    if(a->GetLocType() == ContLoc::ASSIGN_EXPR) {
      a->AddAssignActPreds(preds,*this);
    } 
  }
}

/*********************************************************************/
//add the predicates associated with call actions to call locations
/*********************************************************************/
void ProcManager::AddCallActPreds() const
{
  //get the set of expresions associated with assign actions
  set< pair<Expr,Expr> > preds; Database::GetSpecAssignActExprs(preds);
  for(size_t i = 0;i < allLocs.size();++i) {
    ContLoc *loc = allLocs[i];
    //get the map of names of procedures in call actions to
    //set of predicates associated with the actions
    if(loc->GetLocType() == ContLoc::ASSIGN_CALL) {
      loc->AddCallActPreds(preds,*this);
    }
  }
}

/*********************************************************************/
//add the predicates associated with return actions to return
//locations
/*********************************************************************/
void ProcManager::AddRetActPreds() const
{
  //get the set of expresions associated with return actions
  set<Expr> preds; Database::GetSpecRetActExprs(preds);
  ProcInfo &procInfo = Database::procInfos[name];
  for(size_t i = 0;i < allLocs.size();++i) {
    ContLoc *a = allLocs[i];
    ProcInfo *b = a->GetProcInfo();
    //if this is not the info of the main procedure, there are
    //no return action predicates to add
    if((&procInfo) != b) continue;
    if(a->GetLocType() == ContLoc::RETURN) {
      a->AddRetActPreds(preds,*this);
    }
  }
}

/*********************************************************************/
//add the predicates associated with the spec LTL or SE-AW formula to
//all locations
/*********************************************************************/
void ProcManager::AddLogicPreds() const
{
  for(size_t i = 0;i < allLocs.size();++i) {
    ContLoc *loc = allLocs[i];
    for(set<Expr>::const_iterator j = props.begin();j != props.end();++j) {
      vector<Predicate> pvec;
      if(loc->AddPred(Predicate(*j,Predicate::REQUIRED,pair<const ContLoc*,int>(loc,1)),pvec)) {
	Util::Message(3,"predicate " + j->ToString() + " added to control location " + loc->ToString() + "\n");
      }
    }
  }
}

/*********************************************************************/
//load implementation states
/*********************************************************************/
bool ProcManager::LoadImplStates(set<Action> &newChanActs)
{
  if(seedMap.count(seedBranches) != 0) {
    char stateBuf[128],transBuf[128];
    snprintf(stateBuf,128,"implstate-%d-%d-%d",getpid(),index,seedMap[seedBranches]);
    snprintf(transBuf,128,"impltrans-%d-%d-%d",getpid(),index,seedMap[seedBranches]);
    struct stat statStruct;
    if((stat(stateBuf,&statStruct) == 0) && (stat(transBuf,&statStruct) == 0)) {
      mpProcManager = this;
      mpReachImpl = &reachImpl;
      mpAbsActs = &(absLts.acts);
      mpNewChanActs = &newChanActs;
      FILE *stateIn = fopen(stateBuf,"r");
      FILE *transIn = fopen(transBuf,"r");
      if((stateIn == NULL) || (transIn == NULL)) return false;
      Modelin = stateIn;
      Modelparse();
      fclose(stateIn);
      /*
      //get the implementation states
      while(true) {
	ContLoc *loc = NULL;
	fscanf(in,"%p",&loc);
	if(loc == 0x0) break;
	PVal oldPV;
	short oldSS,oldSI;
	oldPV.Read(in);
	fscanf(in,"%hd",&oldSS);
	fscanf(in,"%hd",&oldSI);
	const ImplState *is = loc->GetImplState(oldPV,oldSS,oldSI);
	reachImpl.insert(is);
	set<Expr> props;
	while(true) {
	  const BasicExpr *prop = NULL;
	  fscanf(in,"%p",&prop);
	  if(prop == NULL) break;
	  props.insert(Expr(prop));
	}
	loc->SetPropositions(is,props);
      }
      //get the actions
      while(true) {
	short actId = Action::ACTION_NONE;
	fscanf(in,"%hd",&actId);
	if(actId == Action::ACTION_NONE) break;
	Action act(actId);
	if(act.IsBcast() || act.IsSend() || 
	   (act.IsAssign() && IsConcGlobalVar(act.GetLhsExpr()))) newChanActs.insert(act);
	absLts.acts.insert(act);
      }
      fclose(Modelin);
      */
      //get the transitions
      Modelin = transIn;
      Modelparse();
      fclose(transIn);
      /*
      while(true) {
	ContLoc *loc = NULL;
	fscanf(in,"%p",&loc);
	if(loc == 0x0) break;
	Util::Message(4,"restoring succs and actions for " + loc->ToString() + 
		      " with predicates : " + loc->GetPreds().ToString() + "\n");
	loc->RestoreSuccsAndActions(in);
      }
      */
      return true;
    } else return false;
  } else return false;
}

/*********************************************************************/
//save the implementation states
/*********************************************************************/
void ProcManager::SaveImplStates()
{
  Timer msTimer; msTimer.Start();
  int x = -1;
  if(seedMap.count(seedBranches) == 0) {
    x = seedMap.size();
    seedMap[seedBranches] = x;
  } else x = seedMap[seedBranches];
  char stateBuf[128];
  snprintf(stateBuf,128,"implstate-%d-%d-%d",getpid(),index,x);
  FILE *out = fopen(stateBuf,"w");
  //output magic number
  fprintf(out,"#MAGIC_STATE_FILE#\n");
  //output the reachable impl states
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    fprintf(out,"%p <",(*i)->GetContLoc());
    (*i)->GetPredVal().Write(out);
    fprintf(out,"> %hd %hd ",(*i)->GetSpecStateId(),(*i)->GetSpecInfoId());
    set<Expr> props; (*i)->GetPropositions(props);
    for(set<Expr>::const_iterator j = props.begin();j != props.end();++j) {
      fprintf(out,"%p ",j->GetExpr());
    }
    fprintf(out,"0x0 ");
  }
  fprintf(out,"0x0 ");
  //output the actions
  for(set<Action>::const_iterator i = absLts.acts.begin();i != absLts.acts.end();++i) {
    fprintf(out,"%hd ",i->GetIndex());
  }
  fprintf(out,"()\n");
  //all done
  fclose(out);
  //save the transitions
  x = -1;
  if(seedMap.count(seedBranches) == 0) {
    x = seedMap.size();
    seedMap[seedBranches] = x;
  } else x = seedMap[seedBranches];
  char transBuf[128];
  snprintf(transBuf,128,"impltrans-%d-%d-%d",getpid(),index,x);
  out = fopen(transBuf,"w");
  //output magic number
  fprintf(out,"#MAGIC_TRANS_FILE#\n");
  for(size_t i = 0;i < usefulLocs.size();++i) {
    Util::Message(4,"backing up succs and actions for " + usefulLocs[i]->ToString() + 
		  " with predicates : " + usefulLocs[i]->GetPreds().ToString() + "\n");
    fprintf(out,"%p ",usefulLocs[i]);
    usefulLocs[i]->BackupSuccsAndActions(out);
  }
  fprintf(out,"0x0\n");
  fclose(out);
  msTimer.Stop();
  Util::Message(2,"model saved in %.1f milliseconds ...\n",msTimer.Read() * 1000);
}

/*********************************************************************/
//compute the set of implementation states for each control location
/*********************************************************************/
void ProcManager::InitializeImplStates(set<Action> &newChanActs)
{
  //display the seed branches
  Util::Message(3,"********* begin seed branches ***********\n");
  for(set<ContLoc*>::const_iterator i = seedBranches.begin();i != seedBranches.end();++i) {
    Util::Message(3,(*i)->ToString() + "\n");
  }
  Util::Message(3,"********* end seed branches ***********\n");
  //setup the locations for computing implementation states
  SetupLocs();
  //compute the spec infos
  for(size_t i = 0;i < usefulLocs.size();++i) {
    usefulLocs[i]->ComputeSpecInfos();
  }  
  //compute the reachable implementation states
  reachImpl.clear();
  absLts.acts.clear();
  assert(Database::USE_REACH_IMPL);
  //first check the cache if necessary
  if(Database::CACHE_IMPL_TRANS && LoadImplStates(newChanActs)) {
    Util::Message(2,"model loaded from file ...\n");
    return;
  }
  //if not in cache compute
  set<const ImplState*> front = initLoc->GetImplStates(guard);
  while(!front.empty()) {
    reachImpl.insert(front.begin(),front.end());
    set<const ImplState*> newFront = front;
    front.clear();
    for(set<const ImplState*>::const_iterator i = newFront.begin();i != newFront.end();++i) {	
      set<Expr> tp; ComputeTrueProps((*i)->GetContLoc(),(*i)->GetPredVal(),tp);
      (*i)->GetContLoc()->SetPropositions(*i,tp);
      map< Action,set<const ImplState*> > sa;
      (*i)->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
      //if this is a call-site, add the alphabet of the inlined spec
      //LTS to the alphabet of the model
      if((*i)->GetSpecInfoId() != -1) {	
	const set<Action> &inlinedSpecActs = Database::ltsInfos[(*i)->GetSpecInfoId()].GetActions();
	for(set<Action>::const_iterator i = inlinedSpecActs.begin();i != inlinedSpecActs.end();++i) {
	  if(!(*i == ActionManager::GetSpecEpsilonAction())) {
	    if(i->IsBcast() || i->IsSend() || (i->IsAssign() && IsConcGlobalVar(i->GetLhsExpr()))) {
	      newChanActs.insert(*i);
	    }
	    absLts.acts.insert(*i);
	  }
	}
      }
      for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
	for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {
	  if(reachImpl.count(*l) == 0) front.insert(*l);
	  if(k->first.IsBcast() || k->first.IsSend() ||
	     (k->first.IsAssign() && IsConcGlobalVar(k->first.GetLhsExpr()))) {
	    newChanActs.insert(k->first);
	  }
	  absLts.acts.insert(k->first);
	}
      }
    }
  }
  //no spec epsilon actions should be present
  assert(absLts.acts.count(ActionManager::GetSpecEpsilonAction()) == 0);
  //clear caches associated with predsets
  PredSet::ClearCache();
  //update the cache if necessary
  if(Database::CACHE_IMPL_TRANS) SaveImplStates();
}

/*********************************************************************/
//update the implementation states on the basis of pending channel
//actions. return the new channel actions.
/*********************************************************************/
void ProcManager::UpdateImplStates(const set<Action> &pendChanActs,set<Action> &newChanActs)
{
  //split the actions into global variable actions and normal channel
  //actions
  set<Action> globActs,chanActs;
  for(set<Action>::const_iterator i = pendChanActs.begin();i != pendChanActs.end();++i) {
    if(i->IsAssign()) globActs.insert(*i);
    else {
      assert(i->IsBcast() || i->IsSend());
      chanActs.insert(*i);
    }
  }
  //initialise the frontier
  set<const ImplState*> front;
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    map< Action,set<const ImplState*> > sa;
    (*i)->GetSuccsAndActions(globActs,chanActs,sa);
    //if this is a call-site, add the alphabet of the inlined spec
    //LTS to the alphabet of the model
    if((*i)->GetSpecInfoId() != -1) {	
      const set<Action> &inlinedSpecActs = Database::ltsInfos[(*i)->GetSpecInfoId()].GetActions();
      for(set<Action>::const_iterator i = inlinedSpecActs.begin();i != inlinedSpecActs.end();++i) {
	if(!(*i == ActionManager::GetSpecEpsilonAction())) {
	  if(i->IsBcast() || i->IsSend() || (i->IsAssign() && IsConcGlobalVar(i->GetLhsExpr()))) {
	    newChanActs.insert(*i);
	  }
	  absLts.acts.insert(*i);
	}
      }
    }
    for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
      for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {
	if(reachImpl.count(*l) == 0) front.insert(*l);
	if(k->first.IsBcast() || k->first.IsSend() ||
	   (k->first.IsAssign() && IsConcGlobalVar(k->first.GetLhsExpr()))) {
	  newChanActs.insert(k->first);
	}
	absLts.acts.insert(k->first);
      }
    }
  }
  //compute the reachable states
  while(!front.empty()) {
    reachImpl.insert(front.begin(),front.end());
    set<const ImplState*> newFront = front;
    front.clear();
    for(set<const ImplState*>::const_iterator i = newFront.begin();i != newFront.end();++i) {	
      set<Expr> tp; ComputeTrueProps((*i)->GetContLoc(),(*i)->GetPredVal(),tp);
      (*i)->GetContLoc()->SetPropositions(*i,tp);
      map< Action,set<const ImplState*> > sa;
      (*i)->GetSuccsAndActions(globActs,chanActs,sa);
      //if this is a call-site, add the alphabet of the inlined spec
      //LTS to the alphabet of the model
      if((*i)->GetSpecInfoId() != -1) {	
	const set<Action> &inlinedSpecActs = Database::ltsInfos[(*i)->GetSpecInfoId()].GetActions();
	for(set<Action>::const_iterator i = inlinedSpecActs.begin();i != inlinedSpecActs.end();++i) {
	  if(!(*i == ActionManager::GetSpecEpsilonAction())) {
	    if(i->IsBcast() || i->IsSend() || (i->IsAssign() && IsConcGlobalVar(i->GetLhsExpr()))) {
	      newChanActs.insert(*i);
	    }
	    absLts.acts.insert(*i);
	  }
	}
      }
      for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
	for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {
	  if(reachImpl.count(*l) == 0) front.insert(*l);
	  if(k->first.IsBcast() || k->first.IsSend() ||
	     (k->first.IsAssign() && IsConcGlobalVar(k->first.GetLhsExpr()))) {
	    newChanActs.insert(k->first);
	  }
	  absLts.acts.insert(k->first);
	}
      }
    }
  }
  //no spec epsilon actions should be present
  assert(absLts.acts.count(ActionManager::GetSpecEpsilonAction()) == 0);
  //clear caches associated with predsets
  PredSet::ClearCache();
}

/*********************************************************************/
//create the set of inferred predicates corresponding to the branch
//location passed as argument
/*********************************************************************/
void ProcManager::ComputeInferredPreds(ContLoc *seedLoc)
{
  if(locToPreds.count(seedLoc) == 0) {
    ClearPredSets();
    vector<Predicate> pvec;
    seedLoc->AddPred(Predicate(seedLoc->GetLocExpr(),Predicate::REQUIRED,pair<const ContLoc*,int>(seedLoc,1)),pvec);
    InitialiseLocs();
    ComputeRelevantPreds();
    RemoveUselessContLocs();
    for(vector<ContLoc*>::const_iterator i = allLocs.begin();i != allLocs.end ();++i) {
      locToPreds[seedLoc][*i] = (*i)->GetPreds();
    }
    locToUseful[seedLoc].insert(usefulLocs.begin(),usefulLocs.end());
  }
}

/*********************************************************************/
//set up the control locations
/*********************************************************************/
void ProcManager::SetupLocs()
{
  Timer lsTimer; lsTimer.Start();
  Util::Message(3,"begin computing necessary predicates for component #%d called %s ...\n",index,name.c_str());
  //if new predicates are to be inferred just from the counterexample
  if(Database::PREDS_FROM_CE) {
    ClearPredSets();
    if(Database::CHECK_INVALID_PTR_DEREF || Database::CHECK_ASSERTION_FAILURE) {
      Expr pred;
      if(Database::CHECK_INVALID_PTR_DEREF) {
	Expr vpp = ExprManager::GetIdExpr(Database::VALID_PTR_PRED);
	list<Expr> el; el.push_back(markedLoc->GetIpdExpr());
	pred = ExprManager::GetParExpr(vpp,el);
      } else {
	pair< Expr,list<Expr> > comps; 
	ExprManager::GetCalledProcDetails(markedLoc->GetLocRhs(),comps);
	assert(comps.second.size() == 1);
	pred = comps.second.front();
      }
      vector<Predicate> pvec;
      markedLoc->AddPred(Predicate(pred,Predicate::REQUIRED,pair<const ContLoc*,int>(markedLoc,1)),pvec);
    }
    for(set<ContLoc*>::const_iterator i = seedBranches.begin();i != seedBranches.end();++i) {
      vector<Predicate> pvec;
      (*i)->AddPred(Predicate((*i)->GetLocExpr(),Predicate::REQUIRED,pair<const ContLoc*,int>(*i,1)),pvec);
    }
    ComputeNecessaryPredicates();
    InitialiseLocs();
    ComputeRelevantPreds();
  } 
  //if new predicates are to be inferred using information other than
  //just the counterexample
  else {
    if(Database::CEGAR) {
      //if we are doing predicate optimization then compute inferred
      //predicates for all branch locations
      if(Database::CEGAR_TYPE == Database::CEGAR_OPTIMIZE_PRED) {
	for(set<ContLoc*>::const_iterator i = branchLocs.begin();i != branchLocs.end();++i) {
	  ComputeInferredPreds(*i);
	}
      }
      //otherwise compute inferred predicates for only new seed branches
      else {
	set<ContLoc*> newSeedLocs;
	for(set<ContLoc*>::const_iterator i = seedBranches.begin();i != seedBranches.end();++i) {
	  if(locToPreds.count(*i) == 0) newSeedLocs.insert(*i);
	}
	//collect the set of predicates at each control for each branch
	//location that is a new seed not seen before
	for(set<ContLoc*>::const_iterator i = newSeedLocs.begin();i != newSeedLocs.end ();++i) {
	  ComputeInferredPreds(*i);
	}
      }
      //compute necessary predicates
      if(necPreds.empty()) {
	ClearPredSets();
	ComputeNecessaryPredicates();
	InitialiseLocs();
	ComputeRelevantPreds();
	RemoveUselessContLocs();
	for(vector<ContLoc*>::const_iterator i = allLocs.begin();i != allLocs.end ();++i) {
	  necPreds[*i] = (*i)->GetPreds();
	}
	necUseful.insert(usefulLocs.begin(),usefulLocs.end());
      }
      //now compute the real set of predicates
      ClearPredSets();
      if(Database::CHECK_INVALID_PTR_DEREF || Database::CHECK_ASSERTION_FAILURE) {
	Expr pred;
	if(Database::CHECK_INVALID_PTR_DEREF) {
	  Expr vpp = ExprManager::GetIdExpr(Database::VALID_PTR_PRED);
	  list<Expr> el; el.push_back(markedLoc->GetIpdExpr());
	  pred = ExprManager::GetParExpr(vpp,el);
	} else {
	  pair< Expr,list<Expr> > comps; 
	  ExprManager::GetCalledProcDetails(markedLoc->GetLocRhs(),comps);
	  assert(comps.second.size() == 1);
	  pred = comps.second.front();
	}
	vector<Predicate> pvec;
	markedLoc->AddPred(Predicate(pred,Predicate::REQUIRED,pair<const ContLoc*,int>(markedLoc,1)),pvec);
	InitialiseLocs();
	ComputeRelevantPreds();
      } else InitialiseLocs();
      for(set<ContLoc*>::const_iterator i = seedBranches.begin();i != seedBranches.end();++i) {
	for(vector<ContLoc*>::const_iterator j = allLocs.begin();j != allLocs.end();++j) {
	  (*j)->GetPreds().Merge(locToPreds[*i][*j]);
	}
      }
      for(vector<ContLoc*>::const_iterator i = allLocs.begin();i != allLocs.end();++i) {
	(*i)->GetPreds().Merge(necPreds[*i]);
      }
      //if((Database::CONF_TYPE == Database::CONF_LTL) || (Database::CONF_TYPE == Database::CONF_SEAW)) AddLogicPreds();
      set<ContLoc*> ulocs = necUseful;
      for(set<ContLoc*>::const_iterator i = seedBranches.begin();i != seedBranches.end();++i) {
	ulocs.insert(locToUseful[*i].begin(),locToUseful[*i].end());
      }
      for(vector<ContLoc*>::const_iterator i = allLocs.begin();i != allLocs.end();++i) {
	if((*i)->GetLocType() == ContLoc::ASSIGN_EXPR) {
	  if(ulocs.count(*i) != 0) (*i)->SetUseful(true);
	}
      }
    } else {
      ClearPredSets();
      if(Database::CHECK_INVALID_PTR_DEREF || Database::CHECK_ASSERTION_FAILURE) {
	Expr pred;
	if(Database::CHECK_INVALID_PTR_DEREF) {
	  Expr vpp = ExprManager::GetIdExpr(Database::VALID_PTR_PRED);
	  list<Expr> el; el.push_back(markedLoc->GetIpdExpr());
	  pred = ExprManager::GetParExpr(vpp,el);
	} else {
	  pair< Expr,list<Expr> > comps;
	  ExprManager::GetCalledProcDetails(markedLoc->GetLocRhs(),comps);
	  assert(comps.second.size() == 1);
	  pred = comps.second.front();
	}
	vector<Predicate> pvec;
	markedLoc->AddPred(Predicate(pred,Predicate::REQUIRED,pair<const ContLoc*,int>(markedLoc,1)),pvec);
      }
      for(set<ContLoc*>::const_iterator i = seedBranches.begin();i != seedBranches.end();++i) {
	vector<Predicate> pvec;
	(*i)->AddPred(Predicate((*i)->GetLocExpr(),Predicate::REQUIRED,pair<const ContLoc*,int>(*i,1)),pvec);
      }
      ComputeNecessaryPredicates();
      InitialiseLocs();
      ComputeRelevantPreds();
    }
  }
  Util::Message(3,"done computing necessary predicates for component #%d called %s ...\n",index,name.c_str());
  RemoveUselessContLocs();
  lsTimer.Stop();
  Util::Message(3,"locations set up in %.1f milliseconds ...\n",lsTimer.Read() * 1000);
}

/*********************************************************************/
//initialise stuff before seting up control locations
/*********************************************************************/
void ProcManager::InitialiseLocs() const
{
  //restore all important fields of the control locations,cleanup
  //implementation states, specification state names, successor
  //implementation states and simulation relation and set the useful
  //fields to false.
  for(size_t i = 0;i < allLocs.size();++i) {
    allLocs[i]->Restore();
    if(allLocs[i]->GetLocType() == ContLoc::ASSIGN_EXPR) {
      allLocs[i]->SetUseful(false);
    }
  }
}

/*********************************************************************/
//clear predicate sets at all control locations
/*********************************************************************/
void ProcManager::ClearPredSets()
{ 
  for(size_t i = 0;i < allLocs.size();++i) {
    allLocs[i]->GetPreds().Clear();
  }
}

/*********************************************************************/
//compute the set of predicates relevant for each location. the first
//argument is the call graph. the second argument is the info about
//the main procedure. the third argument is the predicate qualifying
//the initial state. also compute usefulness of locations w.r.t. the
//computed set of predicates.
/*********************************************************************/
void ProcManager::ComputeRelevantPreds()
{ 
  //initialise the working set
  map< ContLoc*,pair< vector<Predicate>,vector<Predicate> > > wset;
  for(size_t i = 0;i < allLocs.size();++i) {
    ContLoc *cloc = allLocs[i];
    const vector<Predicate> &pvec = cloc->GetPreds().GetPreds();
    const set<ContLoc*> &plocs = cloc->GetPredecessors();
    for(set<ContLoc*>::const_iterator j = plocs.begin();j != plocs.end();++j) {
      if(cloc == (*j)->GetTSucc()) {
	wset[*j].first.insert(wset[*j].first.end(),pvec.begin(),pvec.end());
      } else if(cloc == (*j)->GetESucc()) {
	wset[*j].second.insert(wset[*j].second.end(),pvec.begin(),pvec.end());
      } else assert(false);
    }
  }
  
  //compute the relevant predicates
  while(!wset.empty()) {
    ContLoc *cloc = wset.begin()->first;
    const pair< vector<Predicate>,vector<Predicate> > &pvecs = wset.begin()->second;
    vector<Predicate> pvec;
    cloc->UpdatePreds(pvecs,pvec);
    wset.erase(wset.begin());
    if(!pvec.empty()) {
      const set<ContLoc*> &plocs = cloc->GetPredecessors();
      for(set<ContLoc*>::const_iterator i = plocs.begin();i != plocs.end();++i) {
	if(cloc == (*i)->GetTSucc()) {
	  wset[*i].first.insert(wset[*i].first.end(),pvec.begin(),pvec.end());
	} else if(cloc == (*i)->GetESucc()) {
	  wset[*i].second.insert(wset[*i].second.end(),pvec.begin(),pvec.end());
	} else assert(false);
      }
    }
  }

  //compute the usefulness of the locations
  for(size_t i = 0;i < allLocs.size();++i) {
    allLocs[i]->ComputeUseful(*this);
  }
}

/*********************************************************************/
//remove useless assign expression locations and branch locations
//whose true and false successors are the same. finally recreate the
//global control location array.
/*********************************************************************/
void ProcManager::RemoveUselessContLocs()
{
  //remove useless locations
  if(!initSkipped.empty()) {
    initLoc = initSkipped.front();
    initSkipped.clear();
  }
  list<ContLoc*> contLocs;
  for(size_t i = 0;i < allLocs.size();++i) {
    allLocs[i]->GetTSkipped().clear();
    allLocs[i]->GetESkipped().clear();
    allLocs[i]->GetCallSkipped().clear();
    contLocs.push_back(allLocs[i]);	
  }

  size_t removed = 0;
  if(Database::NO_USELESS_LOCS) {
    bool flag = false;
    do {
      flag = false;
      for(list<ContLoc*>::iterator i = contLocs.begin();i != contLocs.end();) {
	ContLoc *a = *i;
	if((a != a->GetTSucc()) && (a != initLoc) &&
	   (((a->GetLocType() == ContLoc::ASSIGN_EXPR) && (!a->IsUseful())) ||
	    (Database::CHECK_ASSERTION_FAILURE && (a->GetLocType() == ContLoc::ASSIGN_CALL) && (a != markedLoc)) ||
	    ((a->GetLocType() == ContLoc::BRANCH) && (a->GetTSucc() == a->GetESucc())))) {
	  for(list<ContLoc*>::iterator j = contLocs.begin();j != contLocs.end();++j) {
	    if(a == *j) continue;
	    (*j)->ReplaceSucc(a,a->GetTSucc());
	  }
	  if(initLoc == a) {
	    initSkipped.push_back(a);
	    initLoc = a->GetTSucc();
	  }
	  i = contLocs.erase(i);
	  ++removed;
	  flag = true;
	} else {
	  ++i;
	}
      }
    } while(flag);
  }

  //create the useful location vector
  usefulLocs.clear();
  size_t id = 0;
  for(list<ContLoc*>::iterator i = contLocs.begin();i != contLocs.end();++i,++id) {
    usefulLocs.push_back(*i);
    (*i)->SetLocId(id * Database::rootProcNames.size() + index);
  }
  assert(usefulLocs.size() == (allLocs.size() - removed));
  Util::Message(4,"array of useful control locations created ...\n");
}

/*********************************************************************/
//returns true if the argument is a silent action - neither a
//communication action nor a specification action
/*********************************************************************/
bool ProcManager::IsSilentAct(const Action &act) const
{
  const set<Action> &specActs = Database::GetSpecActions();
  if(specActs.count(act) != 0) return false;
  if(act.IsInvalidPtrDeref() || act.IsAssertionFailure()) return false;
  if(act.IsReturn()) return !act.IsDefaultRet();
  if(act.IsAssign()) return ((!Database::HANDLE_DATA_COMM) || (!IsConcGlobalVar(act.GetLhsExpr())));
  Action compAct;
  if(act.IsBcast() || act.IsSend() || act.IsComplement()) {
    compAct = act.Complement();
    if(act.IsBcast() || act.IsSend()) {
      if(specActs.count(Action(act.GetChannel())) != 0) return false;
      if(!(Database::HANDLE_DATA_COMM || act.GetMessage().empty())) return true;
    } else if(act.IsComplement()) {
      if(specActs.count(Action(compAct.GetChannel())) != 0) return false;
      if(!(Database::HANDLE_DATA_COMM || compAct.GetMessage().empty())) return true;
    } 
  } else if(act.IsRecv()) {
    return !(Database::HANDLE_DATA_COMM || act.GetStore().empty());
  } else {
    assert(act.IsBasic());
    compAct = act;
  }
  for(size_t i = 0;i < Database::components.size();++i) {
    if((i != index) && (Database::components[i]->IsL2AbsAct(compAct))) return false;
  }
  return true;
}

/*********************************************************************/
//return true if the argument is a silent transition. a silent
//transition is a non-self loop labeled by a silent action and has
//identical atomic proposition labeling on its source and destination
//states. also the source state must have exactly one outgoing
//transition. a silent action is one that does not appear in the
//specification and is also not a communication action.
/*********************************************************************/
bool ProcManager::IsSilentTrans(const ImplTrans &trans,const map< const ImplState*,set<ImplTrans> > &succMap) const
{
  if(trans.source == trans.dest) return false;
  map< const ImplState*,set<ImplTrans> >::const_iterator it = succMap.find(trans.source);
  assert(it != succMap.end());
  if(it->second.size() != 1) return false;
  if(!IsSilentAct(trans.act)) return false;
  set<Expr> prop1; trans.source->GetPropositions(prop1);
  set<Expr> prop2; trans.dest->GetPropositions(prop2);
  return (prop1 == prop2);
}

/*********************************************************************/
//return true if the argument is a silent transition. a silent loop is
//labeled by a silent action and has an out degree of at least 2. a
//silent action is one that does not appear in the specification and
//is also not a communication action.
/*********************************************************************/
bool ProcManager::IsSilentLoop(const ImplTrans &trans,const map< const ImplState*,set<ImplTrans> > &succMap) const
{
  if(trans.source != trans.dest) return false;
  if(!IsSilentAct(trans.act)) return false;
  map< const ImplState*,set<ImplTrans> >::const_iterator it = succMap.find(trans.source);
  assert(it != succMap.end());
  return (it->second.size() > 1);
}

/*********************************************************************/
//shortcut a transition and update local data structures
/*********************************************************************/
void ProcManager::ShortcutTrans(const ImplTrans &trans1,const ImplTrans &trans2,map< const ImplState*,set<ImplTrans> > &succMap,set<ImplTrans> &allTrans)
{
  assert((skipPathMap.count(trans1) != 0) && (skipPathMap.count(trans2) != 0));
  ImplTrans newTrans(trans1.source,trans2.act,trans2.dest);
  if(skipPathMap.count(newTrans) == 0) {
    //update local data structures
    skipPathMap[newTrans] = skipPathMap[trans1];
    skipPathMap[newTrans].PushBack(trans1.act,trans1.dest);
    skipPathMap[newTrans].Append(skipPathMap[trans2]);
    succMap[trans1.source].insert(newTrans);
    //update data structures in the control location
    map< Action,set<const ImplState*> > x;
    x[newTrans.act].insert(newTrans.dest);
    newTrans.source->GetContLoc()->AddSuccsAndActions(newTrans.source->GetKey(),x);
    allTrans.insert(newTrans);
  }
}

/*********************************************************************/
//remove a transition and update local data structures
/*********************************************************************/
void ProcManager::RemoveTrans(const ImplTrans &trans,map< const ImplState*,set<ImplTrans> > &succMap,set<ImplTrans> &allTrans)
{
  //update local data structures
  skipPathMap.erase(trans);
  succMap[trans.source].erase(trans);
  //update data structures in the control location
  map< Action,set<const ImplState*> > x;
  x[trans.act].insert(trans.dest);
  trans.source->GetContLoc()->RemSuccsAndActions(trans.source->GetKey(),x);
  allTrans.erase(trans);
}

/*********************************************************************/
//remove silent transitions from the transition relation. store enough
//information so that these transitions may be reconstructed later on.
/*********************************************************************/
void ProcManager::EliminateSilentTrans()
{
  //gather initial data
  skipPathMap.clear();
  set<ImplTrans> allTrans;
  map< const ImplState*,set<ImplTrans> > succMap;
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    map< Action,set<const ImplState*> > saa;
    (*i)->GetSuccsAndActions(set<Action>(),set<Action>(),saa);
    for(map< Action,set<const ImplState*> >::const_iterator j = saa.begin();j != saa.end();++j) {
      for(set<const ImplState*>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	ImplTrans tr(*i,j->first,*k);
	skipPathMap[tr] = ImplPath();
	allTrans.insert(tr);
	succMap[*i].insert(tr);
      }
    }
  }
  if(!Database::NO_SILENT_TRANS) return;
  //do forever
  while(true) {
    //collect silent transitions
    set<ImplTrans> silent;
    for(set<ImplTrans>::const_iterator i = allTrans.begin();i != allTrans.end();++i) {
      if(IsSilentTrans(*i,succMap)) silent.insert(*i);
    }
    if(silent.empty()) break;
    for(set<ImplTrans>::const_iterator i = silent.begin();i != silent.end();++i) {
      for(set<ImplTrans>::const_iterator j = succMap[i->dest].begin();j != succMap[i->dest].end();++j) {
	ShortcutTrans(*i,*j,succMap,allTrans);
      }
      RemoveTrans(*i,succMap,allTrans);
    }
  }
  //remove silent self-loops if there is some outgoing transition
  while(true) {
    //collect silent self-loops
    set<ImplTrans> silent;
    set<const ImplState*> loopSrc;
    for(set<ImplTrans>::const_iterator i = allTrans.begin();i != allTrans.end();++i) {
      if((loopSrc.count(i->source) == 0) && IsSilentLoop(*i,succMap)) {
	silent.insert(*i);
	loopSrc.insert(i->source);
      }
    }
    if(silent.empty()) break;
    for(set<ImplTrans>::const_iterator i = silent.begin();i != silent.end();++i) {
      RemoveTrans(*i,succMap,allTrans);
    }
  }
}

/*********************************************************************/
//restore the silent transitions into a counterexample dag
/*********************************************************************/
void ProcManager::RestoreSilentTrans(ConcCEDag &concDag) const
{
  vector<L1AbsState*> stateLabel = concDag.stateLabel;
  concDag.stateLabel.clear();
  vector<Action> actLabel = concDag.actLabel;
  concDag.actLabel.clear();
  vector< set<int> > trans = concDag.trans;
  concDag.trans.clear(); 
  map<int,int> oldToNew;
  assert(RestoreSilentWithRoot(0,oldToNew,stateLabel,actLabel,trans,concDag) == 0);
  for(size_t i = 0;i < stateLabel.size();++i) delete stateLabel[i];
}

/*********************************************************************/
//restore silent transitions starting from a root node
/*********************************************************************/
int ProcManager::RestoreSilentWithRoot(const int root,map<int,int> &oldToNew,const vector<L1AbsState*> &stateLabel,
				       const vector<Action> &actLabel,const vector< set<int> > &trans,
				       ConcCEDag &concDag) const
{
  if(oldToNew.count(root) == 0) {
    oldToNew[root] = concDag.stateLabel.size();
    concDag.stateLabel.push_back(stateLabel[root]->Clone());
    concDag.trans.push_back(set<int>());
    const Action &act = actLabel[root];  
    if(IsL2AbsAct(act)) {
      const ImplState *from = static_cast<CL1AbsState*>(stateLabel[root])->GetImplState();
      ImplTrans nextTrans;
      if(trans[root].empty()) {
	if(Database::CONF_TYPE != Database::CONF_DLOCK) {
	  set<const ImplState*> succs; from->GetSuccsOnAction(act,succs);
	  assert(!succs.empty());
	  const ImplState *to = *(succs.begin());
	  nextTrans = ImplTrans(from,act,to);	
	}
      } else {
	int nextNode = *(trans[root].begin());
	const ImplState *to = static_cast<CL1AbsState*>(stateLabel[nextNode])->GetImplState();
	nextTrans = ImplTrans(from,act,to);
      }
      if(nextTrans.source != NULL) {
	map<ImplTrans,ImplPath>::const_iterator it = skipPathMap.find(nextTrans);
	assert(it != skipPathMap.end());
	for(list< pair<Action,const ImplState*> >::const_iterator j = it->second.data.begin();j != it->second.data.end();++j) {
	  int currNode = concDag.stateLabel.size();
	  concDag.trans.back().insert(currNode);
	  concDag.stateLabel.push_back(new CL1AbsState(j->second));
	  concDag.actLabel.push_back(j->first);
	  concDag.trans.push_back(set<int>());
	}
      }
    }
    concDag.actLabel.push_back(act);
    size_t lastId = concDag.trans.size() - 1;
    for(set<int>::const_iterator i = trans[root].begin();i != trans[root].end();++i) {
      int newSucc = RestoreSilentWithRoot(*i,oldToNew,stateLabel,actLabel,trans,concDag);
      concDag.trans[lastId].insert(newSucc);
    }
  }
  return oldToNew[root];
}

/*********************************************************************/
//create the initial abstract states by grouping together
//implementation states with the same set of outgoing actions
/*********************************************************************/
void ProcManager::CreateInitialAbstractStates()
{
  absLts.stateNum = 0;
  absLts.propMap.clear();
  absLts.refMap.clear();
  map< pair< set<Expr>,set<Action> >,size_t > propActCombs;
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    ImplState *istate = (ImplState*)(*i);
    //if we need to use lts abstraction
    if(Database::USE_LTS_ABSTRACTION) {
      pair< set<Expr>,set<Action> > propActs;
      istate->GetPropositions(propActs.first);
      map< pair< set<Expr>,set<Action> >,size_t >::iterator it = propActCombs.find(propActs);
      if(it == propActCombs.end()) {
	propActCombs[propActs] = absLts.stateNum;
	istate->SetAbsId(absLts.stateNum);
	absLts.propMap[absLts.stateNum] = propActs.first;
	istate->GetRefusal(absLts.refMap[absLts.stateNum]);
	++absLts.stateNum;
      } else {
	istate->SetAbsId(it->second);
	set<Action> ref;
	istate->GetRefusal(ref);
	absLts.refMap[it->second].insert(ref.begin(),ref.end());
      }
    }
    //otherwise each abstract states contains just a single
    //implementation state
    else {
      istate->SetAbsId(absLts.stateNum);      
      istate->GetPropositions(absLts.propMap[absLts.stateNum]);
      istate->GetRefusal(absLts.refMap[absLts.stateNum]);
      ++absLts.stateNum;
    }
  }
}

/*********************************************************************/
//compute abstract initial states and actions
/*********************************************************************/
void ProcManager::ComputeAbsLtsInit()
{
  Util::Message(2,"number of abstract implementation states = %d\n",absLts.stateNum);
  //compute initial states
  absLts.init.clear();
  set<const ImplState*> initImpls = initLoc->GetImplStates(guard);
  for(set<const ImplState*>::const_iterator i = initImpls.begin();i != initImpls.end();++i) {
    absLts.init.insert((*i)->GetAbsId());
  }
  //clear the transitions
  absLts.succMap.clear();
  absLts.predMap.clear();
}

/*********************************************************************/
//compute abstract transitions
/*********************************************************************/
void ProcManager::ComputeAbsLtsTrans()
{
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    map< Action,set<const ImplState*> > sa;
    (*i)->GetSuccsAndActions(set<Action>(),set<Action>(),sa);
    for(map< Action,set<const ImplState*> >::const_iterator k = sa.begin();k != sa.end();++k) {
      for(set<const ImplState*>::const_iterator l = k->second.begin();l != k->second.end();++l) {		
	absLts.succMap[(*i)->GetAbsId()][k->first].insert((*l)->GetAbsId());
	absLts.predMap[(*l)->GetAbsId()][k->first].insert((*i)->GetAbsId());
      }
    }
  }
}

/*********************************************************************/
//given a control location and a predicate valuation compute the set
//of true propositions in the corresponding impl state
/*********************************************************************/
void ProcManager::ComputeTrueProps(const ContLoc *loc,const PVal &pv,set<Expr> &res) const
{  
  if((Database::CONF_TYPE == Database::CONF_LTL) || (Database::CONF_TYPE == Database::CONF_SEAW)) {
    set<Expr> state; loc->GetPreds().ToExpr(loc->GetPreds().PValToPredVal(pv),state);
    for(set<Expr>::const_iterator i = props.begin();i != props.end();++i) {
      set<Expr> x; x.insert(*i);
      if(ExprManager::ProveImplies(state,x)) res.insert(*i);
    }
  }
}

/*********************************************************************/
//draw the control flow graph as a dot file
/*********************************************************************/
void ProcManager::DrawCFG() const
{
  FILE *out = Database::dotFilePtr;
  fprintf(out,"  //CFG of Component %s index %d\n",name.c_str(),index);
  fprintf(out,"  subgraph cluster_cfg_%s_%d {\n",name.c_str(),index);
  fprintf(out,"    color = white;\n");
  fprintf(out,"    cont_loc_%p [peripheries = 2];\n",initLoc);   
  for(vector<ContLoc*>::const_iterator i = allLocs.begin();i != allLocs.end();++i) {
    fprintf(out,"    cont_loc_%p [label = \"%s\"];\n",(*i),(*i)->ToString().c_str());    
    if((*i)->GetLocType() == ContLoc::BRANCH) {
      fprintf(out,"    cont_loc_%p -> cont_loc_%p [label = \"then\"];\n",*i,(*i)->GetTSucc());
      fprintf(out,"    cont_loc_%p -> cont_loc_%p [label = \"else\"];\n",*i,(*i)->GetESucc());
    } else if((*i)->GetLocType() != ContLoc::FINAL){
      fprintf(out,"    cont_loc_%p -> cont_loc_%p;\n",*i,(*i)->GetTSucc());
    }
  }
  fprintf(out,"    label = \"CFG of component #%d \\nProcedure name: %s\";\n",index,name.c_str());
  fprintf(out,"    labelloc = \"b\";\n");
  fprintf(out,"  }\n\n");
}

/*********************************************************************/
//return the set of level-1 abstract initial states. virtual method of
//Component.
/*********************************************************************/
void ProcManager::GetL1AbsInit(L1StSet &res) const
{ 
  set<const ImplState*> x = initLoc->GetImplStates(guard);
  if(x.size() == reachImpl.size()) GetCompleteL1StSet(res);
  else {
    set<L1AbsState*> y;
    for(set<const ImplState*>::const_iterator i = x.begin();i != x.end();++i) {
      y.insert(new CL1AbsState(*i));
    }
    res = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
}

/*********************************************************************/
//return the set of level-2 abstract initial states. virtual method of
//Component.
/*********************************************************************/
void ProcManager::GetL2AbsInit(set<L2AbsState> &res) { res = absLts.GetInit(); }

/*********************************************************************/
//return the set of level-2 abstract initial states from the previous
//iteration. virtual method of Component.
/*********************************************************************/
void ProcManager::GetPrevL2AbsInit(set<L2AbsState> &res) { res = absLts.GetPrevInit(); }

/*********************************************************************/
//the number of level-1 abstract states. virtual method of Component.
/*********************************************************************/
int ProcManager::GetL1AbsStateNum() { return reachImpl.size(); }

/*********************************************************************/
//return the number of level-2 abstract states. virtual method of
//Component.
/*********************************************************************/
int ProcManager::GetL2AbsStateNum() { return absLts.GetStateNum(); }

/*********************************************************************/
//set of successors of a level-2 abstract state on an action. virtual
//method of Component.
/*********************************************************************/
void ProcManager::GetL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res)
{
  absLts.GetSuccsOnAction(state,act,res);
}

/*********************************************************************/
//check validity of a level-2 abstract transition
/*********************************************************************/
bool ProcManager::IsL2AbsTrans(const L2AbsState &from,const Action &act,const L2AbsState &to)
{
  return absLts.IsTrans(from,act,to);
}

/*********************************************************************/
//set of predecessors of a level-2 abstract state on an
//action. virtual method of Component.
/*********************************************************************/
void ProcManager::GetL2AbsPredsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res)
{
  absLts.GetPredsOnAction(state,act,res);
}

/*********************************************************************/
//check if the argument action is an action of the level-2 abstraction
/*********************************************************************/
bool ProcManager::IsL2AbsAct(const Action &act) const
{
  return absLts.HasAct(act);
}

/*********************************************************************/
//return the set of level-2 abstract actions. virtual method of
//Component.
/*********************************************************************/
void ProcManager::GetL2AbsActs(set<Action> &res) const { res = absLts.GetActs(); }

/*********************************************************************/
//return the set of level-2 abstract actions enabled from a particular
//level-2 abstract state. virtual method of Component.
/*********************************************************************/
void ProcManager::GetL2AbsActsOut(const L2AbsState &state,set<Action> &res) const 
{
  absLts.GetActsOut(state,res);
}

/*********************************************************************/
//return the set of level-2 abstract actions incoming to a particular
//level-2 abstract state. virtual method of Component.
/*********************************************************************/
void ProcManager::GetL2AbsActsIn(const L2AbsState &state,set<Action> &res) const 
{
  absLts.GetActsIn(state,res);
}

/*********************************************************************/
//return the number of level-2 abstract states in the previous
//iteration. virtual method of Component.
/*********************************************************************/
int ProcManager::GetPrevL2AbsStateNum() { return absLts.GetPrevStateNum(); }

/*********************************************************************/
//get the set of level-2 abstract states whose successors might have
//changed due to action-guided abstraction refinement. virtual method
//of Component.
/*********************************************************************/
void ProcManager::GetL2AbsSuccsChanged(set<L2AbsState> &res)
{
  res = absLts.GetSuccsChanged();
}

/*********************************************************************/
//set of successors of a level-2 abstract state on an action from
//previous iteration. virtual method of Component.
/*********************************************************************/
void ProcManager::GetPrevL2AbsSuccsOnAction(const L2AbsState &state,const Action &act,set<L2AbsState> &res)
{
  absLts.GetPrevSuccsOnAction(state,act,res);
}

/*********************************************************************/
//return the set of propositions labeling a levell-2 abstract
//state. virtual method of Component.
/*********************************************************************/
void ProcManager::GetL2AbsProps(const L2AbsState &state,set<Expr> &res) 
{ 
  absLts.GetPropositions(state,res); 
}

/*********************************************************************/
//return true if the two sets of level-2 abstract states correspond to
//different sets of control locations. virtual method of Component.
/*********************************************************************/
bool ProcManager::DistinctL2AbsStates(const set<L2AbsState> &arg1,const set<L2AbsState> &arg2)
{
  set<ContLoc*> loc1,loc2;
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    if(arg1.count((*i)->GetAbsId()) != 0) {
      loc1.insert((*i)->GetContLoc());
    }
    if(arg2.count((*i)->GetAbsId()) != 0) {
      loc2.insert((*i)->GetContLoc());
    }
  }
  return (loc1 != loc2);
}

/*********************************************************************/
//return the set of all level-1 abstract states. virtual method of
//Component.
/*********************************************************************/
void ProcManager::GetCompleteL1StSet(L1StSet &res) const
{
  res = L1StSet::GetComplete();
}

/*********************************************************************/
//intersect two sets of level-1 abstract states and return the result
//in the first argument
/*********************************************************************/
void ProcManager::IntersectL1StSets(L1StSet &lhs,const L1StSet &rhs) const
{  
  lhs.Intersect(rhs);
}

/*********************************************************************/
//compute predecessors of a set of level-1 abstract states
/*********************************************************************/
L1StSet ProcManager::GetL1AbsPredsOnAction(const L1StSet &states,const Action &act) const
{
  set<const ImplState*> res;
  if(states.GetAll()) {
    for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
      set<const ImplState*> x; (*i)->GetPredsOnAction(act,x);
      res.insert(x.begin(),x.end());
    }
  } else {
    for(set<L1AbsState*>::const_iterator i = states.GetEle().begin();i != states.GetEle().end();++i) {
      const ImplState *is = static_cast<const CL1AbsState*>(*i)->GetImplState();
      set<const ImplState*> x; is->GetPredsOnAction(act,x);
      res.insert(x.begin(),x.end());
    }
  }
  L1StSet x;
  if(res.size() == reachImpl.size()) GetCompleteL1StSet(x);
  else {
    set<L1AbsState*> y;
    for(set<const ImplState*>::const_iterator i = res.begin();i != res.end();++i) {
      y.insert(new CL1AbsState(*i));
    }
    x = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
  return x;
}

/*********************************************************************/
//compute successors of a level-1 abstract state
/*********************************************************************/
void ProcManager::GetL1AbsSuccsOnAction(const L1AbsState &state,const Action &act,L1StSet &res) const
{
  const ImplState *is = static_cast<const CL1AbsState&>(state).GetImplState();
  set<const ImplState*> x; is->GetSuccsOnAction(act,x);
  if(x.size() == reachImpl.size()) GetCompleteL1StSet(res);
  else {
    set<L1AbsState*> y;
    for(set<const ImplState*>::const_iterator i = x.begin();i != x.end();++i) {
      y.insert(new CL1AbsState(*i));
    }
    res = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
}

/*********************************************************************/
//compute successors of a set of level-1 abstract states
/*********************************************************************/
L1StSet ProcManager::GetL1AbsSuccsOnAction(const L1StSet &states,const Action &act) const
{
  set<const ImplState*> res;
  if(states.GetAll()) {
    for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
      set<const ImplState*> x; (*i)->GetSuccsOnAction(act,x);
      res.insert(x.begin(),x.end());
    }
  } else {
    for(set<L1AbsState*>::const_iterator i = states.GetEle().begin();i != states.GetEle().end();++i) {
      const ImplState *is = static_cast<const CL1AbsState*>(*i)->GetImplState();
      set<const ImplState*> x; is->GetSuccsOnAction(act,x);
      res.insert(x.begin(),x.end());
    }
  }
  L1StSet x;
  if(res.size() == reachImpl.size()) GetCompleteL1StSet(x);
  else {
    set<L1AbsState*> y;
    for(set<const ImplState*>::const_iterator i = res.begin();i != res.end();++i) {
      y.insert(new CL1AbsState(*i));
    }
    x = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
  return x;
}

/*********************************************************************/
//project a set of level-1 abstract states on a level-2 abstract state
/*********************************************************************/
L1StSet ProcManager::ProjectL1StSetOnL2AbsState(const L1StSet &one,const L2AbsState &two) const
{
  set<const ImplState*> res;
  if(one.GetAll()) {
    for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
      if((*i)->GetAbsId() == two) res.insert(*i);
    }
  } else {
    for(set<L1AbsState*>::const_iterator i = one.GetEle().begin();i != one.GetEle().end();++i) {
      const ImplState *is = static_cast<const CL1AbsState*>(*i)->GetImplState();
      if(is->GetAbsId() == two) {
	res.insert(is);
      }
    }
  }
  L1StSet x;
  if(res.size() == reachImpl.size()) GetCompleteL1StSet(x);
  else {
    set<L1AbsState*> y;
    for(set<const ImplState*>::const_iterator i = res.begin();i != res.end();++i) {
      y.insert(new CL1AbsState(*i));
    }
    x = L1StSet(false,y);
    for(set<L1AbsState*>::const_iterator i = y.begin();i != y.end();++i) delete *i;
  }
  return x;
}

/*********************************************************************/
//compute the map from level-2 to leve-1 abstract states
/*********************************************************************/
void ProcManager::CreateL2ToL1AbsStateMap(map<L2AbsState,const L1AbsState*> &res) const
{
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    res[(*i)->GetAbsId()] = new CL1AbsState(*i);
  }
}

/*********************************************************************/
//pick a level-1 abstract state from a set
/*********************************************************************/
L1AbsState *ProcManager::PickL1AbsState(const L1StSet &arg) const
{
  if(arg.GetAll()) return new CL1AbsState(*(reachImpl.begin()));
  else {
    const set<L1AbsState*> &x = arg.GetEle();
    assert(!x.empty());
    return (*(x.begin()))->Clone();
  }
}

/*********************************************************************/
//check if a set of level-1 abstract states contains a level-1
//abstract state
/*********************************************************************/
bool ProcManager::L1AbsContains(const L1StSet &arg1,const L1AbsState &arg2) const
{
  if(arg1.GetAll()) return true;
  else {
    const set<L1AbsState*> &x = arg1.GetEle();
    if(x.empty()) return false;
    else return (*(x.begin()))->SetContains(x,arg2);
  }
}

/*********************************************************************/
//get the set of all level-1 abstract states
/*********************************************************************/
void ProcManager::GetAllL1AbsStates(set<L1AbsState*> &res) const
{
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    res.insert(new CL1AbsState(*i));
  }
}

/*********************************************************************/
//get the level-2 abstract state corresponding to a level-1 abstract
//state
/*********************************************************************/
void ProcManager::L1ToL2AbsState(const L1AbsState &arg,L2AbsState &res) const
{
  res = static_cast<const CL1AbsState&>(arg).GetImplState()->GetAbsId();
}

/*********************************************************************/
//create the L1StSet object corresponding to a set of level-1 abstract
//states
/*********************************************************************/
void ProcManager::CreateL1StSet(const set<L1AbsState*> &arg,L1StSet &res) const
{
  if(arg.size() == reachImpl.size()) GetCompleteL1StSet(res);
  else res = L1StSet(false,arg);
}

/*********************************************************************/
//reset the info for action-guided abstraction refinement
/*********************************************************************/
void ProcManager::ClearL2AbsRefineInfo()
{
  absLtsRefine.clear();
}

/*********************************************************************/
//update the info for action guided abstraction refinement
/*********************************************************************/
bool ProcManager::UpdateL2AbsRefineInfo(const L2AbsState &l2s,const L1StSet &l1s)
{
  //if the set of states to be separated is empty do nothing
  if(l1s.Empty()) return false;
  //if the set of states to be separated is everything do nothing
  L1StSet all; GetCompleteL1StSet(all);
  L1StSet proj = ProjectL1StSetOnL2AbsState(all,l2s);
  if(l1s == proj) return false;
  //separate
  if(absLtsRefine.count(l2s) == 0) {
    for(set<L1AbsState*>::const_iterator i = l1s.GetEle().begin();i != l1s.GetEle().end();++i) {
      absLtsRefine[l2s].insert(static_cast<const CL1AbsState*>(*i)->GetImplState());
    }
    return true;
  } else return false;
}

/*********************************************************************/
//level-1 abstract state to string
/*********************************************************************/
void ProcManager::L1AbsStateToString(const L1AbsState &arg,string &res) const
{
  res = static_cast<const CL1AbsState&>(arg).GetImplState()->ToString();
}

/*********************************************************************/
//set of level-1 abstract states to string
/*********************************************************************/
void ProcManager::L1StSetToString(const L1StSet &arg,string &res) const
{
  set<L1AbsState*> l1s;
  if(arg.GetAll()) GetAllL1AbsStates(l1s);
  else l1s = arg.GetEle();
  res = "[\n";
  for(set<L1AbsState*>::const_iterator i = l1s.begin();i != l1s.end();++i) {
    string x; L1AbsStateToString(**i,x);
    res += x + "\n";
    if(arg.GetAll()) delete (*i);
  }
  res += "]";
}

/*********************************************************************/
//compute string representation of a level-1 abstract state
/*********************************************************************/
void ProcManager::L1AbsStateToString(const L1AbsState &curr,const L1AbsState *next,const bool stutter,string &res) const
{      
  const CL1AbsState &ccurr = static_cast<const CL1AbsState&>(curr);
  const CL1AbsState *cnext = static_cast<const CL1AbsState*>(next);
  const ContLoc *loc = ccurr.GetContLoc();
  res = loc->ToString();
  if((Database::CE_DISPLAY_TYPE == Database::CE_LOC_CONS) || (Database::CE_DISPLAY_TYPE == Database::CE_LOC_CONS_ACT)) {
    res +=  " : " + ExprManager::ExprSetToString(ccurr.GetCons());
  }
  if(loc->GetLocType() == ContLoc::BRANCH) {
    if(cnext != NULL) {
      if(cnext->GetContLoc() == loc->GetTSucc()) {
	res += " : TRUE";
      } else if(cnext->GetContLoc() == loc->GetESucc()) {
	res += " : FALSE";
      } else {
	if(!(stutter && (loc == cnext->GetContLoc()))) {
	  Util::Message(2,loc->ToString() + " === stutter = " + (stutter ? "yes" : "no") + " ==> " + cnext->GetContLoc()->ToString() + "\n");
	  assert(false);
	}
	res += " : STUTTER";
      }
    }
  } else {
    if(stutter) {
      assert((cnext == NULL) || (loc == cnext->GetContLoc()));
      res += " : STUTTER";
    }
  }
}

/*********************************************************************/
//convert a level-1 CE dag to a set of paths
/*********************************************************************/
void ProcManager::L1AbsDagToPaths(const vector<L1AbsState*> &stateLabel,const vector<Action> &actLabel,const vector< set<int> > &trans,const vector<int> &topo,const set<Action> &absActs,set<ConcPath> &res) const
{
  typedef pair<pair< pair< const ContLoc*,set<Expr> >,pair<short,char> >,Action> CConcEle;
  typedef list<CConcEle> CConcPath;
  vector< set<CConcPath> > locPaths(stateLabel.size());
  for(vector<int>::const_iterator i = topo.begin();i != topo.end();++i) {
    for(set<int>::const_iterator j = trans[*i].begin();j != trans[*i].end();++j) {
      locPaths[*i].insert(locPaths[*j].begin(),locPaths[*j].end());
    }
    const Action &act = actLabel[*i];
    if(absActs.count(act) != 0) {
      set<CConcPath> x = locPaths[*i];
      if(x.empty()) x.insert(CConcPath());
      locPaths[*i].clear();
      for(set<CConcPath>::iterator j = x.begin();j != x.end();++j) {
	CConcPath y = *j;
	const CL1AbsState *cas = static_cast<const CL1AbsState*>(stateLabel[*i]);
	pair< const ContLoc*,set<Expr> > p1(cas->GetContLoc(),cas->GetCons());
	pair<short,char> p2(cas->GetSpecStateId(),cas->GetSpecInfoId());
	pair< pair< const ContLoc*,set<Expr> >,pair<short,char> > p3(p1,p2);
	y.push_front(CConcEle(p3,act));
	locPaths[*i].insert(y);
      }
    }
  }
  for(set<CConcPath>::const_iterator i = locPaths[0].begin();i != locPaths[0].end();++i) {
    ConcPath x;
    for(list<CConcEle>::const_iterator j = i->begin();j != i->end();++j) {
      L1AbsState *y = new CL1AbsState(NULL,j->first.first.first,j->first.first.second,j->first.second.first,j->first.second.second);
      x.push_back(ConcEle(y,j->second));
    }
    res.insert(x);
  }
}

/*********************************************************************/
//get the set of fair loop actions
/*********************************************************************/
void ProcManager::GetFairLoopActs(set< pair<Action,Action> > &res) const
{
  for(vector<ContLoc*>::const_iterator i = allLocs.begin();i != allLocs.end();++i) {
    if(((*i)->GetLocType() == ContLoc::BRANCH) && (*i)->IsFairLoop()) {
      res.insert((*i)->GetFairLoopActs());
    }
  }
}

/*********************************************************************/
//return true if the argument string is a global variable
/*********************************************************************/
bool ProcManager::IsConcGlobalVar(const Expr &arg) const
{
  string str = arg.ToString();
  if(str.at(0) == '$') return false;
  const char *x = str.c_str();
  return (strstr(x,"::") == NULL);
}

/*********************************************************************/
//return the refusal set of a level-2 abstract state
/*********************************************************************/
void ProcManager::GetL2AbsRefusal(const L2AbsState &arg,set<Action> &res) const
{
  absLts.GetRefusal(arg,res);
}

/*********************************************************************/
//check if the second argument is the refusal set of the first argument
/*********************************************************************/
bool ProcManager::IsL1AbsRefusal(const L1AbsState &state,const set<Action> &refusal) const
{
  const ImplState *is = static_cast<const CL1AbsState&>(state).GetImplState();
  for(set<Action>::const_iterator i = refusal.begin();i != refusal.end();++i) {
    set<const ImplState*> x; is->GetSuccsOnAction(*i,x);
    if(!x.empty()) return false;
  }
  return true;
}

/*********************************************************************/
//output the current model as an FSP process. the name of the initial
//state is INIT_x where x is the component index
/*********************************************************************/
string ActToFsp(const Action &act,int index)
{
  Action epsAct = ActionManager::GetImplEpsilonAction(index);
  if(act == epsAct) {
    char buf[64];
    sprintf(buf,"epsilon_%d",index+1);
    return string(buf);
  } else if(act.IsAssign()) {
    char buf[64];
    sprintf(buf,"assign_%d",index+1);
    return string(buf);   
  } else return act.ToString();
}

void ProcManager::OutputFsp() const 
{
  Action epsAct = ActionManager::GetImplEpsilonAction(index);
  set<const ImplState*> front = initLoc->GetImplStates(guard);
  if(front.empty()) {
    fprintf(stderr,"INIT_%d = ( initial_act_%d -> STOP )",index,epsAct.GetIndex());
    Util::Message(2,"INIT_%d = ( initial_act_%d -> STOP )",index,epsAct.GetIndex());
  } else {
    fprintf(stderr,"INIT_%d = ( ",index+1);
    Util::Message(2,"INIT_%d = ( ",index+1);
    for(set<const ImplState*>::const_iterator i = front.begin();i != front.end();++i) {
      fprintf(stderr,"initial_act_%d -> State_%p ",epsAct.GetIndex(),*i);
      Util::Message(2,"initial_act_%d -> State_%p ",epsAct.GetIndex(),*i);
      ++i; 
      if(i != front.end()) { fprintf(stderr,"| "); Util::Message(2,"| "); }
      --i;
    }
    fprintf(stderr,")");
    Util::Message(2,")");
  }
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    list< pair<Action,const ImplState*> > saaList;
    map< Action,set<const ImplState*> > saa;
    (*i)->GetSuccsAndActions(set<Action>(),set<Action>(),saa);
    for(map< Action,set<const ImplState*> >::const_iterator j = saa.begin();j != saa.end();++j) {
      for(set<const ImplState*>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	saaList.push_back(pair<Action,const ImplState*>(j->first,*k));
      }
    }
    fprintf(stderr,",\n");
    Util::Message(2,",\n");
    if(saaList.empty()) {
      fprintf(stderr,"State_%p = ( act_%d -> STOP )",*i,epsAct.GetIndex());
      Util::Message(2,"State_%p = ( act_%d -> STOP )",*i,epsAct.GetIndex());
    } else {
      fprintf(stderr,"State_%p = ( ",*i);
      Util::Message(2,"State_%p = ( ",*i);
      for(list< pair<Action,const ImplState*> >::const_iterator j = saaList.begin();j != saaList.end();++j) {
	fprintf(stderr,"act_%d -> State_%p ",j->first.GetIndex(),j->second);
	Util::Message(2,"act_%d -> State_%p ",j->first.GetIndex(),j->second);
	++j; 
	if(j != saaList.end()) {
	  fprintf(stderr,"| "); 
	  Util::Message(2,"| "); 
	}
	--j;       
      }
      fprintf(stderr,")");
      Util::Message(2,")");
    }
  }
  fprintf(stderr,".\n\n");
  Util::Message(2,".\n\n");
}

/*********************************************************************/
//draw the control flow graph with inferred predicates
/*********************************************************************/
void ProcManager::DrawInferredPreds() const
{
  FILE *out = Database::dotFilePtr;
  fprintf(out,"  //Inferred predicates of component %s index %d\n",name.c_str(),index);
  fprintf(out,"  subgraph cluster_inf_preds_%s_%d_%d {\n",name.c_str(),index,Statistics::iterNum);
  fprintf(out,"    color = white;\n");
  for(vector<ContLoc*>::const_iterator i = allLocs.begin();i != allLocs.end();++i) {
    string recStr = Util::EscapeCharacter((*i)->ToString(),'|');    
    const vector<Predicate> &preds = (*i)->GetPreds().GetPreds();
    for(vector<Predicate>::const_iterator j = preds.begin();j != preds.end();++j) {
      for(vector<Expr>::const_iterator k = j->GetValToExpr().begin();k != j->GetValToExpr().end();++k) {
	recStr += " | " + Util::TrimString(Util::EscapeCharacter(k->ToString(),'|'));
      }
    }
    recStr = Util::EscapeCharacter(recStr,'<');    
    recStr = Util::EscapeCharacter(recStr,'>');    
    recStr = Util::EscapeCharacter(recStr,'{');    
    recStr = Util::EscapeCharacter(recStr,'}');        
    fprintf(out,"    inf_preds_%p_%d [shape=Mrecord,label = \"{%s}\"];\n",(*i),Statistics::iterNum,recStr.c_str());    
    if((*i)->GetLocType() == ContLoc::BRANCH) {
      fprintf(out,"    inf_preds_%p_%d -> inf_preds_%p_%d [label = \"then\"];\n",*i,Statistics::iterNum,(*i)->GetTSucc(),Statistics::iterNum);
      fprintf(out,"    inf_preds_%p_%d -> inf_preds_%p_%d [label = \"else\"];\n",*i,Statistics::iterNum,(*i)->GetESucc(),Statistics::iterNum);
    } else if((*i)->GetLocType() != ContLoc::FINAL){
      fprintf(out,"    inf_preds_%p_%d -> inf_preds_%p_%d;\n",*i,Statistics::iterNum,(*i)->GetTSucc(),Statistics::iterNum);
    }
  }
  fprintf(out,"    label = \"Inferred predicates \\nComponent #%d \\nProcedure name: %s\\nIteration #%d\";\n",index,name.c_str(),Statistics::iterNum);
  fprintf(out,"    labelloc = \"b\";\n");
  fprintf(out,"  }\n\n");
}

/*********************************************************************/
//output the implementation machine obtained by predicate abstraction
//as a dot file
/*********************************************************************/
void ProcManager::DrawPredAbsLts() const
{
  FILE *out = Database::dotFilePtr;
  fprintf(out,"  //LTS obtained by predicate abstraction of component %s index %d\n",name.c_str(),index);
  fprintf(out,"  subgraph cluster_pred_abs_lts_%s_%d_%d {\n",name.c_str(),index,Statistics::iterNum);
  fprintf(out,"    color = white;\n");
  string tv = Util::NewTempVar();
  fprintf(out,"    %s [label = \"INIT\",peripheries = 2];\n",tv.c_str());
  set<const ImplState*> front = initLoc->GetImplStates(guard);
  for(set<const ImplState*>::const_iterator i = front.begin();i != front.end();++i) {
    fprintf(out,"    %s -> pred_abs_lts_%p_%d [label =\"epsilon_%d\"];\n",tv.c_str(),*i,Statistics::iterNum,index);
  }  
  for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
    const ContLoc *loc = (*i)->GetContLoc();
    const PredSet &preds = loc->GetPreds();
    string recStr = Util::EscapeCharacter(loc->ToString(),'|');    
    PredVal pval = preds.PValToPredVal((*i)->GetPredVal());
    set<Expr> eset; preds.ToExpr(pval,eset);
    for(set<Expr>::const_iterator j = eset.begin();j != eset.end();++j) {
      recStr += " | " + Util::TrimString(Util::EscapeCharacter(j->ToString(),'|'));
    }
    int ssi = (*i)->GetSpecStateId();
    if(ssi != -1) {
      recStr += " | " + Database::ltsInfos[(*i)->GetSpecInfoId()].GetIdState(ssi);
    }
    recStr = Util::EscapeCharacter(recStr,'<');    
    recStr = Util::EscapeCharacter(recStr,'>');    
    recStr = Util::EscapeCharacter(recStr,'{');    
    recStr = Util::EscapeCharacter(recStr,'}');        
    fprintf(out,"    pred_abs_lts_%p_%d [shape=Mrecord,label = \"{%s}\"];\n",*i,Statistics::iterNum,recStr.c_str());    
    map< Action,set<const ImplState*> > saa;
    (*i)->GetSuccsAndActions(set<Action>(),set<Action>(),saa);
    for(map< Action,set<const ImplState*> >::const_iterator j = saa.begin();j != saa.end();++j) {
      for(set<const ImplState*>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	fprintf(out,"    pred_abs_lts_%p_%d -> pred_abs_lts_%p_%d [label = \"%s\"];\n",*i,Statistics::iterNum,*k,Statistics::iterNum,j->first.ToString().c_str());
      }
    }
  }
  fprintf(out,"    label = \"LTS after predicate abstraction \\nComponent #%d \\nProcedure name: %s\\nIteration #%d\";\n",index,name.c_str(),Statistics::iterNum);
  fprintf(out,"    labelloc = \"b\";\n");
  fprintf(out,"  }\n\n");
}

/*********************************************************************/
//output the model created by predicate abstraction
/*********************************************************************/
void ProcManager::OutputModel() const
{
  FILE *out = Database::modelFilePtr;
  if(Database::OUTPUT_MODEL_TYPE == Database::OUTPUT_MODEL_FSP) {
    fprintf(out,"//****************************************************************//\n");
    fprintf(out,"//Begin Model: Iteration #%d, Component #%d, Procedure name: %s\n",Statistics::iterNum,index,name.c_str());
    fprintf(out,"//****************************************************************//\n\n");
    set<const ImplState*> front = initLoc->GetImplStates(guard);
    for(set<const ImplState*>::const_iterator i = front.begin();i != front.end();++i) {
      fprintf(out,"INIT\n####epsilon_%d####\n%p\n\n",index,*i);
    }
    for(set<const ImplState*>::const_iterator i = reachImpl.begin();i != reachImpl.end();++i) {
      map< Action,set<const ImplState*> > saa;
      (*i)->GetSuccsAndActions(set<Action>(),set<Action>(),saa);
      for(map< Action,set<const ImplState*> >::const_iterator j = saa.begin();j != saa.end();++j) {
	for(set<const ImplState*>::const_iterator k = j->second.begin();k != j->second.end();++k) {
	  fprintf(out,"%p\n####%s####\n%p\n\n",*i,j->first.ToString().c_str(),*k);	
	}
      }
    }
    fprintf(out,"//****************************************************************//\n");
    fprintf(out,"//End Model: Iteration #%d, Component #%d, Procedure name: %s\n",Statistics::iterNum,index,name.c_str());
    fprintf(out,"//****************************************************************//\n");
    fprintf(out,"\n\n");
  } else assert(false);
}

/*********************************************************************/
//converts the absLts (abstract model) to an NFA and returns it in the
//argument pointer
/*********************************************************************/
void ProcManager::GetNFA(NFA* nfa) 
{
  assert(nfa != NULL);
  set<const ImplState*> front = initLoc->GetImplStates(guard);
  for(set<const ImplState*>::const_iterator i = front.begin();i != front.end();i++) {
    string initName = (*i)->ToString();
    nfa->addInitState(initName);
  }
   
  for(set<const ImplState*>::const_iterator itR = reachImpl.begin();itR != reachImpl.end();itR++) {
    map< Action,set<const ImplState*> > saa;
    (*itR)->GetSuccsAndActions(set<Action>(),set<Action>(),saa);
    //no need: nfa->new_state((*itR)->ToString() );
    for(map< Action,set<const ImplState*> >::const_iterator j = saa.begin();j != saa.end();j++) {
      string act = (j->first).ToString();
      assert(act.find(Database::EPSILON + "_") != 0 );
      cout<<"action from magic is: "<<act<<endl;
      string a,token,b; //the new alphabet
      int colonPos = act.find("::");
      //remove all local actions, 2 for P0::epsilon, 3 for other actions
      if( colonPos == 2 || colonPos == 3) a="epsilon"; 
      else {    
	istringstream tokenizer(act);
	while( tokenizer >> token)  {
	  //TODO: make it more general
	  b = token;
	  a+= b; //remove all intermediate spaces
	}
      }
      for(set<const ImplState*>::const_iterator k = j->second.begin();k != j->second.end();k++) {
	string source = (*itR)->ToString();
	string target = (*k)->ToString();
	nfa->new_trans(source,target,a);
      }
    }
  }
}

/*********************************************************************/
//end of ProcManager.cpp
/*********************************************************************/
