/******************************** CPPFile *****************************

* FileName [ProcInliner.cpp]

* PackageName [main]

* Synopsis [Method definitions of ProcInliner class.]

* SeeAlso [ProcInliner.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <set>
#include <list>
#include <vector>
#include <map>
#include <typeinfo>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "ProcAbs.h"
#include "Action.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ProcInfo.h"
#include "CallGraph.h"
#include "TheoremProver.h"
#include "ProcInliner.h"
#define YYSTYPE int
#include "StdcParser.h"
using namespace magic;

/*********************************************************************/
//do all the stuff
/*********************************************************************/
void ProcInliner::Inline(const CallGraph &callGraph)
{
  //sort the call-graph topologically. this will also detect if the
  //call-graph contains any cycles.
  list<string> sorted;
  TopologicalSort(callGraph,sorted);

  //the set of all functions and alias locations encountered
  set<string> allFuncs;

  //inline procedures in reverse topological order
  for(list<string>::const_iterator i = sorted.begin();i != sorted.end();++i) {
    ProcInfo &c = Database::procInfos[*i];
    list<ContLoc*> &contLocs = c.GetContLocs();  
    //create predecessor map
    map< ContLoc*,set<ContLoc*> > predMap;
    for(list<ContLoc*>::const_iterator j = contLocs.begin();j != contLocs.end();++j) {
      int locType = (*j)->GetLocType();
      if(locType == ContLoc::FINAL) {}
      else if((locType == ContLoc::RETURN) || (locType == ContLoc::ASSIGN_EXPR) ||
	      (locType == ContLoc::ASSIGN_CALL)) {
	predMap[(*j)->GetTSucc()].insert(*j);
      } else if(locType == ContLoc::BRANCH) {
	predMap[(*j)->GetTSucc()].insert(*j);
	predMap[(*j)->GetESucc()].insert(*j);
      } else assert(false);
    }
    //the set of added locations
    set<ContLoc*> addedLocs;
    //process assignments
    ProcessAssigns(c,addedLocs,predMap);
    //process calls
    list< pair<ContLoc*,string> > inlineList;
    ProcessCalls(c,addedLocs,allFuncs,predMap,inlineList); 
    contLocs.insert(contLocs.end(),addedLocs.begin(),addedLocs.end());
    //map from call-sites to ids used to distinguish between different
    //call-sites that call the same routine
    map<string,size_t> callSiteToId;
    //do the inlining
    for(list< pair<ContLoc*,string> >::const_iterator j = inlineList.begin();j != inlineList.end();++j) {
      if(callSiteToId.count(j->second) == 0) callSiteToId[j->second] = 0;
      else ++callSiteToId[j->second];
      char idStr[32];
      snprintf(idStr,32,"%d",callSiteToId[j->second]);
      InlineProc(j->first,j->second,idStr);
    }
  }
  //add the function disjoint constraints
  for(set<string>::const_iterator i = allFuncs.begin();i != allFuncs.end();++i) {
    Expr one = ExprManager::GetIdExpr(*i);
    set<string>::const_iterator j = i; ++j;
    for(;j != allFuncs.end();++j) {
      Expr two = ExprManager::GetIdExpr(*j);
      Expr cond = ExprManager::GetBinaryExpr(one,two,MAGIC_NE_OP);
      TheoremProver::AddConstraint(cond.GetExpr());
    }
  }
}

/*********************************************************************/
//process assignments by branching on possible aliases
/*********************************************************************/
void ProcInliner::ProcessAssigns(ProcInfo &procInfo,set<ContLoc*> &addedLocs,map< ContLoc*,set<ContLoc*> > &predMap)
{
  list<ContLoc*> &contLocs = procInfo.GetContLocs();  
  for(list<ContLoc*>::iterator j = contLocs.begin();j != contLocs.end();++j) {
    //if this is an assignment
    if(((*j)->GetLocType() == ContLoc::ASSIGN_EXPR) || ((*j)->GetLocType() == ContLoc::ASSIGN_CALL)) {
      //create vector of all target locations
      set<string> pto = procInfo.FindPointsTo((*j)->GetLocLhs(),-1,(*j)->GetLhsAliasKey());
      size_t dupCount = pto.size();
      if(dupCount > 1) {
	vector<ContLoc*> dupBrs,dupCalls;
	//create the first duplicate branch
	ContLoc *brLoc = new ContLoc(&procInfo,ContLoc::BRANCH);
	addedLocs.insert(brLoc);
	dupBrs.push_back(brLoc);
	if(procInfo.GetInitLoc() == (*j)) procInfo.SetInitLoc(brLoc);
	for(set<ContLoc*>::const_iterator k = predMap[*j].begin();k != predMap[*j].end();++k) {
	  int locType = (*k)->GetLocType();
	  if(locType == ContLoc::FINAL) {}
	  else if((locType == ContLoc::RETURN) || (locType == ContLoc::ASSIGN_EXPR) ||
		  (locType == ContLoc::ASSIGN_CALL)) {
	    if((*k)->GetTSucc() == (*j)) {
	      (*k)->SetTSucc(brLoc);
	      predMap[brLoc].insert(*k);
	    }
	  } else if(locType == ContLoc::BRANCH) {
	    if((*k)->GetTSucc() == (*j)) {
	      (*k)->SetTSucc(brLoc);
	      predMap[brLoc].insert(*k);
	    }
	    if((*k)->GetESucc() == (*j)) {
	      (*k)->SetESucc(brLoc);
	      predMap[brLoc].insert(*k);
	    }
	  } else assert(false);
	}
	//first duplicate call already exists
	brLoc->SetTSucc((*j));
	predMap[*j].clear();
	predMap[*j].insert(brLoc);
	dupCalls.push_back((*j));
	//create other duplicate calls
	for(size_t k = 1;k < dupCount;++k) {
	  ContLoc *dupCall = new ContLoc(**j);
	  predMap[(*j)->GetTSucc()].insert(dupCall);
	  addedLocs.insert(dupCall);
	  dupCalls.push_back(dupCall);
	  if(k == (dupCount - 1)) {
	    dupBrs.back()->SetESucc(dupCall);
	    predMap[dupCall].insert(dupBrs.back());
	  } else {
	    ContLoc *dupBr = new ContLoc(&procInfo,ContLoc::BRANCH);
	    dupBrs.back()->SetESucc(dupBr);
	    predMap[dupBr].insert(dupBrs.back());
	    addedLocs.insert(dupBr);
	    dupBrs.push_back(dupBr);
	    dupBrs.back()->SetTSucc(dupCall);
	    predMap[dupCall].insert(dupBrs.back());
	  }
	}
	assert((dupCalls.size() == dupCount) && (dupBrs.size() == (dupCount - 1)));
	//link branches to assigns and assigns to aliases
	vector<string> targetLocs;
	targetLocs.insert(targetLocs.end(),pto.begin(),pto.end());
	for(size_t k = 0;k < dupCalls.size();++k) {
	  Expr aliasExpr = ExprManager::GetIdExpr(targetLocs[k]);
	  if(k < dupBrs.size()) {
	    Expr one = ExprManager::GetUnaryExpr((*j)->GetLocLhs(),'&');
	    Expr two = ExprManager::GetUnaryExpr(aliasExpr,'&');	      
	    Expr cond = ExprManager::GetBinaryExpr(one,two,MAGIC_EQ_OP);
	    dupBrs[k]->SetLocExpr(cond);
	  }
	  dupCalls[k]->AddLhsAlias(aliasExpr);
	}
      } else if(dupCount == 1) {
	(*j)->AddLhsAlias(ExprManager::GetIdExpr(*(pto.begin())));
      }
    }
  }
}

/*********************************************************************/
//process assignments by branching on possible aliases
/*********************************************************************/
void ProcInliner::ProcessCalls(ProcInfo &procInfo,set<ContLoc*> &addedLocs,set<string> &allFuncs,
			       map< ContLoc*,set<ContLoc*> > &predMap,
			       list< pair<ContLoc*,string> > &inlineList)
{
  list<ContLoc*> &contLocs = procInfo.GetContLocs();  
  for(list<ContLoc*>::iterator j = contLocs.begin();j != contLocs.end();++j) {
    //check if it is a procedure call and if so get the
    //name of the called procedure and the arguments
    pair< Expr,list<Expr> > comps;
    //if this is a call-site
    if((*j)->IsCallSite(comps)) {
      //if this call-site is not abstracted, inline
      pair< set<string>,set<string> > csi = procInfo.GetCallSiteInfo(comps.first,(*j)->GetIndCallKey());
      //create vector of all target procedures
      vector<string> targetProcs;
      targetProcs.insert(targetProcs.end(),csi.first.begin(),csi.first.end());
      set<string> tp = csi.second;
      tp.erase(Database::DUMMY_C_PROC);
      targetProcs.insert(targetProcs.end(),tp.begin(),tp.end());
      //add the functions encountered
      allFuncs.insert(targetProcs.begin(),targetProcs.end());
      if(csi.second.count(Database::DUMMY_C_PROC) != 0) {
	targetProcs.push_back(Database::DUMMY_C_PROC);
      }
      size_t dupCount = csi.first.size() + csi.second.size();
      if(dupCount > 1) {
	vector<ContLoc*> dupBrs,dupCalls;
	//create the first duplicate branch
	ContLoc *brLoc = new ContLoc(&procInfo,ContLoc::BRANCH);
	addedLocs.insert(brLoc);
	dupBrs.push_back(brLoc);
	if(procInfo.GetInitLoc() == (*j)) procInfo.SetInitLoc(brLoc);
	for(set<ContLoc*>::const_iterator k = predMap[*j].begin();k != predMap[*j].end();++k) {
	  int locType = (*k)->GetLocType();
	  if(locType == ContLoc::FINAL) {}
	  else if((locType == ContLoc::RETURN) || (locType == ContLoc::ASSIGN_EXPR) ||
		  (locType == ContLoc::ASSIGN_CALL)) {
	    if((*k)->GetTSucc() == (*j)) {
	      (*k)->SetTSucc(brLoc);
	      predMap[brLoc].insert(*k);
	    }
	  } else if(locType == ContLoc::BRANCH) {
	    if((*k)->GetTSucc() == (*j)) {
	      (*k)->SetTSucc(brLoc);
	      predMap[brLoc].insert(*k);
	    }
	    if((*k)->GetESucc() == (*j)) {
	      (*k)->SetESucc(brLoc);
	      predMap[brLoc].insert(*k);
	    }
	  } else assert(false);
	}
	//first duplicate call already exists
	brLoc->SetTSucc((*j));
	predMap[*j].clear();
	predMap[*j].insert(brLoc);
	dupCalls.push_back((*j));
	//create other duplicate calls
	for(size_t k = 1;k < dupCount;++k) {
	  ContLoc *dupCall = new ContLoc(**j);
	  predMap[(*j)->GetTSucc()].insert(dupCall);
	  addedLocs.insert(dupCall);
	  dupCalls.push_back(dupCall);
	  if(k == (dupCount - 1)) {
	    dupBrs.back()->SetESucc(dupCall);
	    predMap[dupCall].insert(dupBrs.back());
	  } else {
	    ContLoc *dupBr = new ContLoc(&procInfo,ContLoc::BRANCH);
	    dupBrs.back()->SetESucc(dupBr);
	    predMap[dupBr].insert(dupBrs.back());
	    addedLocs.insert(dupBr);
	    dupBrs.push_back(dupBr);
	    dupBrs.back()->SetTSucc(dupCall);
	    predMap[dupCall].insert(dupBrs.back());
	  }
	}
	assert((dupCalls.size() == dupCount) && (dupBrs.size() == (dupCount - 1)));
	//get the function pointer expression
	const BasicExpr *a = comps.first.GetExpr();
	if(typeid(*a) == typeid(UnaryExpr)) {
	  const UnaryExpr *b = static_cast<const UnaryExpr*>(a);
	  assert(b->GetOp() == '*');
	  comps.first = Expr(b->GetExpr());
	}
	//link branches to calls and calls to abstractions
	for(size_t k = 0;k < dupCalls.size();++k) {
	  if(k < dupBrs.size()) {
	    Expr cond = ExprManager::GetBinaryExpr(comps.first,ExprManager::GetIdExpr(targetProcs[k]),MAGIC_EQ_OP);
	    dupBrs[k]->SetLocExpr(cond);
	  }
	  if(csi.first.count(targetProcs[k]) != 0) {
	    inlineList.push_back(pair<ContLoc*,string>(dupCalls[k],targetProcs[k]));
	  } else {
	    dupCalls[k]->SetAbsProc(targetProcs[k]);
	  }
	}
      } else {
	assert(dupCount == 1);
	if(csi.second.empty()) {
	  inlineList.push_back(pair<ContLoc*,string>(*j,*(csi.first.begin())));
	} else (*j)->SetAbsProc(*(csi.second.begin()));
      }
    }
  }
}

/*********************************************************************/
//topologically sort the call-graph. also verify that the call graph
//is acyclic. the argument is used to store the procedures in reverse
//topological order i.e. the root procedure comes last.
/*********************************************************************/
void ProcInliner::TopologicalSort(const CallGraph &callGraph,list<string> &sorted)
{
  const set<string> &all = callGraph.GetNodes();
  const map< string,set<string> > &edges = callGraph.GetEdges();
  set<string> seen;
  set<string> done;
  for(vector<string>::const_iterator i = Database::rootProcNames.begin();
      i != Database::rootProcNames.end();++i) {
    DFS(all,seen,done,edges,*i,sorted);
  }
}

/*********************************************************************/
//the dfs routine for topological sorting and checking for cirularity
/*********************************************************************/
void ProcInliner::DFS(const set<string> &all,set<string> &seen,set<string> &done,
		      const map< string,set<string> > &edges,const string &node,
		      list<string> &sorted)
{
  seen.insert(node);
  map< string,set<string> >::const_iterator i = edges.find(node);
  if(i != edges.end()) {
    const set<string> &succs = i->second;
    for(set<string>::const_iterator j = succs.begin();j != succs.end();++j) {
      if(seen.count(*j) == 0) {
	DFS(all,seen,done,edges,*j,sorted);
      } else {
	if(done.count(*j) == 0) {
	  Util::Error("call-graph is cyclic ...\n");
	}
      }
    }
  }
  done.insert(node);
  sorted.push_back(node);
}

/*********************************************************************/
//given a control location which represents a procedure call and the
//name of the procedure called, add appropriate call and return links.
/*********************************************************************/
void ProcInliner::InlineProc(ContLoc *callSite,const string &procName,const string &siteId)
{
  //find the info for the caller procedure
  ProcInfo *callerInfo = callSite->GetProcInfo();

  //get the list of arguments
  pair< Expr,list<Expr> > comps;
  assert(callSite->IsCallSite(comps));
  list<Expr> &argList = comps.second;

  //find the info for the called procedure
  assert(Database::procInfos.count(procName) != 0);
  ProcInfo &calleeInfo = Database::procInfos[procName];

  //create the list of parameters and local variables, list of
  //parameters and also the set of local variables
  list<string> parLocs = calleeInfo.GetParams();
  list<Expr> parList; ExprManager::StringListToExprList(parLocs,parList);
  if(argList.size() > parList.size()) {
    Util::Message(2,"WARNING: more arguments passed to %s from %s than required ...\n",calleeInfo.GetName().c_str(),callerInfo->GetName().c_str());
  } else if(argList.size() < parList.size()) {
    Util::Error("ERROR: fewer arguments passed to %s from %s than required ...\n",calleeInfo.GetName().c_str(),callerInfo->GetName().c_str());
  }
  parLocs.insert(parLocs.end(),calleeInfo.GetLocals().begin(),calleeInfo.GetLocals().end());
  list<Expr> parLocList; ExprManager::StringListToExprList(parLocs,parLocList);
  set<string> locals;
  locals.insert(calleeInfo.GetLocals().begin(),calleeInfo.GetLocals().end());

  //create the list of renamed parameters and local variables
  string prefix = procName + "#" + siteId + "::";
  list<string> renamed;
  for(list<string>::const_iterator i = parLocs.begin();i != parLocs.end();++i) {
    renamed.push_back(prefix + (*i));
    callerInfo->AddLocal(prefix + (*i));
  }
  list<Expr> renamedList; ExprManager::StringListToExprList(renamed,renamedList);

  //the number of newly created control locations
  size_t newLocNum = 0;

  //create the assignments of the arguments to the
  //parameters. maintain a map of parameters which are assigned to
  //addresses.
  list<ContLoc*> parAssigns;
  map<Expr,Expr> addressPars;
  list<Expr>::const_iterator i = parList.begin();
  for(list<Expr>::const_iterator j = argList.begin();j != argList.end();++i,++j) {
    ContLoc *cl = new ContLoc(callerInfo,ContLoc::ASSIGN_EXPR);
    Expr a = ExprManager::Replace(parLocList,renamedList,*i);
    cl->SetLocLhs(a);
    cl->SetLocRhs(*j);
    parAssigns.push_back(cl);
    ++newLocNum;
    const BasicExpr *e1 = j->GetExpr();
    if(typeid(*e1) == typeid(UnaryExpr)) {
      const UnaryExpr *e2 = static_cast<const UnaryExpr*>(e1);
      if(e2->op == '&') {
	addressPars[a] = Expr(e2->expr);
      }
    }
  }
  for(list<ContLoc*>::iterator i = parAssigns.begin();i != parAssigns.end();) {
    ContLoc *cl = *i; ++i;
    if(i != parAssigns.end()) {
      cl->SetTSucc(*i);
    }
  }

  //create a duplicate copy of the called procedure locations. keep
  //the association between the old and new locations in a map.
  map<ContLoc*,ContLoc*> oldToNew;
  for(list<ContLoc*>::const_iterator i = calleeInfo.GetContLocs().begin();
      i != calleeInfo.GetContLocs().end();++i) {
    oldToNew[*i] = new ContLoc(**i);
    ++newLocNum;
  }

  //this variable indicates whether the inlined function is void or
  //not. it is initially zero. if the inlined procedure is void it is
  //set to 1 and otherwise to 2.
  int voidFlag = 0;

  //this flag indicates if there are any locations to be inlined
  bool inlineFlag = false;

  //get the successor and lhs of the call-site
  ContLoc *callerSucc = callSite->GetTSucc();
  Expr callerLhs = callSite->GetLocLhs();

  //update the new locations by renaming the expressions and changing
  //the successors appropriately
  for(map<ContLoc*,ContLoc*>::iterator i = oldToNew.begin();i != oldToNew.end();++i) {
    ContLoc *cl = i->second;
    int type = cl->GetLocType();
    //final locations - eliminate
    if(type == ContLoc::FINAL) {
      delete cl;
      i->second = NULL;
      --newLocNum;
    }
    //return locations - create appropriate assignments
    else if(type == ContLoc::RETURN) {
      //if this is a void return eliminate it
      if(cl->GetLocExpr().IsEmptyExpr()) {
	assert(voidFlag != 2);
	voidFlag = 1;
	delete cl;
	i->second = NULL;
	--newLocNum;
      }
      //if some value is being returned then turn this location into
      //an assignment of the return value to the return variable
      else {
	assert(voidFlag != 1);
	voidFlag = 2;
	cl->SetLocType(ContLoc::ASSIGN_EXPR);
	cl->SetLocLhs(callerLhs);
	Expr a = ExprManager::Replace(parLocList,renamedList,cl->GetLocExpr());
	cl->SetLocRhs(a);
	cl->SetLocExpr(Expr());
	inlineFlag = true;
      }
    }
    //assignment to expressions
    else if(type == ContLoc::ASSIGN_EXPR) {
      Expr a = ExprManager::Replace(parLocList,renamedList,cl->GetLocLhs());
      cl->SetLocLhs(a);
      Expr b = ExprManager::Replace(parLocList,renamedList,cl->GetLocRhs());
      cl->SetLocRhs(b);
      set<Expr> c;
      for(set<Expr>::const_iterator j = cl->GetLhsAliases().begin();j != cl->GetLhsAliases().end();++j) {
	c.insert(ExprManager::Replace(parLocList,renamedList,*j));
      }
      cl->SetLhsAliases(c);
      assert(oldToNew.count(cl->GetTSucc()) != 0);
      cl->SetTSucc(oldToNew[cl->GetTSucc()]);
      inlineFlag = true;
    }
    //call assignments
    else if(type == ContLoc::ASSIGN_CALL) {
      Expr a = ExprManager::Replace(parLocList,renamedList,cl->GetLocLhs());
      cl->SetLocLhs(a);
      Expr b = ExprManager::Replace(parLocList,renamedList,cl->GetLocRhs());
      cl->SetLocRhs(b);
      set<Expr> c;
      for(set<Expr>::const_iterator j = cl->GetLhsAliases().begin();j != cl->GetLhsAliases().end();++j) {
	c.insert(ExprManager::Replace(parLocList,renamedList,*j));
      }
      cl->SetLhsAliases(c);
      assert(oldToNew.count(cl->GetTSucc()) != 0);
      cl->SetTSucc(oldToNew[cl->GetTSucc()]);

      //update the callee abstraction map
      pair< Expr,list<Expr> > comps1; ExprManager::GetCalledProcDetails(i->first->GetLocRhs(),comps1);
      map< Expr,pair< set<string>,set<string> > > &calleeMap1 = calleeInfo.GetCalleeMap();
      assert((calleeMap1.count(comps1.first) != 0) && (!(calleeMap1[comps1.first].second.empty())));
      pair< Expr,list<Expr> > comps2; ExprManager::GetCalledProcDetails(cl->GetLocRhs(),comps2);
      map< Expr,pair< set<string>,set<string> > > &calleeMap2 = callerInfo->GetCalleeMap();
      calleeMap2[comps2.first].second.insert(calleeMap1[comps1.first].second.begin(),calleeMap1[comps1.first].second.end());
      inlineFlag = true;
    }
    //branches
    else if(type == ContLoc::BRANCH) {
      Expr a = ExprManager::Replace(parLocList,renamedList,cl->GetLocExpr());
      cl->SetLocExpr(a);
      assert(oldToNew.count(cl->GetTSucc()) != 0);
      cl->SetTSucc(oldToNew[cl->GetTSucc()]);
      assert(oldToNew.count(cl->GetESucc()) != 0);
      cl->SetESucc(oldToNew[cl->GetESucc()]);
      inlineFlag = true;
    }
    //impossible
    else assert(false);
  }

  //get the control locations of the caller and its size
  list<ContLoc*> &callerLocs = callerInfo->GetContLocs();
  size_t oldLocNum = callerLocs.size();

  //get the predecessors of the call-site
  set<ContLoc*> callerPreds;
  for(list<ContLoc*>::iterator i = callerLocs.begin();i != callerLocs.end();++i) {
    ContLoc *cl = (*i);
    int type = cl->GetLocType();
    if(type == ContLoc::FINAL) {
    } else if(type == ContLoc::RETURN) {
    } else if((type == ContLoc::ASSIGN_EXPR) || (type == ContLoc::ASSIGN_CALL)) {
      if(cl->GetTSucc() == callSite) callerPreds.insert(cl);
    } else if(type == ContLoc::BRANCH) {
      if((cl->GetTSucc() == callSite) || (cl->GetESucc() == callSite)) {
	callerPreds.insert(cl);
      }
    } else assert(false);
  }
  assert(callerPreds.count(callSite) == 0);

  //link up the parameter assignments to the rest of the inlined code
  if((!parAssigns.empty()) && inlineFlag) {
    parAssigns.back()->SetTSucc(oldToNew[calleeInfo.GetInitLoc()]);
  }

  //link up the initial location of the inlined code
  ContLoc *initInlined = parAssigns.empty() ? (inlineFlag ? oldToNew[calleeInfo.GetInitLoc()] : callerSucc) : 
    parAssigns.front();
  for(set<ContLoc*>::iterator i = callerPreds.begin();i != callerPreds.end();++i) {
    ContLoc *cl = (*i);
    int type = cl->GetLocType();
    if((type == ContLoc::ASSIGN_EXPR) || (type == ContLoc::ASSIGN_CALL)) {
      cl->SetTSucc(initInlined);
    } else if(type == ContLoc::BRANCH) {
      if(cl->GetTSucc() == callSite) cl->SetTSucc(initInlined);
      else if(cl->GetESucc() == callSite) cl->SetESucc(initInlined);
      else assert(false);
    } else assert(false);
  }

  //link up the final location of the inlined code
  if(inlineFlag) {
    for(map<ContLoc*,ContLoc*>::iterator i = oldToNew.begin();i != oldToNew.end();++i) {
      ContLoc *cl = i->first;
      int type = cl->GetLocType();
      if(type == ContLoc::FINAL) {
	assert(i->second == NULL);
      } else if(type == ContLoc::RETURN) {
	if(i->second != NULL) {
	  i->second->SetTSucc(callerSucc);
	}
      } else if((type == ContLoc::ASSIGN_EXPR) || (type == ContLoc::ASSIGN_CALL)) {
	if((cl->GetTSucc()->GetLocType() == ContLoc::RETURN) && (oldToNew[cl->GetTSucc()] == NULL)) {
	  i->second->SetTSucc(callerSucc);
	}
      } else if(type == ContLoc::BRANCH) {
	if((cl->GetTSucc()->GetLocType() == ContLoc::RETURN) && (oldToNew[cl->GetTSucc()] == NULL)) {
	  i->second->SetTSucc(callerSucc);
	}
	if((cl->GetESucc()->GetLocType() == ContLoc::RETURN) && (oldToNew[cl->GetESucc()] == NULL)) {
	  i->second->SetESucc(callerSucc);
	}
      } else assert(false);
    }
  } else {
    if(!parAssigns.empty()) {
      parAssigns.back()->SetTSucc(callerSucc);
    }
  }

  //remove the call-site from the control locations
  callerLocs.remove(callSite);
  delete callSite;

  //compute the set of parameters that were not modified and whose
  //address was never taken
  set<Expr> assignedLvs,addressLvs;
  for(map<ContLoc*,ContLoc*>::iterator i = oldToNew.begin();i != oldToNew.end();++i) {
    ContLoc *loc = i->second;
    if(loc == NULL) continue;
    int locType = loc->GetLocType();
    if((locType == ContLoc::ASSIGN_EXPR) || (locType == ContLoc::ASSIGN_CALL)) {
      assignedLvs.insert(loc->GetLocLhs());
      set<Expr> x = Util::ComputeExprLvalues(loc->GetLocLhs().GetExpr());
      addressLvs.insert(x.begin(),x.end());
      set<Expr> y = Util::ComputeExprLvalues(loc->GetLocRhs().GetExpr());
      addressLvs.insert(y.begin(),y.end());
    } else if(locType == ContLoc::BRANCH) {
      set<Expr> x = Util::ComputeExprLvalues(loc->GetLocExpr().GetExpr());
      addressLvs.insert(x.begin(),x.end());
    } else assert(false);
  }
  set<Expr> aliasPars;
  for(map<Expr,Expr>::const_iterator i = addressPars.begin();i != addressPars.end();++i) {
    if(assignedLvs.count(i->first) == 0) {
      Expr x = ExprManager::GetUnaryExpr(i->first,'&');
      if(addressLvs.count(x) == 0) {
	aliasPars.insert(i->first);
      }
    }
  }

  //replace all occurences of *PAR with ARG
  list<Expr> aliasParList,aliasList;
  for(set<Expr>::const_iterator i = aliasPars.begin();i != aliasPars.end();++i) {
    aliasParList.push_back(ExprManager::GetUnaryExpr(*i,'*'));
    aliasList.push_back(addressPars[*i]);
  }
  for(map<ContLoc*,ContLoc*>::iterator i = oldToNew.begin();i != oldToNew.end();++i) {
    ContLoc *loc = i->second;
    if(loc == NULL) continue;
    int locType = loc->GetLocType();
    if((locType == ContLoc::ASSIGN_EXPR) || (locType == ContLoc::ASSIGN_CALL)) {
      Expr x = ExprManager::Replace(aliasParList,aliasList,loc->GetLocLhs());
      loc->SetLocLhs(x);
      Expr y = ExprManager::Replace(aliasParList,aliasList,loc->GetLocRhs());
      loc->SetLocRhs(y);
    } else if(locType == ContLoc::BRANCH) {
      Expr x = ExprManager::Replace(aliasParList,aliasList,loc->GetLocExpr());
      loc->SetLocExpr(x);
    }
  }

  //add the new locations
  callerLocs.insert(callerLocs.end(),parAssigns.begin(),parAssigns.end());
  for(map<ContLoc*,ContLoc*>::iterator i = oldToNew.begin();i != oldToNew.end();++i) {
    if((i->second) != NULL) {
      i->second->SetProcInfo(callerInfo);
      callerLocs.push_back(i->second);
    }
  }

  //update the initial location if necessary
  if(callerInfo->GetInitLoc() == callSite) callerInfo->SetInitLoc(initInlined);

  //sanity - check that the sizes match
  assert(callerLocs.size() == (oldLocNum + newLocNum - 1));
}
   
/*********************************************************************/
//end of ProcInliner.cpp
/*********************************************************************/
