/******************************** CPPFile *****************************

* FileName [ProcInfo.cpp]

* PackageName [main]

* Synopsis [Method definitions of ProcInfo class.]

* SeeAlso [ProcInfo.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cstdio>
#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <map>
#include <vector>
#include <typeinfo>
using namespace std;

#include "BigInt.h"
#include "Util.h"
#include "Node.h"
#include "Action.h"
#include "Alias.h"
#include "ProcAbs.h"
#include "ProcAddInfo.h"
#include "LtlFormula.h"
#include "LtlManager.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
#include "ContLoc.h"
#include "ProcInfo.h"
#ifdef MAGIC_FULL
#include "PamAPI.h"
using namespace ctool;
#endif //MAGIC_FULL
using namespace magic;

/*********************************************************************/
//constructors
/*********************************************************************/
ProcInfo::ProcInfo() { context = 0; fileIndex = -1; }

ProcInfo::ProcInfo(const Proc &pr,const string &fn,const ProcAddInfo &pa)
  : name(pr.name),ast(*(pr.body)),params(pr.params),locals(pr.locals),preds(pa.preds),
    fairLoops(pa.fairLoops),inlines(pa.inlines),aliases(pa.aliases),callerAbs(pa.abstractions)
{
  assert(pr.name == pa.name);
  context = pa.context;
#ifdef MAGIC_FULL
  fileIndex = (Database::USE_PAM) ? PamAPI::GetFileIndex(fn) : 0;
#else
  fileIndex = 0;
#endif //MAGIC_FULL
}

ProcInfo::ProcInfo(const Proc &pr,const string &fn)
  : name(pr.name),ast(*(pr.body)),params(pr.params),locals(pr.locals)
{
  context = 0;
#ifdef MAGIC_FULL
  fileIndex = (Database::USE_PAM) ? PamAPI::GetFileIndex(fn) : 0;
#else
  fileIndex = 0;
#endif //MAGIC_FULL
}

ProcInfo::ProcInfo(const ProcAddInfo &pa)
  : name(pa.name),preds(pa.preds),fairLoops(pa.fairLoops),inlines(pa.inlines),
    aliases(pa.aliases),callerAbs(pa.abstractions)
{
  context = pa.context;
}

ProcInfo::ProcInfo(const ProcInfo &rhs) { *this = rhs; }

ProcInfo::~ProcInfo() { Cleanup(); }

/*********************************************************************/
//operators
/*********************************************************************/
const ProcInfo &ProcInfo::operator = (const ProcInfo &rhs)
{
  Cleanup();
  name = rhs.name;
  fileIndex = rhs.fileIndex;
  ast = rhs.ast;
  params = rhs.params;
  locals = rhs.locals;
  preds = rhs.preds;
  fairLoops = rhs.fairLoops;
  inlines = rhs.inlines;
  aliases = rhs.aliases;
  callerAbs = rhs.callerAbs;
  calleeMap = rhs.calleeMap;
  context = rhs.context;
  contLocs = rhs.contLocs;
  initLoc = rhs.initLoc;
  finalLoc = rhs.finalLoc;
  return *this;
}

/*********************************************************************/
//add a new local variable
/*********************************************************************/
void ProcInfo::AddLocal(const string &l) { locals.push_back(l); }

/*********************************************************************/
//cleanup allocated memory - currently the control locations
/*********************************************************************/
void ProcInfo::Cleanup()
{
  for(list<ContLoc*>::iterator i = contLocs.begin();i != contLocs.end();++i) {
    delete *i;
  }
  contLocs.clear();
  initLoc = finalLoc = NULL;
}

void ProcInfo::SetContLocs(const list<ContLoc*> &cl)
{
  for(list<ContLoc*>::iterator i = contLocs.begin();i != contLocs.end();++i) {
    delete *i;
  }
  contLocs = cl;
}

/*********************************************************************/
//display the set of control locations of this procedure
/*********************************************************************/
void ProcInfo::DisplayContLocs() const
{
  Util::Message(0,"******* control locations of procedure %s ***************\n",name.c_str());
  for(list<ContLoc*>::const_iterator i = contLocs.begin();i != contLocs.end();++i) {
    Util::Message(0,"***********************************************************\n");
    Util::Message(0,"%s\n",(*i)->ToString().c_str());
    Util::Message(0,"***********************************************************\n");
    string ts = ((*i)->GetTSucc() == NULL) ? "NULL" : (*i)->GetTSucc()->ToString();
    Util::Message(0,"true successor : %s\n",ts.c_str());
    string es = ((*i)->GetESucc() == NULL) ? "NULL" : (*i)->GetESucc()->ToString();
    Util::Message(0,"false successor : %s\n",es.c_str());
  }
  Util::Message(0,"*********************************************************\n");
}

/*********************************************************************/
//returns a pair <X,Y> where X is the set if procedure names to be
//inlined at this callsite and Y is the set of procedure abstractions
//relevant at this call-site
/*********************************************************************/
pair< set<string>,set<string> > ProcInfo::GetCallSiteInfo(const Expr &expr,int ind)
{
  //if there is an entry in the calleeAbs map then this site is
  //abstracted
  if(calleeMap.count(expr) == 0) {
    //first i should check if this is a function pointer or a direct
    //function call and extract the pointer name and points to set
    set<string> pTo;
    if(typeid(*(expr.GetExpr())) == typeid(UnaryExpr)) {
      const UnaryExpr *a = static_cast<const UnaryExpr*>(expr.GetExpr());
      assert(a->op == '*');
      pTo = FindPointsTo(a->expr,ind,-1);
    } else pTo.insert(Util::TrimString(expr.ToString()));
    //this procedure must be abstracted. construct the calleeAbs map.
    for(set<string>::const_iterator i = pTo.begin();i != pTo.end();++i) {
      if(Database::procInfos.count(*i) != 0) {
	const ProcInfo &x = Database::procInfos[*i];
	if(x.GetAst().data != NULL) {
	  if(Database::INLINE_LIBS || (inlines.count(*i) != 0)) {
	    calleeMap[expr].first.insert(*i);
	  } else calleeMap[expr].second.insert(*i);
	} else calleeMap[expr].second.insert(*i);
      } else calleeMap[expr].second.insert(*i);
    }
    if(calleeMap[expr].first.empty() && calleeMap[expr].second.empty()) {
      calleeMap[expr].second.insert(Database::DUMMY_C_PROC);
    }
  }
  return calleeMap[expr];
}

/*********************************************************************/
//find the variables to which the argument variable can point to. the
//second argument indicates the key assigned by PAM if the first
//argument is an indirect function call. if the first argument is not
//an indirect function call then the second argument is -1. the third
//argument indicates the key assigned by PAM if the first argument is
//a lhs. otherwise the third argument is -1.
/*********************************************************************/
set<string> ProcInfo::FindPointsTo(const Expr &ptr,int indCall,int lhsAlias)
{
  set<string> res;
  string ptrStr = Util::TrimString(ptr.ToString());
  for(list<Alias>::const_iterator i = aliases.begin();i != aliases.end();++i) {
    set<string> a = i->FindPointsTo(ptrStr);
    res.insert(a.begin(),a.end());
  }

#ifdef MAGIC_FULL 
  if(Database::USE_PAM) {
    bool isParOrLoc = false;
    const BasicExpr *a = ptr.GetExpr();
    set<string> lv = Util::ComputeIdLvalues(a);
    set<string> x;
    x.insert(params.begin(),params.end());
    for(set<string>::const_iterator i = lv.begin();i != lv.end();++i) {
      if(x.count(*i) != 0) {
	isParOrLoc = true;
	break;
      }
    }
    if(!isParOrLoc) {
      x.clear();
      x.insert(locals.begin(),locals.end());
      for(set<string>::const_iterator i = lv.begin();i != lv.end();++i) {
	if(x.count(*i) != 0) {
	  isParOrLoc = true;
	  break;
	}
      }
    }
    set<string> y = PamAPI::FindPointsTo(ptr,name,isParOrLoc ? fileIndex : -1,indCall,lhsAlias);
    res.insert(y.begin(),y.end());
  }
#endif //MAGIC_FULL

  return res;
}

/*********************************************************************/
//given a call-site and a procedure name invoked at the site, return
//the corresponding procedure abstraction guards and names
/*********************************************************************/
set< pair<Expr,string> > ProcInfo::GetCallAbsInfo(const Expr &site,const string &proc) const
{
  set< pair<Expr,string> > res;
  if(calleeMap.count(site) != 0) {
    map< Expr,pair< set<string>,set<string> > >::const_iterator it = calleeMap.find(site);
    if(it != calleeMap.end()) {
      for(set<string>::const_iterator i = it->second.second.begin();i != it->second.second.end();++i) {
	if(Database::procInfos.count(*i) != 0) {
	  const list<ProcAbs> &pabs = Database::procInfos[*i].callerAbs;
	  for(list<ProcAbs>::const_iterator j = pabs.begin();j != pabs.end();++j) {
	    res.insert(pair<Expr,string>(j->GetGuard(),j->GetLtsName()));
	  }
	}
      }
    }
  }
  return res;
}

/*********************************************************************/
//compute the set of LTS names that abstract a procedure call
//statement. the arguments are the ast of the call statement and a
//list of expressions representing the context in which the call is
//made.
/*********************************************************************/
set<string> ProcInfo::GetCallAbsLtsNames(const Expr &call,const string &proc,const set<Expr> &context) const
{
  //first get the name of the procedure called
  pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(call,comps);
  //get the actions from the procedure's abstraction info
  set<string> res; set<ProcAbs> pabs;  
  if(Database::procInfos.count(proc) != 0) {
    pabs.insert(Database::procInfos[proc].callerAbs.begin(),Database::procInfos[proc].callerAbs.end());
  } else pabs.insert(ProcAbs::GetEpsilonAbstraction());
  for(set<ProcAbs>::const_iterator i = pabs.begin();i != pabs.end();++i) {
    if(i->AbstractionPossible(comps.second,context)) {
      res.insert(i->GetLtsName());
    }
  }
  return res;
}

/*********************************************************************/
//given a call site and an inlined LTS name return the corresponding
//guard that needs to be satisfied for the LTS to be enabled
/*********************************************************************/
Expr ProcInfo::GetCallAbsLtsGuard(const Expr &call,const string &proc,const string &ltsName) const
{
  //first get the name of the procedure called
  pair< Expr,list<Expr> > comps; ExprManager::GetCalledProcDetails(call,comps);
  //get the actions from the procedure's abstraction info
  set<ProcAbs> pabs;  
  if(Database::procInfos.count(proc) != 0) {
    pabs.insert(Database::procInfos[proc].callerAbs.begin(),Database::procInfos[proc].callerAbs.end());
  } else pabs.insert(ProcAbs::GetEpsilonAbstraction());
  for(set<ProcAbs>::const_iterator i = pabs.begin();i != pabs.end();++i) {
    if(i->GetLtsName() == ltsName) {
      return ExprManager::ReplaceDummyVarsOne(i->GetGuard(),comps.second);
    }
  }
  assert(false);
}

/*********************************************************************/
//make sure that the procedure abstractions supplied are mutually
//exclusive and complete. otherwise add a default abstraction to make
//them complete.
/*********************************************************************/
void ProcInfo::CheckAbstractions()
{
  //replace dummy variables in guard
  list<Expr> actuals; ExprManager::StringListToExprList(params,actuals);
  //if there are no parameters - generate some dummy parameters. for
  //example this could happen if the function is a library routine
  //that is not defined but for whom an abstraction has been provided
  if(actuals.empty()) {
    set<string> guardIds;
    for(list<ProcAbs>::const_iterator i = callerAbs.begin();i != callerAbs.end();++i) {
      set<string> x = Util::ComputeIdLvalues(i->GetGuard().GetExpr());
      guardIds.insert(x.begin(),x.end());
    }
    int max = 0;
    for(set<string>::const_iterator i = guardIds.begin();i != guardIds.end();++i) {
      if(i->c_str()[0] == '$') {	
	int index = atoi(i->c_str() + 1);
	max = (index > max) ? index : max;
      }
    }
    for(int i = 0;i < max;++i) {
      actuals.push_back(ExprManager::GetIdExpr(Util::NewTempVar()));
    }
  }

  set<Expr> guards;
  set<Expr> dummys;
  for(list<ProcAbs>::const_iterator i = callerAbs.begin();i != callerAbs.end();++i) {
    dummys.insert(i->GetGuard());
    guards.insert(ExprManager::ReplaceDummyVarsOne(i->GetGuard(),actuals));
  }

  //special case: no abstractions supplied
  if(guards.empty()) {
    Expr a = ExprManager::GetIntConstExpr(1);
    callerAbs.push_back(ProcAbs(name + "_abs",a,Database::DEFAULT_SPEC));
    return;
  }

  //check that guards are mutually exclusive
  for(set<Expr>::const_iterator i = guards.begin();i != guards.end();) {
    set<Expr> list1;
    list1.insert(*i);
    ++i;
    for(set<Expr>::const_iterator j = i;j != guards.end();++j) {
      set<Expr> list2;
      list2.insert(ExprManager::NegateExpr(*j));
      if(!ExprManager::ProveImplies(list1,list2)) {
	Util::Error("abstraction guards for procedure %s are not mutually exclusive ... exiting !\n",name.c_str());
      }
    }
  }
  
  //check that the guards are complete
  set<Expr> list1;
  if(!ExprManager::ProveImplies(list1,guards)) {
    Expr a = ExprManager::DisjunctExprSet(dummys);
    Expr b = ExprManager::NegateExpr(a);
    callerAbs.push_back(ProcAbs(name + "_abs",b,Database::DEFAULT_SPEC));
  }
}

/*********************************************************************/
//end of ProcInfo.cpp
/*********************************************************************/
