/**************************** CPPHeaderFile ***************************

* FileName [Predicate.h]

* PackageName [main]

* Synopsis [Header file for Predicate class.]

* Description [This class encapsulates a predicate. A predicate is a
* collection of mutually disjoint pure C expressions.]

* SeeAlso [Predicate.cpp]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __PREDICATE_H__
#define __PREDICATE_H__

namespace magic {

//other classes needed
class Expr;
class ContLoc;

class Predicate
{
 private:
  //the mapping of values to expressions is maintained as a vector
  //of expressions where the value is the index
  vector<Expr> valToExpr;
  
  //the type of the predicate - required or tentative
  int type;
  
  //this flag is relevant only for general predicates. it indicates
  //if the set of expressions represented by this predicate is
  //complete i.e. their disjunction is TRUE.
  bool complete;

  //the number of valuations
  short valNum;

  //the history of the expressions in this predicate. it is a vector
  //with one element for each expression. the element for expression E
  //is a map <X,Y> where X is the control location appearing in the
  //history of E and Y is the number of times X has appeared.
  vector< map<const ContLoc*,int> > history;
  
 public:
  //the predicate types
  static const int NONE;
  static const int REQUIRED;
  static const int TENTATIVE;
  
  Predicate() { type = NONE; complete = true; valNum = 1; }
  Predicate(const Expr &e,int t,const pair<const ContLoc*,int> &h);
  Predicate(const vector<Expr> &v,int t,const vector< map<const ContLoc*,int> > &h);
  Predicate(const Predicate &rhs);
  const Predicate &operator = (const Predicate &rhs);
  bool operator == (const Predicate &rhs) const;

  string ToString() const;

  bool IsTrue() const;
  bool IsFalse() const;
  bool IsEmpty() const { return valToExpr.empty(); }
  bool IsComplete() const { return complete; }
  const vector<Expr> &GetValToExpr() const { return valToExpr; }
  bool Merge(Predicate &rhs);
  void ComputeCompletenessAndValNum();
  size_t GetSize() const { return valToExpr.size(); }
  bool ContainsLvalue(const set<string> &arg) const;
  int GetType() const { return type; }
  void SetType(int t) { type = t; }
  bool ComputeWP(const list<Expr> &lhsList,const list<Expr> &rhsList,Predicate &res) const;
  void Conjunct(const Expr &expr,Predicate &res) const;
  short GetInitValuation() const;
  short GetValuationNum() const { return valNum; }
  bool IsFinalValuation(short val) const;
  short GetNextValuation(short val) const;
  void ToExpr(short val,set<Expr> &res) const;
  pair< set<Expr>,set<Expr> > ToExprSets(short val) const;
  void GetPossibleValuations(const pair< set<Expr>,set<Expr> > &before,
			     const list<Expr> &lhsList,const list<Expr> &rhsList,vector<short> &res) const;
  void GetConsistentValuations(const Expr &arg,vector<short> &res) const;
  void Sanitize(const ContLoc *loc,Predicate &res) const;
};

} //namespace magic

#endif //__PREDICATE_H__

/*********************************************************************/
//end of Predicate.h
/*********************************************************************/
