/**************************** CPPHeaderFile ***************************

* FileName [PredSet.h]

* PackageName [main]

* Synopsis [Header file for PredSet class.]

* Description [This class encapsulates a set of predicates.]

* SeeAlso [PredSet.cpp]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __PREDSET_H__
#define __PREDSET_H__

namespace magic {

//other classes needed
class Predicate;
class BigInt;

/*********************************************************************/
//this class encapsulates a predicate valuation. it is essentially an
//unbounded integer with some additional operations.
/*********************************************************************/
class PVal : public BigInt
{
 private:
  static const string MAX_PVAL;

 public:
  PVal() : BigInt() {}
  PVal(const BigInt &b) : BigInt(b) {}
  PVal(const signed long int i) : BigInt(i) {}
  PVal(const string &str,const int base) : BigInt(str,base) {}
  PVal(const PVal &rhs) : BigInt(rhs) {}
  ~PVal() {}
  
  const PVal &operator = (const PVal &rhs);
  bool operator == (const PVal &rhs) const;
  bool operator < (const PVal &rhs) const;

  static PVal MaxPVal();
};

//a type for easy manipulation of predicate valuations
typedef pair< PVal,vector<short> > PredVal;

/*********************************************************************/
//this class encapsulates a set of predicates
/*********************************************************************/
class PredSet
{
 private:
  //the vector of predicates
  vector<Predicate> preds;
  
  //the number of valuations
  PVal valNum;

  //the factors for the valuations of various predicates
  vector<PVal> factors;

  //the cache used to store relationships between predicates
  static map< pair< pair<const PredSet*,const PredSet*>,pair< const list<Expr>,const list<Expr> > >,map< pair<size_t,short>,pair<size_t,short> > > cache;
  
  void ComputeValNumAndFactors();
  pair< set<Expr>,set<Expr> > ToExprSets(const PredVal &val) const;
  void CollectPossibleValuations(vector<PVal> &ret,
				 const vector< vector <short> > &vals,
				 const PVal &sum,size_t pos) const;
  void ClearThisCache();
  void CreateCache(const PredSet &before,const list<Expr> &lhsList,const list<Expr> &rhsList,
		   map< pair<size_t,short>,pair<size_t,short> > &entry) const;

 public:
  PredSet();
  PredSet(const PredSet &rhs) { *this = rhs; }
  const PredSet &operator = (const PredSet &rhs);

  inline static void ClearCache() { cache.clear(); }

  string ToString() const;
  bool AddPred(const Predicate &p);
  const vector<Predicate> &GetPreds() const { return preds; }
  const PVal &GetValNum() const { return valNum; }
  PredVal GetInitPredValuation() const;
  bool IsValidPredValuation(const PredVal &val) const;
  PredVal GetNextPredValuation(const PredVal &val) const;
  void ToExpr(const PredVal &val,set<Expr> &res) const;
  void GetPossibleValuations(const PredSet &before,const PVal &bsval,
			     const list<Expr> &lhsList,
			     const list<Expr> &rhsList,vector<PVal> &res) const;
  void GetConsistentValuations(const Expr &arg,vector<PVal> &res) const;
  PredVal PValToPredVal(const PVal &arg) const;
  void Merge(const PredSet &rhs);
  void Clear();
  bool IsEmpty() const;
};

} //namespace magic

#endif //__PREDSET_H__

/*********************************************************************/
//end of PredSet.h
/*********************************************************************/
