/******************************** CPPFile *****************************

* FileName [PredSet.cpp]

* PackageName [main]

* Synopsis [Method definitions of PredSet class.]

* SeeAlso [PredSet.h]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University. All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#include <cassert>
#include <gmp.h>
#include <string>
#include <list>
#include <set>
#include <vector>
#include <map>
using namespace std;

#include "BigInt.h"
#include "Node.h"
#include "Database.h"
#include "Predicate.h"
#include "PredSet.h"
using namespace magic;

/*********************************************************************/
//define static fields of PVal
/*********************************************************************/
const string PVal::MAX_PVAL = "999999999999999";

/*********************************************************************/
//operators
/*********************************************************************/
const PVal &PVal::operator = (const PVal &rhs)
{
  mpz_set(data,rhs.data);
  return *this;
}

bool PVal::operator == (const PVal &rhs) const
{
  return (mpz_cmp(data,rhs.data) == 0);
}

bool PVal::operator < (const PVal &rhs) const
{
  return (mpz_cmp(data,rhs.data) < 0);
}

/*********************************************************************/
//methods of class PVal
/*********************************************************************/
PVal PVal::MaxPVal() { return PVal(MAX_PVAL,10); }

/*********************************************************************/
//static members
/*********************************************************************/
map< pair< pair<const PredSet*,const PredSet*>,pair< const list<Expr>,const list<Expr> > >,map< pair<size_t,short>,pair<size_t,short> > > PredSet::cache;

/*********************************************************************/
//default constructor creates and empty predicate set
/*********************************************************************/
PredSet::PredSet()
{
  ComputeValNumAndFactors();
}

/*********************************************************************/
//operators
/*********************************************************************/
const PredSet &PredSet::operator = (const PredSet &rhs)
{
  preds = rhs.preds;
  valNum = rhs.valNum;
  factors = rhs.factors;
  ClearThisCache();
  return *this;
}

/*********************************************************************/
//string representation
/*********************************************************************/
string PredSet::ToString() const
{
  string res = "[";
  for(vector<Predicate>::const_iterator i = preds.begin();i != preds.end();) {
    res += i->ToString();
    ++i;
    if(i != preds.end()) res += ",";
  }
  return res + "]";
}

/*********************************************************************/
//add a new predicate with a number of redundancy checks. return true
//if the predicate was actually added and false otherwise
/*********************************************************************/
bool PredSet::AddPred(const Predicate &p)
{
  //check if the predicate can be proved to be true or false. in that
  //case we do not add the predicate.
  if(p.IsTrue() || p.IsFalse()) return false;
  
  //make a copy of the predicate
  Predicate q = p;
  
  //try to merge this predicate into already existing predicates
  bool ret = false;
  for(vector<Predicate>::iterator i = preds.begin();i != preds.end();++i) {
    if(i->Merge(q)) ret = true;
    if(q.IsEmpty()) break;
  }
  
  //some expressions remain. compute its completeness add this predicate.
  if(!q.IsEmpty()) {
    q.ComputeCompletenessAndValNum();
    preds.push_back(q);
    ret = true;
  }
  
  ComputeValNumAndFactors();
  if(ret) ClearThisCache();
  return ret;
}

/*********************************************************************/
//compute the array of base indices
/*********************************************************************/
void PredSet::ComputeValNumAndFactors()
{
  valNum = PVal(1);
  factors.clear();
  for(vector<Predicate>::iterator i = preds.begin();i != preds.end();++i) {
    factors.push_back(valNum);
    PVal::Mul(valNum,valNum,PVal(i->GetValuationNum()));
  }
}

/*********************************************************************/
//get the initial predicate valuation. a predicate valuation is a
//BigInteger where the bits represent the truth values of the
//predicates. each predicate is represented by a group of bits, one
//for each expression of the predicate. at any time atmost one of the
//bits can be set meaning that the corresponding expression is
//true. if the predicate is incomplete all its bits could also be
//reset.
/*********************************************************************/
PredVal PredSet::GetInitPredValuation() const
{
  PredVal res;
  res.first = PVal(0);
  for(size_t i = 0;i < preds.size();++i) {
    short x = preds[i].GetInitValuation();
    PVal y; PVal::Mul(y,PVal(x),factors[i]);
    PVal::Add(res.first,res.first,y);
    res.second.push_back(x);
  }
  return res;
}

/*********************************************************************/
//given a predicate valuation, returns true if it is a valid predicate
//valuation and false otherwise. a valuation is valid iff it is less
//than the total number of valuations.
/*********************************************************************/
bool PredSet::IsValidPredValuation(const PredVal &val) const
{
  return (val.first < valNum);
}

/*********************************************************************/
//given a valuation to the predicates, return the next valuation
/*********************************************************************/
PredVal PredSet::GetNextPredValuation(const PredVal &val) const
{
  PredVal res;
  PVal ufirst(0);

  //the flag indicating if some predicate was not flipped.
  bool flag = false;
  
  //do the flipping
  for(size_t i = 0;i < preds.size();++i) {
    short next = 0;
    if(flag) {
      next = val.second[i];
    } else {
      next = preds[i].GetNextValuation(val.second[i]);
      if(next != 0) {
	flag = true;
      }
    }
    PVal x; PVal::Mul(x,PVal(next),factors[i]);
    PVal::Add(ufirst,ufirst,x);
    res.second.push_back(next);
  }
  
  //if all predicates flipped then the valuation is illegal
  res.first = flag ? ufirst : valNum;
  return res;
}

/*********************************************************************/
//return the expression corresponding to a valuation. this is returned
//as a list of expressions, one for each predicate. the expression for
//a predicate is negated iff its value is false according to the
//valuation.
/*********************************************************************/
void PredSet::ToExpr(const PredVal &val,set<Expr> &res) const
{
  for(size_t i = 0;i < preds.size();++i) {
    set<Expr> b; preds[i].ToExpr(val.second[i],b);
    res.insert(b.begin(),b.end());
  }
}

/*********************************************************************/
//given another predicate set and a valuation to it before a parallel
//assignment, and the lhs-list and rhs-list of the assignment, compute
//the set of possible valuations to this predicate set after the
//assignment. if the lhs-list and rhs-list are both empty then it
//means no assignment occurs.
/*********************************************************************/
void PredSet::GetPossibleValuations(const PredSet &before,const PVal &bsval,
				    const list<Expr> &lhsList,const list<Expr> &rhsList,
				    vector<PVal> &res) const
{
  //handle special case when this set is empty
  if(preds.empty()) {
    res.push_back(PVal());
    return;
  }
  //convert the short valuation to a proper valuation
  PredVal bval = before.PValToPredVal(bsval);
  pair<const PredSet*,const PredSet*> key1(&before,this);
  pair<const list<Expr>,const list<Expr> > key2(lhsList,rhsList);
  pair< pair<const PredSet*,const PredSet*>,pair< const list<Expr>,const list<Expr> > > key(key1,key2);
  if(cache.count(key) == 0) {
    cache[key] = map< pair<size_t,short>,pair<size_t,short> >();
    CreateCache(before,lhsList,rhsList,cache[key]);
  }
  const map< pair<size_t,short>,pair<size_t,short> > &entry = cache[key];  
  vector< vector<short> > vals(preds.size());
  for(size_t i = 0;i < before.preds.size();++i) {
    map< pair<size_t,short>,pair<size_t,short> >::const_iterator it = entry.find(pair<size_t,short>(i,bval.second[i]));
    if(it != entry.end()) {
      vector<short> vbs;
      vbs.push_back(it->second.second);
      vals[it->second.first] = vbs;
    }
  }  
  //create the predicate for the context represented by before
  //and bval	
  pair< set<Expr>,set<Expr> > blist = before.ToExprSets(bval);
  //for each predicate compute the set of possible valuations
  for(size_t i = 0;i < preds.size();++i) {
    if(vals[i].empty()) {
      preds[i].GetPossibleValuations(blist,lhsList,rhsList,vals[i]);
    }
  }  
  //collect all possible valuations in ret
  CollectPossibleValuations(res,vals,PVal(),0);
}

/*********************************************************************/
//given a predicate valuation, return a list of expressions which have
//been assigned true and a list of expressions which have been
//assigned false.
/*********************************************************************/
pair< set<Expr>,set<Expr> > PredSet::ToExprSets(const PredVal &val) const
{
  pair< set<Expr>,set<Expr> > ret;
  for(size_t i = 0;i < preds.size();++i) {
    pair< set<Expr>,set<Expr> > y = preds[i].ToExprSets(val.second[i]);
    ret.first.insert(y.first.begin(),y.first.end());
    ret.second.insert(y.second.begin(),y.second.end());
  }
  return ret;
}

/*********************************************************************/
//get the set of valuations consisten with the argument expression
/*********************************************************************/
void PredSet::GetConsistentValuations(const Expr &arg,vector<PVal> &res) const
{
  if(ExprManager::IsTrue(arg)) {
    for(PredVal i = GetInitPredValuation();IsValidPredValuation(i);i = GetNextPredValuation(i)) {
      res.push_back(i.first);
    }
    return;
  }
  //handle special case when this set is empty
  if(preds.empty()) {
    res.push_back(PVal());
    return;
  }
  //for each predicate compute the set of possible valuations
  vector< vector<short> > vals(preds.size());
  for(size_t i = 0;i < preds.size();++i) {
    preds[i].GetConsistentValuations(arg,vals[i]);
  }  
  //collect all possible valuations in ret
  CollectPossibleValuations(res,vals,PVal(),0);
}

/*********************************************************************/
//construct the list of complete valuations given the list of
//valuations for individual predicates. the first argument is the list
//where all the valuations are collected. the second argument is the
//vector of valuations for each predicate. the third argument is the
//running sum so far. the fourth argument is the index of vals to be
//considered next.
/*********************************************************************/
void PredSet::CollectPossibleValuations(vector<PVal> &ret,
					const vector< vector <short> > &vals,
					const PVal &sum,size_t pos) const
{
  const vector <short> &a = vals[pos];
  for(size_t i = 0;i < a.size();++i) {
    PVal x; PVal::Mul(x,factors[pos],PVal(a[i]));
    PVal b; PVal::Add(b,sum,x);
    if(pos == vals.size() - 1) ret.push_back(b);
    else CollectPossibleValuations(ret,vals,b,pos+1);
  }
}

/*********************************************************************/
//clear the caches. this is necessary whenever this predicate set is
//modified since the cache values are then no longer valid.
/*********************************************************************/
void PredSet::ClearThisCache()
{
  for(map< pair< pair<const PredSet*,const PredSet*>,pair< const list<Expr>,const list<Expr> > >,map< pair<size_t,short>,pair<size_t,short> > >::iterator i = cache.begin();i != cache.end();++i) {
    if((i->first.first.first == this) || (i->first.first.second == this) ||
       (i->first.first.first == this) || (i->first.first.second == this)) {
      cache.erase(i);
    }
  }
}

/*********************************************************************/
//create the caches
/*********************************************************************/
void PredSet::CreateCache(const PredSet &before,const list<Expr> &lhsList,const list<Expr> &rhsList,
			  map< pair<size_t,short>,pair<size_t,short> > &entry) const
{
  entry.clear();
  bool lhsEmpty = lhsList.empty(); 
  const vector<Predicate> &bpreds = before.GetPreds();
  for(size_t i = 0;i < preds.size();++i) {
    const Predicate &pred = preds[i];
    short shift = pred.IsComplete() ? 0 : 1;
    const vector<Expr> &vte = pred.GetValToExpr();
    for(size_t j = 0;j < vte.size();++j) {
      Expr wp = lhsEmpty ? vte[j] : ExprManager::ComputeWP(lhsList,rhsList,vte[j]);
      for(size_t k = 0;k < bpreds.size();++k) {
	const Predicate &bpred = bpreds[k];
	short bshift = bpred.IsComplete() ? 0 : 1;
	const vector<Expr> &bvte = bpred.GetValToExpr();
	for(size_t l = 0;l < bvte.size();++l) {
	  if(bvte[l] == wp) entry[pair<size_t,short>(k,l + bshift)] = pair<size_t,short>(i,j + shift);
	}
      }
    }
  }
}

/*********************************************************************/
//convert a valuation given in bitset to a proper valuation
/*********************************************************************/
PredVal PredSet::PValToPredVal(const PVal &arg) const
{
  PredVal res;
  res.first = arg;
  PVal x = arg;
  for(size_t i = 0;i < preds.size();++i) {
    short y = preds[i].GetValuationNum();
    PVal quo,rem;
    PVal::Div(quo,rem,x,PVal(y));
    res.second.push_back(static_cast<short>(rem.ToSL()));
    x = quo;
  }
  return res;
}

/*********************************************************************/
//merge with another set of predicates
/*********************************************************************/
void PredSet::Merge(const PredSet &rhs)
{
  bool flag = false;
  for(vector<Predicate>::const_iterator i = rhs.preds.begin();i != rhs.preds.end();++i) {
    if(AddPred(*i)) flag = true;
  }
  ComputeValNumAndFactors();
  if(flag) ClearThisCache();
}

/*********************************************************************/
//clear the set of predicates
/*********************************************************************/
void PredSet::Clear()
{
  preds.clear();
  ComputeValNumAndFactors();
  ClearThisCache();
}

/*********************************************************************/
//return whether set of predicates is empty
/*********************************************************************/
bool PredSet::IsEmpty() const
{
  return preds.empty();
}

/*********************************************************************/
//end of PredSet.cpp
/*********************************************************************/
