/**************************** CPPHeaderFile ***************************

* FileName [PredAbsRefiner.h]

* PackageName [main]

* Synopsis [Header file for PredAbsRefiner class.]

* Description [This class encapsulates a predicate abstraction refiner
* for a component of the program.]

* SeeAlso [PredAbsRefiner.cpp]

* Author [Sagar Chaki]

* Copyright [ Copyright (c) 2002 by Carnegie Mellon University.  All
* Rights Reserved. This software is for educational purposes only.
* Permission is given to academic institutions to use, copy, and
* modify this software and its documentation provided that this
* introductory message is not removed, that this software and its
* documentation is used for the institutions' internal research and
* educational purposes, and that no monies are exchanged. No guarantee
* is expressed or implied by the distribution of this code. Send
* bug-reports and/or questions to: chaki+@cs.cmu.edu. ]

**********************************************************************/

#ifndef __PREDABSREFINER_H__
#define __PREDABSREFINER_H__

namespace magic {

//other classes needed
class ProcManager;

//Type for a state in pre-optimization
typedef pair < const ImplState *, string > POState;
//Type for a path in pre-optimization;
typedef pair <POState, vector < pair <Action, POState> > > POPath;

/*********************************************************************/
//iterator to generate successive subsets of branch locations of a
//fixed size
/*********************************************************************/
class PredIter
{
 private:
  //the set of locations from which we want to construct subsets
  vector<ContLoc*> all;

  //the current iterators
  vector<size_t> indices;

 public:
  PredIter() { assert(false); }
  PredIter(const set<ContLoc*> &a,size_t s);
  PredIter(const PredIter &rhs) { *this = rhs; }
  const PredIter &operator = (const PredIter &rhs);
  bool End() const;
  set<ContLoc*> Next();
};

/*********************************************************************/
//the predicate abstraction refiner
/*********************************************************************/
class PredAbsRefiner
{
 private:
  //the procedure manager we are concerned with
  ProcManager &procManager;

  //flag to indicate if new predicates were discovered
  bool newPreds;

  void UpdateSeedBranchesUseful();
  void UpdateSeedBranchesEliminate(bool message);
  void UpdateSeedBranchesGreedy();
  bool IsSuperSetOfAny(const set<ContLoc*> &locSub,const set<set <ContLoc*> > &constraints);
  void UpdateConstraintFile();
  void UpdateCNF(set< set<ContLoc*> > &cnf,const set<ContLoc*> &clause) const;
  void EliminateRedundant(set< set<ContLoc*> > &arg) const;
  pair< set< set <ContLoc*> >,set<ContLoc*> > SplitSet(const list< set<ContLoc*> > &clause);
  void SolveOptimization();
  void UpdateConstraints(const set< set<ContLoc*> > &clauses) const;
  void WriteConstraints() const;
  void UpdatePreds(const ConcCEDag &concDag,const set<ContLoc*> &seeds);
  void CNFInsert(const set< set<ContLoc*> > &clause,
			map <ContLoc*, int> &varMap,
			map <int, ContLoc*> &mapVar,
			map <set <ContLoc*>, int > &disjunctMap,
			map <int, set <ContLoc*> > &mapDisjunct,
			int &varNo, set <set <int> > &clauses);
  bool CanEliminate(const set<ContLoc*> &locSet,const ConcCEDag &concDag);
  bool CanEliminate(const ConcCEDag &concDag);
  ConcCEDag Concretize (POPath path);
  bool AddsToCoverage(POPath path, map <POState, int> & coverMap);
  vector <ConcCEDag> CFACounterexamplesBFS ();
  vector <ConcCEDag> CFACounterexamplesDFS ();
  vector <ConcCEDag> CFACounterexamples ();
  void PreOptUpdateConstraintFile(vector <ConcCEDag> counterexamples);

 public:
  static const int REFINED_NEW_PREDS;
  static const int REFINED_NO_NEW_PREDS;

  PredAbsRefiner(ProcManager &p) : procManager(p) { newPreds = false; }
  PredAbsRefiner(const PredAbsRefiner &rhs) : procManager(rhs.procManager) { newPreds = false; }

  void PreOptimize ();

  int Run();
};

} //namespace magic

#endif //__PREDABSREFINER_H__

/*********************************************************************/
//end of PredAbsRefiner.h
/*********************************************************************/
